
;
;+
;
;============================  SHADOW VUI   ==================================
;
; SHADOW VUI  is a Visual User Interface for the ray tracing code SHADOW
;
; HELP: see on-line help
;
; COPYRIGHT:
;	SHADOWVUI  is distributed as an XOP extension.
;	PLEASE REFER TO THE XOP COPYRIGHT NOTICE BEFORE USING IT.
;       http://www.esrf.eu/UsersAndScience/Experiments/TBS/SciSoft/xop2.3/
;
; CREDITS:
;	Published calculations made with SHADOW and SHADOWVUI should refer:
;
;	F. Cerrina and M. Sanchez del Rio
;	"Ray tracing of X-ray optical systems"
;       Handbook of Optics (Volume V):
;       Atmospheric Optics, Modulators, Fiber Optics, X-Ray and Neutron Optics
;	Chapter 35 (pags 35.1-35.6) (2009)
;       ISBN: 0071633138 / 9780071633130
;       http://www.mhprofessional.com/handbookofoptics/vol5.php
;	
;	M. Sanchez del Rio and R. J. Dejus  
;       "Status of XOP: an x-ray optics software toolkit" 
;       SPIE Proceedings Vol. 5536 (2004) pp.171-174
;	http://dx.doi.org/10.1117/12.560903
;
; LAST MODIFICATION: msr/msr/14-04-2010
;
;-
;
; -----------------------------------------------------------------------
;	Author: M. Sanchez del Rio (srio@esrf.fr) , ESRF
;	00/07/10 srio@esrf.fr version 1.0Beta3 for xop 2.0. Documented.
;	00/11/16 srio@esrf.fr version 1.0Beta3.1 (few bugs fixed)
;	01/03/07 srio@esrf.fr version 1.0Beta3.2 (a bug (wiggler) fixed,
;		added rowland+rcil)
;	01/06/13 srio@esrf.fr Allows to close windows without quitting
;		the application. Memorize the workspace path for load/save.
;	01/07/02 srio@esrf.fr Allows to import a systemfile.dat file
;		with the system definition start.xx v. 1.0Beta3.3
;
;	01/10/22 srio@esrf.fr Version 1.0Beta4.0. Added BLViewer and other details
;	03/08/29 srio@esrf.fr Version 1.0Beta4.1. Added manuals in distribution.
;	04/03/19 srio@esrf.fr Version 1.0Beta4.2. Fixed a bug when writing 
;			files for oe>9.
;	04/09/23 srio@esrf.fr Version 1.0Beta4.2.1 Added rowland and rcil
;	06/02/20 srio@esrf.fr Version 1.0Beta5
;	06/04/13 srio@esrf.fr Version 1.01 added xsh_waviness and more.
;	08/09/10 srio@esrf.eu Version 1.06 added zoom in plotxy,histo1.
;		Added routines for creating Restrax files. 
;	09/03/06 srio@esrf.eu Version 1.07 added spectrum calculation
;	2010/04/14 srio@esrf.eu Version 1.08: Fixed some bugs importing 
;           start.xx files, changes size of xsh_{plotxy,histo1,ray_prop},
;           added FWHM calculation to ray_prop, upgrade HISTO1 file, 
;           updated Shadow's ref. 
;	2010/10/15 srio@esrf.eu Version 1.09: Adapted to Shadow3 PreBeta
;
;========================================================================
;
;		
;========================================================================
;
PRO  Shadow_Help,Group=group

text = [' ',' ', $
' ',$
'                 SHADOW VUI                  ',$
'(A Visual User Interface to SHADOW under IDL)',$
' ',$
'                         by                  ',$
' ',$
'             Manuel Sanchez del Rio          ',$
' ',' ', $
'   European Synchrotron Radiation Facility   ', $
'      BP 220, F-38043 Grenoble Cedex 9       ', $
' ', $
'            email: srio@esrf.eu              ', $
'             Tel: +33-476882513              ', $
'             Fax: +33-476882542              ', $
' ',' ', $
'SHADOWVUI is distributed as an extension of the XOP package:   ',$
' ',$
'http://www.esrf.eu/UsersAndScience/Experiments/TBS/SciSoft/xop2.3/',$
'','', $
'SHADOW is an x-ray tracing program available at:', $
'','', $
'http://www.nanotech.wisc.edu/shadow', $ 
' ']

tmp = Widget_Message(Dialog_Parent=group,$
  text,/Info,Title='SHADOW VUI')

END ; Shadow_Help
;		
;========================================================================
;
Function shadow_version,version=version,beta=beta
version=1.09
;beta=5
;out = string(version,' Beta',beta,Format='(F3.1,A,I1)')
out = string(version,Format='(F4.2)')
return,out
end

;		
;========================================================================
;

PRO shadow_quit,stateId,noAsk=noAsk

COMMON shadow_ifc,parent1

IF N_Elements(stateId) EQ 0 THEN stateId=parent1
IF Widget_Info(stateId,/Valid_Id) NE 1 THEN BEGIN
   itmp = Dialog_Message('SHADOW_QUIT: No shadowVUI window (id:'+$
      StrCompress(stateId,/Rem)+')')
   RETURN
ENDIF

Widget_Control,stateid,Get_UValue=state

IF keyword_Set(noAsk) THEN BEGIN
   itmp = 'Yes'
ENDIF ELSE BEGIN
  itmp = Dialog_Message(/Question,Dialog_Parent=stateId,/Cancel,$
	  ['All the SHADOW files will remain in the current directory.',$
	  ' ',$
	  'Do you want to clean intermediate files used by SHADOW VUI?'])
ENDELSE

if itmp EQ 'Cancel' then RETURN

for i=0,3 do  $ 
	  if Ptr_Valid(state.PtrSrc(i)) then Ptr_Free,state.PtrSrc(i)
for i=0,14 do $ ; n_elements(state.PtrTrc)-1 do $
if Ptr_Valid(state.PtrTrc(i)) then Ptr_Free,state.PtrTrc(i)
	ptrMacros = *state.ptrPtrMacros	
FOR i=0,N_Elements(ptrMacros)-1 DO $ 
IF Ptr_Valid(ptrMacros[i]) THEN Ptr_Free,ptrMacros[i]
IF Ptr_Valid(ptrPtrMacros) THEN Ptr_Free,ptrPtrMacros
IF Ptr_Valid(state.ptrWSDoc) THEN Ptr_Free,state.ptrWSDoc
if itmp EQ 'Yes' then begin
	  case sdep() of
	    'UNIX': BEGIN
		command = 'rm xsh_*_tmp.* '
	  	message,/Info,'Executing: '+command
	  	spawn,command
		END
	    'WINDOWS': BEGIN
		command = ['del xsh_*_tmp.* ','del xsh_env_file.inp']
		xsh_run,command
		command = 'del shadowvui.bat'
	  	message,/Info,'Executing: '+command
	  	spawn,command
		END
	    else: 
	  endcase
endif
	
Widget_Control,stateId,/Destroy
END ; shadow_quit

;		
;========================================================================
;

PRO shadow_update_macrotext,state,index,Empty=empty

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!err_string
   itmp = Dialog_Message('SHADOW_UPDATE_MACROTEXT: Error caught: '+$
	!err_string,/Error)
   RETURN
ENDIF

IF Keyword_Set(empty) THEN BEGIN
  Widget_Control,state.wids.macrosText,Set_Value='Macros: '
  RETURN
ENDIF

; line = (*state.ptrMacros(event.Value))[0]
line = ( *( (*state.ptrPtrMacros)[index] ) )[0]
Widget_Control,state.wids.macrosText,Set_Value='Macros: ( #'+$
  StrCompress(index+1,/Rem)+' '+StrMid(line,0,60)+' )'
END ; shadow_update_macrotext

;		
;========================================================================
;
PRO Shadow_RunSource, input, index, Only_Run=only_run,  Group=group

COMMON shadow_ifc,parent1


Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!err_string
   itmp = Dialog_Message('SHADOW_RUNSOURCE: Error caught: '+$
	!err_string,/Error,Dialog_Parent=group)
   RETURN
ENDIF


CASE type(input) OF
  0: wid=parent1 ; undefined
  3: wid=input   ; widget_parent
  8: BEGIN ; state (structure)
     wid=0L
     state=input
     END
  else: Message,'Bad input'
ENDCASE
IF Widget_Info(wid,/Valid_Id) THEN Widget_Control,wid,Get_UValue=state

IF N_Elements(index) EQ 0 THEN BEGIN
  index=-1
  Widget_Control,state.wids.src,Get_Value=index
ENDIF

IF N_Elements(only_Run) EQ 0 THEN only_Run=0

CASE index OF
	  0: BEGIN
		if not(Ptr_Valid(state.PtrSrc(0))) then begin
		  PtrTmp = Ptr_New(xsh_defaults('XSH_SOURCE_G'))
		  state.PtrSrc(0) = PtrTmp
                  IF Widget_Info(wid,/Valid_Id) THEN $
                     Widget_Control,wid,Set_UValue=state
		endif
		XSh_Source,state.PtrSrc(0),Group=group, $
			Only_Run=only_Run
		END
	  1: BEGIN
		if not(Ptr_Valid(state.PtrSrc(1))) then begin
		  PtrTmp = Ptr_New(xsh_defaults('XSH_SOURCE_BM'))
		  state.PtrSrc(1) = PtrTmp
                  IF Widget_Info(wid,/Valid_Id) THEN $
                     Widget_Control,wid,Set_UValue=state
		endif
		XSh_Source,state.PtrSrc(1),Group=group, $
			Only_Run=only_Run,/BM
		END
	  2: BEGIN
		if not(Ptr_Valid(state.PtrSrc(2))) then begin
		  PtrTmp = Ptr_New(xsh_defaults('XSHWIG'))
		  state.PtrSrc(2) = PtrTmp
                  IF Widget_Info(wid,/Valid_Id) THEN $
                     Widget_Control,wid,Set_UValue=state
		endif
		XShWig,state.PtrSrc(2),Group=group, $
			Only_Run=only_Run
		END
	  3: BEGIN
		if not(Ptr_Valid(state.PtrSrc(3))) then begin
		  PtrTmp = Ptr_New(xsh_defaults('XSHUNDUL'))
		  state.PtrSrc(3) = PtrTmp
                  IF Widget_Info(wid,/Valid_Id) THEN $
                     Widget_Control,wid,Set_UValue=state
		endif
		XShUndul,state.PtrSrc(3),Group=group, $
			Only_Run=only_Run
		END
	  else:
ENDCASE




END ; Shadow_RunSource
;		
;========================================================================
;
PRO Shadow_RunTrace, input, Group=group

COMMON shadow_ifc,parent1
COMMON shadow3, shadow3_mode

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!err_string
   itmp = Dialog_Message('SHADOW_RUNTRACE: Error caught: '+$
	!err_string,/Error,Dialog_Parent=group)
   RETURN
ENDIF

IF XRegistered('XSHOE') THEN BEGIN
  itmp = Dialog_Message(/Error,Dialog_Parent=group,$
    ['SHADOW VUI has detected opened windows for o.e.s',$
    'Please close all of them before starting SHADOW/trace'])
  RETURN
ENDIF


CASE type(input) OF
  0: BEGIN ; undefined
     Widget_Control,parent1,Get_UValue=state
     END
  3: BEGIN ; widget_parent
     Widget_Control,input,Get_UValue=state
     END
  8: BEGIN ; state (structure)
     state=input
     END
  else: Message,'Bad input'
ENDCASE

IF (state.ifc.oe_n LE 0) THEN BEGIN
  itmp = Dialog_Message(Dialog_Parent=group,$
    ['No optical elements defined.','Use "Add oe" before'])
  RETURN
ENDIF

OpenW,lun,'systemfile.dat',/Get_Lun
FOR i=1,state.ifc.oe_n DO BEGIN
  IF i LE 9 THEN file = 'start.0'+strcompress(i,/Rem) ELSE $
                 file = 'start.'+strcompress(i,/Rem)
  ;file = 'start.0'+strcompress(i,/Rem)
  PrintF,lun,file
  Write_GFile,*state.PtrTrc(i-1),file
ENDFOR
Free_lun,lun

Widget_control,/hourglass
IF shadow3_mode EQ 0 THEN BEGIN
  command = 'echo 0 | trace -m menu'
ENDIF ELSE BEGIN
  openw,unit,'shadow3.inp',/get_lun
  printf,unit,'trace'
  printf,unit,'systemfile'
  printf,unit,'0'
  printf,unit,'exit'
  free_lun,unit
  command = 'shadow3 < shadow3.inp'
ENDELSE
XSh_Run,command

END ; Shadow_RunTrace

;		
;========================================================================
;
; accepts also an array of indices. If index is not defined, then it
; runs all macros
;
PRO Shadow_RunMacro, input, index,  Group=group

COMMON shadow_ifc,parent1

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!err_string
   itmp = Dialog_Message('SHADOW_RUNMACRO: Error caught: '+$
	!err_string,/Error,Dialog_Parent=group)
   RETURN
ENDIF


CASE type(input) OF
  0: BEGIN ; undefined
     Widget_Control,parent1,Get_UValue=state
     END
  3: BEGIN ; widget_parent
     Widget_Control,input,Get_UValue=state
     END
  8: BEGIN ; state (structure)
     state=input
     END
  else: Message,'Bad input'
ENDCASE
ptrMacros = *state.ptrPtrMacros
IF N_Elements(index) EQ 0 THEN index = LIndGen(N_Elements(ptrMacros))

Widget_Control,/Hourglass
FOR i=0L,N_Elements(index)-1 DO BEGIN
  Message,/Info,'Running macro # '+StrCompress(index[i]+1)
  prog=*( ptrMacros[index[i]] )
  IF N_Elements(prog) EQ 1 AND prog[0] EQ '' THEN BEGIN  ; running nothing
  ENDIF ELSE BEGIN
    tmp = Xop_Macro_Compact(prog,Group=group,/Execute)
  ENDELSE
ENDFOR

END ; Shadow_RunMacro

;		
;========================================================================
;
;
; source, trace and macros: Flag: 0=No, 1=Yes
PRO Shadow_Run, input, Source=source, Trace=trace, Macros=macros, $
  All=all, Group=group, oneMacro=oneMacro


COMMON shadow_ifc,parent1

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!err_string
   itmp = Dialog_Message('SHADOW_RUN: Error caught: '+$
	!err_string,/Error,Dialog_Parent=group)
   RETURN
ENDIF

CASE type(input) OF
  0: BEGIN ; undefined
     wid = parent1
     END
  3: BEGIN ; widget_parent
     wid=input
     END
  8: BEGIN ; state (structure)
     wid = input.wids.main
     END
  else: Message,'Bad input'
ENDCASE


IF Keyword_Set(all) THEN BEGIN
  source = 1
  trace = 1
  macros = 1
ENDIF

IF Keyword_Set(source) THEN BEGIN
  Widget_Control,wid,Get_UValue=state
  src_index=-1
  Widget_Control,state.wids.src,Get_Value=src_index
  Shadow_RunSource,input,src_index,/Only_Run,Group=group
ENDIF

IF Keyword_Set(trace) THEN Shadow_RunTrace,input,Group=group

IF Keyword_Set(macros) THEN Shadow_RunMacro,input,Group=group

IF N_Elements(oneMacro) GT 0 THEN Shadow_RunMacro,input,oneMacro,Group=group

END ; Shadow_Run

;		
;========================================================================
;

PRO shadow_update,state,add=add,delete=delete,group=group,$
  alldelete=alldelete

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!err_string
   itmp = Dialog_Message('SHADOW_UPDATE: Error caught: '+$
	!err_string,/Error)
   RETURN
ENDIF
f_insert = 0
f_del  = 0
nn =  state.ifc.oe_n
nsel =  state.ifc.oe_sel
nn_old = nn
nsel_old = nsel

if keyword_set(add) then begin
  ;if Widget_Info(state.wids.select,/Valid) then begin
  ;  Widget_Control,state.wids.oe,Get_Value=nsel
  ;  nsel=nsel+1
  ;endif else nsel=0
  if nsel NE nn then begin
    itmp = Dialog_Message(Dialog_Parent=group,$
      ['This option will insert a new o.e. after the o.e. number '+$
	strcompress(nsel),'Please confirm:'],/Cancel)
    if itmp EQ 'Cancel' then return
    f_insert = 1
  endif
  ; add screen and poit to the added one
  nsel = nsel+1
  nn = nn+1
endif

if keyword_set(delete) then begin
  if not( Widget_Info(state.wids.oe,/Valid) ) then return
  Widget_Control,state.wids.oe,Get_Value = delete_n
  delete_n = delete_n + 1
  nsel = delete_n - 1 > 0
  itmp = Dialog_Message(Dialog_Parent=group,$
    ['This option will remove o.e. number '+strcompress(delete_n),$
       'Please confirm:'],/Cancel)
  if itmp EQ 'Cancel' then return
  if delete_n NE nn then f_del = 1
  nn =  nn-1
endif

if keyword_set(alldelete) then begin
  if not( Widget_Info(state.wids.oe,/Valid) ) then return
  itmp = Dialog_Message(Dialog_Parent=group,$
    ["This option will remove all o.e.'s in your system",$
       'Please confirm:'],/Cancel)
  if itmp EQ 'Cancel' then return
  nn = 0
  nsel = 0
endif

if Widget_Info(state.wids.oe,/Valid) then $
  widget_control,state.wids.oe,/Destroy

if nn gt 0 then begin
  oes = 'oe '+strcompress(indgen(nn)+1)
  wtmp = CW_BGroup(state.wids.oebase,oes,/Exclusive,UVALUE='TRCSEL',$
	COLUMN=10, Set_Value=nsel-1>0,/No_Release)
  state.wids.oe = wtmp
endif

;if f_insert EQ 1 then begin
;  for i=nn,nsel+1,-1 do print,'???? moved oe to ',i,' from',i-1
;  ;; for i=nn,nsel+1,-1 do xshscr_copy,state.oestr,i-1,i,/verbose
;  print,'??? inserted o.e. #',nsel
;  ;; xshscr_copy,state.oestr,10,nsel,/verbose
;endif
;if f_del EQ 1 then begin
;  print,'??? deleted o.e. #',delete_n
;  for i=delete_n+1,nn+1 do print,'moved o.e. to ',i-1,' from',i
;  ;; for i=delete_n+1,nn+1 do xshscr_copy,state.oestr,i,i-1,/verbose
;endif

;
; manage the oe pointers
;
if keyword_set(alldelete) then begin
  for i=0,nn_old-1 do $
    if Ptr_Valid(state.PtrTrc(i)) then Ptr_Free,state.PtrTrc(i)
  print,'[][] deleted ALL oes'
  goto,out
endif

if (keyword_set(add) AND (f_insert EQ 0)) then begin ; add oe at end
  ; delete current data if exists
  if Ptr_Valid(state.PtrTrc(nn-1)) then Ptr_Free,state.PtrTrc(nn-1)
  ; assign new data
  ;
  if nsel_old EQ 0 then begin ; empty system
    state.PtrTrc(nn-1) = Ptr_New(xsh_defaults('XSHOE'))
  endif else begin
    itmp = Dialog_Message(Dialog_Parent=group,/Question,$
     ['The new o.e. may have either the default values or the values copied',$
      'from the selected o.e. (#'+strcompress(nsel_old,/rem)+').',$
      'Do you want to copy the values from selected o.e. ?'])
    case itmp of
      'Yes': state.PtrTrc(nn-1) = Ptr_New( *state.PtrTrc(nsel_old-1) )
      'No': state.PtrTrc(nn-1) = Ptr_New(xsh_defaults('XSHOE'))
      else:
    endcase
  endelse
  ;
  ;state.PtrTrc(nn-1) = Ptr_New(xsh_defaults('XSHOE'))
  print,'[][] added last oe: ',nn
  goto,out
endif

if f_insert EQ 1 then begin
  for i=nn,nsel+1,-1 do print,'[][] moved oe to ',i,' from',i-1
  for i=nn,nsel+1,-1 do state.PtrTrc(i-1) = state.PtrTrc(i-1-1)
  print,'[][] inserted o.e. #',nsel
  state.PtrTrc(nsel-1) = Ptr_New(xsh_defaults('XSHOE'))
  goto,out
endif

if f_del EQ 1 then begin
  print,'[][] deleted o.e. #',delete_n
  if Ptr_Valid(state.PtrTrc(delete_n-1)) then Ptr_Free,state.PtrTrc(delete_n-1)
  for i=delete_n+1,nn+1 do print,'[][] moved o.e. to ',i-1,' from',i
  for i=delete_n+1,nn+1 do  state.PtrTrc(i-1-1) = state.PtrTrc(i-1)
  goto,out
endif


out:

state.ifc.oe_n = nn
state.ifc.oe_sel = nsel

Widget_Control,state.wids.src,Set_Value=state.ifc.src_sel 
end



;		
;========================================================================
;


PRO shadow_import_system,state,file=file, group=group,path=path,Get_path=get_path

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!err_string
   itmp = Dialog_Message('SHADOW_IMPORT_SYSTEM: Error caught: '+$
	!err_string,/Error,Dialog_Parent=group)
   RETURN
ENDIF

if not(keyword_set(file)) then  begin
back:
  file = Dialog_Pickfile(group=group,title=$
	'Select a systemfile.dat file containing SHADOW start.xx files', $
	Filter='*.dat',Path=path,Get_Path=get_path)
  if strcompress(file,/Rem) EQ '' then return
endif


txt = read_textfile(file)
oe_n = N_Elements(txt)
Print,'SHADOW_IMPORT_SYSTEM:  Files to read: '
FOR i=0L,oe_n-1 DO Print,'   '+txt[i]
   

if oe_n GT 0 then begin
  message,/info,'Updating system'
  state.ifc.oe_n = oe_n
  state.ifc.oe_sel = oe_n
  for i=0,state.ifc.oe_n-1 do begin
    Print,'Importing file: ',txt[i]
    if Ptr_Valid(state.PtrTrc(i)) then Ptr_Free,state.PtrTrc(i)
    ; srio@esrf.eu 2010-04-14 debugging: failed to import start.xx files
    ; where float are defined as integer. 
    tmp0 = xsh_defaults('xshoe')
    tmp = read_gfile(txt[i])
    copy_structure,tmp,tmp0
    state.PtrTrc(i) = Ptr_New(tmp0)
  endfor
endif

Shadow_Update,state


itmp = Dialog_Message(/Info,Dialog_Parent=group,$
  'Successfully imported '+StrCompress(oe_n)+' elements.')

END



;		
;========================================================================
;
PRO shadow_loadws,input, group=group, norun=norun, $
  file=file,dir=dir
;, path=path,Get_path=get_path

COMMON shadow_ifc,parent1


Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!err_string
   itmp = Dialog_Message('SHADOW_LOADWS: Error caught: '+$
	!err_string,/Error,Dialog_Parent=group)
   RETURN
ENDIF

CASE type(input) OF
  0: BEGIN ; undefined
     Widget_Control,parent1,Get_UValue=state
     END
  3: BEGIN ; widget_parent
     Widget_Control,input,Get_UValue=state
     END
  8: BEGIN ; state (structure)
     state=input
     END
  else: Message,'Bad input'
ENDCASE


if not(keyword_set(file)) then  begin
	CASE dir OF
	'Load Workspace...': BEGIN
	  IF  state.wsPath EQ '' THEN CD,Current=path ELSE path=state.wsPath
	  ;shadow_loadws,state,group=event.top,Path=path,Get_Path=newdir
	  ;state.wsPath=newdir
	END
	'From Example Dir...': BEGIN
	  ds = SDep(/ds)
	  path = Xop_GetEnv('XOP_HOME')+ds+'extensions'+ds+'shadowvui'+ $
		ds+'workspaces' 
	  ; remove double slashes (confuses dialog_pickfile)
	  path = StrSubstitute(path,SDep(/ds)+SDep(/ds),SDep(/ds))
	  ;shadow_loadws,state,group=event.top,PATH=path
	END
	'From Tutorial Dir...': BEGIN
	  ds = SDep(/ds)
	  path = $
	   Xop_GetEnv('XOP_HOME')+ds+'examples'+ds+'tutorial'
	  ; remove double slashes (confuses dialog_pickfile)
	  path = StrSubstitute(path,SDep(/ds)+SDep(/ds),SDep(/ds))
	  ;shadow_loadws,state,group=event.top,PATH=path
	END
	else:
	ENDCASE
  file = Dialog_Pickfile(group=group,title=$
	'Select a file containing SHADOW VUI workspace',Filter='*.ws',$
	Path=path,Get_Path=get_path)
  if strcompress(file,/Rem) EQ '' then return
endif

IF checkFile(file) NE 1 THEN BEGIN
  itmp = Dialog_Message(/Error,'File not found: '+file, $
    Dialog_Parent=state.wids.main)
  RETURN
ENDIF
shadow_vui = 0
restore,file,/Verbose

iver = 0
if shadow_vui.version NE shadow_version() then begin
  itmp = Dialog_Message(Dialog_Parent=group, $
    ['Restored workspace version ('+shadow_vui.version+' ) differes from '+ $
     'SHADOW VUI version ('+shadow_version()+') ','Should not be any problem.'])
  iver = 1
endif

if shadow_vui.src_sel GE 0 then begin
  message,/info,'Updating source'
  state.ifc.src_sel = shadow_vui.src_sel
  if Ptr_Valid(state.PtrSrc(state.ifc.src_sel)) then $
	Ptr_Free,state.PtrSrc(state.ifc.src_sel)
  if type(shadow_vui.source) EQ 8 then $
	state.PtrSrc(state.ifc.src_sel) = Ptr_New(shadow_vui.source)
endif

if shadow_vui.oe_n GT 0 then begin
  message,/info,'Updating system'
  state.ifc.oe_n = shadow_vui.oe_n
  state.ifc.oe_sel = shadow_vui.oe_sel
  for i=0,state.ifc.oe_n-1 do begin
    if Ptr_Valid(state.PtrTrc(i)) then Ptr_Free,state.PtrTrc(i)
    state.PtrTrc(i) = Ptr_New(shadow_vui.system(i))
  endfor
endif

;
; Macros
;
tnames = Tag_Names(shadow_vui)
tmp = Where(StrLowCase(tNames) EQ 'macros')
IF tmp[0] NE -1 THEN BEGIN
  ptrPtrMacros = shadow_vui.macros
  state.ptrPtrMacros=ptrPtrMacros
ENDIF ELSE BEGIN
  ptrMacros = *state.ptrPtrMacros
  FOR i=0L,N_Elements(ptrMacros)-1 DO BEGIN
    IF Ptr_Valid(ptrMacros[i]) THEN *ptrMacros[i]=''
  ENDFOR
  *state.ptrPtrMacros = ptrMacros
ENDELSE

; bug fixed 16/11/2000: nMacros=N_Elements(*ptrPtrMacros)
nMacros=N_Elements(*state.ptrPtrMacros)
Widget_Control,state.wids.Macros,/Destroy
macros_labels = strcompress(indgen(nMacros)+1)
wMacros=CW_BGroup(state.wids.macrosB,macros_labels,UVALUE='MACROSSEL',/Exclusive,/Row, $
Set_Value=iMacro,/No_Release)
state.wids.macros=wMacros

;
; WorkSpace description
;
*(state.PtrWSDoc) = ['** Enter here the workspace description **']
tmp = Where(StrLowCase(tNames) EQ 'wsdoc')
IF tmp[0] NE -1 THEN BEGIN
  ptrWSDoc = shadow_vui.wsdoc
  state.PtrWSDoc = ptrWSDoc
  text = *(state.ptrWSDoc)
  ishow=0
  IF N_Elements(text) EQ 1 AND  $
     (text[0] NE '** Enter here a description for the workspace **') THEN $
     ishow =1
  IF N_Elements(text) GT 1 THEN ishow =1
  ;itmp = Dialog_Message(/Question, Dialog_Parent=group, $
  ;  'Display workspace documentation')
  ;IF itmp EQ 'Yes' THEN XDisplayFile1,Text=Text,Group=group
  IF ishow THEN XDisplayFile1,Text=Text,Group=group, $
	Title='Wokspace description '+file
ENDIF
 
Widget_Control,state.wids.macros,Set_Value=0
Widget_Control,state.wids.macrosText,Set_Value='Macros: '

Shadow_Update,state

; After the update of the structure, we need to register it for being able 
; to run macros containing "shadow_setifcpar" without knowing parent1
; added srio@esrf.fr 2006-5-11
Widget_Control,state.wids.main,Set_UValue=state

; 
; unpack_files
;

itmp = Where(StrUpCase(tag_names(SHADOW_VUI)) EQ 'PACKED_FILES')
IF itmp[0] NE -1 THEN BEGIN
  IF Type(SHADOW_VUI.packed_files) EQ 8 THEN BEGIN
    unpack_files,SHADOW_VUI.packed_files,/free,/inform,/confirm,Group=group
  ENDIF
ENDIF

IF Not(keyword_Set(norun)) THEN BEGIN
  itmp = Dialog_Message(/Question, Dialog_Parent=group, $
    'Run loaded workspace (source+trace+macros)? ')
  IF itmp EQ 'Yes' THEN Shadow_Run,state,/All,Group=group
ENDIF

tmp = StrParse(file,SDep(/ds),tmpList)
filews = tmpList(N_Elements(tmpList)-1)
Widget_Control,state.wids.main,TLB_Set_Title= $
      filews+' - Shadow VUI '+shadow_version()

END
;		
;========================================================================
;

PRO shadow_savews,state,file=file, group=group, Path=path,Get_Path=get_path


Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!err_string
   itmp = Dialog_Message('SHADOW_SAVEWS: Error caught: '+$
	!err_string,/Error,Dialog_Parent=group)
   RETURN
ENDIF

if not(keyword_set(file)) then  begin
back:
  file = Dialog_Pickfile(group=group,title=$
	'Select a file to save SHADOW VUI workspace',file='shadowvui.ws', $
	Path=path,Get_Path=get_path,Filter='*.ws')
  if strcompress(file,/Rem) EQ '' then return
endif

if checkfile(file) then begin
  itmp = Dialog_Message(Dialog_Parent=group,['File exists: '+file,$
	'Overwrite it?'], /Question)
  if itmp EQ 'No' then goto,back
endif

src_sel = state.ifc.src_sel
if Ptr_Valid(state.PtrSrc(src_sel)) then $
  source = *state.PtrSrc(src_sel) else source = 0

oe_sel = state.ifc.oe_sel
oe_n = state.ifc.oe_n

system = 0
if oe_n GT 0 then begin
  if Ptr_Valid(state.PtrTrc(0)) then begin
    system = replicate( *state.PtrTrc(0),oe_n ) 
    if oe_n  GT 1 then begin
      for i=1,oe_n-1 do begin
        if Ptr_Valid(state.PtrTrc(i)) then begin
           ; changed srio@esrf.eu 2009-11-13 to avoid occasinal 
           ; problema in writing workspace.
           ;system(i) = *state.PtrTrc(i)
           tmp = system(i)
           copy_structure,*state.PtrTrc(i),tmp
           system(i) = tmp
        endif
      endfor
    endif
  endif 
endif

macros = state.ptrPtrMacros
wsdoc = state.ptrWSDoc

itmp = Dialog_Message(/Question,[ $
	'Individual files can be attached or packed to the workspace (ws)',$
	'This is useful to pack preprocessor or data files.',$
	'They will be unpacked at the working directory when loading the ws.',$
	' ','Do you want to attach files? '],Dialog_Parent=group)
IF itmp EQ 'Yes' THEN pack_files,packed_files=packed_files,group=group $
	         ELSE packed_files=0

shadow_vui = {$
  version:shadow_version(),$
  src_sel:src_sel, $
  oe_sel:oe_sel, $
  oe_n:oe_n, $
  source:source, $
  system:system, $
  macros:macros, $
  wsdoc:wsdoc, $
  packed_files:packed_files }

;
; added compress srio 2006-5-12
;
save,/Verbose,shadow_vui,file=file,/Compress
tmp = StrParse(file,SDep(/ds),tmpList)

filews = tmpList(N_Elements(tmpList)-1)
Widget_Control,state.wids.main,TLB_Set_Title= $
      filews+' - Shadow VUI '+shadow_version()

END
;		
;========================================================================
;

PRO shadow_event,event

common shadow3,shadow3_mode

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!err_string
   itmp = Dialog_Message('SHADOW_EVENT: Error caught: '+$
	!err_string,/Error)
   GoTo,out
ENDIF


IF Tag_Names(event,/Structure_Name) EQ 'WIDGET_KILL_REQUEST' THEN BEGIN
  shadow_quit,event.top
  RETURN
ENDIF


stateId = event.handler
Widget_Control,event.id,Get_UValue=uval

;
; events that do not need state
;
CASE uval OF
  'LOADWS': BEGIN
	Widget_Control,event.id,Get_Value=tmp
	shadow_loadws,stateId,group=event.top,dir=tmp
	RETURN
	END
  'EXIT': BEGIN
        shadow_quit,event.top
        RETURN
       END
  else:
ENDCASE

;
; events that do need state
;
Widget_Control,stateid,Get_UValue=state  ;,/No_Copy

CASE uval OF
  'WORKINGDIR': BEGIN
	tmp = 0
	Widget_Control,state.wids.workingdir,Get_Value=tmp
	cd,tmp(0)
	message,/info,'Changed current directory to: '+tmp(0)
	itmp = Dialog_Message(/Info,Dialog_Parent=event.top,$
	   'Working directory is now: '+tmp(0))
	END
  ; File Menu
  'IMPORT_SOURCE': BEGIN
	txt = ['SHADOWVUI can import start.00 files containing: ',$
	' ',$
	' i) A geometrical source',$
	' ii) A bending magnet source',$
	' ',$
	'It cannot import wiggler nor undulator sources. ',$
	' ',$
	'To import a source, please use the button "Load GFile..." ',$
	'in either the Geometrical or BM screen. ']
	itmp = Dialog_Message(/Info,txt,Dialog_Parent=event.top)
	END
  'IMPORT_SYSTEM': BEGIN
	IF  state.wsPath EQ '' THEN CD,Current=pwd ELSE pwd=state.wsPath
	shadow_import_system,state,group=event.top,PATH=pwd,Get_Path=newdir
	state.wsPath=newdir
	Widget_Control,stateId,Set_UValue=state
	END
  'SAVEWS': BEGIN
	Widget_Control,event.id,Get_Value=val
	CASE val OF
	'Save Workspace...': BEGIN
	  IF  state.wsPath EQ '' THEN CD,Current=pwd ELSE pwd=state.wsPath
	  shadow_savews,state,group=event.top,Path=pwd,Get_Path=newdir
	  state.wsPath=newdir
	  Widget_Control,stateId,Set_UValue=state
	  END
	'Save Workspace as Default...': BEGIN
	  file=XOP_GetEnv('XOP_DEFAULTS_DIR')+SDep(/DS)+'shadowvui_default.ws'
	  itmp = Dialog_Message(/Info,/Cancel,Dialog_Parent=event.top,$
	 	['This option writes the current workspace in the ',$
		 'in the default directory: ','',file,'', $
		 'to be loaded when starting ShadowVUI.',$
		 '(To remove default, just delete this file.)',$
		 '','Note that ShadowVui may also need some additional',$
		 'files (from pre-processors) that must be in the working',$
		 'directory to avoid run-time errors!'])
	  IF itmp EQ 'Cancel' THEN GoTo,out
	  shadow_savews,state,group=event.top,File=file
	  Widget_Control,stateId,Set_UValue=state
	  END
	'Delete Default File...': BEGIN
	  file=XOP_GetEnv('XOP_DEFAULTS_DIR')+SDep(/DS)+'shadowvui_default.ws'
	  IF checkfile(file) EQ 0 THEN BEGIN
	    itmp = Dialog_Message(Dialog_Parent=event.top,/Info, $
		'No default defined.')
	    GoTo,out
 	  ENDIF ELSE BEGIN
	    Delete_Files,file,/Confirm
	  ENDELSE
	  END
	ENDCASE
	END
  'PREFERENCES': BEGIN
	xsh_preferences,Group=event.top
	END
  'WSDOC': BEGIN
	text = *(state.ptrWSDoc)
	XDisplayFile1,Text=text,Title='WorkSpace description', $
	  /Modal, ACTION=action, Dialog_Parent=event.top
	IF action EQ 'DONT' THEN GoTo,out
	*(state.ptrWSDoc) = text
	END
  ; sources
  'SRCSEL': BEGIN
	; warning: the selected source index starts with 0
	; whereas the oe index starts with 1!!!
	Widget_Control,state.wids.src,Get_Value=tmp
	state.ifc.src_sel=tmp
	Widget_Control,stateId,Set_UValue=state
	END
  'SOURCE': BEGIN
	src_string=''
	Widget_Control,event.id,Get_Value=src_string
	only_Run=0
	case src_string of
	  'Modify...': Widget_Control,state.wids.src,Get_Value=index
	  'Run SHADOW/source': BEGIN
		Widget_Control,state.wids.src,Get_Value=index
		only_Run=1
		END
	  'Geometrical...':		index = 0
	  'Bending Magnet...':		index = 1
	  'Wiggler...':			index = 2
	  'Undulator...':		index = 3
	  else:
	endcase

	Shadow_RunSource,stateId,index,Only_Run=only_run,group=event.top
	Widget_Control,state.wids.src,Set_Value=index
	END
  'SRCPLOTXY': BEGIN
	Widget_Control,event.id,Set_DropList_Select=0 
	xcol=0 & ycol=0
	CASE event.index OF
	  0: GoTo,out
	  1: BEGIN 
		xcol=1 & ycol=3 
		END
	  2: BEGIN 
		xcol=4 & ycol=6 
		END
	  3: BEGIN 
		xcol=1 & ycol=4 
		END
	  4: BEGIN 
		xcol=3 & ycol=6 
		END
	  else:
	ENDCASE
	xsh_plotxy,'begin.dat',xCol,yCol,Group=event.top,$
	  wtitle='PlotXY: begin.dat'
	END

  'SRCHISTO1': BEGIN
	Widget_Control,event.id,Set_DropList_Select=0 
	col=0 
	CASE event.index OF
	  0: GoTo,out
	  1: col=1
	  2: col=3
	  3: col=4
	  4: col=6
	  5: col=11
	  else:
	ENDCASE
	XSh_Histo1,'begin.dat',col,Group=event.top,$
	  wtitle='Histo1: begin.dat'
	END


  'SRCANA': BEGIN
	Widget_Control,event.id,Get_Value=val
	case val of
	  'InfoSh': infosh,'begin.dat',group=event.top,/Ask, $
		Dialog_Parent=event.top
	  'SourcInfo': sourcinfo,Group=event.top
	  'Spectrum': xsh_spectrum,parent=event.top
	  else:
	endcase
	END
  ; trace
  'TRCSEL': BEGIN
	Widget_Control,state.wids.oe,Get_Value=tmp
	state.ifc.oe_sel=tmp+1
	Widget_Control,stateId,Set_UValue=state
	END
  'TRACE': BEGIN
	trc_string=''
	Widget_Control,event.id,Get_Value=trc_string
	case trc_string of
	  'Add Optical Element...':	action = 'ADD'
	  'Modify/Edit O.E...':		action = 'EDIT'
	  'Delete O.E...':		action = 'DELETE'
	  'Delete System...':		action = 'DELETEALL'
	  'Run SHADOW/trace':		action = 'RUNTRACE'
	  'Load gfiles...':		action = 'LOADG'
	  'Save gfiles...':		action = 'SAVEG'
	  'GO terminal':		action = 'TERM_GO'
	  'SHELL terminal':		action = 'TERM_SHELL1'
	  'SHELL terminal with SHADOW initialization':	action = 'TERM_SHELL2'
	  'SHELL terminal and load a SHADOW program':	action = 'TERM_SHADOW_LOAD'
	  'SHELL terminal with SHADOW command':	action = 'TERM_SHADOW'
	  ;	
	  'Add oe':			action = 'ADD'
	  'Modify oe...':		action = 'EDIT'
	  'Delete oe':			action = 'DELETE'
	  'Delete all':			action = 'DELETEALL'
	  'Run SHADOW/trace':		action = 'RUNTRACE'
	  else: message,/info,'Case not found: '+trc_string
	endcase
	case action of
	  'ADD': BEGIN
		shadow_update,state,/add,Group=event.top
		Widget_Control,stateId,Set_UValue=state
		END
	  'EDIT': BEGIN
		if (state.ifc.oe_n LE 0) then begin
		  itmp = Dialog_Message(Dialog_Parent=event.top,$
		    ['No optical elements defined.','Use "Add oe" before'])
		  goto,out
		endif
		Widget_Control,/HourGlass
		MyPtr = state.PtrTrc(state.ifc.oe_sel-1)
		if Ptr_Valid(MyPtr) then begin
		  xshoe, MyPtr, Group = event.top, wtitle=$
		   'Parameters for O.E. # '+strcompress(state.ifc.oe_sel)
		endif else begin
		  itmp = Dialog_Message(/Error,Dialog_Parent=event.top,$
		   'SHADOW_EVENT: Pointer to empty data.')
		  goto,out
		endelse
		END
	  'DELETE': BEGIN
		shadow_update,state,/delete,Group=event.top
		Widget_Control,stateId,Set_UValue=state
		END
	  'DELETEALL': BEGIN
		shadow_update,state,/alldelete,Group=event.top
		Widget_Control,stateId,Set_UValue=state
		END
	  'RUNTRACE': BEGIN
		Shadow_RunTrace,state,Group=event.top
		Widget_Control,stateId,Set_UValue=state
		END
	  'LOADG': BEGIN
		itmp = Dialog_Message(Dialog_Parent=event.top,$
			'Not yet implemented')
		END
	  'SAVEG': BEGIN
		itmp = Dialog_Message(Dialog_Parent=event.top,$
			'Not yet implemented')
		END
;	  'TERM_GO': BEGIN
;		xsh_run,'xterm -e GO '
;		END
	  'TERM_SHELL1': BEGIN
		case sdep() of
		'WINDOWS':  spawn
		'UNIX': spawn,'xterm'
		else: 
		endcase
		END
	  'TERM_SHELL2': BEGIN
		case sdep() of
		'WINDOWS':  xsh_run,'start'
		'UNIX': xsh_run,'xterm'
		else: 
		endcase
		END
	  'TERM_SHADOW_LOAD': BEGIN
              IF shadow3_mode EQ 0 THEN BEGIN
		path=getenv('SHADOW_ROOT')+SDep(/ds)+'bin'
		; remove double slashes (confuses dialog_pickfile)
		path = StrSubstitute(path,SDep(/ds)+SDep(/ds),SDep(/ds))

		file = Dialog_PickFile(/Must_Exist,Path=path, $
			Title='Select a program to execute')
		IF file EQ '' THEN GoTo,out
		n = StrParse(file[0],'/',list)
		command = list[n]
		case sdep() of
		'WINDOWS':  BEGIN
			; command = 'recolor'
			action = 0
			xedit,command,Text='SHADOW command',Title=$
			  'SHADOW command',Action=action, $
			  Dialog_Parent=event.top,XSize=80
			if action EQ 'CANCEL' then goto,out
			xsh_run,command
			END
		'UNIX': BEGIN
			; command = 'recolor'
			action = 0
			xedit,command,Text='SHADOW command', $
			  Title='SHADOW command',$
			  Action=action, Dialog_Parent=event.top, $
			  XSize=80
			if action EQ 'CANCEL' then goto,out
			xsh_run,'xterm -e '+command
			END
		else: 
		endcase
              ENDIF ELSE BEGIN
		case sdep() of
		'WINDOWS':  xsh_run,'shadow3'
		'UNIX': xsh_run,'xterm -e shadow3'
		else: 
		endcase
              ENDELSE
		END
	  'TERM_SHADOW': BEGIN

              IF shadow3_mode EQ 0 THEN BEGIN
		case sdep() of
		'WINDOWS':  BEGIN
			command = 'recolor'
			action = 0
			xedit,command,Text='SHADOW command',Title=$
			  'SHADOW command',Action=action, $
			  Dialog_Parent=event.top,XSize=80
			if action EQ 'CANCEL' then goto,out
			xsh_run,command
			END
		'UNIX': BEGIN
			command = 'recolor'
			action = 0
			xedit,command,Text='SHADOW command', $
			  Title='SHADOW command',$
			  Action=action, Dialog_Parent=event.top, $
			  XSize=80
			if action EQ 'CANCEL' then goto,out
			xsh_run,'xterm -e '+command
			END
		else: 
		endcase
              ENDIF ELSE BEGIN
		case sdep() of
		'WINDOWS':  xsh_run,'shadow3'
		'UNIX': xsh_run,'xterm -e shadow3'
		else: 
		endcase
              ENDELSE
		END


	  else: message,/info,'Action not found/implemented: '+action
	endcase
	END

  'TRCPLOTXY': BEGIN
	Widget_Control,event.id,Set_DropList_Select=0 
	xcol=0 & ycol=0
	CASE event.index OF
	  0: GoTo,out
	  1: BEGIN 
		IF state.ifc.oe_sel LE 9 THEN $
		  file = 'mirr.0'+strcompress(state.ifc.oe_sel,/rem) ELSE $
		  file = 'mirr.'+strcompress(state.ifc.oe_sel,/rem)
		xcol=2 & ycol=1 
		END
	  2: BEGIN 
		IF state.ifc.oe_sel LE 9 THEN $
		  file = 'star.0'+strcompress(state.ifc.oe_sel,/rem) ELSE $
		  file = 'star.'+strcompress(state.ifc.oe_sel,/rem)
		xcol=1 & ycol=3 
		END
	  3: BEGIN 
		IF state.ifc.oe_sel LE 9 THEN $
		  file = 'star.0'+strcompress(state.ifc.oe_sel,/rem) ELSE $
		  file = 'star.'+strcompress(state.ifc.oe_sel,/rem)
		xcol=4 & ycol=6 
		END
	  4: BEGIN 
		IF state.ifc.oe_sel LE 9 THEN $
		  file = 'star.0'+strcompress(state.ifc.oe_sel,/rem) ELSE $
		  file = 'star.'+strcompress(state.ifc.oe_sel,/rem)
		xcol=1 & ycol=4 
		END
	  5: BEGIN 
		IF state.ifc.oe_sel LE 9 THEN $
		  file = 'star.0'+strcompress(state.ifc.oe_sel,/rem) ELSE $
		  file = 'star.'+strcompress(state.ifc.oe_sel,/rem)
		xcol=3 & ycol=6 
		END
	  else:
	ENDCASE
	xsh_plotxy,file,xCol,yCol,Group=event.top,$
	  wtitle='PlotXY: '+file
	END

  'TRCHISTO1': BEGIN
	Widget_Control,event.id,Set_DropList_Select=0 
	file = 'star.0'+strcompress(state.ifc.oe_sel,/rem)
	col=0 
	CASE event.index OF
	  0: GoTo,out
	  1: col=1
	  2: col=3
	  3: col=4
	  4: col=6
	  5: col=11
	  else:
	ENDCASE
	XSh_Histo1,file,col,Group=event.top,$
	  wtitle='Histo1: '+file
	END
  'TRCANA': BEGIN
	;Widget_Control,event.id,Get_Value=val
	Widget_Control,event.id,Set_DropList_Select=0 
	file = 'star.0'+strcompress(state.ifc.oe_sel,/rem)
	case event.index of
	  0: GoTo,out
	  ;InfoSh (rays)
	  1: infosh,file, /NoLost, group=event.top, $
		/Ask, Dialog_Parent=event.top
	  ;MirInfo (o.e.)
	  2: mirinfo, state.ifc.oe_sel,Group=event.top
	  ;SysInfo (system)
	  3: sysinfo, state.ifc.oe_n,Group=event.top
	  4: xsh_summary,Group=event.top
	  else: Message,'Case not found: '+String(event.index)
	endcase
	END

  'BLVIEWER': blviewer, File='systemfile.dat',/End_File, Group=event.top

  ; Run menu


  'RUN': BEGIN
	sWhat=''
	Widget_Control,event.id,Get_Value=sWhat
	CASE sWhat OF
          'Everything (source+trace+macros)': BEGIN
		source=1 & trace=1 & macros=1
		END
          'SHADOW (source+trace)': BEGIN
		source=1 & trace=1 & macros=0
		END
          'Source': BEGIN
		source=1 & trace=0 & macros=0
		END
          'Trace': BEGIN
		source=-1 & trace=1 & macros=0
		END
          'All macros': BEGIN
		source=0 & trace=0 & macros=1
		END
	  else: BEGIN
		source=0 & trace=0 & macros=0 
		END
	ENDCASE
	Shadow_Run,state,source=source,trace=trace,macros=macros, Group=event.top
	END



  ; Results menu
  'PLOTXY': XSh_Plotxy,Group=event.top,/NoExecute
  'HISTO1': XSh_Histo1,Group=event.top,/NoExecute
  'RAY_PROP': XSh_Ray_Prop,Group=event.top,/NoExecute
  'INFOSH': InfoSh,/Ask,group=event.top,Dialog_Parent=event.top
  'BLVIEWER': blviewer,Group=event.top
  'SOURCINFO': SourcInfo,Group=event.top
  'MIRINFO': MirInfo,state.ifc.oe_sel,Group=event.top
  'SYSINFO': SysInfo,state.ifc.oe_sel,Group=event.top
  'DISTANCES': xsh_summary,Group=event.top
  ; Util menu
  'PREREFL': xsh_prerefl,Group=event.top
  'PRE_MLAYER': xsh_pre_mlayer,Group=event.top
  'BRAGG': xsh_bragg,Group=event.top
  'WAVINESS': xsh_waviness,Group=event.top
  'XSH_CONIC': xsh_conic,Group=event.top
  'SRCOMP': BEGIN
	IF shadow3_mode NE 0 THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		'Not (yet) implemented in Shadow3')
	  return
	ENDIF
	xsh_srcomp,Group=event.top
	END
  'WIGGLER_SPECTRUM': BEGIN
	IF shadow3_mode NE 0 THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		'Not (yet) implemented in Shadow3')
	  return
	ENDIF
	xsh_wiggler_spectrum,Group=event.top
	END
  'ABREFC': BEGIN
	IF shadow3_mode NE 0 THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		'Not (yet) implemented in Shadow3')
	  return
	ENDIF
	IF shadow3_mode NE 0 THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		'Not (yet) implemented in Shadow3')
	  return
	ENDIF
	xsh_abrefc,Group=event.top
	END
  'MLAYER': BEGIN
	IF shadow3_mode NE 0 THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		'Not (yet) implemented in Shadow3')
	  return
	ENDIF
	xsh_mlayer,Group=event.top
	END
  'TRANSMIT': BEGIN
	IF shadow3_mode NE 0 THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		'Not (yet) implemented in Shadow3')
	  return
	ENDIF
	xsh_transmit,Group=event.top
	END
  'RCIL': rcil,Group=event.top
  'ROWLAND': rowland,Group=event.top
  ; tools menu
  'XLOADCT': xloadct,Group=event.top
  'XPLOT': xplot,Group=event.top,/No_block
  'XOP': xop,Group=event.top,/No_block
  'SHADOWVUI': shadow ; ,Group=event.top
  'IDLMACRO': myidlmacro,Group=event.top
  'SCICALC': scicalc,Group=event.top
  'XDISPLAYFILE1': BEGIN
	file = Dialog_Pickfile(group=event.top,title=$
        'Select a file to display/edit')
	if file NE '' then xdisplayfile1,file,Group=event.top
	END
  ; help menus
  'HLPLIB': BEGIN
	ds = SDep(/ds)
	path = Xop_GetEnv('XOP_HOME')+ds+'extensions'+ds+'shadowvui'+ds+'doc'
	; remove double slashes (confuses dialog_pickfile)
	path = StrSubstitute(path,SDep(/ds)+SDep(/ds),SDep(/ds))

	file = Dialog_PickFile(Dialog_Parent=event.top, $
	  Filter='*.txt',Title='Select a txt file',Path=path)
	IF file EQ '' THEN GoTo,out
	XDisplayFile1,file,Group=event.top,Dialog_Parent=event.top
	END

  'HLPSHADOWDOC': BEGIN
	ds = SDep(/ds)
	path = Xop_GetEnv('XOP_HOME')+ds+'extensions'+ds+'shadowvui'+ds+'doc'+$
          ds+'SHADOW'
	; remove double slashes (confuses dialog_pickfile)
	path = StrSubstitute(path,SDep(/ds)+SDep(/ds),SDep(/ds))

	file = Dialog_PickFile(Dialog_Parent=event.top, $
	  Filter='*.*',Title='Select a file',Path=path)
	IF file EQ '' THEN GoTo,out
	XDisplayFile1,file,Group=event.top,Dialog_Parent=event.top
	END
;  'HLPVUI': BEGIN
;	itmp = Dialog_message(Dialog_Parent=event.top,'Not yet implemented')
;	END
  'HLPABOUT': Shadow_Help,Group=event.top
  'HLPMANUALS': BEGIN
        Widget_Control,/HourGlass
        xop_help,'shadowvui.pdf'
;        ds = SDep(/ds)
;        IF sdep() EQ 'WINDOWS' THEN BEGIN
;          spawn,'start "'+xop_getenv('XOP_HOME')+ds+'extensions'+ds+'shadowvui'+ds+'doc'+ds+'shadowvui.pdf"'
;        ENDIF ELSE BEGIN
;          spawn,'acroread '+xop_getenv('XOP_HOME')+ds+'extensions'+ds+'shadowvui'+ds+'doc'+ds+'shadowvui.pdf &'
;        ENDELSE
	END
;  'HLPWEB1': BEGIN
;	command = 'netscape -install http://www.esrf.fr/computing/expg/'+$
;	 'subgroups/theory/shadow/vui/shadowvui.html'
;	action=0
;	xedit,command,infotext='Command to open WWW browser: ',action=action,$
;	  text = ' ',Title='WWW conection'
;	if action EQ 'CANCEL' then goto,out
;	message,/Info,'Executing: '+command
;	spawn,command
;	END
  'HLPWEB2': BEGIN
	command = 'netscape -install '+$
	 'http://www.xraylith.wisc.edu/shadow/shadow.html' 
	action=0
	xedit,command,infotext='Command to open WWW browser: ',action=action,$
	  text = ' ',Title='WWW conection'
	if action EQ 'CANCEL' then goto,out
	message,/Info,'Executing: '+command
	spawn,command
	END
  'MACROEDIT': BEGIN
	IF XRegistered('xop_macro') THEN BEGIN
	  itmp = Dialog_Message(/Question,Dialog_Parent=event.top,$
	  ['SHADOW VUI has detected that a macro is being edited.',$
	   'Please confirm you want to open another macro editor.'])
	  IF itmp EQ 'No' THEN GoTo,out
	ENDIF
	iMacro=-1
	Widget_Control,state.wids.macros,Get_Value=iMacro
	Xop_Macro, ptrCode=(*state.ptrPtrMacros)[iMacro], Group=event.top, $ 
	  Title='ShadowVui Macro '+StrCompress(iMacro+1)
	END
  'MACRORUN': BEGIN
	iMacro=-1
	Widget_Control,state.wids.macros,Get_Value=iMacro
	Shadow_RunMacro,state,iMacro,Group=event.top
	END
  'MACROSSEL': BEGIN
	Shadow_Update_MacroText,state,event.value
	Widget_Control,stateId,Set_UValue=state
	END
  'MACROADD': BEGIN
	; get selected macro number
	iMacro=-1
	Widget_Control,state.wids.macros,Get_Value=iMacro
	; add pointer
	ptrMacros = *state.ptrPtrMacros
	ptrMacros = [ptrMacros,Ptr_New('')]
	nMacros = N_Elements(ptrMacros)
	; refresh the macro widget area
	Widget_Control,state.wids.Macros,/Destroy
	macros_labels = strcompress(indgen(nMacros)+1)
	wMacros=CW_BGroup(state.wids.macrosB,macros_labels,UVALUE='MACROSSEL',/Exclusive,/Row, $
  	Set_Value=iMacro,/No_Release)
	Shadow_Update_MacroText,state,iMacro
	; store new values
	*state.ptrPtrMacros=ptrMacros
	state.wids.macros=wMacros
	Widget_Control,stateId,Set_UValue=state
	END


  'MACRODELETE': BEGIN
	; get selected macro number
	iMacro=-1
	Widget_Control,state.wids.macros,Get_Value=iMacro
	; add pointer
	ptrMacros = *state.ptrPtrMacros
	nMacros = N_Elements(ptrMacros)
	IF nMacros EQ 1 THEN BEGIN
	  itmp = Dialog_Message(Dialog_Parent=event.top, $
	    'Cannot delete macro #1. At least one macro must be defined')
	  GoTo,out
	ENDIF
	CASE iMacro OF 
	  0:    ptrMacros = ptrMacros[1:nMacros-1] 
	  nMacros-1: ptrMacros = ptrMacros[0:nMacros-2] 
	  else:  ptrMacros = [ptrMacros[0:iMacro-1],ptrMacros[iMacro+1:nMacros-1]]
	ENDCASE
	nMacros = N_Elements(ptrMacros)
	iMacro=iMacro-1>0
	; refresh the macro widget area
	Widget_Control,state.wids.Macros,/Destroy
	macros_labels = strcompress(indgen(nMacros)+1)
	wMacros=CW_BGroup(state.wids.macrosB,macros_labels,UVALUE='MACROSSEL',/Exclusive,/Row, $
  	  Set_Value=iMacro,/No_Release)
	Shadow_Update_MacroText,state,iMacro
	; store new values
	*state.ptrPtrMacros=ptrMacros
	state.wids.macros=wMacros
	Widget_Control,state.wids.macros,Set_Value=iMacro
	Widget_Control,stateId,Set_UValue=state
	END
  ELSE:
ENDCASE


out:
;IF widget_info(stateid,/Valid_Id) THEN $
;  IF N_Elements(state) GT 0 THEN $
;    Widget_Control,stateid,Set_UValue=state
END


;		
;========================================================================
;
Pro SHADOW,Group=group,Parent=stateId

COMMON shadow_ifc,parent1

catch, error_status
 if error_status ne 0 then begin
   catch, /cancel
   message,/info,'Error caught: '+!err_string
   if sdep(/w) then itmp = dialog_message('SHADOW: Error caught: '+$
	!err_string,/error)
   Return
endif

;
; apply preferencies
;
if (!d.n_colors gt 256) then device,decomposed=0
ifile = 0
if getenv('XOP_HOME') NE '' then begin ; in embedded mode
  ds = sdep(/ds)
  file = getenv('XOP_HOME')+ds+'extensions'+ds+'shadowvui'+ds+$
	'shadowvui_preferences.pro'
  ifile = 1
endif else if filepath1('shadowvui_preferences.pro') NE '' then begin
  ds = sdep(/ds)
  file = filepath1('shadowvui_preferences.pro')+$
	ds+'shadowvui_preferences.pro'
  ifile = 1
endif else message,/info,$
	'Preferences file not found. Use File/Preferences menu to define'

if ifile EQ 1 and checkfile(file) EQ 1 then begin
  message,/info,'Reading preferences from file: '+file
  a = read_textfile(file)
  b= where(strmid(strcompress(a,/Rem),0,1) NE ';',count)
  if count NE 0 then begin
    for i=0,n_elements(b)-1 do itmp = execute(a(b(i)))
  endif
endif

;
; start creating widgets

wShadowMain=Widget_Base(Column=1,Title='Shadow VUI '+shadow_version(),$
	MBAR=wMenuBar, $
        /TLB_KILL_REQUEST_EVENTS, Event_Pro='shadow_quit')

wSourceBase=Widget_Base(wShadowMain,Col=1,/Frame)
wTraceBase=Widget_Base(wShadowMain,Col=1,/Frame)
wMacrosBase=Widget_Base(wShadowMain,Col=1,/Frame)
wWorkingDirBase=Widget_Base(wShadowMain,Col=1,/Frame)

;
; build the menu bar
;
 
;
; File Menu
;
wtmp0 =  WIDGET_BUTTON(wMenuBar, VALUE='ShadowVUI', /MENU)
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='Load Workspace...', $
        UVALUE='LOADWS')
  wtmp1=WIDGET_BUTTON(wtmp0, VALUE='Load Example Workspace', $
        /MENU)
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='From Example Dir...', UVALUE='LOADWS')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='From Tutorial Dir...', UVALUE='LOADWS')
  wtmp1=WIDGET_BUTTON(wtmp0, VALUE='Import', /Menu)
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Source...', UVALUE='IMPORT_SOURCE')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='System...', UVALUE='IMPORT_SYSTEM')
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='Save Workspace...', $
        UVALUE='SAVEWS',/Separator)
  wtmp1=WIDGET_BUTTON(wtmp0, VALUE='Default Workspace',/Menu)
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Save Workspace as Default...', $
	UVALUE='SAVEWS')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Delete Default File...', $
	UVALUE='SAVEWS')
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='Preferences...', UVALUE='PREFERENCES', $
	/Separator)
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='Quit', UVALUE='EXIT',/SEPA)

;
; Edit Menu
;
wtmp0 =  WIDGET_BUTTON(wMenuBar, VALUE='Edit', /MENU)
  ;
  ; Source Menu
  ;
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='WorkSpace description', UVALUE='WSDOC')
  wtmp1 =  WIDGET_BUTTON(wtmp0, VALUE='Source', /MENU)
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Geometrical...', UVALUE='SOURCE')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Bending Magnet...', UVALUE='SOURCE')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Wiggler...', UVALUE='SOURCE')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Undulator...', UVALUE='SOURCE')
  ;
  ; trace Menu
  ;
  wtmp1 =  WIDGET_BUTTON(wtmp0, VALUE='Trace', /MENU)
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Add Optical Element...', UVALUE='TRACE')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Modify/Edit O.E...', UVALUE='TRACE')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Delete O.E...', UVALUE='TRACE')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Delete System...', UVALUE='TRACE')

;
; Run Menu
;
wtmp0 =  WIDGET_BUTTON(wMenuBar, VALUE='Run', /MENU)
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='Everything (source+trace+macros)', UVALUE='RUN')
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='SHADOW (source+trace)', UVALUE='RUN')
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='Source', UVALUE='RUN')
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='Trace', UVALUE='RUN')
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='All macros', UVALUE='RUN')

;
; Results Menu
;
wtmp0 =  WIDGET_BUTTON(wMenuBar, VALUE='Results', /MENU)
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='PlotXY tool...', UVALUE='PLOTXY')
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='Histo1 tool...', UVALUE='HISTO1')
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='Ray_Prop (animation tool)...', $
	UVALUE='RAY_PROP')
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='InfoSh...', UVALUE='INFOSH')
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='BLViewer', UVALUE='BLVIEWER')
  wtmp1=WIDGET_BUTTON(wtmp0, VALUE='Information', /MENU)
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='SourcInfo', UVALUE='SOURCINFO')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='MirInfo', UVALUE='MIRINFO')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='SysInfo', UVALUE='SYSINFO')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Distances', UVALUE='DISTANCES')



;
; Pre-processors Menu
;
wtmp0 =  WIDGET_BUTTON(wMenuBar, VALUE='PreProcessors', /MENU)
    wtmp=WIDGET_BUTTON(wtmp0, VALUE='Mirror/Filter refl/Abs (PreRefl)', $
	UVALUE='PREREFL')
    wtmp=WIDGET_BUTTON(wtmp0, VALUE='Multilayers (Pre_MLayer)', $
	UVALUE='PRE_MLAYER')
    wtmp=WIDGET_BUTTON(wtmp0, VALUE='Crystals (Bragg)', $
	UVALUE='BRAGG')
    wtmp=WIDGET_BUTTON(wtmp0, VALUE='Waviness', $
	UVALUE='WAVINESS')
    wtmp=WIDGET_BUTTON(wtmp0, VALUE='xsh_conic', $
	UVALUE='XSH_CONIC')
;
; Util Menu
;
wtmp0 =  WIDGET_BUTTON(wMenuBar, VALUE='Util', /MENU)
  wtmp1=WIDGET_BUTTON(wtmp0, VALUE='Source', /MENU)
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Bending Magnet Spectrum (SRComp)', $
	UVALUE='SRCOMP')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Wiggler Spectrum (Wiggler_Spectrum)', $
	UVALUE='WIGGLER_SPECTRUM')
  wtmp1=WIDGET_BUTTON(wtmp0, VALUE='Optical', /MENU)
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Absorption & Reflectivity (AbRefC)', $
	UVALUE='ABREFC')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Multilayers (MLayer)',UVALUE='MLAYER')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Transmission on Source (Transmit)',$
	UVALUE='TRANSMIT')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Tang and Sag radii (rcil)',$
	UVALUE='RCIL',/Separator)
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Rowland conditions (rowland)',$
	UVALUE='ROWLAND')
;
; Tools Menu
;
wtmp0 =  WIDGET_BUTTON(wMenuBar, VALUE='Tools', /MENU)
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='New ShadowVUI window',UVALUE='SHADOWVUI')
  wtmp=WIDGET_BUTTON(wtmp0, VALUE= 'Display File... ', UVALUE='XDISPLAYFILE1')
  wtmp=WIDGET_BUTTON(wtmp0, VALUE= 'Calculator', UVALUE='SCICALC')
;  wtmp=WIDGET_BUTTON(wtmp0, VALUE= 'IDL Macro', UVALUE='IDLMACRO')
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='Change color table...',UVALUE='XLOADCT')
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='XPlot (plotting tool)',UVALUE='XPLOT')
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='XOP',UVALUE='XOP')
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='SHELL terminal', UVALUE='TRACE',/Separator)
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='SHELL terminal and load a SHADOW program', $
	UVALUE='TRACE')
  wtmp=WIDGET_BUTTON(wtmp0, VALUE='SHELL terminal with SHADOW command', $
	UVALUE='TRACE')
;
; Help Menu
;
wtmp0 =  WIDGET_BUTTON(wMenuBar, VALUE='Help', /HELP)
  wtmp1=WIDGET_BUTTON(wtmp0, VALUE='About SHADOW VUI', UVALUE='HLPABOUT')
  wtmp1=WIDGET_BUTTON(wtmp0, VALUE='Manuals and Tutorials', UVALUE='HLPMANUALS')
  wtmp1=WIDGET_BUTTON(wtmp0, VALUE='ShadowVUI Library functions', UVALUE='HLPLIB')
  wtmp1=WIDGET_BUTTON(wtmp0, VALUE='Shadow demo+doc text files', UVALUE='HLPSHADOWDOC')
;  wtmp1=WIDGET_BUTTON(wtmp0, VALUE='SHADOW VUI', UVALUE='HLPVUI')
;  wtmp1=WIDGET_BUTTON(wtmp0, VALUE='Web link to SHADOW VUI', UVALUE='HLPWEB1')
;  wtmp1=WIDGET_BUTTON(wtmp0, VALUE='Web link to SHADOW Home Page', $
;	UVALUE='HLPWEB2')

;
; Source section
;
wtmp=Widget_Label(wSourceBase,VALUE='Source: ',/Align_Left)
wtmp0=Widget_Base(wSourceBase,/Row)
  wsrc=CW_BGroup(wtmp0,['Geometrical','BM','Wiggler','Undulator'],$
	UVALUE='SRCSEL',/Exclusive,/Row,/No_Release)
wtmp0=Widget_Base(wSourceBase,/Row)
  wtmp=Widget_Button(wtmp0,VALUE='Modify...',UVALUE='SOURCE') 
  wtmp=Widget_Button(wtmp0,VALUE='Run SHADOW/source',UVALUE='SOURCE') 
wtmp0=Widget_Base(wSourceBase,/Row)
  ;wtmp=Widget_Button(wtmp0,Value='PlotXY',UVALUE='SRCANA')
  wtmp=Widget_DropList(wtmp0,Value= $
	['PlotXY:',"(x,z)","(x',z')","(x,x')","(z,z')"],UVALUE='SRCPLOTXY')
  ;wtmp=Widget_Button(wtmp0,Value='Histo1',UVALUE='SRCANA')
  wtmp=Widget_DropList(wtmp0,Value= $
	['Histo1:',"x","z","x'","z'","E"],UVALUE='SRCHISTO1')
  wtmp=Widget_Button(wtmp0,Value='InfoSh',UVALUE='SRCANA')
  wtmp=Widget_Button(wtmp0,Value='SourcInfo',UVALUE='SRCANA')
  wtmp=Widget_Button(wtmp0,Value='Spectrum',UVALUE='SRCANA')

;
; Trace section
;
wtmp=Widget_Label(wTraceBase,VALUE='Optical System: ',/Align_Left)
woebase =Widget_Base(wTraceBase)
oes = 'oe '+strcompress(indgen(10))
woe=CW_BGroup(woebase,oes,UVALUE='TRCSEL',/Exclusive,/Row,/No_Release)
wtmp0=Widget_Base(wTraceBase,/Row)
  wtmp=Widget_Button(wtmp0,VALUE='Add oe',UVALUE='TRACE')
  wtmp=Widget_Button(wtmp0,VALUE='Modify oe...',UVALUE='TRACE')
  wtmp=Widget_Button(wtmp0,VALUE='Delete oe',UVALUE='TRACE')
  wtmp=Widget_Button(wtmp0,VALUE='Delete all',UVALUE='TRACE')
  wtmp=Widget_Button(wtmp0,VALUE='Run SHADOW/trace',UVALUE='TRACE')
wtmp0=Widget_Base(wTraceBase,/Row)
  wtmp=Widget_DropList(wtmp0,Value= $
        ['PlotXY:',"Footprint","(x,z)","(x',z')","(x,x')","(z,z')"], $
	UVALUE='TRCPLOTXY')
  wtmp=Widget_DropList(wtmp0,Value= $
        ['Histo1:',"x","z","x'","z'","E"],UVALUE='TRCHISTO1')
  ; wtmp=Widget_Button(wtmp0,Value='PlotXY',UVALUE='TRCANA')
  ; wtmp=Widget_Button(wtmp0,Value='Histo1',UVALUE='TRCANA')
  wtmp=Widget_DropList(wtmp0,Value= $
    ['Info on:','InfoSh','MirInfo','SysInfo','Distances'], $
	UVALUE='TRCANA')
  wtmp=Widget_Button(wtmp0,Value='BLViewer',UVALUE='BLVIEWER')

;
; Macros section
;
nMacros=1
wMacrosText=Widget_Label(wMacrosBase,VALUE='Macros: ',/Align_Left, $
  /Dynamic_Resize)

wMacrosB =Widget_Base(wMacrosBase)
; macros_labels = 'Macro '+strcompress(indgen(nMacros))
macros_labels = strcompress(indgen(nMacros)+1)
wMacros=CW_BGroup(wMacrosB,macros_labels,UVALUE='MACROSSEL',/Exclusive,/Row, $
  Set_Value=0,/No_Release)

wtmp0=Widget_Base(wMacrosBase,/Row)
  wtmp=Widget_Button(wtmp0,VALUE='Add macro',UVALUE='MACROADD')
  wtmp=Widget_Button(wtmp0,VALUE='Edit',UVALUE='MACROEDIT')
  wtmp=Widget_Button(wtmp0,VALUE='Delete macro',UVALUE='MACRODELETE')
  wtmp=Widget_Button(wtmp0,VALUE='Run macro',UVALUE='MACRORUN')

;
; Working directory section
;
if getenv('SHADOWVUI_WD') ne '' then cd,getenv('SHADOWVUI_WD')
cd,c=current
;wtmp0=Widget_Base(wWorkingDirBase,/Col)
wtmp=Widget_Label(wWorkingDirBase,VALUE='Working directory: ',/Align_Left)
wWorkingDir = cw_pickfile(wWorkingDirBase,Title='',$
	UVALUE = 'WORKINGDIR',VALUE=current,/Directory,XSIZE=55)


Widget_Control,wShadowMain,/Realize

Widget_Control,/HourGlass

ptrWSDoc = Ptr_New(['** Enter here a description for the workspace **'])
ptrSrc = PtrArr(4)
ptrTrc = PtrArr(15)
;
; pointer to array of pointers. Each element of the array points to 
; a different macro code.
;
tmp = PtrArr(nMacros,/Allocate_Heap)
FOR i=0L,nMacros-1 DO *tmp[i]=''
ptrPtrMacros=Ptr_New(tmp)

wids =	{ main:wShadowMain, oe:woe, oebase:woebase, src:wsrc, workingdir:wWorkingDir, $
          macros:wMacros, macrosB:wMacrosB, macrosText:wMacrosText }
ifc = {oe_sel:1, oe_n:1, src_sel:1, macro_sel:1 }

if ifc.oe_n GT 0 then begin
  tmp = replicate(xsh_defaults('XSHOE'),ifc.oe_n)
  for i=0,ifc.oe_n-1 do begin
    ; message,/info,'Assigning data to Ptr of oe: '+strcompress(i+1)
    PtrTrc(i) = Ptr_New(tmp(i))
  endfor
endif
   

state = {wids:wids, ifc:ifc, PtrSrc:ptrSrc, PtrTrc:ptrTrc, $
  PtrPtrMacros:ptrPtrMacros,ptrWSDoc:ptrWSDoc, wspath:''}

shadow_update,state


parent1=wShadowMain
stateId=wShadowMain
Widget_Control,parent1,Set_UValue=state  
XManager,'SHADOW',wShadowMain,/No_Block,Group=group

defaultFile=XOP_GetEnv('XOP_DEFAULTS_DIR')+SDep(/DS)+'shadowvui_default.ws'
IF CheckFile(defaultFile) THEN BEGIN
  shadow_loadws,parent1,file=defaultfile
ENDIF

END
