pro FASTFTR,ftrin,fourier,NPOINT=npoint,RRANGE=rrange,KSTEP=kstep
;+
; NAME:
;	FASTFTR
;
; PURPOSE:
;	This procedure calculates the Fast Fourier Transform of a set
;
; CATEGORY:
;	XAID xafs data analysis package. 
;
; CALLING SEQUENCE:
;
;	FASTFTR, ftrin, ftrout
;
; INPUTS:
;	ftrin:  a 2 or 3 col set with k,real(chi),imaginary(chi)
;	ftrout:	a 4-column set with the  congugated variable (col 0), 
;	the modulus (col 1), the real part (col 2) and imaginary 
;	part (col 3) of the Fourier Transform
;
; OPTIONAL INPUTS:
;	
; KEYWORD PARAMETERS:
;	RRANGE=[rmin,rmax] : range of the congugated variable for 
;		the transformation (default = [0.,6.])
;	NPOINT= number of points of the the fft calculation (default = 4096)
;	KSTEP = step of the k variable for the interpolation
;
; OUTPUTS:
;	This function returns a 4-columns array (ftrout) with
;	the congugare variable (R) in column 0, the modulus of the
;	FT in col 1, the real part in col 2 and the imaginary part in
;	col 3.
;
; OPTIONAL OUTPUTS:
;
; COMMON BLOCKS:
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	uses the FFT IDL's function
;
; EXAMPLE:
;		FASTFTR,setin,fourier
;
; MODIFICATION HISTORY:
; 	Written by:	Manuel Sanchez del Rio. ESRF
;	March, 1993	
;-
;
on_error,2
;
; default values
;
if not(keyword_set(npoint)) then npoint=4096
if not(keyword_set(kstep)) then kstep=0.04
if not(keyword_set(rrange)) then begin
  rmin = 0.0
  rmax = 6.0
endif else begin
  rmin = rrange(0)
  rmax = rrange(1)
endelse
;
; some definitions
;
npoint2=n_elements(ftrin(0,*))
xmin=ftrin(0,0)
xmax=ftrin(0,npoint2-1)
b=fltarr(2,npoint)
;
; creates the b set with the interpolated values of ftrin
;
b(0,*)=fix(findgen(npoint))*kstep
b(1,*)=interpol(ftrin(1,*),ftrin(0,*),b(0,*))
for i=0,npoint-1 do begin
  if (b(0,i) gt xmax) then b(1,i)=0.0
  if (b(0,i) lt xmin) then b(1,i)=0.0
endfor
;
; calculates the fft and generates the congugated variable (rr)
;
ff=fft(b(1,*),-1)
rstep=!pi/npoint/kstep
rr=fix(findgen(npoint))*rstep
;
; prepare the results
;
coef=npoint*kstep/sqrt(!pi)*sqrt(2.)
f12= coef*float(ff)              ; real part of fft
f13= coef*imaginary(ff)*(-1.)    ; imaginary part of fft
;
; cut the results to the selected interval in r (rrange)
;
f13=f13(where((rr gt rmin)and(rr lt rmax)))
f12=f12(where((rr gt rmin)and(rr lt rmax)))
f10=rr(where((rr gt rmin)and(rr lt rmax)))
f11=sqrt( f12^2 + f13^2 )                      ; modulus
;
; plot the result (for testing) 
;
;plot,rr,coef*imaginary(ff)*(-1.),xrange=[rmin,rmax],yra=[-.15,.15]
;oplot,rr,coef*abs(ff)
;oplot,rr,-1.*coef*abs(ff)
;
; define the result array
;
fourier=fltarr(4,n_elements(f10))  
fourier(0,*)=f10
fourier(1,*)=f11
fourier(2,*)=f12
fourier(3,*)=f13
;
return
end
