
;=========================== copyright =========================================
;                       XAID (XAfs with IDl) package.                          ;
;     Copyright European Synchrotron Radiation Facility (1993-1996).           ;
; This software may be used, copied, or redistributed as long as it is not     ;
; sold and this copyright notice is reproduced on each copy made.              ;
; The software is provided as is without any express or implied warranties     ;
; whatsoever.                                                                  ;
; Other limitations apply as described in the file README.copyright.           ;
;                                                                              ;
; REFERENCE: "XAID: A package for XAFS data analysis based on IDL"             ;
;       Manuel Sanchez del Rio. To be published at the proceedings of          ;
;       the 9th International Conference on X-ray Absorption Fine Structure    ;
;       26th - 30th August 1996,  E S R F - Grenoble  France                   ;
;==============================================================================;
PRO WINDOW_FTR,setin,windout,setweighted,WINDOW=window,  $
WINDPAR=windpar,WRANGE=wrange,NAMES=names
;
;+
; NAME:
;	WINDOW_FTR
;
; PURPOSE:
;	This procedure calculates and applies a weighting window to a set
;
; CATEGORY:
;	XAID xafs data analysis package. 
;
; CALLING SEQUENCE:
;
;	WINDOW_FTR, setin, windout
;
; INPUTS:
;	setin:	fltarr(ncol,npoints) set of data
;	windout:	fltarr(ncol,npoints) set with the calculated window
;
; OPTIONAL INPUTS:
;	setweighted:	the input set multiplied by the window
;	
; KEYWORD PARAMETERS:
;	NAMES = Return in this named variable the list of avilable 
;		windows (only that). Example: 
;		IDL> names = 1
;		IDL> window_ftr,names=names
;	WINDOW = kind of window:
;		1 Gaussian Window (default)
;		2 Hanning Window
;		3 No Window
;		4 Parzen (triangular)
;		5 Welch
;		6 Hamming
;		7 Tukey
;		8 Papul
;		9 Kaiser
;	WINDPAR Parameter for windowing
;		If WINDOW=(2,4,5,6) this sets the width of
;		the appodization (default=0.2)
;		If WINDOW=9 sets the t parameter 0<t<=10.
;	WRANGE = [xmin,xmax] the limits of the window. If wrange
;		is not set, the take the minimum and maximum values
;		of the abscisas. The window has value zero outside
;		this interval.
;
; OUTPUTS:
;	a set with the window and (optionally) the weighted set
;
; OPTIONAL OUTPUTS:
;
; COMMON BLOCKS:
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	Straightforward
;
; EXAMPLE:
;		WINDOW_FTR,setin,wind,window=5
;
; MODIFICATION HISTORY:
; 	Written by:	Manuel Sanchez del Rio. ESRF
;	March, 1993	
;	96-08-14 MSR (srio@esrf.fr) adds names keyword.
;	06-03-14 srio@esrf.fr always exits "names"
;-
;
on_error,2
;
names = ['1 Gaussian', '2 Hanning','3 None', $
	'4 Parzen','5 Welch','6 Hamming','7 Tukey',$
	'8 Papul','9 Kaiser']
IF N_PARAMS() EQ 0 THEN RETURN

if not(keyword_set(window)) then window=3
if not(keyword_set(windpar)) then windpar=0.2
;
tk=setin(0,*)
;
if not(keyword_set(wrange)) then begin
  xmax=max(tk)
  xmin=min(tk)
endif else begin
  xmin=float(wrange(0))
  xmax=float(wrange(1))
endelse
;
xp=(xmax+xmin)/2.
xm=xmax-xmin
apo1=xmin+windpar
apo2=xmax-windpar
;
npoint=n_elements(tk)
wind=fltarr(npoint)
case (window) of 
  1: begin                    ; gaussian window
       wind=((tk-xp)/xm)^2 
       wind=exp(-wind*9.2)
     end
  2: begin                    ; hanning window
     for i=0,npoint-1 do begin
       if ((tk(i) gt apo1) and (tk(i) lt apo2)) then begin
         a = 1.
         wind(i) = a
       endif
       if (tk(i) lt apo1) then begin
         a=.5*(1.-cos(!pi*(tk(i)-xmin)/windpar))
         wind(i)=a
       endif
       if (tk(i) gt apo2) then begin
         a=.5*(1.+cos(!pi*(tk(i)-apo2)/windpar))
         wind(i)=a
       endif
     endfor
     end
  3: begin                    ; boxcar window
       wind=wind*0.0+1.0
     end
  4: begin                    ; Parzen (triangular) window
     for i=0,npoint-1 do begin
       if ((tk(i) gt apo1) and (tk(i) lt apo2)) then begin
         a = 1.
         wind(i) = a
       endif
       if (tk(i) lt apo1) then begin
         a=(tk(i)-xmin)/windpar
         wind(i)=a
       endif
       if (tk(i) gt apo2) then begin
         a=1.-( (tk(i)-apo2)/windpar )
         wind(i)=a
       endif
     endfor
     end
  5: begin                    ; Welch window
     for i=0,npoint-1 do begin
       if ((tk(i) gt apo1) and (tk(i) lt apo2)) then begin
         a = 1.
         wind(i) = a
       endif
       if (tk(i) lt apo1) then begin
         a=1.-(((tk(i)-apo1)/windpar)^2)
         wind(i)=a
       endif
       if (tk(i) gt apo2) then begin
         a=1.-( (tk(i)-apo2)/windpar )^2
         wind(i)=a
       endif
     endfor
     end
  6: begin                    ; hamming
     for i=0,npoint-1 do begin
       if ((tk(i) gt apo1) and (tk(i) lt apo2)) then begin
         a = 1.
         wind(i) = a
       endif
       if (tk(i) lt apo1) then begin
         a=.54+0.46*cos(!pi*(tk(i)-xmin)/windpar)
         a=1.08-a
         wind(i)=a
       endif
       if (tk(i) gt apo2) then begin
         a=.54-0.46*cos(!pi*(tk(i)-apo2)/windpar)
         a=1.08-a
         wind(i)=a
       endif
     endfor
     end
  7: begin                    ; Tukey
     for i=0,npoint-1 do begin
       if ((tk(i) gt apo1) and (tk(i) lt apo2)) then begin
         a = 1.
         wind(i) = a
       endif
       if (tk(i) lt apo1) then begin
         a=( cos(0.5*!pi*(tk(i)-xmin)/windpar) )^2
         a=1.-a
         wind(i)=a
       endif
       if (tk(i) gt apo2) then begin
         a=( cos(-0.5*!pi*(tk(i)-apo2)/windpar) )^2
         wind(i)=a
       endif
     endfor
     end
  8: begin                    ; Papul (check please !!!!!!)
     for i=0,npoint-1 do begin
       if ((tk(i) gt apo1) and (tk(i) lt apo2)) then begin
         a = 1.
         wind(i) = a
       endif
       if (tk(i) lt apo1) then begin
         a=(1./!pi)*sin(!pi*(tk(i)-xmin)/windpar) + $
             (1.-(tk(i)-xmin)/windpar)*cos(!pi*(tk(i)-xmin)/windpar)
         a=1.-a
         wind(i)=a
       endif
       if (tk(i) gt apo2) then begin
         a=(1./!pi)*sin(!pi*(tk(i)-apo2)/windpar) + $
             (1.-(tk(i)-apo2)/windpar)*cos(!pi*(tk(i)-apo2)/windpar)
         wind(i)=a
       endif
     endfor
     end
  9: begin                    ; kasel
       wind=beseli( windpar*sqrt(1.-((tk-xp)/xm*2.)^2),0 )/  $
       beseli(windpar,0)
     end
endcase
;
; 
if keyword_set(wrange) then begin
  for i=0,npoint-1 do begin
    if ((tk(i) lt xmin) or (tk(i) gt xmax)) then wind(i)=0.0
  endfor
endif
;
windout=setin
windout(1,*)=wind
;
setweighted=setin
setweighted(1,*)=setin(1,*)*wind
;
return
end
