;=========================== copyright =========================================
;                       XAID (XAfs with IDl) package.                          ;
;     Copyright European Synchrotron Radiation Facility (1993-1996).           ;
; This software may be used, copied, or redistributed as long as it is not     ;
; sold and this copyright notice is reproduced on each copy made.              ;
; The software is provided as is without any express or implied warranties     ;
; whatsoever.                                                                  ;
; Other limitations apply as described in the file README.copyright.           ;
;                                                                              ;
; REFERENCE: "XAID: A package for XAFS data analysis based on IDL"             ;
;       Manuel Sanchez del Rio. To be published at the proceedings of          ;
;       the 9th International Conference on X-ray Absorption Fine Structure    ;
;       26th - 30th August 1996,  E S R F - Grenoble  France                   ;
;==============================================================================;
PRO        BFTRSET,fourier,backftr,KRANGE=krange,RRANGE=rrange, $
WINDOW=window, WINDPAR=windpar, WRANGE=wrange, INQUIRE=inquire, $
METHOD=method,PLOTBFTR=plotbftr,RSHELL=rshell,MOUSECUT=mousecut
;+
; NAME:
;	BFTRSET
;
; PURPOSE:
;	This procedure is a driver to calculate the Inverse Fourier Transform 
;	of a Fourier Transform of an Exafs spectrum
;
; CATEGORY:
;	XAID xafs data analysis package. 
;
; CALLING SEQUENCE:
;
;	BFTRSET, fourier, backftr
;
; INPUTS:
;	fourier:	set of data with the Fourier Transform
;		as written by FTRSET
;	backftr:	set of data with the output 4-column set:
;		k (column 0), real part pf the bftr (column 1), modulus
;		of the BFTR (column 2) and phase (column 3)
;
; OPTIONAL INPUTS:
;	
; KEYWORD PARAMETERS:
;	RRANGE=[rmin,rmax] : range of the abscissas variable for 
;		the transformation	
;	KRANGE=[kmin,kmax] : range of the conjugated variable (k) for
;		the anti-transformation
;	WINDOW = if selected, the result is divided by the weighted
;		window used in the transformation. Kind of window:
;		1 Gaussian Window (default)
;		2 Hanning Window
;		3 No Window
;		4 Parzen (triangular)
;		5 Welch
;		6 Hamming
;		7 Tukey
;		8 Papul
;		9 Kaiser
;	WINDPAR Parameter for windowing (default=0.2)
;		If WINDOW=(2,4,5,6) this sets the width of
;		the appodization 
;		If WINDOW=9 sets the t parameter 0<t<=10.
;	WRANGE = [min,max] values (in k) used for the windowing 
;		during the FTR transform
;	INQUIRE  when set to 1 ask about change parameters
;	METHOD method used for calculating the Fourier Transform:
;		0 Standard Fourier transform (default)
;		1 Fast Fourier Transform
;	PLOTBFTR  when set then plot the bftr (1=plot real,
;		2=modulus,3=phase)
;	RSHELL = rshell: R  value for correcting the phase  
;		newphase = phase - 2 k R  (default R=0.0)
;	MOUSECUT when set to 1, then cut the fourier set using the mouse
;
; OUTPUTS:
;	This function returns a 4-columns array (back-fourier) with
;	the conjugated variable (k) in column 0, the real part of the
;	BFTR in col 1, the modulus in col 2 and the phase in
;	col 3.
;
; OPTIONAL OUTPUTS:
;
; COMMON BLOCKS:
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	Calls BFTR osr FASTBFTR and divide the output by the selected window
;
; EXAMPLE:
;		BFTRSET,fourier,backftr,/inq
;
; MODIFICATION HISTORY:
; 	Written by:	Manuel Sanchez del Rio. ESRF
;	March, 1993	
;-
on_error,2
;
; Keywords setup
;
if not(keyword_set(krange)) then begin
  kmin = 2.0
  kmax = 15.0
endif else begin
  kmin=float(krange(0))
  kmax=float(krange(1))
endelse
;
if not(keyword_set(rrange)) then begin
  rmin = 0.0
  rmax = 6.0
endif else begin
  rmin=float(rrange(0))
  rmax=float(rrange(1))
endelse
;
if not(keyword_set(wrange)) then begin
  wmin = kmin
  wmax = kmax
endif else begin
  wmin = float(wrange(0))
  wmax = float(wrange(1))
endelse
;
if not(keyword_set(window)) then window = 3
if not(keyword_set(windpar)) then windpar=0.2
if not(keyword_set(method)) then method=0
if not(keyword_set(rshell)) then rshell=0.0
;
; if MOUSECUT keyword is set, then allow to cut the ftr set using the mouse
;
if keyword_set(mousecut) then begin
  i_ok = 0
  repeat begin
    plotset,fourier
    print,' Please, click the rmin value for the bftr'
    cursor,rmin,tmp,3
    plotset,[[rmin,-1e10],[rmin,1e10]],/over
    print,' Please, click the rmax value for the bftr'
    cursor,rmax,tmp,3
    plotset,[[rmax,-1e10],[rmax,1e10]],/over
    print,' The selected points are: ',rmin,rmax
    i_ok = getyesornot(' Are you happy with these values ? [y/n] ')
    endrep until i_ok
endif
;
; inquire about changes
;
if keyword_set(inquire) then begin
  command = ' '
  repeat begin
    print,'*************************************************************'
    print,' Data for the Back Fourier Transform: '
    print,' rmin (min for the fourier transform cutting) = ',rmin
    print,' rmax (max                                  ) = ',rmax
    print,' kmin (min for the back fourier interval) = ',kmin
    print,' kmax (max                                = ',kmax
    print,' wmin (min for the window interval) = ',wmin
    print,' wmax (max                        ) = ',wmax
    print,' rshell (R of the shell) = ',rshell
    print,'       method = ',method
    print,'          0 = Standard Back Fourier Transform'
    print,'          1 = Fast Back Fourier Transform'
    print,'       window = ',window
    print,'          1 = Gaussian Window'
    print,'          2 = Hanning Window'
    print,'          3 = boxcar window (no window)'
    print,'          4 = Parzen (triangular) window'
    print,'          5 = Welch window '
    print,'          6 = Hamming window'
    print,'          7 = Tukey window'
    print,'          8 = Papul window'
    print,'          9 = Kaiser window'
    if ((window eq 2) or $
        (window eq 4) or $
        (window eq 5) or $
        (window eq 6) or $
        (window eq 7) or $
        (window eq 8) )  then $
        print,'                windpar (width of appodization) = ',windpar
    if (window eq 9) then $
        print,'                t-value for Bessel function = ',windpar
    print,'*************************************************************'
    i_change = getyesornot(' Do you want to change these parameters ? [y/n] ')
    if (i_change eq 1) then begin
      i_ok = 0
      print,' For changing, type "parameter = newvalue". "-1" to end'
      repeat begin
        read,command
          if (command eq '-1') then begin
            i_ok=1
          endif else begin
            i_flag = execute(command)
            if (i_flag ne 1) then print,'Error asigning variable.'
          endelse
      endrep until i_ok
    endif
  endrep until (i_change eq 0)
endif
;
; call Back Fourier Transform procedure
;
cutset,fourier,fourier2,xrange=[rmin,rmax]
if (method eq 0) then begin
  bftr,fourier2,backftr,npoint=512,krange=[kmin,kmax]
endif else begin
  fastbftr,fourier2,backftr,npoint=4096,krange=[kmin,kmax] ;,rstep=0.04
endelse
;
; correct the phases with 2 k Rshell
;
backftr(3,*)=backftr(3,*) - backftr(0,*)*2.*rshell
;
; divide the result by the selected window
;
if (window ne 3) then begin
  window_ftr,backftr,ww,window=window,windpar=windpar,wrange=[wmin,wmax]
  backftr(1,*) = backftr(1,*)/ww(1,*)
  backftr(2,*) = backftr(2,*)/ww(1,*)
endif
;
; plot the result if required
;
if keyword_set(plotbftr) then begin
  case plotbftr of
    1: plotset,backftr
    2: plotset,backftr([0,2],*)
    3: plotset,backftr([0,3],*)
  endcase
endif
;
end
