;+
; =================================  xaid_mu2chi ===============================
;
;
;
;   xaid_mu2chi is a widget based graphical interface to perform the
;   XAFS background substraction. Among several different  operations 
;   xaid_mu2chi may extract the Eo value from the maximum of the derivative, 
;   get automatically or manually the Jump value, perform polynomial 
;   and multispline fits, and allows three methods of normalization.
;   
;   The main window contains the following controls
;
;	FILE Menu:
;	  SAVE TO FILE: wites an ASCII file with all the calculated
;		data in SPEC format. Columns are: 
;		E [eV]  E-Eo [eV]  k [A^-1]  Mu  Mu-PreEdge  Mu Norm
;		PreEdge  PostEdge  Chi  Chi*k  Chi*k^2  Chi*k^3
;	  INPUT PARAMETERS: These options permit to load/write a 
;		file with the parameters of the XAID_MU2CHI application. 
;		It can also be used for defining user's defaults. 
;	  PRINT:  Send a copy of the current plot to the printer.
;	  PREFERENCES...:  Allows to customize some parameters relevant
;		for the display and Fourier Transform. 
;	  QUIT:	Exit or Quit the program
;
;	RAW_DATA Menu:
;	  These operations apply ONLY to RAW data:
;	  CUT: Permits to reduce the horizontal limits of the data
;	  DERIVATIVES: Shows at the same plot the signal and
;		its two first derivatives
;	  INTERPOLATION: Allows Linear or Spline interpolation for
;		reducing or adding points. 
;	  GENERIC OPERATION: Permits to modify the x and y raw data. 
;		Useful to change units (eV-keV), prenormalization, etc. 
;
;	Tools Menu:
;	  XAID_MU2CHI: Opens a new application window
;	  XAID: Starts main XAID window
;	  XPLOT: Opens an empty XPLOT window.
;	  XOP: Starts main XOP window
;
;       HELP Menu:   Shows documentation to several related XAFS routines.
;
;	Main Left Panel: Input parameters 
;
;	  RAW DATA FILE: To Load a File with Data. The file must contain 
;		the data in ascii formatted in two columns. It
;		may content comments becoming by a non-numeric (nor + - .)
;		character. The program stores the data and reads 
;		automatically all the points and columns
;	  1) PreEdge Linear Fit: Selects the limits in E for the preEdge
;		linear fit
;	  2) Eo and Jump: Gets the Eo from the maximum of the derivative, and 
;		allows the user to change it or input another one.
;	        It also gets an initial Jump value and allows the user to 
;		change it.
;	  3) Post-edge Multi-Spline: allows multi-spline polynomial fits.
;	  4) EXAFS extraction: Selects the extraction algorithm:
;		
;	Main Right Panel: Plot parameters. 
;
;
;		
; ======== Non-interactive use of XAID_Mu2CHI ====================
;
;	NAME:
;		XAID_MU2CHI
;	PURPOSE:
;		 EXAFS extraction
; 	CATEGORY:
;		XAID: XOP XAFS data analysis.
;	CALLING SEQUENCE:
;		XAID_MU2CHI [,filename] 
;	OPTIONAL INPUT PARAMETERS:
;		filename: a string containing the file name to be
;		plotted or an idl variable [fltarr(2,n_points)]
;		with data. The file must be ascii data arranged in 
;		columns. Lines starting with characters [except . + -]
;		are comments and not considered for the plot.
;
;	KEYWORD PARAMETERS:
;		GROUP = The widget ID of the widget that calls Xplot.
;		When this ID is specified, a death of the caller results 
;		in a death of XAID_Mu2Chi.
;		NO_BLOCK = If set, starts Xmanager with the No_Block keyword.
;		WSIZE = The initial size of the graphical display in pixels
;		DELIAID = The widget id for DELIA panel (for internal
;			use of DELIA)
;
;	OUTPUT KEYWORD PARAMETERS:
;		PARENT = The widget id for the main window (see later)
;	OUTPUTS:
;		Open a widget utility and present a graphic.
;	COMMON BLOCKS:
;		COMMON xaid_mu2chi, strOper
;		to keep memory from operation window
;	PROCEDURE:
;		Uses XAID data analysis IDL library.
;	MODIFICATION HISTORY:
;		Written by  Manuel Sanchez del Rio. ESRF. 11 October 1994
;		96-08-22 MSR creates a new version from the old xcnpick.
;		97-01-14 MSR fixes for Windows.
;		97-09-08 MSR adds NO_BLOCK keyword
;		97-09-30 srio@esrf.fr READ_ASCII renamed to RASCII
;		97-10-25 srio@esrf.fr small fixed in main window.
;		98-10-01 srio@esrf.fr adapts for delia. Use of catch.
;		99-02-18 srio@esrf.fr adapts XOP 2.0
;		00-02-15 srio@esrf.fr adapts for Exodus. Centers 
;			the screens. Adds *1000 and :1000 operations.
;			Allows to load a spline.
;		03-12-09 srio@esrf.fr adds linear fit (LINFIT) and
;			normalization to jump.
;		06-03-13 srio@esrf.fr Almost completely rewritten. v 2.0Beta1
;		06-07-20 srio@esrf.fr Avoid loading file name when loading 
;			input files. 
;
;
;   Example of using of xaid_mu2chi from external programs: 
;	The Parent keyword permits to store the application id and 
;	reuse it from other applications. Some routines allow the
;	data manipulation:
;		xaid_mu2chi_loadfile,parent,set
;		xaid_mu2chi_plot,parent
;		xaid_mu2chi_putstr,parent,str
;		;   (or xaid_mu2chi_putstr,parent,str,/NoFile to avoid 
;		;    changing the input file name)
;		str=xaid_mu2chi_getstr(parent)
;		xaid_mu2chi_calc,set,str
;
;	Examples: 
;
;	1) Start and quit:
;		IDL> xaid_mu2chi,parent=p
;		IDL> xaid_mu2chi_quit,p
;	2) Start and load data
;		IDL> xaid_mu2chi,parent=p
;		IDL> xaid_mu2chi_loadfile,p,'CuFoilRef.txt'
;		or
;		xaid_mu2chi_loadfile,p,rascii('CuFoilRef.txt'),/noplot
;
;	3) Load data, modify parameters of left panel, update the plot
;		
;		IDL> xaid_mu2chi,parent=p,'CuFoilRef.txt'
;		IDL> str=xaid_mu2chi_getstr(p)
;		IDL> str.e0=8979
;		IDL> str.jump=1
;		IDL> xaid_mu2chi_putstr,p,str
;		IDL> xaid_mu2chi_plot,p,ix=2,iy=5
;	
;	4) Save file with data
;		
;		IDL> xaid_mu2chi_calc, $
;			rascii('CuFoilRef.txt'),str,/write,file='tmp.spec'
;
;
;  HELP FOR THE USE OF NON-INTERACTIVE ROUTINES: 
;
;
;-


;
;
;========================================================================
;
FUNCTION xaid_mu2chi_version
RETURN,'2.0Beta2'
END ; xaid_mu2chi_version

;
;========================================================================
;
FUNCTION xaid_mu2chi_text,input
txt=''
CASE input OF
  'DataOperations': txt=[ $
     'This option permits to make operations in the x and y data',$
     'The user must input the new expressions for x and y:',$
     '    x=...',$
     '    y=...',$
     '   ',$
     '   Notes: ',$
     '   1) The operations are done sequentially, first for x.',$
     '      For example, if x=x-100 and y=y*x are selected, the',$
     '      y operation uses x with the substraction (x-100) already',$
     '      performed',$
     '   2) It is possible to define new intermediate variables',$
     '   3) It is possible to send several instructions separated by &',$
     '    ',$
     '   Examples: ',$
     '   1) Shift x axis by 100: ',$
     '               x=x-100',$
     '               y=y',$
     '   2) Normalize y in [0,1]: ',$
     '               x=x',$
     '               y=y-Min(y) & y=y/Max(y) ',$
     '   3) Set the zero of x in the Max of the derivative (as in XANES):',$
     '               e0=Max(deriv(y),i) & x=x-x[i]',$
     '               y=y', $
     '   4) Normalize y to the mean value in a window: ',$
     '               Identify the window: you may use xplot to plot the',$
     '               data vs the index, by double-click on the data set.',$
     '               x=x ',$
     '               y0 = Mean(y[400:500]) & y=y/y0 ']
  else:
ENDCASE ; xaid_mu2chi_text
RETURN,txt
END ; xaid_mu2chi_text

;
;========================================================================
;
PRO xaid_mu2chi_quit,id

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'xaid_mu2chi_quit: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

Widget_Control,id, Get_UValue=state
Ptr_Free,state.ptr
Widget_Control,id,/Destroy

END ; xaid_mu2chi_quit



;
;========================================================================
;
FUNCTION xaid_mu2chi_getstr,id

Widget_Control,id,Get_UValue=state
wids = state.wids

	str = state.str
	Widget_Control,wids.file,Get_Value=file
	Widget_Control,wids.preMin,Get_Value=preMin
	Widget_Control,wids.preMax,Get_Value=preMax
	Widget_Control,wids.e0,Get_Value=e0
	Widget_Control,wids.jump,Get_Value=jump
	Widget_Control,wids.npol,Get_Value=npol
	Widget_Control,wids.degree,Get_Value=degree
	Widget_Control,wids.knots,Get_Value=knots
;	Widget_Control,wids.kkmin,Get_Value=kkmin
;	Widget_Control,wids.kkmax,Get_Value=kkmax
	Widget_Control,wids.extraction,Get_Value=extraction
	str.file = file
	str.preMin = preMin
	str.preMax = preMax
	str.jump = jump
	str.e0 = e0
	str.npol = npol
	str.degree = degree
	str.knots = knots
;	str.kkmin = kkmin
;	str.kkmax = kkmax
	tmp = str.extraction
	tmp[0] = StrCompress(extraction,/Rem)
	str.extraction = tmp
RETURN,str
END ; xaid_mu2chi_getstr

;
;========================================================================
;

PRO xaid_mu2chi_putstr,id,str,noFile=noFile

IF Type(str) NE 8 THEN RETURN
Widget_Control,id,Get_UValue=state

IF NOT(Keyword_Set(noFile)) THEN $
	Widget_Control,state.wids.file,Set_Value=str.file
Widget_Control,state.wids.preMin,Set_Value=str.preMin
Widget_Control,state.wids.preMax,Set_Value=str.preMax
Widget_Control,state.wids.e0,Set_Value=str.e0
Widget_Control,state.wids.jump,Set_Value=str.jump
Widget_Control,state.wids.npol,Set_Value=str.npol
Widget_Control,state.wids.degree,Set_Value=str.degree
Widget_Control,state.wids.knots,Set_Value=str.knots
Widget_Control,state.wids.extraction,Set_Value= $
  Fix( (str.extraction)[0] ) 

str2=state.str
copy_structure,str,str2,/Only
state.str=str2
Widget_Control,id,Set_UValue=state

END ; xaid_mu2chi_putstr


;
;========================================================================
;

PRO xaid_mu2chi_plot,id,Print=Print,_Extra=extra, $
  ix=ix,iy=iy,resetzoom=resetzoom, xplot=xplot

;+
;
;  PRO xaid_mu2chi_plot,id,Print=Print,_Extra=extra, $
;    ix=ix,iy=iy,resetzoom=resetzoom, xplot=xplot
;
;  ResetZoom: redefines default limits (recommended)
;  ix: 0 E [eV]
;      1 E-Eo [eV]
;      2 K [A^-1]
;      3 R [A]   (if set, the Fourier transform is displayed vs R)
;
;  iy: 0  Mu
;      1  Mu & PreEdge
;      2  Mu - PreEdge
;      3  Mu Normalized
;      4  Mu - PreEdge & Postedge
;      5  Chi
;      6  Chi*k
;      7  Chi*k
;      8  Chi*k^1
;      9  Chi*k^3
;
;-
catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(Dialog_Parent=id,$
       /Error,'XAID_MU2CHI_PLOT: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   Return
endif


Widget_Control,id,Get_UValue=state
IF N_Elements(dialog_parent) EQ 0 then dialog_parent=id

;
; update delia panel (if open)
;
IF type(state.wids.delia) EQ 8 THEN BEGIN
  ;
  Widget_Control,state.wids.e0,Get_Value=e0
  Widget_Control,state.wids.delia.chi.e0,Set_Value=e0
  ;
  Widget_Control,state.wids.jump,Get_Value=jump
  Widget_Control,state.wids.delia.chi.jump,Set_Value=jump
  ;
  Widget_Control,state.wids.preMin,Get_Value=preMin
  Widget_Control,state.wids.delia.chi.preMin,Set_Value=preMin
  ;
  Widget_Control,state.wids.preMax,Get_Value=preMax
  Widget_Control,state.wids.delia.chi.preMax,Set_Value=preMax
  ;
  Widget_Control,state.wids.npol,Get_Value=npol
  Widget_Control,state.wids.delia.chi.npol, $
  Set_Value=STrCompress(npol,/Rem)
  ;
  Widget_Control,state.wids.degree,Get_Value=degree
  Widget_Control,state.wids.delia.chi.degree,Set_Value=degree
  ;
  Widget_Control,state.wids.knots,Get_Value=knots
  Widget_Control,state.wids.delia.chi.knots,Set_Value=knots
  ;
  Widget_Control,state.wids.extraction,Get_Value=extraction
  Widget_Control,state.wids.delia.chi.extraction,Set_Value=extraction
  ;
  Widget_Control,state.wids.delia.chi.kkmin,Set_Value=state.str.kkmin
  Widget_Control,state.wids.delia.chi.kkmax,Set_Value=state.str.kkmax
ENDIF

xtitle='X'
ytitle='Y'

IF Not(Keyword_set(print)) THEN BEGIN
  wset,state.windownum
  x_old = !x
  y_old = !y
  !x=state.x
  !y=state.y
ENDIF

IF Keyword_Set(resetZoom) THEN BEGIN
  !x.range=[0,0]
  !y.range=[0,0]
ENDIF

!x.style=1
!y.style=1

setmu=*(state.ptr)
x=Reform(setmu[0,*])
y=Reform(setmu[1,*])
mu=y


; 
; abscissas
;
IF N_Elements(ix) EQ 0 THEN BEGIN
  widget_control,state.wids.abscissas,get_value = ix
  ix = Fix(ix)
ENDIF

; get jump
Widget_Control,state.wids.jump,Get_Value=jump
;
; calculate k
;
Widget_Control,state.wids.e0,Get_Value=e0
tmp = e2k(Make_Set(x,y),e0=e0)
kk=Reform(tmp[0,*])

CASE ix OF
     0: BEGIN
	xx=x             ; E
	xtitle='E [eV]'
	END
     1: BEGIN	      ; E-E0
	Widget_Control,state.wids.e0,Get_Value=e0
	xx=x-e0
	xtitle='E-Eo [eV]'
	END
     2: BEGIN	      ; K
	xx=kk
	xtitle='k [A^-1]'
	END
     3: BEGIN	      ; R
	xx=kk
	xtitle='R [A]'
	END
     else: xx=x
ENDCASE
;
; ordinates
;
IF N_Elements(iy) EQ 0 THEN BEGIN
  widget_control,state.wids.ordinates,get_value = iy
  iy = Fix(iy)
ENDIF

CASE iy OF
     0: ytitle='Mu'
     1: ytitle='Mu & PreEdge'
     2: ytitle='Mu-PreEdge'
     3: ytitle='Mu Normalized'
     4: ytitle='Mu-PreEdge & PostEdge'
     5: ytitle='Chi'
     6: ytitle='Chi*k'
     7: ytitle='Chi*k^2'
     8: ytitle='Chi*k^3'
     else:
ENDCASE
IF ix EQ 3 THEN ytitle='FT('+ytitle+')'

;
; Plot  (xx1,yy1,and eventually overplot yy2) 
;
yy1=0

IF iy EQ 0 THEN BEGIN
   xx1=xx
   yy1=y
ENDIF

IF iy EQ 1 THEN BEGIN ; linear fit
   Widget_Control,state.wids.preMin,Get_Value=xmin
   Widget_Control,state.wids.preMax,Get_Value=xmax
   xx1=xx
   yy1=y
   IF xmin EQ xmax THEN BEGIN
     itmp = Dialog_Message(/Info,Dialog_Parent=dialog_parent,$
	'No Pre-Edge selected.')
   ENDIF ELSE BEGIN
     fitset,Make_Set(x,y),fit1,method=0,xrange=[xmin,xmax],/VERBOSE
     ymin=Min([y,Reform(fit1[1,*])],Max=ymax)
     yy2=Reform(fit1[1,*])
     !y.range=[ymin,ymax]
   ENDELSE
ENDIF

; this is done for iy>=2
IF iy GE 2 THEN BEGIN ; substracted linear fit
   Widget_Control,state.wids.preMin,Get_Value=xmin
   Widget_Control,state.wids.preMax,Get_Value=xmax
   mu = y
   IF xmin NE xmax THEN BEGIN
     fitset,Make_Set(x,y),fit1,method=0,xrange=[xmin,xmax],/VERBOSE
     mu = y-Reform(fit1[1,*])
   ENDIF
   IF iy EQ 2 THEN BEGIN
     xx1 = xx
     yy1 = mu
   ENDIF
ENDIF

; up to here:
; mu is with the preEdge removed

IF iy EQ 3 THEN BEGIN ; normalized
   munor = mu
   IF jump EQ 0 THEN BEGIN
     itmp = Dialog_Message(/Error,Dialog_Parent=dialog_parent,$
	'No Jump defined.')
   ENDIF ELSE BEGIN
     muNor=muNor/Abs(jump)     
   ENDELSE
   xx1=xx
   yy1 = muNor
ENDIF
; up to here:
; mu is with the preedge removed


IF iy GE 4 THEN BEGIN  ; mu-preedge  &  postedge
  ; postedge
  widget_control,state.wids.Npol,get_value=tmp
  spl_n = fix(tmp[0])
  widget_control,state.wids.Degree,get_value=tmp
  spl_degree = 0
  itmp = execute('spl_degree = '+tmp[0])
  sknots=''
  widget_control,state.wids.knots,get_value=sknots
  spl_knots = 0
  itmp = execute('spl_knots = '+sknots[0])

  spl = postedge_evaluate(Make_Set(kk,mu),spl_n=spl_n,$
    spl_degree=spl_degree, spl_knots=spl_knots, $
    dialog_parent=dialog_parent)

  IF N_Elements(spl) LE 1 THEN BEGIN
    itmp = Dialog_Message(/Error,Dialog_Parent=dialog_parent, $
	'Error calculating Post-Edge spline.')
    spl = mu*0
  ENDIF

  IF iy EQ 4 THEN BEGIN
	xx1 = xx
	yy1 = mu
  ENDIF
  nn = N_Elements( Where( finite(spl[1,*],/NaN) ) EQ 1  ) 
  IF nn GT 1 THEN BEGIN
     itmp = Dialog_message(/Error,Dialog_Parent=dialog_parent, $
	'Error in evaluating PostEdge splines: Got '+StrCompress(nn,/Rem)+ $ 
	' NaN')
  ENDIF ELSE BEGIN
    IF iy EQ 4 THEN yy2=Reform(spl[1,*])
  ENDELSE
ENDIF

; variables up to here: 
;  kk: k
;  y: normalized data
;  mu: preedge removed 
;  spl[2,npoints] : postedge

IF iy GE 5 THEN BEGIN  ; chi
  Widget_Control,state.wids.extraction,Get_Value=iextraction
  iextraction=Fix(iextraction)
  yfit = reform(spl[1,*])

  CASE  iextraction OF
   0: BEGIN ; experimental
      IF jump EQ 0 THEN BEGIN
	itmp = Dialog_Message(/Error,Dialog_Parent=dialog_parent, $
	   'Undefined jump: It is needed for the experimental extraction')
	chi = mu 
      ENDIF ELSE BEGIN
        chi  = (mu - yfit) / jump
      ENDELSE
      END
   1: BEGIN ; constant
      chi  = (mu - yfit) / yfit
      END
   2: BEGIN ; lengeler
      IF jump EQ 0 THEN BEGIN
	itmp = Dialog_Message(/Error,Dialog_Parent=dialog_parent, $
	   'Undefined jump: It is needed for the L-E extraction')
	chi = mu 
      ENDIF ELSE BEGIN
        Widget_Control,state.wids.e0,Get_Value=e0
        chi  = (mu - yfit) / jump /( 1. - (8./3.)*(x-e0)/e0 )
      ENDELSE
      END
  ENDCASE
  ; plot points inside k interval
  ;widget_control,state.wids.kkmin,get_value=kmin
  ;widget_control,state.wids.kkmax,get_value=kmax
  IF ix EQ 2 THEN BEGIN
    ;!x.range=[kmin,kmax]
    !x.range=[state.str.kkmin,state.str.kkmax]
  ENDIF
  IF iy EQ 5 THEN BEGIN
	;plot,xx,chi,_extra=extra, $
	;xtitle=xtitle,ytitle=ytitle
	xx1 = xx
	yy1 = chi
  ENDIF
ENDIF

IF iy EQ 6 THEN BEGIN
	xx1=xx
	yy1=chi*kk
ENDIF

IF iy EQ 7 THEN BEGIN
	xx1=xx
	yy1=chi*kk*kk
ENDIF

IF iy EQ 8 THEN BEGIN
	xx1=xx
	yy1=chi*kk*kk*kk
ENDIF

;
; FT is ix=3 (abscissas=R)
;
IF ix EQ 3 THEN BEGIN ; fourier
  ;widget_control,state.wids.kkmin,get_value=kmin
  ;widget_control,state.wids.kkmax,get_value=kmax
  ftrset,Make_Set(kk,yy1),ft,method=1, $
    xrange=[state.str.kkMin,state.str.kkMax], $
    rrange=[state.str.rmin,state.str.rmax], kpower=0, $
    window=Fix((state.str.ftwindow)[0]),windpar=state.str.ftpar,$ 
    npoint=4096,kstep=0.04
  xx1 = Reform(ft[0,*])
  yy1 = Reform(ft[1,*])
  ;IF N_Elements(yy2) GT 1 THEN BEGIN
  ;   ftrset,Make_Set(kk,yy2),ft,method=1,xrange=[kMin,kMax], $
  ;     rrange=[0,8], kpower=0,window=1,windpar=0,$ 
  ;     npoint=4096,kstep=0.04
  ;ENDIF
ENDIF

;
; Make the plot
;
plot,xx1,yy1,_extra=extra,xtitle=xtitle,ytitle=ytitle
IF N_Elements(yy2) GT 1 THEN oplot,xx1,yy2,color=2

IF keyword_set(xplot) THEN BEGIN
  IF N_Elements(yy2) GT 1 THEN BEGIN
    xplot,Make_Set(xx1,yy2,yy1),/No_Block,coltitles=[xtitle,'fit',ytitle]
  ENDIF ELSE BEGIN
    xplot,Make_Set(xx1,yy1),/No_Block,coltitles=[xtitle,ytitle]
  ENDELSE
ENDIF

IF Not(Keyword_set(print)) THEN BEGIN
  state.x=!x  
  state.y=!y
  Widget_Control,id,Set_UValue=state
ENDIF
;print,'>> Out plot'
END ; xaid_mu2chi_plot

;
;========================================================================
;

PRO xaid_mu2chi_loadfile,id,file,Dialog_Parent=dialog_parent, noplot=noplot
;
;+
;
;  PRO xaid_mu2chi_loadfile,id,file, $
;			Dialog_Parent=dialog_parent, noplot=noplot 
;
;-

Widget_Control,id,Get_UValue=state
IF Type(file) EQ 7 THEN BEGIN
   IF checkfile(file) NE 1 THEN BEGIN
    itmp = Dialog_message(/Error,'File not found: '+file, $
	Dialog_Parent=dialog_parent)
    Return
   ENDIF
   set=rascii(file)
   print,'XAID_MU2CHI_LOADFILE: loading file: ',file
   Widget_Control,state.wids.file,Set_Value=file
ENDIF ELSE BEGIN ; from variable
   set=file
ENDELSE

*(state.ptr)=set
Widget_Control,id,Set_UValue=state

IF Not(Keyword_Set(noplot)) THEN xaid_mu2chi_plot,id,/reset


END
;
;========================================================================
;
PRO xaid_mu2chi_event, Event

COMMON xaid_mu2chi, strOper


catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(Dialog_Parent=event.top,$
	/Error,'XAID_MU2CHI_EVENT: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   Return
endif



IF Tag_Names(event,/Structure_Name) EQ 'WIDGET_KILL_REQUEST' THEN BEGIN
  xaid_mu2chi_Quit,event.top
  RETURN
ENDIF

Widget_Control, event.id, get_UValue=UValue
;IF UValue NE 'Cursor' THEN print,'>>>>>>>>>UVALUE: ',UValue


CASE uvalue OF 
  ; 
  ; Items in the menu bar
  ;
  'Load File': BEGIN
	Widget_Control,event.id,Get_Value=file
	xaid_mu2chi_loadFile,event.top,file,Dialog_Parent=event.top
	END

  'FileInputs': BEGIN
	Widget_Control,event.top,Get_UValue=state
	Widget_Control,event.id,Get_Value=val
	str = xaid_mu2chi_getstr(event.top)

	CASE val OF
	'Save file': BEGIN
          Xop_Input_Save,str,Group=event.top, /Write, $
            File='xaid_mu2chi.xop',Comment='; xop/xaid_mu2chi(v'+$
            xaid_mu2chi_version()+') input file on '+SysTime()
	  END
	'Save as default': BEGIN
          Xop_Input_Save,str,Group=event.top, $
            Default='xaid_mu2chi.xop',Comment='; xop/xaid_mu2chi(v'+$
            xaid_mu2chi_version()+') input file on '+SysTime()
	  END
	'Load from file': BEGIN
          if sdep() EQ 'UNIX' then filter='*.xop' else filter=0
          str = Xop_Input_Load(Title=$
          'Select xaid_mu2chi input file...',$
          /NoConf,Filter=filter,Group=event.top)
	  xaid_mu2chi_putstr,event.top,str,/NoFile
	  xaid_mu2chi_plot,event.top,/reset
	  END
	else:
	ENDCASE
    END
  'Save File': BEGIN
    Widget_Control,event.top,Get_UValue=state
    set = *(state.ptr)
    str = xaid_mu2chi_getstr(event.top)
    xaid_mu2chi_calc,set,str,out=out,/write,xplot=2,Dialog_Parent=event.top
    END

  'Print': BEGIN
    id = StrCompress(event.top,/Rem)+'L'
    Xprint,BUFFER='xaid_mu2chi_plot,'+id+',/Print',Group=Event.Top
    END
  'Preferences': BEGIN
    Widget_Control,event.top,Get_UValue=state
    str=state.str
    tmp = { $
	kkmin:str.kkmin, $
	kkmax:str.kkmax, $
	rmin:str.rmin, $
	rmax:str.rmax, $
	ftwindow:str.ftwindow, $
	ftpar:str.ftpar }
    n = N_Tags(tmp)
    Titles=['K min for plot','K max for plot','R min for Fourier',$
	'R max for Fourier','Window for Fourier','Appodization for Fourier']
    Flags=Replicate('1',n)
    Flags[n-1]='(w('+StrCompress(n-2,/Rem)+') EQ 1) OR '+$
               '(w('+StrCompress(n-2,/Rem)+') EQ 3) OR '+$
               '(w('+StrCompress(n-2,/Rem)+') EQ 4) OR '+$
               '(w('+StrCompress(n-2,/Rem)+') EQ 5) '
    XScrMenu,tmp,/Interp,/NoType,Action=action , $
	  group=event.top, NCol=2, $ ;Help=xaid_mu2chi_text('DataOperations'), $
          Flags=flags, WTitle='XAID_MU2CHI Preferences', Titles=titles, $
	  Dialog_Parent=event.top ; ,FieldLen=50
    IF action EQ 'DONT' THEN Return
    copy_structure,tmp,str,/onlyFirstField
    state.str=str
    Widget_Control,event.top, Set_UValue=state
    xaid_mu2chi_plot,event.top,/reset

    END
  'Quit': BEGIN
    xaid_mu2chi_quit,event.top
    return
    END
  'Xplot': BEGIN
    Widget_Control,event.top,Get_UValue=state
    xplot,*(state.ptr),wtitle='XAID_mu2chi Mu',GROUP=event.top,$
	ColTitles=['X','Y'], no_block=state.no_block
    END
  'Cut': BEGIN
    itmp = Dialog_Message(Dialog_Parent=event.top,$
	/info,['This option permits to cut the ',$
	'raw spectrum to analyze only the desirable interval.',$
	'Please select two poins with the mouse.'],/CANCEL)
    if itmp EQ 'Cancel' then Return
    Widget_Control,event.top,Get_UValue=state
    Widget_Control,state.wids.abscissas,Set_Value=0
    Widget_Control,state.wids.ordinates,Set_Value=0
    xaid_mu2chi_plot,event.top,/reset
    wait,0.1
    tmp1 = 1
    while (tmp1 EQ 1) do begin
      cursor,x1,y1,3
      plotfile,[[x1,-1000000],[x1,1000000]],ki=2
      cursor,x2,y2,3
      plotfile,[[x2,-1000000],[x2,1000000]],ki=2
      xaid_mu2chi_plot,event.top,xrange=[x1,x2],/xstyle

      tmp1 = Wmenu2(['Cancel','Repeat','Keep cut set (no undo)'], $
	Dialog_Parent=event.top)
      case tmp1 of
        0: xaid_mu2chi_plot,event.top
        1:
        2: begin
	   Widget_Control,event.top,Get_UValue=state
	   set1 = *(state.ptr)
           cutset,set1,tmpout,xrange=[x1,x2]
	   *(state.ptr) = tmpout
           state.zoom_sel=0
	   Widget_Control,event.top,Set_UValue=state
           xaid_mu2chi_plot,event.top
	   Return
           end
      endcase
    endwhile
    END
  'derivatives': BEGIN
    Widget_Control,event.top,Get_UValue=state
    set1 = *(state.ptr)
    derivset,set1,tmp
    tmp1=fltarr(5,n_elements(tmp(0,*)))
    tmp1(0:3,*) = tmp
    tmp1(4,*)=set1(1,*)
    xplot,tmp1,ycol=2,coltitles=['x',"f'(x)","f''(x)","f''''(x)",'f(x)'],$
	xtitle='-1',ytitle='-1',group=event.top,no_block=state.no_block
    END
  'Interp': BEGIN
    Widget_Control,event.top,Get_UValue=state
    set1 = *(state.ptr)
    npoints = n_elements(set1[0,*])
    stdevval = stddev(set1[0,*])
    mean1 = mean(set1[0,*])
    vecttmp = shift(set1[0,*],-1) - set1[0,*]
    vecttmp = vecttmp(0:npoints-2)
    stdevval = stddev(vecttmp)
    mean1 = mean(vecttmp)
    text = [strcompress(npoints,/rem)+' points in current spectrum',$
    '------------- min -----  max -------  mean ------- stdev ---',$
    ' E  :     '+strcompress(min(set1(0,*)),/rem)+'       '+$
	strcompress(max(set1(0,*)),/rem)+'       '+$
	strcompress(mean1,/rem)+'       '+strcompress(stdevval,/rem),$
    ' mu :     '+strcompress(min(set1(1,*)),/rem)+ '       '+ $
	strcompress(max(set1(1,*)),/rem)+'       '+ $
	strcompress(mean1,/rem)+'       '+strcompress(stdevval,/rem),$
    ' DE :     ' +strcompress(min(vecttmp),/rem)+'       '+$
	strcompress(max(vecttmp),/rem)+'       '+$
	strcompress(mean1,/rem)+'       '+strcompress(stdevval,/rem), $
    '------------------------------------------------------------']

    itmp = Dialog_Message(Dialog_Parent=event.top,$
	/info,/cancel,['This option can be used to',$
	'interpolate new points in raw data .',$
	'     ',text]) 
    if itmp EQ 'Cancel' then Return
    tmpmenu = { kind:['0','Linear','Spline'], npts:512 }
    XscrMenu,tmpmenu,/NoType,/Interp,Titles=['Interpolation: ',$
        'Number of points'],Wtitle='Interpolation',Action=action,$
	Dialog_Parent=event.top
    if action eq 'DO' then begin
      n_interp=tmpmenu.npts
      int_step = (max(set1(0,*))-min(set1(0,*)))/float(n_interp-1)
      tmpout = fltarr(2,n_interp)
      tmpout(0,*) = findgen(n_interp)*int_step + min(set1(0,*))
      if (n_interp LT npoints) then begin
        n_interp = npoints
        print,'XAID_Mu2Chi: Set interp number of points to minimum: ',npoints
      endif
      if (tmpmenu.kind(0) eq '0') then begin
        print,'XAID_Mu2Chi: Liner interp selected.'
        tmpout(1,*) = interpol(set1(1,*),set1(0,*),tmpout(0,*))
      endif else begin
        print,'XAID_Mu2Chi: Spline interp selected.'
        tmpout(1,*) = spline(set1(0,*),set1(1,*),tmpout(0,*))
      endelse
      wset,state.windownum
      plot,tmpout[0,*],tmpout[1,*],Title='Interpolated data'
      stdevval = stddev(tmpout(0,*))
      stdevval = stddev(tmpout(1,*))
      vecttmp = shift(tmpout(0,*),-1) - tmpout(0,*)
      vecttmp = vecttmp(0:npoints-2)
      stdevval = stddev(vecttmp)
      mean1 = mean(vecttmp)

      print,n_interp,' points in current spectrum'
      print,'------------- min -----  max -------  mean ------- stdev ---'
      print,' E  :     ',min(tmpout(0,*)),max(tmpout(0,*)),mean1,stdevval
      print,' mu :     ',min(tmpout(1,*)),max(tmpout(1,*)),mean1,stdevval
      print,' DE :     ',min(vecttmp),max(vecttmp),mean1,stdevval
      print,'------------------------------------------------------------'


    text = [strcompress(n_interp,/rem)+' points in current spectrum',$
    '------------- min -----  max -------  mean ------- stdev ---',$
    ' E  :     '+strcompress(min(tmpout(0,*)),/rem)+'       '+$
	strcompress(max(tmpout(0,*)),/rem)+'       '+$
	strcompress(mean1,/rem)+'       '+strcompress(stdevval,/rem),$
    ' mu :     '+strcompress(min(tmpout(1,*)),/rem)+ '       '+ $
	strcompress(max(tmpout(1,*)),/rem)+'       '+ $
	strcompress(mean1,/rem)+'       '+strcompress(stdevval,/rem),$
    ' DE :     ' +strcompress(min(vecttmp),/rem)+'       '+$
	strcompress(max(vecttmp),/rem)+'       '+$
	strcompress(mean1,/rem)+'       '+strcompress(stdevval,/rem), $
    '------------------------------------------------------------']


      ;tmp1 = Wmenu2(['Cancel','Substitute'])
      tmp1 = Dialog_Message(Dialog_Parent=event.top,$
	/question,[text,' ','Do you want to '+$
	'keep the interpolated spectrum? (no undo)'])
      case tmp1 of
        'No': Xaid_Mu2Chi_Plot,event.top
        'Yes': *(state.ptr) = tmpout
      endcase
    endif
    END
  'Oper': BEGIN  
      Widget_Control,event.top,Get_UValue=state
      set1 = *(state.ptr)
      x = Reform(set1[0,*])
      y = Reform(set1[1,*])
      IF N_Elements(strOper) EQ 0 THEN BEGIN
	  strOper = { $
;		apply:['0','On all scans','On selected scans only'],$
		expressionX:'x=x',expressionY:'y=y'}
      ENDIF
      flags=Replicate('1',N_tags(strOper))
      titles=['Expression: New x=f(x)', $
	        'Expression: New y=f(x,y)']
      XScrmenu,strOper,/Interp,/NoType,Action=action, $
	  group=event.top, Help=xaid_mu2chi_text('DataOperations'), $
          Flags=flags, Titles=titles, WTitle='Operations in raw data',$
	  Dialog_Parent=event.top,FieldLen=50
      IF action EQ 'DONT' THEN RETURN

      __itmp = Execute(strOper.expressionx)
      IF __itmp NE 1 THEN BEGIN
	     __tmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		'Error executing: '+strOper.expressionx)
	     RETURN
      ENDIF
      __itmp = Execute(strOper.expressiony)
      IF __itmp NE 1 THEN BEGIN
	     __tmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		'Error executing: '+strOper.expressiony)
	     RETURN
      ENDIF
      wset,state.windownum
      plot,x,y,Title='Spectrum after Generic Operations'
      tmp1 = Dialog_Message(Dialog_Parent=event.top,$
	/question,['Do you want to keep the new spectrum? (no undo)'])
      IF tmp1 EQ 'Yes' THEN *(state.ptr) = Make_Set(x,y)
      Xaid_Mu2Chi_Plot,event.top
      END

  'About': XAID_Help,Group=event.top
  'Help': BEGIN
    ;PRINT, 'Event for Help'
     widget_control,event.id,get_value=tmp
     xhelp,tmp,GROUP=event.top
    END

  ; 
  ; Left Panel
  ;
  'Get Eo': BEGIN
    Widget_Control,event.top, Get_UValue=state
    set1 = *(state.ptr)
    e0 = gete0(set1,/sort)
    xaid_mu2chi_plot,event.top
    plotset,[[e0,-1e5],[e0,1e5]],/over

    itmp = wmenu2([   'Max(Deriv): '+strcompress(e0), $
                      'Get a value with the cursor... ',$
                      'Input an external value... ',$
		      'Cancel'],Dialog_Parent=event.top)
    case itmp of
        0: 
        1: BEGIN
          ;widget_control,state.wids.abscissas,set_value = 0
	  ;xaid_mu2chi_plot,event.top,/reset
          ;xaid_mu2chi_plot,event.top

	  itmp = Dialog_Message(/Info, $
		'Click one point',Dialog_Parent=event.top)

	  wait,0.1
          e0 = gete0(set1,/sort,method=1)
          print,' The selected Eo is: ',e0
          END
        2: BEGIN
          tmpmenu = { e0:e0 }
          XScrMenu,tmpmenu,/NOTYPE,TITLES='Eo value: ',ACTION=action,$
		Dialog_Parent=event.top,WTitle='E0'
          if action eq 'DO' then e0 = tmpmenu.e0 ELSE Return
          END
        3: BEGIN
	  xaid_mu2chi_plot,event.top
	  return
	  END
    endcase

    widget_control,state.wids.e0,set_value=e0
;    if type(state.wids.delia) EQ 8 then $
;      widget_control,state.wids.delia.chi.e0,set_value=e0
    END

  'Jump': BEGIN
    Widget_Control,event.top, Get_UValue=state
    set1 = *(state.ptr)
    jump = getjump(set1,Dialog_Parent=event.top)
    itmp = wmenu2([   'Default: '+strcompress(jump), $
                      'Get jump with the cursor... ',$
                      'Input an external value... ',$
		      'Cancel'],Dialog_Parent=event.top)
    xaid_mu2chi_plot,event.top
    case itmp of
        0: 
        1: jump = getjump(set1,Dialog_Parent=event.top,iMode=1)
        2: jump = getjump(set1,Dialog_Parent=event.top,iMode=2)
        else: Return
    endcase


    widget_control,state.wids.jump,set_value=jump
;    if type(state.wids.delia) EQ 8 then $
;      widget_control,state.wids.delia.chi.jump,set_value=jump

    END

  'Linear Fit': BEGIN
    Widget_Control,event.top, Get_UValue=state
    set1 = *(state.ptr)
    itmp = Dialog_Message(Dialog_Parent=event.top,$
	/info,/cancel,['Click the two limit'+ $
	' points.'])
    if itmp eq 'Cancel' then return
    ymin=Min(set1[1,*],Max=ymax)
    yrange=[ymin-Abs(ymax-ymin),ymax]
    xaid_mu2chi_plot,event.top,yrange=yrange
    wait,0.1
    cursor,x1,y1,3
    oplot,[x1,x1],[y1,y1],psym=1,color=2
    cursor,x2,y2,3
    oplot,[x2,x2],[y2,y2],psym=1,color=2
    wait,0.5

    ; store values
    Widget_Control,state.wids.preMin,Set_Value=x1
    Widget_Control,state.wids.preMax,Set_Value=x2
;    if type(state.wids.delia) EQ 8 then begin
;      widget_control,state.wids.delia.chi.preMin,set_value=x1
;      widget_control,state.wids.delia.chi.preMax,set_value=x2
;    endif
    xaid_mu2chi_plot,event.top,iy=1,yrange=yrange
    END
  'PostEdge': BEGIN
    ;
    itmp = Dialog_Message(/Question, $
       ['This option allows to get interactively the knots from the ', $
        'current graphic by clicking with the mouse. ',$
        ' ',$
        'SELECT A RECTANGULAR AREA WITH **LEFT** BUTTON TO SELECT THE VIEW (ZOOM)', $
        'CLICK WITH **RIGHT** BUTTON TO GET A POINT', $
        'click with **CENTER** button to get the last point', $
        '(the degree of the splines will be set 2 for all)', $
        ' ',$
        'Do you want to proceed? ',$
        '    Yes=start list of knots', $
        '    No=Cancel action and eventually end an ongoing list of knots.'], $
        Dialog_Parent=event.top, Title='List of points')
    IF itmp EQ 'No' THEN BEGIN
      Widget_Control,event.top,Get_UValue=state
      state.getListOfKnots=0
      Widget_Control,event.top, Set_UValue=state
      Return
    ENDIF ELSE BEGIN
      ; prepare the plot vs k
      Widget_Control,event.top,Get_UValue=state
      Widget_Control,state.wids.abscissas,Set_Value=2
      Widget_Control,state.wids.ordinates,Set_Value=4
      state.getListOfKnots=1
      Widget_Control,event.top, Set_UValue=state
      xaid_mu2chi_plot,event.top,/reset
    ENDELSE

    END

  ; 
  ; Right Panel
  ;
  'Plot': xaid_mu2chi_plot,event.top,/reset
  'XPlot': xaid_mu2chi_plot,event.top,/xplot
  'Zoom': BEGIN
        printBuffer = 'xaid_mu2chi_plot,'+StrCompress(event.top,/Rem)+'L'
        printBuffer = printBuffer+',/Print,'+$
          'YRange=[parms.ymin,parms.ymax],'+$
          'XRange=[parms.xmin,parms.xmax]'
        parms={xmin:0.0,xmax:0.0,ymin:0.0,ymax:0.0}
        xwindow,b=printBuffer,/Edit,/ZoomFlag, $
          parms=parms,Group=event.top,$
          wtitle='XAID_Mu2Chi ZOOM',Dialog_Parent=event.top

	END
     
  'Resize': BEGIN
    ;widget_control,state.wids.wDr,scr_xsize=event.x-10,scr_ysize=event.y-70
    if !version.os_family eq 'Windows' then $
    widget_control,state.wids.wDr,scr_xsize=event.x-5,scr_ysize=event.y-68 $
    else $
    widget_control,state.wids.wDr,scr_xsize=event.x-5,scr_ysize=event.y-102
    uvalue = ''
    if  (n_elements(set1) ne 1) then begin
	xaid_mu2chi_plot,event.top
    endif
    END
  'Cursor': BEGIN
     Widget_Control,event.top,Get_UValue=state
     wset,state.windownum
     !x=state.x
     !y=state.y
     ; 
     ; cursor coordinates
     ;
     coords_d = [event.x,event.y]
     if n_elements(*(state.ptr)) gt 1 then $
       coords = convert_coord(event.x,event.y,/device,/to_data) else $
       coords = coords_d
     widget_control,state.XY,SET_VALUE='X: '+$
       strcompress(coords(0),/REM)+'   Y: '+$
       strcompress(coords(1),/REM) ;+ $
;' Zoom_Sel: '+StrCompress(state.zoom_Sel,/Rem)+ $
;' Knots: '+StrCompress(state.GetListOfKnots,/Rem)
     ;
     ; zoom
     ;
     if event.type EQ 0 AND event.press EQ 1 then begin 
       ;initialize zoom origin
       state.zoom_sel=1
       for ii=0,1 do state.zoomxy(ii)=coords_d(ii)
       Widget_Control,event.top,Set_UValue=state
     endif
     if state.zoom_sel EQ 1 then begin ; zoom rectangle
       distx=abs(coords_d(0)-state.zoomxy(0))
       disty= abs(coords_d(1)-state.zoomxy(1))
       if distx NE 0.0 and disty NE 0.0 then begin
         tvbox,min([coords_d(0),state.zoomxy(0)]), $
         min([coords_d(1),state.zoomxy(1)]), $
         distx,disty,-2
       endif
     endif
     if ((event.type EQ 1) AND (state.zoom_sel EQ 1) AND $
	 (event.release EQ 1) ) then begin 
       ;end zoom region
       state.zoom_sel=0
       coords0 = convert_coord(state.zoomxy(0),state.zoomxy(1),/device,/to_data)
       x0 = min([coords0(0),coords(0)])
       y0 = min([coords0(1),coords(1)])
       distx=abs(coords0(0)-coords(0))
       disty= abs(coords0(1)-coords(1))
       xrange = [x0,x0+distx]
       yrange = [y0,y0+disty]
       state.x.range=xrange
       state.y.range=yrange
       state.zoom_Sel=0
       Widget_Control,event.top,Set_UValue=state
       xaid_mu2chi_plot,event.top ; ,xrange=xrange,yrange=yrange
       ;plotfile,state.handle,xrange=xrange,yrange=yrange
       ;state.x=!x
       ;state.y=!y
       ;;state.zoomxy=coords
     endif
     ;
     ; list of knots
     ;
     if state.getListOfKnots EQ 1 then begin ; loading list of knots
       IF event.press EQ 2 OR event.press EQ 4 THEN BEGIN
         IF event.press EQ 2 THEN BEGIN
	   print,'CENTER BUTTON',coords
           state.GetListOfKnots=0
	   Widget_Control,event.top,Set_UValue=state
         ENDIF
         IF event.press EQ 4 THEN BEGIN
	   print,'RIGHT BUTTON',coords
         ENDIF

         ; update list
         Widget_Control,state.wids.knots,Get_Value=knots
         knots = StrCompress(knots,/Rem)
         IF StrMid(knots,0,/Rev) EQ ']' THEN BEGIN
		tmp='['+String(coords[0],Format='(F5.2)')
         ENDIF ELSE BEGIN
		tmp=knots+','+String(coords[0],Format='(F5.2)')
         ENDELSE
         Widget_Control,state.wids.knots,Set_Value=tmp

         IF event.press EQ 2 THEN BEGIN  ; finish the list of knots
	    tmp=tmp+']'
            Widget_Control,state.wids.knots,Set_Value=tmp
	    spl_n  = n_elements(strsplit(tmp,','))-1>1
            Widget_Control,state.wids.npol,Set_Value=StrCompress(spl_n,/Rem)
            Widget_Control,state.wids.degree,Set_Value= $
		vect2string(Replicate(2,spl_n))

            wait,0.5
            xaid_mu2chi_plot,event.top
         ENDIF
         
       ENDIF

     endif
    END
    'Execute': BEGIN
      widget_control,event.id,get_value=tmp
      command = tmp
      message,/info,'Executing: '+command
      itmp = execute(command)
    END
   '':
   else:
ENDCASE

END ; xaid_mu2chi_event

;=========================================================================

PRO xaid_mu2chi, setin, GROUP=Group, NODRAW=nodraw, NO_BLOCK=no_block, $
  deliaid=deliaid,WSize=wSize,parent=parent

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(Dialog_Parent=group,$
	/Error,'XAID_MU2CHI: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   return
endif

IF N_Elements(no_block) EQ 0 THEN no_block=1

;
; color setting
;
Device,Get_Decomposed=usingDecomposed
IF usingDecomposed EQ 1 THEN BEGIN
  itmp = Dialog_Message(/Question, $
     ['XAID_MU2CHI does not work properly with decomposed colors',$
     'Turn decomposed colors off?'])
  IF itmp EQ 'Yes' THEN Device,Decomposed=0  
ENDIF
Tek_Color

;
; default inputs
;
str = xaid_defaults('xaid_mu2chi')

IF Keyword_Set(setin) THEN BEGIN
  if type(setin) EQ 7 then BEGIN
     str.file=setin
  ENDIF ELSE str.file='<from variable>'
ENDIF
IF N_ELEMENTS(Group) EQ 0 THEN GROUP=0

;
; define widgets
;


wbase = widget_base(/Row,TITLE='XAID_Mu2Chi '+xaid_mu2chi_version(),$
  MBAR=wMenuBar, $ ; /TLB_SIZE_EVENTS, $
  /TLB_KILL_REQUEST_EVENTS, $
  Event_Pro='xaid_mu2chi_quit',UVALUE='Resize')

;Menu bar
wFileMenu =  WIDGET_BUTTON(wMenuBar, VALUE='File', /MENU)
  wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Save to File', Uvalue= 'Save File')
  wtmp1=WIDGET_BUTTON(wFileMenu, VALUE='Input parameters',/Menu)
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Load from file', $
	Uvalue= 'FileInputs')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Save file', Uvalue= 'FileInputs')
    wtmp=WIDGET_BUTTON(wtmp1, VALUE='Save as default', Uvalue= 'FileInputs')
  wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Print', Uvalue= 'Print')
  wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Preferences...', Uvalue='Preferences')
  wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Quit', Uvalue= 'Quit',/Separator)
;wViewMenu =  WIDGET_BUTTON(wMenuBar, VALUE='View', /MENU)
;  wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Refresh', Uvalue= 'Plot')
;  wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Raw Data', Uvalue= 'Data')
wOperation1Menu =  WIDGET_BUTTON(wMenuBar, VALUE='Raw_Data', /MENU)
  wtmp=WIDGET_BUTTON(wOperation1Menu, VALUE='Cut', Uvalue= 'Cut')
  wtmp=WIDGET_BUTTON(wOperation1Menu, VALUE='derivatives', $
	Uvalue= 'derivatives')
  wtmp=WIDGET_BUTTON(wOperation1Menu, VALUE='Interpolation', $
	Uvalue= 'Interp')
  wtmp=WIDGET_BUTTON(wOperation1Menu, VALUE='Generic Operation', $
	Uvalue= 'Oper')
wTools =  WIDGET_BUTTON(wMenuBar, VALUE='Tools', /MENU)
  wtmp=WIDGET_BUTTON(wTools, VALUE='xaid_mu2chi', Uvalue= 'Execute')
  wtmp=WIDGET_BUTTON(wTools, VALUE='xaid', Uvalue= 'Execute')
  wtmp=WIDGET_BUTTON(wTools, VALUE='xplot', Uvalue= 'Execute')
  wtmp=WIDGET_BUTTON(wTools, VALUE='xop', Uvalue= 'Execute')

wHelpMenu = WIDGET_BUTTON(wMenuBar, VALUE='Help', /HELP)
   wHelptmp = WIDGET_BUTTON(wHelpMenu, VALUE='About XAID', UVALUE='About')
   wHelptmp = WIDGET_BUTTON(wHelpMenu, VALUE='xaid_mu2chi', UVALUE='Help')

;
; Left Panel
;
wLeftPanel=Widget_Base(wBase,/Frame,/Column)

  wtmp = widget_label(wLeftPanel,/Align_Center,$
    Value='Parameters to calculate EXAFS signal')
  wtmp = widget_label(wLeftPanel,/Align_Left,$
    Value=' ')
  wfile = cw_pickfile(wLeftPanel,Value=str.file,UValue='Load File', $
	/Button_Last,Title='Raw Data File: ',XSize=35)

  ; 
  ; pre-edge
  ;
  wtmp0 = widget_base(wLeftPanel,/Col,/Frame)
  wtmp = widget_label(wtmp0,/Align_Left,Value=$
      '1) PreEdge Linear Fit (Does not apply if Min=Max): ')

  wtmp1 = widget_base(wtmp0,/Row)
      wPreMin = cw_field(wtmp1,Title=' E Min:',Value=str.preMin, $
	/Float,XSize=10,UValue='Plot',/Return_Events)
      wPreMax = cw_field(wtmp1,Title=' E Max:',Value=str.preMax, $
	/Float,XSize=10,UValue='Plot',/Return_Events)
      wtmp = Widget_Button(wtmp1,Value='Get...',UValue='Linear Fit')


  ; 
  ; E0 and Jump
  ;
  wtmp0 = widget_base(wLeftPanel,/Col,/Frame)
  wtmp = widget_label(wtmp0,/Align_Left,Value=$
      '2) Eo and Jump (Does not apply if zero): ')
  wtmp1 = widget_base(wtmp0,/Row)
    wE0 = cw_field(wtmp1,Title='Eo:',Value=str.e0, $
	/Float,XSize=10,UValue='Plot',/Return_Events)
    wtmp = Widget_Button(wtmp1,Value='Get...',UValue='Get Eo')
    wJump = cw_field(wtmp1,Title='    Jump: ',Value=str.jump, $
	/Float, XSize=10,UValue='Plot',/Return_Events)
    wtmp = Widget_Button(wtmp1,Value='Get...',UValue='Jump')


  ; 
  ; Post Edge
  ;
  wtmp0 = widget_base(wLeftPanel,/Col,/Frame)
    wtmp = widget_label(wtmp0,/Align_Left,Value=$
      '3) Post Edge Multi-spline: ')
    wtmp1 = widget_base(wtmp0,/Row)
      wNPol = cw_field(wtmp1,Title='N polynomials: ',Value=str.npol, $
	/Int, XSize=10,UValue='Plot',/Return_Events)
      wtmp = widget_label(wtmp1,/Align_Left,Value='Degrees:')
      wDegree = widget_text(wtmp1,Value=str.degree, $
	XSize=15,/Edit,UValue='Plot')
    wtmp1 = widget_base(wtmp0,/Row)
      wtmp = widget_label(wtmp1,/Align_Left,Value='Knots (in k):')
      wknots = widget_text(wtmp1,Value=str.knots, $
	XSize=30,/Edit,UValue='Plot')
      wtmp = widget_button(wtmp1,Value='Get...',UValue='PostEdge')
  ; 
  ; Extraction
  ;
  wtmp0 = widget_base(wLeftPanel,/Col,/Frame)
    wtmp = widget_label(wtmp0,/Align_Left,Value=$
      '4) EXAFS extraction: ')
;    wtmp = Widget_Label(wtmp0,/Align_Left,Value=$
;	' Chi interval for plot (kmin=kmax for default): ')
    wtmp1 = Widget_Base(wtmp0,/Row)
;      wKKMin = cw_field(wtmp1,Title= 'k min: ', $
;	Value=str.kkmin,/Float,XSize=10,UValue='Plot')
;      wKKMax = cw_field(wtmp1,Title='k max: ',$
;  	Value=str.kkmax,/Float,XSize=10,UValue='Plot')
    wExtraction = CW_DROPLIST(wtmp0,Title='Type: ', $
	VALUE=str.extraction,UVALUE='Plot')

;---------------------
;
; Right Panel (plot)
;

if not(keyword_set(nodraw)) then $
  IF N_Elements(wSize) EQ 0 THEN wSize=[500,400]

wRightPanel = widget_Base(wBase, /Col)
  wDr = widget_draw(wRightPanel, XSIZE=wSize[0], YSIZE=wSize[1], RETAIN=2, $
	MOTION_EVENTS=1,BUTTON_EVENTS=1,UVALUE='Cursor')

  BASE20 = WIDGET_BASE(wRightPanel, ROW=1, MAP=1, UVALUE='BASE20')
  wtmp = Widget_Button(BASE20,Value='Refresh',UValue='Plot')
  wtmp = Widget_Button(BASE20,Value='Zoom Window',UValue='Zoom')
  wtmp = Widget_Button(BASE20,Value='XPlot',UValue='XPlot')
  LABEL22 = WIDGET_LABEL( BASE20, UVALUE='LABEL22', VALUE=$
	'X:        Y:        ',/DYNAMIC_RESIZE)

  wButtons = 0L
  wtmp0 = Widget_Base(wRightPanel,/Row)
    wtmp = Widget_Label(wtmp0,Value='X value: ')
    wAbscissas = CW_DROPLIST(wtmp0,VALUE=['0', $
	'E [eV]','E-Eo','K [A^-1]', 'R'],UVALUE='Plot')
    wtmp = Widget_Label(wtmp0,Value='Y value: ')
    wOrdinates = CW_DROPLIST(wtmp0,VALUE=['0', $
	'Mu','Mu & PreEdge','Mu-PreEdge', 'Mu Normalized', $
	'Mu-PreEdge & Spline','Chi','Chi*k','Chi*k^2','Chi*k^3'], $
	UVALUE='Plot')


WIDGET_CONTROL, wbase, /REALIZE

npp = n_params()
if npp EQ 0 then begin
     ptr = Ptr_New(0)
endif else begin
     ptr = Ptr_New(setin)
endelse
if not(keyword_set(deliaId)) then deliaId=0L
WIDGET_CONTROL, wDr, GET_VALUE=drId

wids = { Dr:wdr, delia:deliaId, $
	abscissas:wAbscissas, ordinates:wOrdinates, $
	LeftPanel:wLeftPanel, file:wFile, $
	preMin:wPreMin, preMax:wPreMax, e0:we0, Jump:wJump, $
	NPol:wNPol, Degree:wDegree, Knots:wKnots, $
;	kkMin:wKKMin, kkMax:wkkMax, $
	extraction:wExtraction } 

state = { XY:LABEL22, Windownum:DrId, $
	ptr:ptr, str:str,  $
        x:!x, y:!y, wids:wids, zoom_sel:0, zoomxy:[0.0,0.0], $
	getListOfKnots:0, no_block:no_block}

parent=wbase
widget_control,wbase,set_uvalue=state
IF Keyword_Set(setin) THEN xaid_mu2chi_loadfile,wbase,setin
xmanager, 'xaid_mu2chi', wbase, GROUP=group, NO_BLOCK=no_block

END
