;=========================== copyright =========================================
;                       XAID (XAfs with IDl) package.                          ;
;     Copyright European Synchrotron Radiation Facility (1993-1996).           ;
; This software may be used, copied, or redistributed as long as it is not     ;
; sold and this copyright notice is reproduced on each copy made.              ;
; The software is provided as is without any express or implied warranties     ;
; whatsoever.                                                                  ;
; Other limitations apply as described in the file README.copyright.           ;
;                                                                              ;
; REFERENCE: "XAID: A package for XAFS data analysis based on IDL"             ;
;       Manuel Sanchez del Rio. To be published at the proceedings of          ;
;       the 9th International Conference on X-ray Absorption Fine Structure    ;
;       26th - 30th August 1996,  E S R F - Grenoble  France                   ;
;==============================================================================;
PRO BFTR,fourier,backftr,NPOINT=npoint,KRANGE=krange
;+
; NAME:
;	BFTR
;
; PURPOSE:
;	This procedure calculates the Back Fourier Transform of a set
;
; CATEGORY:
;	XAID xafs data analysis package. 
;
; CALLING SEQUENCE:
;
;	BFTR, fourier, backftr
;
; INPUTS:
;	fourier:  a 4 col set with r,modulus,real and imaginary part
;		of a Fourier Transform of an Exafs spectum, as produced
;		by FTR or FASTFTR procedures
;	backftr: a 4-column set with the  conjugated variable k (col 0), 
;	the real part (col 1), the modulus (col 2) and phase (col 3)
;	of the Back Fourier Transform
;
; OPTIONAL INPUTS:
;	
; KEYWORD PARAMETERS:
;	KRANGE=[kmin,kmax] : range of the conjugated variable k for 
;		the anti-transformation (default = [2.,15.])
;	NPOINT= number of points of the conjugated variable for the output
;		(default = 512)
;
; OUTPUTS:
;	This procedure returns a 4-columns array (backftr) with
;	the conjugated variable (k) in column 0, the real part of the
;	BFT in col 1, the modulus in col 2 and the phase in col 3.
;
; OPTIONAL OUTPUTS:
;
; COMMON BLOCKS:
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	Reciprocal to FTR
;
; EXAMPLE:
;		BFTR,fourier,backftr,krange=[2,12]
;
; MODIFICATION HISTORY:
; 	Written by:	Manuel Sanchez del Rio. ESRF
;	March, 1993	
;-
;
on_error,2
;
if not(keyword_set(npoint)) then npoint = 512
if not(keyword_set(krange)) then begin
  smin = 2.0
  smax = 15.0
endif else begin
  smin = float(krange(0))
  smax = float(krange(1))
endelse
;
tr=fourier(2,*)
ti=fourier(3,*)*(-1.)
tar=fourier(0,*)
nr=n_elements(tar)
;print,' nr good points in r ',nr
;
dr = tar(1)-tar(0)
dr2= tar(2)-tar(1)
diff=abs(dr2 - dr2)
if (diff gt 1e-4) then begin
  print,' Error from BFTR: ftr array contains bad points -> Abort'
  return
endif
;
fmul=2.             ; multiplicative factor to take into account the fact
                    ; that our ftr only contains half part of the total ftr
;
dk = (smax - smin) / (npoint-1)
;
kk=fltarr(npoint)
k1=fltarr(npoint)
kk(0)=-1.
k1(0)=1.
for i=1,npoint-1 do kk(i)=kk(i-1)*(-1.)
for i=1,npoint-1 do k1(i)=kk(i)+3.
tk=fltarr(npoint)
br=fltarr(npoint)
bi=fltarr(npoint)
for i=0,npoint-1 do tk(i)=dk*i+smin
;
for i=0,npoint-1 do begin
  tk_val=tk(i)
  a=(tar*2.*tk_val)*(-1.)
  ca=cos(a)
  sa=sin(a)
  c=tr*ca-ti*sa
  s=tr*sa+ti*ca
  c1 = total(c*k1)
  s1 = total(s*k1)
  br(i)=fmul*(c1-c(nr-1))*dr/3./sqrt(2.*!pi)
  bi(i)=fmul*(s1-s(nr-1))*dr/3./sqrt(2.*!pi)
endfor
;
; create the output set
;
plot,tk,br
pause
plot,tk,bi
pause
backftr=fltarr(4,npoint)
backftr(0,*)=tk                  ; the conjugated variable (k [A^-1])
backftr(1,*)=br                  ; the real part of backftr or atra
backftr(2,*)= sqrt(br^2 + bi^2)  ; the modulus of backftr
backftr(3,*)=  atan(br/bi)       ; the phase 
;
; now correct the phase to make it continuous
;
for i=1,npoint-1 do begin
  if ( abs(backftr(3,i)-backftr(3,i-1)) gt 0.9*!pi) then begin
    for j=i,npoint-1 do backftr(3,j)=backftr(3,j)+!pi
  endif
endfor
;
return
end
