#! /bin/csh
#
# GO: the master SHADOW driver.
#
# Author: Mumit khan <khan@xraylith.wisc.edu>
#
# Source: src/drivers/GO
#
# ----------------------------------------------
#               SHADOW
#    Center for X-ray Lithography
#  University of Wisconsin-Madison
#  3731 Schneider Dr., Stoughton, WI, 53589
# ------------------------------------------------
#
# Log: GO
# Revision 1.3  1991/07/06  20:09:17  khan
# Grenoble and after. Minor changes
#
# Revision 1.2  90/10/16  22:14:09  khan
# Added umask 002 so group can read-write the files created.
# 
# Revision 1.1  90/07/23  21:04:01  khan
# Initial revision
# 
#

#
# Make sure the environment is correctly set up.
#

# This is for The CXrL Shadow developers only. Users can safely delete this.
umask 002

onintr done

if (! $?SHADOW_ROOT) then
    echo "Must have SHADOW_ROOT defined in the environment"
    echo "source .shadowrc before running shadow."
    exit 1
endif

if (! $?SHADOW_START_DIR) setenv SHADOW_START_DIR './'
if (! $?SHADOW_DATA_DIR)  setenv SHADOW_DATA_DIR $SHADOW_ROOT/data
if (! $?SHADOW_SR_DIR)  setenv SHADOW_SR_DIR $SHADOW_ROOT/data


#
# the MENU program dumps the name of the SOURCE namelist file defined by 
# variable SHADOW_MENU_ENV. This is the replacement strategy for the
# VMS version setting the SOURCE_STAT symbol in the menu program.
#
if (! $?SHADOW_MENU_ENV) then
    setenv SHADOW_MENU_ENV "/tmp/shadow_menu_${USER}"
endif

#
# define the programs to run.
#
set MENU	 = $SHADOW_ROOT/bin/menu
set TRACE 	 = $SHADOW_ROOT/bin/trace
set SOURCE 	 = $SHADOW_ROOT/bin/source
set GEN_SOURCE 	 = $SHADOW_ROOT/bin/gen_source
set INPUT_SOURCE = $SHADOW_ROOT/bin/input_source

set HELP_PROG =  \
    '/u4/src/shadow/src/help/help -d /u4/src/shadow/src/help/SHADOW_HELP'

# define utility programs.
set CLEAR_SCRN 	= clear
set CAT		= cat

# define utility files that are printed to the screen.
set BANNER = $SHADOW_DATA_DIR/banner.txt
set GO     = $SHADOW_DATA_DIR/go.txt
set EXIT   = $SHADOW_DATA_DIR/exit.txt

# 
# The main part begins here.
#

#
# Turn globbing off so we don't make a mess with metacharacters.
#
set noglob

#
# Check for a startup program on the command line. If exists, bypass
# printing the initial screen and go for the guts directly.
#
# Note all the trouble we go thru to make sure that the input line only
# consists of one word. Else the 'switch' fails miserably. Must be a
# better/more elegant way!
#

$CLEAR_SCRN
if ($#argv == 0) then
    cat $SHADOW_DATA_DIR/banner.txt
    echo -n "Hit <ret> when ready: " ; set dummy = $<
    $CLEAR_SCRN
    cat $SHADOW_DATA_DIR/go.txt
else
    set line = $1; goto start
endif

#
# This is the top of the main loop. All road lead to the BEGIN label. If
# a command line program is supplied, we bypass the first query, and go 
# to START.
#

begin:
onintr done
echo -n "Shadow:: " ; set line = $<

start:
set action = ($line)
if ($#action == 0) goto begin

onintr begin
switch ($action[1])
    
    case [eE][xX][iI][tT]:
	goto done
	set all_done
	breaksw

    case "[hH][eE][lL][pP]":
	echo ""
	$HELP_PROG
	echo ""
	breaksw

    
    case "[mM][eE][nN][uU]":
	echo ""
	echo "Prepare MENU ..."
	echo ""
	$MENU
	breaksw

    case "[tT][rR][aA][cC][eE]":
	echo ""
	echo "Ray Tracing Selected. Begin procedure."
	echo ""
trace_mode:
	echo -n "Mode selected [ ? <ret> for HELP ] ? "; set line = $<
	set mode = ($line)
	if ($#mode == 0) goto trace_help 
	switch ($mode[1])
	    case '[mM][eE][nN][uU]':
		unset skip_menu
		echo "MENU selected."
		if ( -f systemfile.dat) then
		    echo -n "Execute existing SYSTEMFILE ? : "
		    set line = $<
		    if ($line[1] == 'y' || $line[1] == 'Y') set skip_menu
		endif
		if (! $?skip_menu) $MENU
		breaksw

	    case '[Pp][rR][oO][mM][pP][tT]':
	    case '[bB][aA][tT][cC][hH]':
		breaksw
	    
	    case '?':
		goto trace_help
	    default:
		echo "What ? Unrecognized input."
		echo " "
trace_help:
		echo "Enter:"
		echo "MENU:		for screen-driven menus"
		echo "PROMPT:		prompted session"
		echo "BATCH:		file-oriented session"
		echo ""
		goto trace_mode
		breaksw
	endsw
	$TRACE -m $mode[1]
	echo ""
	echo "Procedure completed."
	echo ""
	breaksw

    case "[sS][oO][uU][rR][cC][eE]":
	echo ""
	echo "SOURCE selected. Begin procedure."
	echo ""

source_mode:

	echo "This procedure generates a SOURCE for SHADOW."
	echo -n "Mode selected [ ? <ret> for HELP ] ? "; set line = $<
	set mode = ($line)
	if ($#mode == 0) goto source_help 
	set FNAME = ('start.00')
	switch ($mode[1])
	    case '[Pp][rR][oO][mM][pP][tT]':
		$INPUT_SOURCE
		breaksw
	    case '[bB][aA][tT][cC][hH]':
readfile:
		echo -n "File containing source description [ start.00 ] ? "
		set line = $<
		set FNAME = ($line)
		if ($#FNAME == 0) goto readfile
		breaksw
	    
	    case '[mM][eE][nN][uU]':
		unset menu_error
		/bin/rm -f ${SHADOW_MENU_ENV}
		if ( -e ${SHADOW_MENU_ENV} && ! -w ${SHADOW_MENU_ENV} ) then
		    echo ""
		    echo \
		    "GO Error: cannot delete temporary file ${SHADOW_MENU_ENV}"
		    /bin/ls -l ${SHADOW_MENU_ENV}
		    echo \
		    "    Please have your sysadmin delete it, or"
		    echo \
		    "    change SHADOW_MENU_ENV environment variable to point"
		    echo \
		    "    to another file name, eg., ./shadow_menu_env"
		    echo \
		    "    Exiting MENU ..."
		    echo ""
		    goto trouble
		endif
		$MENU
		if (! -f $SHADOW_MENU_ENV) then
		    echo ""
		    echo "	Hmmm! Did you forget to SAVE SOURCE in MENU?"
		    set menu_error 
		    breaksw
		endif
		set FNAME = (`cat $SHADOW_MENU_ENV`)
		if (! -f $FNAME[1]) then
		    echo ""
		    echo "	Hmmm! MENU didn't create source namelist file"
		    set menu_error 
		    breaksw
		endif
		breaksw
	    
	    case '?':
		goto source_help
	    default:
		echo "What ? Unrecognized input."
		echo " "
source_help:
		echo "Enter:"
		echo "MENU:		for screen-driven menus"
		echo "PROMPT:		prompted session"
		echo "BATCH:		file-oriented session"
		echo ""
		goto source_mode
		breaksw
	endsw
	if ($?menu_error) then
	    echo "	Warning! SOURCE not created"
	    echo ""
	    echo -n "Hit <ret> to continue: " ; set dummy = $<
	    $CLEAR_SCRN
	    cat $SHADOW_DATA_DIR/go.txt
	    echo ""
	    breaksw
	endif
	echo "Calling gen_source with $GEN_SOURCE $FNAME[1]"
	$GEN_SOURCE $FNAME[1]
	echo ""
	echo "SOURCE procedure completed."
	echo ""
	breaksw

    case "[sS][eE][tT][dD][iI][rR]":
	echo ""
	echo "SETDIR not implemented yet"
	echo ""
	echo -n "Hit <ret> to continue: " ; set dummy = $<
	$CLEAR_SCRN
	cat $SHADOW_DATA_DIR/go.txt
	breaksw

    default:
	echo " ? Command not recognized ?"
	echo ""
        echo " Legal commands are: "
        echo "EXIT     MENU	TRACE      SOURCE     HELP    SETDIR"
	echo ""
	breaksw
endsw

if (! $?all_done) goto begin
unset noglob

done:
$CLEAR_SCRN
exit 0


trouble:
exit 1
