function READSH, filesha,NFILES=nfiles,SEPARATOR=separator, $
  NOLOST=nolost,APPEND=append,GROUP=group,VERBOSE=VERBOSE
;+
; NAME:
;	READSH
; PURPOSE:
;	to load in an IDL structure a SHADOW  file. It can also be
;	used to write a new idl/shadow structure with fewer rays by
;	using the NOLOST keyword, or to append data into a shadow/
;	idl structure by using the APPEND keyword.
; CATEGORY:
;	SHADOW's utilities.
; CALLING SEQUENCE:
;
;	Result = readsh('shadow_file_name')
;
;	with that, IDL stores in a structure named Result (or whatever you
;	want) the shadow_file_name (for instance, begin.dat).
;
;   For help, type
;
;	HELP,/ST,Result
; INPUTS:
;	SHADOW_FILE_NAME name of the input file (or a idl/shadow
;	  structure; see NOLOST keyword)
; KEYWORD PARAMETERS:
;	NFILES = number of files to be read. Optionally READSH can read
;	a number of files from an iterative shadow run. These files must
;	have the extension -1,-2...-n (i.e.  star.01-1,star.01-2,star.01-3)
;	where - acts as a "seperator". The separator can be any sign 
;	defined in the SEPARATOR keyword
;	SEPARATOR = sign that acts a separator between the file name and
;	the run number (default='-')
;       NOLOST = consider losses in the returned shadow/idl structure
;            0 all rays (default option)
;            1 exclude losses;
;            2 only losses;
;       APPEND Set this keyword to a shadow file or idl shadow structure
;	     to be appended. Thus Result=InFile+AppendedFile.
;	GROUP The parent id for the caller widget (used to positioning
;		error and information windows).
;       VERBOSE Set this keyword to print some info (Default: verbose=1)
;	
; OUTPUTS:
;	A1 an IDL structure 
; COMMON BLOCKS:
;	None.
; SIDE EFFECTS:
;	None
; RESTRICTIONS:
;	None.
; PROCEDURE:
; 	
; EXAMPLES:
;	1) read a shadow file and write it into a shadow/idl structure:
;		a=readsh('begin.dat')
;	2) same but using only good rays:
;		a=readsh('star.01',/nolost)
;	3) save 8 files (star.01-1, ..., star.01-8) in a shadow/idl str:
;		a=readsh('star.01',nfiles=8)
;	4) remove the bad rays from a shadow/idl structure
;		a=readsh(a,/nolost)
;	5) read the file 'star.05' five times and place the result in a:
;		a=1 ; initialize variable
;		for i=1,5 do a=readsh('star.05', append=a)
; MODIFICATION HISTORY:
;	M. Sanchez del Rio. ESRF. Grenoble, May 1991
;        92-07-09 MSR includes a 'spawn' call in order to run sh-idl from idl
;        92-10-17 MSR includes NFILES option
;        93-07-09 MSR includes SEPARATOR keyword
;        94-06-09 MSR if the input is not a file, but a SHADOW-IDL
;		structure, then returns this (the same) structure. This check 
;		is done here, and can therefore be removed from the other 
;		routines.
;        94-11-29 MSR adds the /NOLINK flag, which is passed to getrays
;        95-03-03 MSR removes the /NOLINK flag, unnecessary with the 
;		new getrays function
;	 96-06-18 MSR adds NOLOST keyword, updates doc, small changes.
;	 98-04-02 MSR adds GROUP keyword.
;	 2009-01-12 MSR adds VERBOSE keyword.
;-
on_error,2
;
; reads a set of files when selected
;
;
; return when the input is already an idl structure
;

IF N_Elements(verbose) EQ 0 THEN verbose=1

input_type = (size(filesha)) ((size(filesha)) (0)+1)
if input_type eq 8 then begin
  if keyword_set(nolost) or  keyword_set(append) then  begin
    ncol = filesha.ncol
    npoint = filesha.npoint
    rays = filesha.ray
    name = filesha.name
    goto,COLLAPSE
  endif else return,filesha
endif else name = filesha
;
if keyword_set(nfiles) then begin
;
 if not(keyword_set(separator)) then separator='-'
 rays1 = getrays(filesha+separator+'1',GROUP=group)
 if not(keyword_set(rays1)) then return,0
 sizevec1 = size(rays1)
 ierr1 = 0
 ncol1 = sizevec1(1)
 npoint1 = sizevec1(2)
 IF verbose THEN message,/INFO, $
       '>>file '+filesha+separator+'1'+' read OK: '+strtrim(ncol1,2)+$
       ' columns '+strtrim(npoint1,2)+' rays.'
 rays=dblarr(ncol1,npoint1*nfiles)
 rays(*,0:npoint1-1)=rays1
;
 for i=2,nfiles do begin
   name=filesha+separator+strtrim(i,2) 
   rays_n=getrays(name,GROUP=group)
   if not(keyword_set(rays_n)) then return,0
   sizevec=size(rays_n)
   ierr = 0
   ncol = sizevec(1)
   npoint = sizevec(2)
   if ncol ne ncol1 or npoint ne npoint1 then $
            message, 'Error: incompatible file type'
   IF verbose THEN message,/INFO, $
         '>>file '+name+' read OK: '+strtrim(ncol,2)+$
         ' columns '+strtrim(npoint,2)+' rays.'
   ;istart=long((i-1))*5000
   istart=long((i-1))*npoint1
   iend  = istart+npoint-1
   rays(*,istart:iend) = rays_n
 endfor
 npoint = npoint1*nfiles
;
; normal case: only one file
;
endif else begin
 rays = getrays(filesha,GROUP=group)
 if not(keyword_set(rays)) then return,0
 sizevec = size(rays)
 ierr = 0 
 ncol = sizevec(1)
 npoint = sizevec(2)
endelse

;
;       redefine arrays when excluding losses
;
COLLAPSE:
ierr = 0
if keyword_set(nolost) then begin
  flg = rays(9,*)

  case nolost of
  1: keep = where(flg>0)
  2: keep = where(flg<0)
  else: message,'nolost flag value not valid.'
  endcase
  if n_elements(keep) EQ 1 then begin ; check for no rays
    if keep(0) EQ -1 then begin
      rays=0 
      ncol = 0
      IF verbose THEN message,'Using ONLY good or lost rays(0)',/INFO
    endif else begin
      rays = rays(*,keep)
      IF verbose THEN message,'Using ONLY good or lost rays(1)',/INFO
    endelse
  endif else begin
    rays = rays(*,keep)
    IF verbose THEN message,/INFO,'Using ONLY good or lost rays('+ $
    strcompress(n_elements(keep),/rem)+')'
  endelse
  npoint = n_elements(rays(0,*))
  if npoint EQ 1 and n_elements(rays) EQ 1 then npoint = 0
endif

;
; create structure
;
if not(keyword_set(append)) then begin
  sh_str = {  name:name, ncol: ncol, npoint: npoint, ray:rays }  
endif else begin
  if type(append) NE 8 then begin 
    sh_str = {  name:name, ncol: ncol, npoint: npoint, ray:rays }
    goto,OUT
  endif
  sh_str1 = readsh(append)
    
  if sh_str1.ncol NE ncol then $
	message,'Incompatible number of colums to append.'
  if npoint GT 0 then sh_str = {  name:sh_str1.name, ncol: sh_str1.ncol, $
	npoint: sh_str1.npoint + npoint, $
	ray:transpose([transpose(sh_str1.ray),transpose(rays)])} $
	else sh_str = sh_str1
endelse
OUT:
IF verbose THEN $
message,/INFO,strcompress(sh_str.ncol,/rem)+' cols, '+ $
              strcompress(sh_str.npoint,/rem)+' points, ' + $
              strcompress(ierr,/rem)+' =error flag'
return, sh_str
end
