#! /bin/sh
#
# excon: simple wrapper for actual EXCON executable.
#
# Author: Mumit Khan <khan@xraylith.wisc.edu>
#
# Source: src/utils/excon/excon
#
# ----------------------------------------------
#	       SHADOW
#    Center for X-ray Lithography
#  University of Wisconsin-Madison
#  3731 Schneider Dr., Stoughton, WI, 53589
# ------------------------------------------------
#
# Log: excon
# Revision 1.2  1994/01/20  20:28:08  khan
# fixed multiple ';;' typos in case statements.
#
# Revision 1.1  1994/01/13  21:17:15  khan
# Initial revision
#
#
#

# environment variables that this shell script sets/changes:
# export VARIABLES.

# error message function
error () {
    echo "`basename $0`: $@" 1>&2
    exit 1
}

# usage message function
usage () {
    if [ ! -z "$@" ]; then
	echo "`basename $0`: $@" 1>&2
    fi
    echo "Usage: `basename $0` [-t toolsfile] [-e expfile] [-o report] [-help] [-debug]" 1>&2
    exit 1
}


# don't bother to continue if SHADOW environmet isn't set up properly.
if [ -z "${SHADOW_ROOT}" ]; then
	error \
"SHADOW environment is not properly setup for this script
to run.  Set the environment variables via the \`\`.shadowrc'' shell script 
provided with SHADOW distribution."
fi

if [ -z "${SHADOW_BIN}" ]; then
    SHADOW_BIN="${SHADOW_ROOT}/bin"
fi

if [ -z "${SHADOW_DATA_DIR}" ]; then
    SHADOW_DATA_DIR="${SHADOW_ROOT}/data"
fi

# Now all the programs that we need to run.
EXCON_REAL=${SHADOW_BIN}/excon-real

#
# set up the variables, including the ones from EXCON.
#
RETVAL=0			# return codes from programs.
TOOLSFILE=${SHADOW_DATA_DIR}/shadow.tools
EXPFILE=shadow.exp
REPORTFILE=table
DEBUG=

# Parse command line args.
while [ $# -gt 0 ]; do
    case "$1" in
	-toolsfile|-t)
	    #
  	    # this option overrides the default tools file in 
	    # ${SHADOW_DATA_DIR} and passes it on excon.real.
	    #
	    if [ $# -lt 2 ]; then
		usage "$1 option requires a tools file name"
	    fi
	    shift
	    TOOLSFILE="$1"
	    ;;
	-expfile|-e)
	    #
  	    # this option overrides the default experiment file in 
	    # current directory (shadow.exp) and passes it on excon.real.
	    #
	    if [ $# -lt 2 ]; then
		usage "$1 option requires a experiment file name"
	    fi
	    shift
	    EXPFILE="$1"
	    ;;
	-output|-o)
	    #
  	    # this option overrides the default report file prefix in 
	    # current directory (table) and passes it on excon.real.
	    #
	    if [ $# -lt 2 ]; then
		usage "$1 option requires a report file prefix name"
	    fi
	    shift
	    REPORTFILE="$1"
	    ;;
	-debug|-d)
	    set -x
	    DEBUG="-d"
	    ;;
	-help|-h)
	    usage ""
	    ;;
	*)
	    usage "Illegal command line option $1"
	    ;;
    esac
    shift
done

#
# check sanity and init state info
#

if [ ! -x "${EXCON_REAL}" ]; then
    error "EXCON executable ${EXCON_REAL} is missing or un-executable."
fi

if [ ! -r "${TOOLSFILE}" ]; then
    error "SHADOW Tools file ${TOOLSFILE} does not exist or unreadable."
fi

if [ ! -r "${EXPFILE}" ]; then
    error "Experiment file ${EXPFILE} does not exist or unreadable."
fi

#
# save the old reports (eg., table1 table2 ... etc) by appending .bak 
# extension.
#
OLDREPORTS="`ls ${REPORTFILE}[0-9]* 2>/dev/null`"
for i in $OLDREPORTS; do
    mv $i ${i}.bak
done

# 
# now simply run excon.real with all the parameters.
#

${EXCON_REAL} -t ${TOOLSFILE} -e ${EXPFILE} -o ${REPORTFILE} ${DEBUG}
RETVAL=$?
exit ${RETVAL}
