#! /bin/sh
#
# excon-source: EXCON wrapper for SHADOW GEN_SOURCE program.
#
# Author: Mumit Khan <khan@xraylith.wisc.edu>
#
# Source: src/utils/excon/excon-source
#
# ----------------------------------------------
#	       SHADOW
#    Center for X-ray Lithography
#  University of Wisconsin-Madison
#  3731 Schneider Dr., Stoughton, WI, 53589
# ------------------------------------------------
#
# Log: excon-source
# Revision 1.1  1993/12/15  15:34:36  khan
# Initial revision
#
#
#

# environment variables that this shell script sets/changes:
# export VARIABLES.

# error message function
error () {
    echo "`basename $0`: $@" 1>&2
    exit 1
}

# usage message function
usage () {
    if [ ! -z "$@" ]; then
	echo "`basename $0`: $@" 1>&2
    fi
    echo "Usage: `basename $0` -g G_FILE_NAME [-help] [-debug]" 1>&2
    exit 1
}


# don't bother to continue if SHADOW environmet isn't set up properly.
if [ -z "${SHADOW_ROOT}" ]; then
	error \
"SHADOW environment is not properly setup for this script
to run.  Set the environment variables via the \`\`.shadowrc'' shell script 
provided with SHADOW distribution."
fi

if [ -z "${SHADOW_BIN}" ]; then
    SHADOW_BIN="${SHADOW_ROOT}/bin"
fi


# Now all the programs that we need to run.
G_TO_NML=${SHADOW_BIN}/g-to-nml
GEN_SOURCE=${SHADOW_BIN}/gen_source

#
# set up the variables, including the ones from EXCON.
#
GFILE=				# name of gfile to read from.
LOGFILE=			# store output of program in this file
APPEND=0			# append or overwrite (default).
RETVAL=0			# return codes from programs.
STARTFILE=
ENDFILE=

# Parse command line args.
while [ $# -gt 0 ]; do
    case "$1" in
	-g)
	    if [ $# -lt 2 ]; then
		usage "$1 option requires a gfile name"
	    fi
	    shift
	    GFILE="$1"
	    ;;
	-logfile|-l)
	    #
  	    # this option creates a log file and redirects the programs'
	    # output to this file. 
	    #
	    # Note: Undocumented in usage() function.
	    #
	    if [ $# -lt 2 ]; then
		usage "$1 option requires a log file name"
	    fi
	    shift
	    LOGFILE="$1"
	    ;;
	-append|-a)
	    #
  	    # this option is only useful when logging is turned on. When
	    # specified, the output of child programs' is appended to the
	    # logfile if it already exists.
	    #
	    # Note: Undocumented in usage() function.
	    #
	    APPEND=1
	    ;;
	-debug|-d)
	    set -x
	    ;;
	-help|-h)
	    usage ""
	    ;;
	*)
	    usage "Illegal command line option $1"
	    ;;
    esac
    shift
done

#
# check sanity and init state info
#

if [ -z "${GFILE}" ]; then
    usage "No GFILE name supplied."
fi

if [ ${APPEND} -eq 0 ]; then
    rm -f ${LOGFILE} 2>/dev/null
fi

# 
# FUN begins.
#
# Some explanation on how the files are manipulated:
#
# blah blah blah (when I get some ``free'' time)
#

# FUNCTION to get a parameter value from a gfile.
getgparam() {
    echo `cat "$GFILE" | grep "$1" | awk -F= '{print $2}'`
}

STARTFILE="`getgparam startfile`"

ENDFILE=end.00
EXCON_ENDFILE="`getgparam endfile`"

BEGINFILE=begin.dat
EXCON_BEGINFILE="`getgparam file_source`"

unset getgparam				# not needed any more after this

#
# now run the programs. Notice that I don't really care if LOGFILE is set
# or not, since Bourne shell will not redirect to file with zero length
# name (which is the case when -logfile option is not specified by user).
#

# create the namelist file from supplied gfile if needed.
if test "$SHADOW_START_FORMAT" = "FORTRAN"; then
    ${G_TO_NML} -o ${STARTFILE} -t source ${GFILE} 1>&2 >> ${LOGFILE}
else
    STARTFILE=$GFILE
fi

#
# now run GEN_SOURCE with the created NAMELIST. If ${G_TO_NML} exits with
# an error, don't bother running GEN_SOURCE.
#

if [ $? -eq 0 ]; then		# why bother going on if already screwed!
    ${GEN_SOURCE} ${STARTFILE} 1>&2 >> ${LOGFILE}
fi
RETVAL=$?

if [ -f $BEGINFILE ]; then
    mv $BEGINFILE $EXCON_BEGINFILE
fi

if [ -f $ENDFILE ]; then
    mv $ENDFILE $EXCON_ENDFILE
fi

exit ${RETVAL}
