#! /bin/sh
#
# excon-seed: EXCON wrapper replacement for SHADOW SEED program.
#
# Author: Mumit Khan <khan@xraylith.wisc.edu>
#
# Source: src/utils/excon/excon-seed
#
# ----------------------------------------------
#	       SHADOW
#    Center for X-ray Lithography
#  University of Wisconsin-Madison
#  3731 Schneider Dr., Stoughton, WI, 53589
# ------------------------------------------------
#
# Log: excon-seed
# Revision 1.2  1993/12/21  16:35:19  khan
# Added [-e | -echo] command line option to conditionally echo the new
# seed to the tty as well. Seed is always written to the statefile.
# Also added sanity checks for GFILE variables.
#
# Revision 1.1  1993/12/19  20:49:37  khan
# Initial revision
#
#
#

# environment variables that this shell script sets/changes:
# export VARIABLES.

# error message function
error () {
    echo "`basename $0`: $@" 1>&2
    exit 1
}

# usage message function
usage () {
    if [ ! -z "$@" ]; then
	echo "`basename $0`: $@" 1>&2
    fi
    echo "Usage: `basename $0` -g G_FILE_NAME [-echo] [-help] [-debug]" 1>&2
    exit 1
}


# don't bother to continue if SHADOW environmet isn't set up properly.
if [ -z "${SHADOW_ROOT}" ]; then
	error \
"SHADOW environment is not properly setup for this script
to run.  Set the environment variables via the \`\`.shadowrc'' shell script 
provided with SHADOW distribution."
fi

# Now all the programs that we need to run.

#
# set up the variables, including the ones from EXCON.
#
GFILE=				# name of gfile to read from.
ISTAR1=				# this is the value that everyone wants!
RETVAL=0			# return codes from programs.
ITERATION=			# first iteration is always special
STATEFILE=seed.state		# this is where the last seed is stored 
NEWSEED=
DOECHO=0			# echo the seed to terminal as well?

# Parse command line args.
while [ $# -gt 0 ]; do
    case "$1" in
	-g)
	    if [ $# -lt 2 ]; then
		usage "$1 option requires a gfile name"
	    fi
	    shift
	    GFILE="$1"
	    ;;
	-echo|-e)
	    DOECHO=1
	    ;;
	-debug|-d)
	    set -x
	    ;;
	-help|-h)
	    usage ""
	    ;;
	*)
	    usage "Illegal command line option $1"
	    ;;
    esac
    shift
done

#
# check sanity and init state info
#

if [ -z "${GFILE}" ]; then
    usage "No GFILE name supplied."
fi

if [ ! -r "${GFILE}" ]; then
    error "ERROR: GFILE \`\`${GFILE}'' is not readable or does not exist."
fi

# 
# FUN begins.
#

# FUNCTION to get a parameter value from a gfile.
getgparam() {
    echo `cat "$GFILE" | grep "$1" | awk -F= '{print $2}'`
}

ITERATION="`getgparam '$ITERATION'`"
STATEFILE="`getgparam statefile`"

#
# the whole thing is ruined if the GFILE supplied is a bad one ...
# so check for all the essential variables.
#

if [ ! "$ITERATION" ]; then		# null string
    error "ERROR: GFILE \`${GFILE}' does not have \$ITERATION value."
fi

if [ ! "$STATEFILE" ]; then		# null string
    error "ERROR: GFILE \`${GFILE}' does not have \`statefile' value."
fi

#
# algorithm (from NOTES file).
#
#    - get statefile from g-file
#    - get iteration counter from g-file (supplied by EXCON)
#    - if ($iteration == 1)
#	get ``istar1'' from g-file
#	set newseed = $istar1
#      else
#	get ``istar1'' (ie., previous seed) from $statefile
#	newseed = `get new seed from NAWK program`
#      end
#    - write $newseed to $statefile for next iteration
#    - conditionally echo $newseed to standard output.
#

if [ $ITERATION -eq 1 ]; then
    rm -f $STATEFILE >/dev/null 2>&1
    ISTAR1="`getgparam istar1`"
    NEWSEED=$ISTAR1
else
    ISTAR1="`cat $STATEFILE`"
    NEWSEED="`(nawk 'END{srand(lastseed); print int(rand()*1.0e6)}
	' lastseed=$ISTAR1) </dev/null 2>/dev/null`"
    if test $? -ne 0; then
	NEWSEED="`(awk -v lastseed=$ISTAR1 '
	    END{srand(lastseed); print int(rand()*1.0e6)}
	') </dev/null 2>/dev/null`"
    fi
    if test $? -ne 0; then
	NEWSEED=0
	echo "EXCON-SEED: ERROR running awk/nawk to get new seed!"
    fi
fi

unset getgparam				# not needed any more after this

echo "$NEWSEED" > $STATEFILE 2>/dev/null

if [ $DOECHO -eq 1 ]; then
    echo "$NEWSEED"
fi

RETVAL=$?
exit ${RETVAL}
