PRO Div_Grid,$
  maxAperture,NPoints=nPoints, $
  CTheta=cTheta, NCTheta=nCTheta, $     ; circles
  NRTheta=nRTheta, RTheta=rTheta, $  ; radii
  NHTheta=nHTheta, HTheta=hTheta, $ ; horizontal lines
  NVTheta=nVTheta, VTheta=vTheta, $    ; vertical lines
  Group=group,V=v,File=file

;+
; NAME:
;	DIV_GRID
;
; PURPOSE:
;       This procedure creates a SHADOW source. It is a point source
;	with a grid divergence space. The grid can be a combination of:
;	  i) circles
;	  ii) radial sticks
;	  iii) horizontal sticks
;	  iv) vertical sticks
;
; CATEGORY:
;	SHADOW
;
; CALLING SEQUENCE:
;       Div_Grid,maxAperture
;
; OPTIONAL INPUTS:
;       
;	maxAperture: the angle defining the maximal aperture for
;		any grid.
;
; KEYWORD PARAMETERS (** AT LEAST ONE MUST BE DEFINED**)
;	nCTheta: the number of circles to be creates. The nCTheta
;		circles will range from 0 rad to maxAperture
;	nRTheta: the number of radial sticks to be creates. The nRTheta
;		sticks will have inclination from 0 to 2 pi rads.
;	nHTheta: the number of horizontal sticks to be creates. The nHTheta
;		sticks will be spaced from -maxAperture to maxAperture
;	nHTheta: the number of vertical sticks to be creates. The nVTheta
;		sticks will be spaced from -maxAperture to maxAperture
;
;	cTheta: array with the angular values for circles.
;		If cTheta is defined the nCTheta keyword is irrelevant.
;	rTheta: array with the angular values for radial sticks.
;		If rTheta is defined the nRTheta keyword is irrelevant.
;	hTheta: array with the angular values for horizontal sticks.
;		If hTheta is defined the nHTheta keyword is irrelevant.
;	vTheta: array with the angular values for vertical sticks.
;		If vTheta is defined the nVTheta keyword is irrelevant.
;
;	Group: The widget ID of the caller. 
;	V: A named variable where the velocities are optionally returned
;		( 3xnPoints array )
;	File: The name of SHADOW file (for output) (Defautl='begin.dat')
;
; OUTPUTS:
;       Writes a shadow file.
;
; PROCEDURE:
;       
;	Calls div_circle.pro div_radial.pro and div_line.pro
;
; EXAMPLE:
;
;	Div_Grid,1.,nHTheta=10,nVTheta=10,nCTheta=10,nRTheta=10
;	plotxy,'begin.dat',4,6,Cart=1,Histo=3
;
; MODIFICATION HISTORY:
; 	Written by:	Manuel Sanchez del Rio (srio@esrf.fr), 99-12-23
;
;-

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!error_state.msg
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'DIV_GRID: error caught: '+!error_state.msg)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

IF N_Elements(maxAperture) EQ 0 THEN  maxAperture=1.00D0

; points per pattern
IF N_Elements(nPoints) EQ 0 THEN nPoints=100L


;
; circular patterns initialization
;
IF N_Elements(cTheta) GT 0 THEN nCTheta = N_ELEMENTS(cTheta)
IF N_Elements(nCTheta) EQ 0 THEN nCTheta=0
IF N_Elements(cTheta) EQ 0 THEN cTheta = abscissas(0.0,maxAperture,nCTheta>1)

;
; radial patterns initialization
;
IF N_Elements(rTheta) GT 0 THEN nRTheta = N_Elements(rTheta)
IF N_Elements(nRTheta) EQ 0 THEN nRTheta=0
IF N_Elements(rTheta) EQ 0 THEN BEGIN
  IF nRTheta GT 1 THEN BEGIN
     tmp = (2.0*!dpi)*(nRTheta-1)/nRTheta 
  ENDIF else tmp = 2.0*!dpi
  rTheta = abscissas(0.0D0,tmp,nRTheta>1) 
ENDIF

;
; horizontal patterns initialization
;
IF N_Elements(vTheta) GT 0 THEN nVTheta = N_Elements(vTheta)
IF N_Elements(hTheta) GT 0 THEN nHTheta = N_elements(hTheta)
IF N_Elements(nHTheta) EQ 0 THEN nHTheta=0
IF N_Elements(hTheta) EQ 0 THEN $
  hTheta = Abscissas(-1.0D0*maxAperture,maxAperture,nHTheta>1)

;
; vertical patterns initialization
;
IF N_Elements(nVTheta) EQ 0 THEN nVTheta=0
IF N_Elements(vTheta) EQ 0 THEN $
  vtheta = abscissas(-1.0D0*maxAperture,maxAperture,nVTheta>1)

;
; initializas variables
;
nn = (nCTheta+nRTheta+nHTheta+nVTheta)*nPoints

IF nn EQ 0 THEN BEGIN
  Message,/Info,'Nothing to do. Set at least one keyword.'
  RETURN
ENDIF

vx = DblArr(nn)
vy = DblArr(nn)
vz = DblArr(nn)

;
; circular patterns
;
IF nCTheta GT 0 THEN $
FOR i=0,nCTheta-1 DO BEGIN
  itheta = cTheta[i]
  vtmp = div_grid_circle(itheta,nPoints)
  vx[i*nPoints:(i+1)*nPoints-1]=vtmp[0,*]
  vy[i*nPoints:(i+1)*nPoints-1]=vtmp[1,*]
  vz[i*nPoints:(i+1)*nPoints-1]=vtmp[2,*]
ENDFOR
;
; radial patterns
;
IF nRTheta GT 0 THEN $
FOR i=nCTheta,nCTheta+nRTheta-1 DO BEGIN
  itheta = rTheta[i-nCTheta]
  vtmp = div_grid_radial(maxAperture,itheta,nPoints)
  vx[i*nPoints:(i+1)*nPoints-1]=vtmp[0,*]
  vy[i*nPoints:(i+1)*nPoints-1]=vtmp[1,*]
  vz[i*nPoints:(i+1)*nPoints-1]=vtmp[2,*]
ENDFOR
;
; Horizontal lines
;
IF nhtheta GT 0 THEN $
FOR i=nCTheta+nRTheta,nCTheta+nRTheta+nHTheta-1 DO BEGIN
  itheta = hTheta[i-nCTheta-nRTheta]
  vtmp = div_grid_line(maxAperture,itheta,nPoints,/Horizontal)
  vx[i*nPoints:(i+1)*nPoints-1]=vtmp[0,*]
  vy[i*nPoints:(i+1)*nPoints-1]=vtmp[1,*]
  vz[i*nPoints:(i+1)*nPoints-1]=vtmp[2,*]
ENDFOR
;
; Vertical lines
;
IF nVTheta GT 0 THEN $
FOR i=nCTheta+nRTheta+nHTheta,nCTheta+nRTheta+nHTheta+nVTheta-1 DO BEGIN
  itheta = vTheta[i-nCTheta-nRTheta-nHTheta]
  vtmp = div_grid_line(maxAperture,itheta,nPoints)
  vx[i*nPoints:(i+1)*nPoints-1]=vtmp[0,*]
  vy[i*nPoints:(i+1)*nPoints-1]=vtmp[1,*]
  vz[i*nPoints:(i+1)*nPoints-1]=vtmp[2,*]
ENDFOR

v = make_set(vx,vy,vz)
;
; writes file
;
IF N_Elements(file) EQ 0 THEN file='begin.dat'
writesh,file,v,[4,5,6]

END ; div_grid

