function SYS_ABS, m, cent, sg_no=sg_no, symmetry=sym, isys=isys, ksys=ksys
;+
; NAME:
;	SYS_ABS
;
; PURPOSE:
;	Determine if hkl is a systematic absence in this spacegroup.
;	See below for full testing or only centering testing.
;	Fastest to call with keyworded parameters SYMMETRY, ISYS and
;	KSYS (if only centering testing then supply dummy SYMMETRY
;	from the call "symmetry=READ_SYMMETRY(0)"). 
;
; CATEGORY:
;	Laue data processing, Crystallography
;
; CALLING SEQUENCE:
; 	result= SYS_ABS(m [,centering_no, sg_no=sg_no, symmetry=symmetry, $
;			isys=isys, ksys=ksys])
;
; INPUTS:
;	M: Array (INTARR(3,*)) of the Miller indices.
;	CENTERING_NO: Type of centering (integer):
;	    'P':	centering_no=1
;	    'A':   	centering_no=2
;	    'B':	centering_no=3
;	    'C':	centering_no=4
;	    'I':	centering_no=5
;	    'F':	centering_no=6
;	    'R':	centering_no=7
;
; KEYWORDED PARAMETERS:
;	SG_NO: Spacegroup number. If given then the centering is read
;		with READ_SYMMETRY. Either CENTERING_NO, SG_NO or SYMMETRY
;		must be given. If SG_NO or SYMMETRY is given then the full 
;		symmetry is checked to determin if a reflection is absent. 
;		Otherwise only the centering is checked for absent reflections.
;	SYMMETRY: Instead of giving SG_NO which is used to read a SYMMETRY
;		structure with READ_SYMMETRY the SYMMETRY structure itself
;		can be given. (Use "symmetry=READ_SYMMETRY(sg_no)")
;	ISYS, KSYS: See SET_LATT. If given then SET_LATT is not called.
;
; OUTPUTS:
;	Returns an one-dimensional array of integers set to one for 
;	the systematic absent reflections and zero for the rest.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;
; RESTRICTIONS:
;
; PROCEDURE:
;	From LDM.FOR (Daresbury Laue Software Suite).
;	(LDM_SET_LATT, LDM_SYSABS)
;
; MODIFICATION HISTORY:
;	Written by Thomas Ursby, September 1995
;
;-

IF ((N_PARAMS() lt 2) and (not (KEYWORD_SET(sg_no) or KEYWORD_SET(sym))))$
	THEN MESSAGE,'No symmetry information given!'

IF (not KEYWORD_SET(sym)) THEN BEGIN
  IF KEYWORD_SET(sg_no) THEN BEGIN
    sym= READ_SYMMETRY(sg_no)
    cent= sym.centering_no
  ENDIF ELSE sym= READ_SYMMETRY(0)
ENDIF ELSE cent= sym.centering_no

; Get ISYS and KSYS. Variables used for determin what test on centering.
IF NOT KEYWORD_SET(isys) THEN $
  IF NOT KEYWORD_SET(ksys) THEN SET_LATT, cent, isys, ksys
absent= m(0,*)*0

; Tests on centering
IF (isys ne 0) THEN BEGIN
  IF (isys eq 4) THEN $
    FOR i=0,1 DO absent= ((((m(i,*)+m(i+1,*)) MOD 2) ne 0) or absent) $
  ELSE BEGIN
    j= m(0,*)*0
    FOR i=0,2 DO j= j+m(i,*)*ksys(i)
    absent= ((j MOD isys) ne 0)
  ENDELSE
ENDIF
; Test on full symmetry:
IF sym.valid THEN BEGIN
  IF (sym.symop_no gt 1) THEN BEGIN
    FOR j=1,sym.symop_no-1 DO BEGIN
      t= ((sym.symop_rot(*,*,j) # m) eq m)
      s= t(0,*) and t(1,*) and t(2,*)
      phas= TRANSPOSE(sym.symop_trans(*,j)) # m 
      ; Reflection absent if h'k'l' == hkl and "phase" different:
      absent= ((ABS(phas-FIX(phas)) gt 0.05) and s) or absent
    ENDFOR
  ENDIF
ENDIF 
;ELSE PRINT, 'Full symmetry not given. Only centering tested.'

RETURN, absent

END
