function READ_SYMMETRY, sg_no
;+
; NAME:
;	READ_SYMMETRY
;
; PURPOSE:
;	Read spacegroup information from the CCP4 file symop.lib.
;
; CATEGORY:
;	Crystallography
;
; CALLING SEQUENCE:
; 	symmetry= READ_SYMMETRY(sg_no)
;
; INPUTS:
;	SG_NO: The spacegroup number.
;		If set to zero READ_SYMMETRY returns a structure
;		with default values.
;		If the spacegroup is not found in the symop.lib file then 
;		READ_SYMMETRY returns a structure with default values and
;		gives an error message.
;
; KEYWORDED PARAMETERS:
;
; OUTPUTS:
;	Returns a structure with the space group information:
;	SYMMETRY= {symmetry, sg_no:1, $
;		sg_name:'P1', $
;		pg_name:'PG1', $
;		syst_no: 1, $
;		syst_name:'TRICLINIC', $
;		centering_no: 1, $
;		centering_name:'P', $
;		symop_no: 1, $
;		symop_rot: INTARR(3,192), $
;		symop_trans: FLTARR(3,192), $
;		valid: 0b }
;	where:
;	SYST_NO: Is the system number: 
;	    'TRICLINIC':    syst_no=1
;	    'MONOCLINIC':   syst_no=2
;	    'ORTHORHOMBIC': syst_no=3
;	    'TETRAGONAL':   syst_no=4
;	    'TRIGONAL':     syst_no=5
;	    'HEXAGONAL':    syst_no=6
;	    'CUBIC':        syst_no=7
;	CENTERING_NO: Is the centering type:
;	    'P':	centering_no=1
;	    'A':   	centering_no=2
;	    'B':	centering_no=3
;	    'C':	centering_no=4
;	    'I':	centering_no=5
;	    'F':	centering_no=6
;	    'R':	centering_no=7
;	VALID: Is put to 1 if the spacegroup is found and succesfully read.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;
; RESTRICTIONS:
;
; PROCEDURE:
;	
;
; MODIFICATION HISTORY:
;	Written by Thomas Ursby, September 1995
;
;-

; Define the symmetry object:
symm= {symmetry, sg_no:1, $
		sg_name:'P1', $
		pg_name:'PG1', $
		syst_no:1, $
		syst_name:'TRICLINIC', $
		centering_no: 1, $
		centering_name:'P', $
		symop_no: 1, $
		symop_rot: INTARR(3,3,192), $
		symop_trans: FLTARR(3,192), $
		valid: 0b }

IF (sg_no eq 0) THEN RETURN, symm

; Get path and name of CCP4's symop.lib:
filename= STRCOMPRESS(GETENV('CLIBD')+'/symop.lib', /REMOVE_ALL)
line='' & no=0 & lines_no=0 & pe_lines_no=0
sg_name='' & pg_name='' & syst_name='' & syst_no=0
centering_no=0 & xop='' & yop='' & zop=''
op= STRARR(10) & thisop= STRARR(3) & symop_no=0
; Open file and look for right space group:
OPENR, file, filename, /GET_LUN
REPEAT BEGIN
  READF, file, line
  READS, line, no, lines_no, pe_lines_no, line 
  IF (no ne sg_no) THEN FOR i=1,lines_no DO READF, file, line
ENDREP UNTIL ((no eq sg_no) or eof(file))
IF (no eq sg_no) THEN BEGIN
  ; Space group found. Read the information:
  line= STRTRIM(line,2)
  READS, STRMID(line,0,STRPOS(line,' ')), sg_name
  line= STRTRIM(STRMID(line,STRPOS(line,' '),STRLEN(line)),2)
  READS, STRMID(line,0,STRPOS(line,' ')), pg_name
  line= STRTRIM(STRMID(line,STRPOS(line,' '),STRLEN(line)),2)
  IF (STRPOS(line,' ') ne -1) THEN line=STRMID(line,0,STRPOS(line,' '))
  READS, STRMID(line,0,STRLEN(line)), syst_name
  ; Get the centering type:
  centering_name= STRMID(sg_name,0,1)
  CASE centering_name OF 
    'P':	centering_no=1
    'A':   	centering_no=2
    'B':	centering_no=3
    'C':	centering_no=4
    'I':	centering_no=5
    'F':	centering_no=6
    'R':	centering_no=7
    ELSE: PRINT, 'Failed getting centering type.'
  ENDCASE
  ; Read the lines with the symmetry operators:
  FOR l=1,lines_no DO BEGIN
    READF, file, line
    line= STRUPCASE(line)
    i=0 & pos=STRPOS(line,'*')
    REPEAT BEGIN
      IF (pos eq -1) THEN pos=STRLEN(line)
      op(i)= STRCOMPRESS(STRMID(line,0,pos),/REMOVE_ALL)
      line= STRMID(line,pos+1,STRLEN(line))
      i=i+1 & pos=STRPOS(line,'*')
    ENDREP UNTIL (pos eq -1)
    ; For each operator get the operation for each component:
    FOR k=symop_no,symop_no+i-1 DO BEGIN
      ; Divide into components:
      comma1=STRPOS(op(k-symop_no),',')
      comma2=STRPOS(op(k-symop_no),',',comma1+1)
      thisop(0)= STRMID(op(k-symop_no),0,comma1)
      thisop(1)= STRMID(op(k-symop_no),comma1+1,comma2-comma1-1)
      thisop(2)= STRMID(op(k-symop_no),comma2+1,STRLEN(op(k-symop_no)))
      ; Get the sign and translation for each component:
      symm.symop_rot(*,*,k)= 0 & posxyz=INTARR(3)
      FOR j=0,2 DO BEGIN
        oper= thisop(j)
        posxyz(0)= STRPOS(oper,'X') 
        posxyz(1)= STRPOS(oper,'Y') 
        posxyz(2)= STRPOS(oper,'Z') 
        xyz=WHERE(posxyz ne -1) & pos=(posxyz(xyz))(0)
        STRPUT, oper, ' ', pos 
        IF (STRMID(oper,(pos-1)>0,1) eq '-') THEN sign=-1 ELSE sign=1
        STRPUT, oper, ' ', (pos-1)>0
        oper= STRCOMPRESS(oper,/REMOVE_ALL)
        slash=STRPOS(oper,'/')
        IF (slash ne -1) THEN BEGIN
	  nominator=FLOAT(STRMID(oper,slash-1,1))
	  denominator=FLOAT(STRMID(oper,slash+1,1))
          trans= nominator/denominator
        ENDIF ELSE trans=0.0
        symm.symop_rot(j,xyz,k)= sign
        symm.symop_trans(j,k)= trans
      ENDFOR
    ENDFOR
    symop_no= symop_no+i
  ENDFOR
  CASE syst_name OF
    'TRICLINIC':    syst_no=1
    'MONOCLINIC':   syst_no=2
    'ORTHORHOMBIC': syst_no=3
    'TETRAGONAL':   syst_no=4
    'TRIGONAL':     syst_no=5
    'HEXAGONAL':    syst_no=6
    'CUBIC':        syst_no=7
    ELSE: PRINT, 'Illegal crystal system name: '+syst_name
  ENDCASE
  symm.sg_no= 		sg_no
  symm.sg_name= 	sg_name
  symm.pg_name=		pg_name
  symm.syst_no=		syst_no
  symm.syst_name=	syst_name
  symm.centering_no=	centering_no
  symm.centering_name=	centering_name
  symm.symop_no=	symop_no
  symm.valid= 	1
ENDIF ELSE PRINT, STRCOMPRESS('Space group not found in: '+filename)
FREE_LUN, file

RETURN, symm

END
