function RDBW, lambda, dstar2, lambdapol, bfactor, polscale=polscale,lscale=lscale,lmin=lmin,lmax=lmax,dmax=dmax,max_res=max_res,use_mono=use_mono,$
f_mono=f_mono,get_norm_coeff=get_norm_coeff,set_norm_coeff=set_norm_coeff,norm_coeff=norm_coeff
;+
; NAME:
;	RDBW
;
; PURPOSE:
;	Calculate a factor that gives an estimation of the intensity of
;	a reflection witout considering the structure factor:
;	  POLSCALE / POLY(lambda,lambdapol) * exp(-0.5*b*dstar2) * 0.0025 / (dstar2*lambda^2) [* Fmono^2]
;
; CATEGORY:
;	Laue data processing, Crystallography
;
; CALLING SEQUENCE:
; 	result= RDBW(lambda, dstar2, lambdapol, bfactor, polscale=polscale,[/lscale],[lmin=lmin],$
;			[lmax=lmax],[dmax=dmax],[max_res=max_res],[use_mono=use_mono],[f_mono=f_mono])
;
; INPUTS:
;	LAMBDA: THe wavelenght of the reflection.
;	DSTAR2: 1/d^2 for the reflection.
;	  Both LAMBDA and DSTAR2 can be arrays (of same size).
;	LAMBDAPOL: Polynomial that corresponds to an expected normalisation
;		curve. (As output from Lauenorm, i.e. the curve with
;		which the intensities are multiplied.) The curve is 
;		normalised so that the inverse peaks at 1.0 (in the
;		given lambdamin and lambdamax).
;	BFACTOR: B-factor as expected output in TRUNCATE.
;
; KEYWORDED PARAMETERS:
;	POLSCALE: The factor is multiplied with this (default 1.0).
;	LSCALE : if this keyword is set, then a chebytchev polynom is calculated as in the LSCALE program.
;	LMIN : the minimum wavelength to consider
;	LMAX : the maximum wavelength to consider
;	DMAX : the lowest resolution to consider
;	USE_MONO: if this keyword is set, then information from a monochromatic data set is taken into account. 
;	The monochromatic structure factor amplitudes are then also
;	taken into account when assessing the predicted integrated intensities. Be careful that there is no mis-indexing.
;	If the monochromatic data set is incomplete relative to the Laue data set, a dummy monochromatic structure factor 
;	amplitude is taken, equal to the median value of the monochromatic Fmono's. 
;	A difficulty in using monochromatic reference data set is that proper symmetry equivalent must be found for each measurement.
;	F_MONO: if /USE_MONO is used, F_MONO contains the monochromatic SF's.
;	SET_NORM_COEFF : If the calculation has already been done for the contribution of the lambda curve, it can be passed through SET_NORM_COEFF
;		This allows to save a lot of time if the computation has to be done many times
;	GET_NORM_COEFF : if the computation has to be done many times, the result of the contribution of the lambda curve can be saved in NORM_COEFF
;		if /GET_NORM_COEFF is given.
; OUTPUTS:
;	Returns the factor as above.
;	MAX_RES : Returns the maximum results expected at lowest resolution and top of normalization curve.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;
; RESTRICTIONS:
;
; PROCEDURE:
;	
; MODIFICATION HISTORY:
;	Written by Thomas Ursby, September 1995
;	Modif 17/09/98 : D. Bourgeois
;	Modif 02/03/99 : D. Bourgeois
;-

IF NOT KEYWORD_SET(polscale) THEN polscale=1.0
IF N_ELEMENTS(lmin) EQ 0 THEN lmin=MIN(lambda)
IF N_ELEMENTS(lmax) EQ 0 THEN lmax=MAX(lambda)
IF N_ELEMENTS(dmax) EQ 0 THEN dstar2min=MIN(dstar2) ELSE dstar2min=1/FLOAT(dmax^2)

IF N_ELEMENTS(use_mono) EQ 1 THEN BEGIN
 IF N_ELEMENTS(f_mono) EQ 0 THEN MESSAGE,'Monochromatic Structure Factors are missing !'
ENDIF

n_samples=500
l=FLOAT(lmin)+indgen(n_samples)/FLOAT(n_samples)*(FLOAT(lmax)-FLOAT(lmin))

IF NOT KEYWORD_SET(use_mono) THEN BEGIN

 IF KEYWORD_SET(lscale) THEN BEGIN 
  w_max=WHERE(1/CHEBYCHEV(lambdapol,l) EQ MAX(1/CHEBYCHEV(lambdapol,l)),ct_max)
  max_l = l(w_max(0))
  max_res=polscale * exp(-0.5*bfactor*dstar2min) * 0.0025 / (dstar2min*l(w_max(0))^2) 
  max_res2=polscale * exp(-0.5*bfactor*MIN(dstar2)) * 0.0025 / (MIN(dstar2)*l(w_max(0))^2) 
  PRINT,'Reference intensity is : '+STRING(max_res/max_res2*100)+' % of maximum predicted intensity'
  
  IF KEYWORD_SET(get_norm_coeff) THEN BEGIN
   norm_coeff = polscale/CHEBYCHEV(lambdapol,lambda,lmin=lmin,lmax=lmax,lnorm=l(w_max(0)))
   RETURN, norm_coeff * exp(-0.5*bfactor*dstar2) * 0.0025 / (dstar2*lambda^2) 
  ENDIF 


  IF N_ELEMENTS(set_norm_coeff) GT 0 AND (SIZE(set_norm_coeff))(1) EQ (SIZE(lambda))(1) THEN $
   RETURN, set_norm_coeff * exp(-0.5*bfactor*dstar2) * 0.0025 / (dstar2*lambda^2) $
  ELSE $
   RETURN, polscale/CHEBYCHEV(lambdapol,lambda,lmin=lmin,lmax=lmax,lnorm=l(w_max(0))) * exp(-0.5*bfactor*dstar2) * 0.0025 / (dstar2*lambda^2)

 ENDIF ELSE BEGIN
  w_max=WHERE(POLY(l,lambdapol) EQ MIN(POLY(l,lambdapol)),ct_max)
  max_l = l(w_max(0))
  max_res=polscale/POLY(max_l,lambdapol) * exp(-0.5*bfactor*dstar2min) * 0.0025 / (dstar2min*l(w_max(0))^2) 
  max_res2=polscale/POLY(max_l,lambdapol) * exp(-0.5*bfactor*MIN(dstar2)) * 0.0025 / (MIN(dstar2)*l(w_max(0))^2) 
  PRINT,'Reference intensity is : '+STRING(max_res/max_res2*100)+' % of maximum predicted intensity'
  RETURN, polscale/POLY(lambda,lambdapol) * exp(-0.5*bfactor*dstar2) * 0.0025 / (dstar2*lambda^2) 
 ENDELSE 

ENDIF ELSE BEGIN

 IF KEYWORD_SET(lscale) THEN BEGIN 
  w_max=WHERE(1/CHEBYCHEV(lambdapol,l) EQ MAX(1/CHEBYCHEV(lambdapol,l)),ct_max)
  max_l = l(w_max(0))
  max_res=polscale * exp(-0.5*bfactor*dstar2min) * 0.0025 / (dstar2min*l(w_max(0))^2) * MEDIAN(f_mono^2d) 
  max_res2=polscale * exp(-0.5*bfactor*MIN(dstar2)) * 0.0025 / (MIN(dstar2)*l(w_max(0))^2) * MEDIAN(f_mono^2d) 
  PRINT,'Reference intensity is : '+STRING(max_res/max_res2*100)+' % of maximum predicted intensity'

  IF KEYWORD_SET(get_norm_coeff) THEN BEGIN
   norm_coeff = polscale/CHEBYCHEV(lambdapol,lambda,lmin=lmin,lmax=lmax,lnorm=l(w_max(0)))
   RETURN, norm_coeff * exp(-0.5*bfactor*dstar2) * 0.0025 / (dstar2*lambda^2) * f_mono^2d 
  ENDIF 
  
  IF N_ELEMENTS(set_norm_coeff) GT 0 AND (SIZE(set_norm_coeff))(1) EQ (SIZE(lambda))(1) THEN $
   RETURN, set_norm_coeff * exp(-0.5*bfactor*dstar2) * 0.0025 / (dstar2*lambda^2) * f_mono^2d $
  ELSE $
   RETURN, polscale/CHEBYCHEV(lambdapol,lambda,lmin=lmin,lmax=lmax,lnorm=l(w_max(0))) * exp(-0.5*bfactor*dstar2) * 0.0025 / (dstar2*lambda^2) * f_mono^2d 

 ENDIF ELSE BEGIN
  w_max=WHERE(POLY(l,lambdapol) EQ MIN(POLY(l,lambdapol)),ct_max)
  max_l = l(w_max(0))
  max_res=polscale/POLY(max_l,lambdapol) * exp(-0.5*bfactor*dstar2min) * 0.0025 / (dstar2min*l(w_max(0))^2) * MEDIAN(f_mono^2d) 
  max_res2=polscale/POLY(max_l,lambdapol) * exp(-0.5*bfactor*MIN(dstar2)) * 0.0025 / (MIN(dstar2)*l(w_max(0))^2) * MEDIAN(f_mono^2d) 
  PRINT,'Reference intensity is : '+STRING(max_res/max_res2*100)+' % of maximum predicted intensity'
  RETURN, polscale/POLY(lambda,lambdapol) * exp(-0.5*bfactor*dstar2) * 0.0025 / (dstar2*lambda^2) * f_mono^2d
 ENDELSE 

ENDELSE


END

