function FIND_ROTAXIS, par1, par2, rot_angle
;+
; NAME:
;	FIND_ROTAXIS
;
; PURPOSE:
;	Given the determined rotation matrix for two subsequent
;	spindle settings FIND_ROTAXIS determines the direction
;	of the rotation axis. The rotation axis is oriented so
;	that from PAR1 to PAR2 is a positive rotation (<Pi).
;
; CATEGORY:
;	Laue data processing, Crystallography
;
; CALLING SEQUENCE:
; 	result= FIND_ROTAXIS(par1,par2)
;
; INPUTS:
;	PAR1,PAR2: As from READ_GEASC. Note that the SPINDLE tag
;		of the two structures are added to the respective
;		PHIZ before the rotations axis is determined. 
;
; KEYWORDED PARAMETERS:
;
; OUTPUTS:
;	Returns the rotation axis in the form of its cosine values
;	along the three axes.
;	ROT_ANGLE: The calculated rotation from PAR1 to PAR2. [Degrees]
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;
; RESTRICTIONS:
;
; PROCEDURE:
;	
; MODIFICATION HISTORY:
;	Written by Thomas Ursby, September 1995
;
;-

; Calculate the rotation matrix for the two orientations:
M1= CALC_ROTMAT(par1)
M2= CALC_ROTMAT(par2)
deltaM= M2-M1

; Calculate three vectors that all should be along the
; rotation axis (but one of the reciprocal axis might be
; along the rotation axis which causes problems):
r= DBLARR(3,3)
r(*,0)= CROSSP(deltaM(*,0),deltaM(*,1))
r(*,1)= CROSSP(deltaM(*,1),deltaM(*,2))
r(*,2)= CROSSP(deltaM(*,2),deltaM(*,0))

; so choose the largest of the three vectors:
length= TOTAL(r^2,1)
index= WHERE(length ge max(length))
s= (r(*,index(0)))(*)
s= s/sqrt(TOTAL(s^2))

; Determine if the rotation axis should be inverted:
v= CROSSP(M1(*,index(0)),deltaM(*,index(0)))
sign= TOTAL(v*s)
sign= sign/sqrt(TOTAL(sign^2))
s= sign*s

; Normalise:
s= s/sqrt(TOTAL(s^2))

; Get the rotation between PAR1 and PAR2:
v1=CROSSP(s,M1(*,index(0)))
v2=CROSSP(s,M2(*,index(0)))
rot_angle= 180/!dpi*acos( TOTAL(v1*v2)/sqrt(TOTAL(v1^2)*TOTAL(v2^2)))

RETURN, s

END

