function FIND_PEAKS1,struc,image,red,par,ovlp_dist,var_limit,nodal_index, $
	border_cutoff,min_peaks, show=show
;+
; NAME:
;	FIND_PEAKS1
;
; PURPOSE:
;	Finds out peaks in a diffraction image that are suitable for 
;	refinement. The returned structure contains informations about 
;	these peaks. They are characterized by the following properties :
;	(i) : non_overlapped peaks
;	(ii) : nodal spots : ie h,k,l <= nodal_index
;	(iii) : good signal to noise ratio : integrated intensity / background
;			should be higher than var_limit
;	
;
; CATEGORY:
;	Data processing
;
; CALLING SEQUENCE:
;	peaks = FIND_PEAKS1(struc,image,par,var_limit,nodal_index,/show)
;
; INPUTS:
;	struc : the reflection list (a structure produced with the
;	        program READ_GEASC2.PRO to read the content of the CCP4 file
;		"*.geasc". 
;	image : the original data image 
;	red : the rebinned image
;	par : the parameters (from READ_GEASC2.PRO + WRITEPAR_LAUE2.PRO)
;	var_limit : minimum I/Sig(I) to select a peak
;		[ex: var_limit = 2 means that the peak integrated 
;		intensity value must be greater than 2 times its variance]
;	show : set this keyword too see what happens
;
; OUTPUTS:
;	peaks : the list of reflections matching the criteria described 
;		above. The .x and .y fields contains the position in image 
;		of the center of mass of the peaks [mm from beam center] 
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None
; PROCEDURE:
;	The reflection list "struc" is first searched for overlaps. Then for
;	nodal index limit. For each remaining spot, a box of size par.boxsize
;	is taken around the predicted position. The background is calculated
;	(plane fitting). The optimal peak region is defined and the center
;	of mass calculated. The spot is retained if its signal to background 
;	is high enough.
;	Note : satutated spots are not removed.
;
; MODIFICATION HISTORY:
;	D.Bourgeois, September 1995.
;-


PRINT,'I/Sig(I) ratio limit for peak selection :',var_limit

;suppress nasty messages for arithmetic errors
junk = CHECK_MATH(trap = 0)

;position of the beam center on the original Laue image [pixels] 
centerX = par.cenx-1
centerY = par.ceny-1

;pixel size [mm]
; scale_y is taken from LAUEGEN
  scaleY = par.yscale
;original pixel size [micrometers]
  pix_size = par.raster
pix_sizeX = 0.001*pix_size
pix_sizeY = 0.001*pix_size/scaleY

;size of "image"
imageX = (size(image))(1)
imageY = (size(image))(2)

;position of the studied area relative to the center of the beam [mm]
minX = pix_sizeX*(-centerX+par.boxsize/2+2)
minY = pix_sizeY*(-centerY+par.boxsize/2+2)
maxX = pix_sizeX*(-centerX+imageX-1-par.boxsize/2-2)
maxY = pix_sizeY*(-centerY+imageY-1-par.boxsize/2-2)

IF KEYWORD_SET(show) THEN BEGIN
main_window = !D.WINDOW
WIN,512,512
surf_window = !D.WINDOW
ENDIF ;show

;create a reduced structure array which contains only the spots of interest
;  The criteria of selection is that the position of the spot be inside
;  the box defined by minX, maxX, minY and maxY
spots = struc(WHERE((struc(*).x GE minX) AND $
		    (struc(*).x LE maxX) AND $
		    (struc(*).y GE minY) AND $
		    (struc(*).y LE maxY)))

PRINT,'Number of input spots : ',(SIZE(struc))(1)

;finds out the overlaps
;FIND_OVLP2,spots,par,par.ovlp_dist,par.ovlp_min_dist
FIND_QUICKOVLP,spots,par,ovlp_dist

spots = spots(WHERE(spots.ovlp EQ 0,ct))
PRINT,'Number of non overlapped spots : ',ct


;finds out the miller indices
spots = spots(WHERE((ABS(spots.m(0)) LE nodal_index) AND (ABS(spots.m(1)) LE nodal_index) AND (ABS(spots.m(2)) LE nodal_index), ct))
PRINT,'Number of spots of nodal index < ',nodal_index,' : ',ct


;define the potentially matched spots
matched = spots
deviation = FLTARR(ct)

cr = STRING("15b)
;Go through all reflections of spots
PRINT,'Matching spots ...'

iteration = 0 & bf=1.0 & vf=1.0
REPEAT BEGIN
IF (iteration ne 0) THEN BEGIN
  PRINT,'No of badflag: ', no_badflag
  PRINT,'No of bad I/sigma: ', no_badvar
  PRINT,'No of bad border: ', no_badborder
  PRINT,'No of selected: ',j
  IF (ct lt 2*min_peaks) THEN vf=vf*1.25 $
  ELSE IF (no_badvar gt no_badborder) THEN vf=vf*1.25 $
  ELSE bf=bf*1.25
  IF ((vf eq 1.0) and (bf eq 1.0)) THEN MESSAGE,'Can not find peaks!'
ENDIF
iteration= iteration + 1
border_cutoff2= border_cutoff/bf
var_limit2= var_limit/vf
PRINT,'Border cut-off: ', border_cutoff2
PRINT,'I/sigma limit: ', var_limit2
IF ((border_cutoff2<var_limit2) le 0.1) THEN MESSAGE,'Too small cut-offs!'

j = 0
no_badflag=0 & no_badvar=0 & no_badborder=0 & no_sat=0
IF ct LT 2 THEN MESSAGE,'Not enough selected spots !'
FOR i=0,ct-1 DO BEGIN
;time1 = systime(1)
IF i/100 EQ i/100.0 THEN PRINT,FORMAT="($,I5,' ie : ',F3.0,'%',A)",i,100.*i/FLOAT(ct),cr

;take a box around predicted position of size par.boxsize
 ;get the position in pixels
  xpos = ROUND(spots(i).x/pix_sizeX+centerX)
  ypos = ROUND(spots(i).y/pix_sizeY+centerY)

 box = image(xpos-FIX(par.boxsize/2.):xpos+FIX(par.boxsize/2.),ypos-FIX(par.boxsize/2.):ypos+FIX(par.boxsize/2.))

 max_value=max(box)
 IF max_value GE par.satvalue THEN saturated=1 ELSE saturated=0

;calculate background
;time2=systime(2)
 background =  BACK2D_POLY4(box,par.noise_percentage,par.bg_gain,par.dark_current,flag,coeff,bc_mean,n_back,chi_square,max_cycle=2)
;time3=systime(3)

area = box - background

;We request that the maximum value on the borders of area be at least 5 times lower
;that the maximum value in area.
max_border = MAX(area(0,*)) > MAX(area(par.boxsize-1,*)) > MAX(area(*,0)) > MAX(area(*,par.boxsize-1))

;we know request that the maximum pixel value in area not be on a border :
;xy_max = WHERESUB(WHERE(area EQ MAX(area),n_max),area)
;x_max = FLOAT(xy_max)
;y_max = IMAGINARY(xy_max)

;IF n_max EQ 1 THEN bad_flag = (x_max EQ 0) OR (x_max EQ (par.boxsize-1)) OR (y_max EQ 0) OR (y_max EQ (par.boxsize-1)) ELSE bad_flag = (MIN(x_max) EQ 0) OR (MAX(x_max) EQ (par.boxsize-1)) OR (MIN(y_max) EQ 0) OR (MAX(y_max) EQ (par.boxsize-1))

;bad_flag = MAX(bad_flag)

;get the optimal peak region
;cutoff = GET_BOXLEVEL(area,n_back,bc_mean,par.dark_current)
;We define an arbitrary cutoff of 20%
cutoff = 20
region = WHERE(area GE MAX(area)*cutoff/100.,n_peak)

;is there a spike or a dead pixel ?
;spike = GET_SPIKE(area,cutoff,par.boxsize)
;dead_pix = GET_DEADPIX(area,cutoff,par.boxsize)
;IF spike NE 0 THEN print,'Spike detected'
;IF dead_pix NE 0 THEN print,'Dead pixel detected'

;take the spot if signal to background is high enough and background is correct :
tot_region = TOTAL(area(region))

;get the variance
gain= par.gain*1.5418/spots(i).l
var = SQRT(GET_BOXVAR(tot_region,n_peak,bc_mean,n_back,gain,par.bg_gain,par.dark_current))

;IF (spike EQ 0) AND (dead_pix EQ 0) AND (flag EQ 0) AND (tot_region/TOTAL(background(region)) GT var_limit2) THEN BEGIN

no_sat= no_sat + (saturated eq 1)
no_badflag= no_badflag + (flag ne 0)
no_badvar= no_badvar + (tot_region/var le var_limit2)
no_badborder= no_badborder + (MAX(area) le border_cutoff2*max_border)

IF (saturated eq 0) AND (flag EQ 0) AND (tot_region/var GT var_limit2) AND $
	(MAX(area) GT border_cutoff2*max_border) THEN BEGIN

matched(j) = spots(i)
;calculate center of mass
;xy coordinates
xy = WHERESUB(region,box)
com_x = TOTAL(FLOAT(xy)*area(region))/FLOAT(tot_region)
com_y = TOTAL(IMAGINARY(xy)*area(region))/FLOAT(tot_region)

;transform pixel values inside box into [mm] value inside image

;get the center [raster]
matched(j).x = pix_sizeX*(com_x + xpos - par.boxsize/2 - centerX)
matched(j).y = pix_sizeY*(com_y + ypos - par.boxsize/2 - centerY)

;calculate deviation
deviation(j) = (matched(j).x - spots(i).x)^2 + (matched(j).y - spots(i).y)^2

IF KEYWORD_SET(show) THEN BEGIN
ok = ''
;show the predicted background subtracted box
WSET,main_window
dummy = PIXLAUE2(matched(j),red,image,par,draw=4,color=30)
PRINT,'Initial box to look for peak. Predicted position at center of this box'
WSET,surf_window
SURFACE,area
PRINT,'Observed peak at position :',com_x,com_y
PRINT,'Deviation from predicted [mm]:',SQRT(deviation(j))
PRINT,'Observed position in image [pixels] :',com_x + xpos - par.boxsize/2,com_y + ypos - par.boxsize/2 
PRINT,'Box for integration will be centered on [pixels] :',ROUND(matched(j).x/pix_sizeX+centerX),ROUND(matched(j).y/pix_sizeY+centerY)
PRINT,'Integrated intensity :',tot_region
PRINT,'RMS deviation :',var
PRINT,'I/Sig(I) :',tot_region/var
READ,'Ok ? ',ok
;region for peak center determination
used_region = BYTARR(par.boxsize,par.boxsize)
used_region(region)=1
SURFACE,used_region
READ,'Ok ? ',ok
WSET,main_window
ENDIF ; show

j = j+1
ENDIF ; all conditions

;time4=systime(1)
;PRINT,'Time for bc calc :',time3-time2
;PRINT,'Total time :',time4-time1
ENDFOR ; i
ENDREP UNTIL (j ge min_peaks)

PRINT,'No of saturated: ', no_sat
PRINT,'No of badflag: ', no_badflag
PRINT,'No of bad I/sigma: ', no_badvar
PRINT,'No of bad border: ', no_badborder

deviation = deviation(0:j-1)
;std = STDEV(deviation,mean)
mean = TOTAL(SQRT(deviation))/(SIZE(deviation))(1)
std = SQRT(TOTAL(deviation)/(SIZE(deviation))(1))
PRINT,'Average distance from predicted to observed [um] :',mean*1000.
PRINT,'RMS distance from predicted to observed [um] :',std*1000.
RETURN,matched(0:j-1)
END
