function CALC_ROTMAT, par, rotmat
;+
; NAME:
;	CALC_ROTMAT
;
; PURPOSE:
;	Calculate the matrix relating the indices of a rlp to its
;	coordinates in a orthogonal system.
;
; CATEGORY:
;	Laue data processing, Crystallography
;
; CALLING SEQUENCE:
; 	result= CALC_ROTMAT(par [,rotmat])
;
; INPUTS:
;	PAR: As from READ_GEASC. Note that the SPINDLE tag of the PAR 
;		structure is added to PHIZ before the rotation matrix
;		is determined. 
;
; KEYWORDED PARAMETERS:
;
; OUTPUTS:
;	Returns the rotation matrix relating the indices of a rlp to its
;	coordinates in a orthogonal system:
;		  x = M # [h,k,l]
;	ROTMAT: The rotation matrix defined by PhiX, PhiY and PhiZ.
;		  x = rotmat # m1 # [h,k,l]
;		where m1 relates hkl to the coordinates in reciprocal space
;		before any rotation.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;
; RESTRICTIONS:
;
; PROCEDURE:
;	
;
; MODIFICATION HISTORY:
;	Written by Thomas Ursby, September 1995
;
;-

; Give the parameters some more sensible names:
phix= 	par.phis(0)*!dpi/180.0
phiy= 	par.phis(1)*!dpi/180.0
phiz= 	(par.phis(2)+par.spindle)*!dpi/180.0
ast= 	par.crystal(0)
bst= 	par.crystal(1)
cst= 	par.crystal(2)
alphast=par.crystal(3)*!dpi/180.0
betast= par.crystal(4)*!dpi/180.0
gammast=par.crystal(5)*!dpi/180.0

; Some values needed for Mksi:
cosalpha= (cos(betast)*cos(gammast)- cos(alphast)) $
		/ (sin(betast)*sin(gammast))
vstar= ast*bst*cst* sqrt(1 - cos(alphast)^2 - cos(betast)^2 - $
	 cos(gammast)^2 + 2*cos(alphast)*cos(betast)*cos(gammast))
c= ast*bst*sin(gammast)/vstar

; Matrix that relates the Miller indices to the coordinates in reciprocal
; space:
; FOR ast along spindle (Z axis), bst along beam (X axis) , cst along third axis (Y axis)
;PRINT,'Assuming : ast along spindle (Z axis), bst along beam (X axis) , cst along third axis (Y axis)'
Mksi = [[ 0., bst*sin(gammast), -cst*sin(betast)*cosalpha], $
	[ 0., 0., 1/c], $
	[ ast, bst*cos(gammast), cst*cos(betast)]]

; FOR ast along beam (X axis), bst along third axis (Y axis) , cst along spindle (Z axis)
PRINT,'Assuming : ast along beam (X axis), bst along third axis (Y axis) , cst along spindle (Z axis)'
Mksi = [[ ast, bst*cos(gammast), cst*cos(betast)],$
	[ 0., bst*sin(gammast), -cst*sin(betast)*cosalpha], $
	[ 0., 0., 1/c]]

; The matrices that corresponds to the three missetting angles:
Mx = [ 	[ 1.0, 0.0, 0.0], $
	[ 0.0, cos(phix), -sin(phix)], $
	[ 0.0, sin(phix), cos(phix)] ]
My = [ 	[ cos(phiy), 0.0, sin(phiy)], $
	[ 0.0, 1.0, 0.0], $
	[ -sin(phiy), 0.0, cos(phiy)] ]
Mz = [ 	[ cos(phiz), -sin(phiz), 0.0], $
	[ sin(phiz), cos(phiz), 0.0], $
	[ 0.0, 0.0, 1.0] ]
; To get the matrices right in the multiplication we need to (IDL...)
Mksi= TRANSPOSE(Mksi)
Mx= TRANSPOSE(Mx)
My= TRANSPOSE(My)
Mz= TRANSPOSE(Mz)

; ksi is the coordinates in reciprocal space of the rotated reciprocal
; lattice vector. (x is along the beam and z is horisontal (i.e. along
; the calculated y-coordinate).)
rotmat= Mz # My # Mx 
mat= rotmat # Mksi 

RETURN, mat

END
