function CALC_COORD_REFINE, x, y, a, ddda, stepfactor=stepfactor
;+
; NAME:
;	CALC_COORD_REFINE
;
; PURPOSE:
;	Calculate coordinates in respect to the beam center of Laue
;	spots and calculate the difference from observed positions.
;	The coordinates calculated are the same as in LAUEGEN.
;	The user input the Miller indices (in X) and gives observed 
;	coordinates (in Y) and some experimental parameters (in A).
;	If dDdA is given then partial derivatives are calculated
;	and given in dDdA (distance between calculated and observed
;	position in respect to a change in parameter a(x)). 
;	Written for use with refinement of the parameters A.
;
; CATEGORY:
;	Laue data processing, Crystallography
;
; CALLING SEQUENCE:
; 	d= CALC_COORD_REFINE( x, y, a [, ddda])
;
; INPUTS:
;	X: See below on outputs. From the input only the tag m is used
;		(the Miller indices).
;	Y: An array (same size as X) of structure with the tags X,Y
;		that contains the observed coordinates.
;	A: An array containing some parameters needed for the coordinate
;		calculation:
;		a(0)= par.phis.x
;		a(1)= par.phis.y
;		a(2)= par.phis.z + par.spin
;		a(3)= par.cdd
;		a(4)= par.cenx	; In [mm]. Offset relative to the value used 
;		a(5)= par.ceny	; when determining the positions in Y.
;		a(6)= par.yscale ; A factor to multiply the old yscale with.
;	Note that ceny is in "observed position space" so it should be sub-
;	tracted from the old observed positions before these are multiplied
;	with the yscale (a(6)).
;		a(7)= par.twist
;		a(8)= par.tilt
;		a(9)= par.bulge
;		a(10)= par.crystal.angle(0)
;		a(11)= par.crystal.angle(1)
;		a(12)= par.crystal.angle(2)
;		a(13)= par.crystal.axis(0)
;		a(14)= par.crystal.axis(1)
;		a(15)= par.crystal.axis(2)
;	dDdA: If supplied then partial derivatives will be calculated.
;		On input this variable should be an array of size 16
;		with it's elements non-zero if the corresponding partial
;		derivative should be calculated (otherwise the partial
;		derivative is set to zero).
; KEYWORDED PARAMETERS:
;	STEPFACTOR: The default steps used to calculate the derivatives
;		are multiplied by this factor.
;
; OUTPUTS:
;	Returns the distance between calculated and observed position
;	(array of same size as the input X and Y).
;	X: An array (possibly of size 1) of structures with the tags
;		M (INTARR(3) of Miller indices), X (floating number
;		where the calculated x-coordinate is put), Y (floating
;		number where the calculated y-coordinate is put).
;	dDdA: On completion (if given; see input dDdDA) it is an double
;		precision array containing the partial derivative for
;		the required parameters.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	Alters the input X (tags X and Y).
;
; RESTRICTIONS:
;	At present the missetting angles are supposed to be relative
;	a* being along the rotation axis (along the coordinate y-axis)
;	and b* being in the plane of the rotation axis and the beam
;	axis as close to the beam axis as possible (direction of the beam).
;	In other words a*,b* being given in LAUEGEN.
;
; PROCEDURE:
;	Hopefully the same as is used in LAUEGEN.
;
; MODIFICATION HISTORY:
;	Written by Thomas Ursby, June 1995
;
;-

ndata= (size(x))(1)
nopar=16
IF ((size(a))(1) ne nopar) THEN MESSAGE, $
	'Wrong number of paramters in A.'
IF NOT KEYWORD_SET(stepfactor) THEN stepfactor=1.0

IF (N_PARAMS() eq 4) THEN BEGIN ; Calculate the partial derivatives.
  c= ddda & ddda= DBLARR(ndata,nopar)
  ; Very crude way of determining step size...
  step=stepfactor*[0.01, 0.01, 0.01, 0.1, 0.01, 0.01, 0.01, 1.0, 1.0, 1.0, $
	0.01, 0.01, 0.01, 0.01, 0.01, 0.01] 
  amod1=a & amod2=a
  FOR i=0,nopar-1 DO BEGIN
    IF (c(i) ne 0) THEN BEGIN
      amod1(i)= a(i)-step(i)
      amod2(i)= a(i)+step(i)
      ddda(*,i)= (calc_coord_refine(x,y,amod2)-calc_coord_refine(x,y,amod1))/ $
		(2*step(i))
      amod1(i)= a(i)
      amod2(i)= a(i)
    ENDIF
  ENDFOR  
ENDIF 

; Give the parameters some more sensible names:
phix= 	a(0)*!pi/180.0
phiy= 	a(1)*!pi/180.0
phiz= 	a(2)*!pi/180.0
cdd= 	a(3)
cenx= 	a(4)	; In [mm]. Offset relative to the value used when determining
ceny= 	a(5)	; the positions in Y
yscale=	a(6)    ; A factor to multiply the old yscale with.
twist= 	a(7)
tilt= 	a(8)
bulge= 	a(9)
alphast=a(10)*!pi/180.0
betast= a(11)*!pi/180.0
gammast=a(12)*!pi/180.0
ast= 	a(13)
bst= 	a(14)
cst= 	a(15)

; Some values needed for Mksi:
cosalpha= (cos(betast)*cos(gammast)- cos(alphast)) $
		/ (sin(betast)*sin(gammast))
vstar= ast*bst*cst* sqrt(1 - cos(alphast)^2 - cos(betast)^2 - $
	 cos(gammast)^2 + 2*cos(alphast)*cos(betast)*cos(gammast))
c= ast*bst*sin(gammast)/vstar

; Matrix that relates the Miller indices to the coordinates in reciprocal
; space:
Mksi = [[ 0., bst*sin(gammast), -cst*sin(betast)*cosalpha], $
	[ 0., 0., 1/c], $
	[ ast, bst*cos(gammast), cst*cos(betast)]]

; The matrices that corresponds to the three missetting angles:
Mx = [ 	[ 1.0, 0.0, 0.0], $
	[ 0.0, cos(phix), -sin(phix)], $
	[ 0.0, sin(phix), cos(phix)] ]
My = [ 	[ cos(phiy), 0.0, sin(phiy)], $
	[ 0.0, 1.0, 0.0], $
	[ -sin(phiy), 0.0, cos(phiy)] ]
Mz = [ 	[ cos(phiz), -sin(phiz), 0.0], $
	[ sin(phiz), cos(phiz), 0.0], $
	[ 0.0, 0.0, 1.0] ]
; To get the matrices right in the multiplication we need to (IDL...)
Mksi= TRANSPOSE(Mksi)
Mx= TRANSPOSE(Mx)
My= TRANSPOSE(My)
Mz= TRANSPOSE(Mz)

; ksi is the coordinates in reciprocal space of the rotated reciprocal
; lattice vector. (x is along the beam and z is horisontal (i.e. along
; the calculated y-coordinate).)
ksi= Mz # My # Mx # Mksi # x.m

; Calculate the coordinates on a flat non-distorted detector:
IF ((size(ksi))(0) eq 2) THEN BEGIN ; An array or only one reflection?
  k= (ksi(0,*)^2+ksi(1,*)^2+ksi(2,*)^2)/(-2*ksi(0,*))
  x.x = x.x*0.0+ksi(1,*)/(k+ksi(0,*)) * cdd 
  x.y = x.y*0.0+ksi(2,*)/(k+ksi(0,*)) * cdd 
ENDIF ELSE BEGIN
  k= (ksi(0)^2+ksi(1)^2+ksi(2)^2)/(-2*ksi(0))
  x.x = x.x*0.0+ksi(1)/(k+ksi(0)) * cdd 
  x.y = x.y*0.0+ksi(2)/(k+ksi(0)) * cdd 
ENDELSE

; Adjust the coordinates for twist, tilt and bulge:
tilt =  0.01*tilt*!pi/180.0
twist =  0.01*twist*!pi/180.0
bulge = 0.01*bulge*!pi/180.0
K = 1 + (tilt*x.x + twist*x.y + $
		bulge*sqrt((x.x)^2 + (x.y)^2))/cdd
; Corrected positions (we multiply with yscale since after refinement
; the observed positions in mm are corrected by dividing with yscale
; and here we apply yscale to the predicted positions):
x.x = K*x.x + cenx
x.y = K*x.y*yscale + ceny
; Which gives the distances between the observed and calculated positions:
d= sqrt((x.x-y.x)^2+(x.y-y.y)^2)

RETURN, d

END
