pro SPOTLAUE3,struc,image,par,reb, show=show
;+
; NAME:
;	SPOTLAUE3
;
; PURPOSE:
;	Interactively compares real Laue data with expected spots
;
; CATEGORY:
;	Data processing
;
; CALLING SEQUENCE:
;	SPOTLAUE3,struc,image,par,reb,/show
;
; INPUTS:
;	struc : the reflection list (containing integrated intensities)
;	image : the original data image
;	par : the reflection list parameters
;	reb : the rebinning factor of the displayed image
;	show : set this keyword to get a surface plot of the data
;
; OUTPUTS:
;	Spots : the reflection list inside the part of the data called "image"
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;
; PROCEDURE:
;	The user clicks with left mouse on one spot from real data (usually
;	a small portion of the whole array). If there is a simulated spot
;	whose location is near enough the clicked position, the information
;	concerning this spot will be print out from the "struc" parameter.
;	In particular, all distances refer to the center of the Laue image
;	as determined by LAUEGEN.
;
; MODIFICATION HISTORY:
;	D.Bourgeois, January 1994.
;-


!ERR = 0
;beam center
centerX = par.cenx - 1
centerY = par.ceny - 1

;yscale
scaleY = par.yscale

;original pixel size [micrometers]
pix_size = par.raster
pix_sizeX = 0.001*pix_size
pix_sizeY = 0.001*pix_size/scaleY

;position of the lower left coner of "image" (usually a small portion of the
;	whole data set, relative to the whole data set [pixels]
relX = 0*2 ;<---- HERE
relY = 0*2 ;<--- HERE

;this is the size of the box to look in to search for a spot arount the
;click point
boxsize  = 16  ; <---- HERE
;3D display
boxdisplay  = 16  ; <---- HERE

;size of "image"
imageX = (size(image))(1)
imageY = (size(image))(2)

;create a prediction image
simul = INTARR(imageX,imageY)

;position of the studied area relative to the center of the beam [mm]
minX = pix_sizeX*(relX-centerX)
minY = pix_sizeY*(relY-centerY)
maxX = pix_sizeX*(relX-centerX+imageX-1)
maxY = pix_sizeY*(relY-centerY+imageY-1)
;PRINT,'Position of lower left corner of image relative to center of beam (X,Y) [mm] :',minX,minY

;create a window containing the simulated data and a small for 3D surface
;representation
;save the window numbers
main_window = !D.WINDOW
;sim_window = !D.WINDOW
WIN,400,400
surf_window = !D.WINDOW

;create a reduced structure array which contains only the spots of interest
;  The criteria of selection is that the position of the spot be inside
;  the box defined by minX, maxX, minY and maxY
spots = struc(WHERE((struc(*).x GE minX) AND $
		    (struc(*).x LE maxX) AND $
		    (struc(*).y GE minY) AND $
		    (struc(*).y LE maxY)))

PRINT,'Total number of spots selected : ',(SIZE(spots))(1)

;display these spots : for each of them, compute the position in "simul"
;in pixels
FOR i=0,(size(spots))(1)-1 DO BEGIN
	;xpos = ROUND(spots(i).x/pix_sizeX+centerX-relX)
	;ypos = ROUND(spots(i).y/pix_sizeY+centerY-relY)
	xpos = FIX(spots(i).x/pix_sizeX+centerX-relX)
	ypos = FIX(spots(i).y/pix_sizeY+centerY-relY)
	simul(xpos,ypos) = 100 ;spots(i).l
ENDFOR
;WSET,sim_window
;TVSCL,simul

;We come back to the window containing the real data
WSET,main_window

;print information about spots in interactive way
save_err = 0

WHILE save_err NE 4 DO BEGIN
  WSET,main_window
  CURSOR,X,Y,/DEVICE,/DOWN
  save_err=!err
  box = image(MAX([0,reb*X-boxsize/2.+1]):MIN([imageX-1,reb*X+boxsize/2.]),$
	      MAX([0,reb*Y-boxsize/2.+1]):MIN([imageY-1,reb*Y+boxsize/2.]))
;we want to compare the pixel position inside that box with candidate 
;positions for simulated spots
;the positions of the corners of the selected box relative to beam center [mm]
minboxX = minX+pix_sizeX*(MAX([reb*X-boxsize/2.0,0]))
minboxY = minY+pix_sizeY*(MAX([reb*Y-boxsize/2.0,0]))
maxboxX = minX+pix_sizeX*(MIN([reb*X+boxsize/2.0-1,imageX-1]))
maxboxY = minY+pix_sizeY*(MIN([reb*Y+boxsize/2.0-1,imageY-1]))
;PRINT,'Position of lower left corner of box relative to center of beam (X,Y) [mm] :',minboxX,minboxY
;then repeat the same selection procedure
poss_index = WHERE((spots(*).x GE minboxX) AND $
		    (spots(*).x LE maxboxX) AND $
		    (spots(*).y GE minboxY) AND $
		    (spots(*).y LE maxboxY),count)

loc=WHERE(box LT MAX(box)/1.2,count2)
IF count2 NE 0 THEN box(loc) = 0
;get the integrated row and column
row=TOTAL(box,2)
column=TOTAL(box,1)
;compute center of gravity
ind=INDGEN(boxsize)
cenX=TOTAL(ind*row)/TOTAL(row)
cenY=TOTAL(ind*column)/TOTAL(column)
;  distance to center of beam of actual data
;  distX=minboxX+pix_sizeX*cenX
;  distY=minboxY+pix_sizeY*cenY
;  PRINT,'X : ',X, ' Y : ',Y
;  PRINT,'Position of center of gravity [mm] (real data) : ',distX,distY
;  IF count NE 0 THEN PRINT,$
;	'Distance from simulated to center of mass of real [mm] : ',$
;        ABS(distX-spots(poss_index(0)).x),ABS(distY-spots(poss_index(0)).y)
;  PRINT,'Position of center of gravity inside box [pixels] (real data) : ',cenX,cenY
;  PRINT,'Peak value :',MAX(box)
;  PRINT,'Spot location [x,y] :',cenX+reb*X-boxsize/2+1,cenY+reb*Y-boxsize/2+1
;  XYOUTS,(cenX+reb*X-boxsize/2+1)/FLOAT(reb),(cenY+reb*Y-boxsize/2+1)/FLOAT(reb),'*',/DEVICE

 
peak_minX = MAX([0,cenX+reb*X-boxsize/2. - boxdisplay/2.])
peak_maxX = MIN([imageX-1,cenX+reb*X - boxsize/2. + boxdisplay/2.])
peak_minY = MAX([0,cenY+reb*Y-boxsize/2. - boxdisplay/2.])
peak_maxY = MIN([imageY-1,cenY+reb*Y - boxsize/2. + boxdisplay/2.])


 IF KEYWORD_SET(show) THEN BEGIN
  WSET,surf_window

  peak = image(MAX([0,cenX+reb*X-boxsize/2. - boxdisplay/2.]):MIN([imageX-1,cenX+reb*X - boxsize/2. + boxdisplay/2.]),MAX([0,cenY+reb*Y-boxsize/2. - boxdisplay/2.]):MIN([imageY-1,cenY+reb*Y - boxsize/2. + boxdisplay/2.]))

  simpeak = simul(MAX([0,cenX+reb*X-boxsize/2. - boxdisplay/2.]):MIN([imageX-1,cenX+reb*X - boxsize/2. + boxdisplay/2.]),MAX([0,cenY+reb*Y-boxsize/2. - boxdisplay/2.]):MIN([imageY-1,cenY+reb*Y - boxsize/2. + boxdisplay/2.]))

 SURFACE,peak,CHARSIZE=1.5,XTITLE='X Dim [Pixels]',YTITLE='Y Dim [Pixels]',ZTITLE='Intensity [Counts]',/SAVE

; CONTOUR,peak,CHARSIZE=1.5,ZVALUE=1,/T3D,/NOERASE,LEVELS=[(1+indgen(10))/10.*max(peak)]

 CONTOUR,simpeak,CHARSIZE=1.5,ZVALUE=1,/T3D,/NOERASE,LEVELS=[(1+indgen(2))/2.*(max(simpeak)>1)]

 ENDIF






PRINT,''
PRINT,'*******************************************'
PRINT,'Number of spots selected : ',count
PRINT,'*******************************************'


IF count LT 1 THEN PRINT,'No spot found' ELSE BEGIN
 FOR index=0,count-1 DO BEGIN
	PRINT,''
	PRINT,'@@@@@@'
	PRINT,'Spot # :',index+1
	PRINT,'@@@@@@'
	PRINT,'Miller indices : ', spots(poss_index(index)).m
	PRINT,'X and Y position (predicted) [mm]: ', spots(poss_index(index)).x,spots(poss_index(index)).y
	;xpos = ROUND(spots(poss_index(index)).x/pix_sizeX+centerX-relX)
	;ypos = ROUND(spots(poss_index(index)).y/pix_sizeY+centerY-relY)
	xpos = FIX(spots(poss_index(index)).x/pix_sizeX+centerX-relX)
	ypos = FIX(spots(poss_index(index)).y/pix_sizeY+centerY-relY)
	PRINT,'X and Y position (predicted) [pixels in box]: ', xpos - peak_minX,ypos-peak_minY
	PRINT,'Wavelength [A] ; Energy [keV] : ', spots(poss_index(index)).l,12.4/spots(poss_index(index)).l
	PRINT,'Resolution : ', spots(poss_index(index)).res
	PRINT,'Multiplicity : ', spots(poss_index(index)).mul
ovlp =  spots(poss_index(index)).ovlp
IF ovlp GT 0 THEN PRINT,'Spot overlapped with ',ovlp,' other spots !' ELSE $
	PRINT,'Non overlapped spot !'
IF  spots(poss_index(index)).intb(0) EQ  spots(poss_index(index)).intp(0) THEN $
	PRINT,'Box integrated intensity : ', spots(poss_index(index)).intp(0) ELSE BEGIN
PRINT,'Profile fitted integrated intensity : ', spots(poss_index(index)).intp(0)
	PRINT,'Goodness of fit : ', spots(poss_index(index)).gfit
ENDELSE
	PRINT,'Standard deviation : ', spots(poss_index(index)).isigp(0)
	PRINT,'I/Sigma(I) : ', spots(poss_index(index)).intp(0)/spots(poss_index(index)).isigp(0)
	PRINT,'Multiplicity : ', spots(poss_index(index)).mul
flag =  spots(poss_index(index)).flag
IF flag EQ 0 THEN BEGIN
	PRINT,'Correctly integrated spot !'

ENDIF
IF flag EQ 1 THEN PRINT,'Bad background !'
IF flag EQ 2 THEN PRINT,'Bad integrated intensity !'
IF flag EQ 3 THEN PRINT,'Spot predicted overlapped but treated as single !'
IF flag EQ 4 THEN PRINT,'Weak spot !'

 ENDFOR
ENDELSE

ENDWHILE 
PRINT,''
WSET,surf_window
WDELETE
END








