function LOC_REFINE1,struc,image,par,min_snr,weak,sigma,image2, show=show, change_pred=change_pred
;+
; NAME:
;	LOC_REFINE1
;
; PURPOSE:
;
;	Compares real Laue data with expected spots : finds out those of the 
;	simulated spots which are actually matching real data. The program
;	is more advanced than matchlaue.pro. First the predicted spots are
;	sorted so that the more "likely" spots are treated before. For each
;	predicted spots, the program looks if there is a peak in the proximity,
;	(inside a box of 9 pixels) positions itself at the maximum peak value
;	inside that box, and then looks at the neibhouring pixels around that 
;	maximum
;	pixel. If none of these neibhouring pixels are of higher intensity
;	than the maximum pixel, then this pixel is considered as a real peak.
;	It is put to the mean background value as well as the neibhouring
;	pixels, in order not to be picked again by another predicted spot
;	of lower likelyhood. If one of the neibhouring pixels is of higher
;	intensity than the maximum pixel, then the spot is considered as
;	ambiguous : what was picked in that case is probably the tail of a spot
;	corresponding to another predicted spot. The predicted spot under 
;	study is then put on a waiting list, and reprocessed after the rest.
;	If the peak is not picked in the meantime, it will be taken into 
;	account. 
;	Change as compared to MATCHLAUE2.PRO :
;	1/ The background calculation is now done with the routine 
;	BACK2D_POLY4.pro
;	2/ The CHANGE_PRED keyword is only applicable to spots with a signal 
;	over background higher than a minimum value taken as SQRT(2)*(expected 
;	variance)
;	
;
; CATEGORY:
;	Data processing
;
; CALLING SEQUENCE:
;	matched = LOC_REFINE1(struc,image,par,sigma,/show,/change_pred)
;
; INPUTS:
;	struc : the reflection list (a structure produced with the
;	        program REFLIST to read the content of the CCP4 file
;		"*.geasc". 
;	image : the original data image (usually part of the full data)
;
;	par : the parameters (from reflist.pro)
;	min_snr : minimum I/Sig(I) for the maximum pixel value in a peak 
;		to select between "weak"
;		peaks and "not weaks" peaks [ex: min_snr = 2 means that
;		the peak value must be at least twice its variance not to be
;		considered as a weak peak]
;	show : keyword set if the surfaces are to be displayed
;	change_pred : keyword_set if the x,y coordinates are to be changed to match
;		the actual right position of the observed peak
;
; OUTPUTS:
;	matched : the reflection list matching the real data
;	weak : optional output containing the weak (unobserved) spots
;	sigma : optional output giving the SQUARE of the distance
;	        between observed and predicted
;	image2 : optionnal output ; contains an image of the background.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None
; PROCEDURE:
;	See above
;
; MODIFICATION HISTORY:
;	D.Bourgeois, February 1994.
;-

!ERR = 0
;PARAMETERS TO CHANGE SEE THE SIGN " <---- HERE"

bctime=0
tottime=0

;a spot is saturated above
satvalue = par.satvalue

PRINT,'Saturation limit :',satvalue
PRINT,'I/Sig(I) (of central pixel) ratio limit for peak selection :',min_snr


;position of the beam center on the original Laue image [pixels] 
centerX = par.cenx-1
centerY = par.ceny-1

;pixel size [mm]
; scale_y is taken from LAUEGEN
  scaleY = par.yscale
;rebinning factor (if a compressed version of the image is used)
  reb = 1; <---- HERE
;original pixel size [micrometers]
  pix_size = par.raster
pix_sizeX = 0.001*reb*pix_size
pix_sizeY = 0.001*reb*pix_size/scaleY

;position of the lower left coner of "image" (usually a small portion of the
;	whole data set, relative to the whole data set [pixels]
relX = 0*2; <---- HERE
relY = 0*2; <---- HERE
;PRINT,'Position of the lower left corner on primitive image :'
;PRINT,relX,' : X dim [pixels]',relY,' : Y dim [pixels]'

;this is the size of the box to look in to search for a spot arount the
;click point
smallbox_size  = 1 ; <---- HERE
;PRINT,'Box size to look around for peaks [pixels] :',smallbox_size

;box for background
bigbox_size  = {box, x:FIX(par.boxsize.x/2.), y:FIX(par.boxsize.y/2.)}

;PRINT,'Box size for background calculation [rasterx,rastery] :',bigbox_size.x,bigbox_size.y
;PRINT,'Size of search box [pixels^2] :',(2*smallbox_size+1)^2
;PRINT,'Size of background search box [pixels^2] :',4*bigbox_size.x*bigbox_size.y


;size of "image"
imageX = (size(image))(1)
imageY = (size(image))(2)

;position of the studied area relative to the center of the beam [mm]
minX = pix_sizeX*(relX-centerX+bigbox_size.x+2)
minY = pix_sizeY*(relY-centerY+bigbox_size.y+2)
maxX = pix_sizeX*(relX-centerX+imageX-1-bigbox_size.x-2)
maxY = pix_sizeY*(relY-centerY+imageY-1-bigbox_size.y-2)
;PRINT,'Position of lower left corner of image relative to center of beam (X,Y) [mm] :',minX,minY

IF KEYWORD_SET(show) THEN BEGIN
;create a window containing the simulated data and a small for 3D surface
;representation
;save the window numbers
main_window = !D.WINDOW
WIN,512,512
surf_window = !D.WINDOW
ENDIF

;create a reduced structure array which contains only the spots of interest
;  The criteria of selection is that the position of the spot be inside
;  the box defined by minX, maxX, minY and maxY
spots = struc(WHERE((struc(*).x GE minX) AND $
		    (struc(*).x LE maxX) AND $
		    (struc(*).y GE minY) AND $
		    (struc(*).y LE maxY)))

PRINT,'Number of input spots : ',(SIZE(struc))(1)
PRINT,'Number of spots selected : ',(SIZE(spots))(1)
sigma = FLTARR((SIZE(spots))(1))
sigma_b = FLTARR((SIZE(spots))(1))

;sort the reflections in "spots" according to resolution sphere
PRINT,'Sorting the spots according to resolution'
spots = spots(REVERSE(SORT(spots.res)))


;define the potentially matched spots
matched = spots
j = 0L

;define the number of saturated spots
number_sat = 0

;define the number of spots shifted
number_shifted = 0L

;define the potentially weak spots
number_weak = 2000
l = 0
weak = REPLICATE(spots(0),number_weak)


;define a waiting list ; we allow for 75% of waiting spots
number_wait = LONG((SIZE(spots))(1)*0.75)
k = -1
waitspots = REPLICATE(spots(0),number_wait)
waitpar = REPLICATE({maxval0:0.,maxval1:0.,maxval2:0.,bc_coeff:FLTARR(3),maxpos : FLTARR(2), background : FLTARR(par.boxsize.x,par.boxsize.y), flag:0 },number_wait)

;make a backup of the original image
image2 = image

;Go through all reflections of the structure spots
IF KEYWORD_SET(change_pred) THEN BEGIN
	PRINT,'Doing local refinement ...'
ENDIF ELSE PRINT,'Matching spots ...'
cr = STRING("15b)

;tottime=SYSTIME(1)
FOR i=LONG(0),(SIZE(spots))(1)-1 DO BEGIN

;m0=3
;m1=0
;m2=39
;IF ((spots(i).m(0) EQ m0) AND (spots(i).m(1) EQ m1) AND (spots(i).m(2) EQ m2)) THEN show=1 ELSE show=0

IF i/100 EQ i/100.0 THEN PRINT,FORMAT="($,I5,' ie : ',F3.0,'%',A)",i,100.*i/FLOAT((SIZE(spots))(1)),cr
;IF i/100 EQ i/100.0 THEN PRINT,'Time for bc : ',bctime,'  Total time : ',SYSTIME(1)-tottime

 ;get the position in pixels
 ;We have to use FIX, not ROUND : here is the justification : each pixel contains the integrated signal
 ;within that pixel. For exemple if the predicted position is 2.8 then it means that pixel 2 contains
 ;the integrated signal between 2 and 3, which has a higher value that the integrated signal between 3 and 4.
 ;The perfectly centered profile should have a predicted position of 2.5

  xpos = FIX(spots(i).x/pix_sizeX+centerX-relX)
  ypos = FIX(spots(i).y/pix_sizeY+centerY-relY)

; the value of image at this position is :
orig_value = image(xpos,ypos)

 ;scan through a box around this position
smallbox = image(xpos-smallbox_size:xpos+smallbox_size,ypos-smallbox_size:ypos+smallbox_size)

;let's start by defining the background
;get the big box around the pixel
 bigbox = image(xpos-FIX(par.boxsize.x/2.):xpos+FIX(par.boxsize.x/2.),ypos-FIX(par.boxsize.y/2.):ypos+FIX(par.boxsize.y/2.))

;we now have to position "smallbox" inside "bigbox" 
maxpos = COMPLEX(bigbox_size.x,bigbox_size.y)

;bctime0=SYSTIME(1)

;background =  BACK2D_POLY_QUICK(bigbox,par.noise_percentage,par.bg_gain,par.dark_current,flag,coeff,bc_mean,n_back,chi_square,max_cycle=2)
background =  BACK2D_POLY4(bigbox,par.noise_percentage,par.bg_gain,par.dark_current,flag,coeff,bc_mean,n_back,chi_square,max_cycle=2)
;bctime=bctime+SYSTIME(1)-bctime0
;surface,200*(background2-background)/(background2+background),zrange=[0,10],zstyle=1

gain = par.gain*1.5418/spots(i).l

;we treat all spots with a minimum signal over background ratio of min_snr
; IF (MAX(smallbox-par.dark_current) GT min_snr*(background(FLOAT(maxpos),IMAGINARY(maxpos))-par.dark_current)) THEN BEGIN


 IF ABS((MAX(smallbox) - background(FLOAT(maxpos),IMAGINARY(maxpos))))/SQRT(gain*(MAX(smallbox)-par.dark_current)) GT min_snr THEN BEGIN

;start careful study of the spot

;the background subtracted box where the peak is, is now :
bigbox = bigbox-background

;where is the maximum in the small box ?
;the small box should be considered background subtracted
 smallback = background(FLOAT(maxpos)-smallbox_size:FLOAT(maxpos)+smallbox_size,IMAGINARY(maxpos)-smallbox_size:IMAGINARY(maxpos)+smallbox_size)

 smallbox = smallbox-smallback

; maxpos_sbox = WHERESUB(WHERE(smallbox EQ MAX(smallbox)),smallbox)
; maxpos_im2 = [FLOAT(maxpos_sbox(0)),IMAGINARY(maxpos_sbox(0))] + $
;		[MAX([0,xpos-smallbox_size]),MAX([0,ypos-smallbox_size])]

 row=TOTAL(smallbox,2)>0.1
 column=TOTAL(smallbox,1)>0.1
 ;compute center of gravity
 indx=INDGEN((SIZE(smallbox))(1))
 indy=INDGEN((SIZE(smallbox))(2))
 cgx=TOTAL(indx*row)/TOTAL(row)
 cgy=TOTAL(indy*column)/TOTAL(column)

 maxpos_im2 = [cgx,cgy] + $
		[MAX([0,xpos-smallbox_size]),MAX([0,ypos-smallbox_size])]

;maxpos_new will now point to the new maximum inside bigbox

 maxpos_bbox = WHERESUB(WHERE(bigbox EQ MAX(smallbox)),bigbox)

IF (SIZE(maxpos_bbox))(1) GT 1 THEN BEGIN
;there are several possibilities for maxpos
;we take the one closest to the old maximum
closest=WHERE((FLOAT(maxpos-maxpos_bbox)^2+IMAGINARY(maxpos-maxpos_bbox)^2) EQ MIN(FLOAT(maxpos-maxpos_bbox)^2+IMAGINARY(maxpos-maxpos_bbox)^2))
maxpos_bbox=maxpos_bbox(closest(0))
ENDIF ELSE maxpos_bbox=maxpos_bbox(0)

;max_value is defined as the non_background subtracted maximum value in
;small box 
 max_value=image(ROUND(maxpos_im2(0)),ROUND(maxpos_im2(1)))

;Is there a higher intensity around ?
 searchbox=bigbox(FLOAT(maxpos_bbox)-1:FLOAT(maxpos_bbox)+1,IMAGINARY(maxpos_bbox)-1:IMAGINARY(maxpos_bbox)+1)

;where is the maximum of searchbox inside bigbox
maxpos_search_bbox = WHERESUB(WHERE(bigbox EQ MAX(searchbox)),bigbox)
IF (SIZE(maxpos_search_bbox))(1) GT 1 THEN BEGIN
;there are several possibilities for maxpos_search_bbox
;we take the one closest to the old maximum
closest=WHERE((FLOAT(maxpos-maxpos_search_bbox)^2+IMAGINARY(maxpos-maxpos_search_bbox)^2) EQ MIN(FLOAT(maxpos-maxpos_search_bbox)^2+IMAGINARY(maxpos-maxpos_search_bbox)^2))
maxpos_search_bbox=maxpos_search_bbox(closest(0))
ENDIF
maxpos_search_bbox=[FLOAT(maxpos_search_bbox),IMAGINARY(maxpos_search_bbox)]

searchback=background(FLOAT(maxpos_bbox)-1:FLOAT(maxpos_bbox)+1,IMAGINARY(maxpos_bbox)-1:IMAGINARY(maxpos_bbox)+1)

;the position of maxpos_search_bbox in image2 is maxpos_search_bbox(0)+xpos-bigbox_size.x,
;and same for y
 
 row=TOTAL(searchbox,2)>0.1
 column=TOTAL(searchbox,1)>0.1
 ;compute center of gravity
 indx=INDGEN((SIZE(searchbox))(1))
 indy=INDGEN((SIZE(searchbox))(2))

;the center of gravity is given by : 
 ; note the + 0.5 due to "integrated pixels"
 ; ie if we find cgx is 1 for exemple, it means that the peak
 ; center is in the middle of pixel 1, ie 1.5

 cgx=TOTAL(indx*row)/TOTAL(row) + 0.5  
 cgy=TOTAL(indy*column)/TOTAL(column) + 0.5

 maxpos_im2 = [cgx+xpos-bigbox_size.x+FLOAT(maxpos_bbox)-1,cgy+ypos-bigbox_size.y+IMAGINARY(maxpos_bbox)-1]
; maxpos_s_im2 = [maxpos_search_bbox(0)+xpos-bigbox_size.x,maxpos_search_bbox(1)+ypos-bigbox_size.y]
 maxpos_s_im2 = [cgx+xpos-bigbox_size.x+FLOAT(maxpos_bbox)-1,cgy+ypos-bigbox_size.y+IMAGINARY(maxpos_bbox)-1]

 max_search=image(FIX(maxpos_s_im2(0)),FIX(maxpos_s_im2(1)))

 ;in case yes, we put the spot on a waiting list, associated with the 
;value 'max_search' and the position of max_search
  IF MAX(searchbox) GT MAX(smallbox) THEN BEGIN
   k = k + 1
   waitspots(k) =  spots(i)
   waitpar(k).bc_coeff = [0,n_back,bc_mean]
   waitpar(k).maxval0 = orig_value  ;original value not background subtracted
   waitpar(k).maxval1 = max_search  ;not background subtracted
   waitpar(k).maxval2 = MAX(searchbox)  ;background subtracted
   waitpar(k).background = background
   waitpar(k).maxpos = maxpos_s_im2
   waitpar(k).flag = flag

  ENDIF ELSE BEGIN
;in case no, we put the spot on the matching list and we take out the found
;peak

IF KEYWORD_SET(CHANGE_PRED) THEN BEGIN ;change pred
;is max_value of sufficient signal to noise ?
;the expected variance is gain*SQRT(numberof X-rays) = gain*SQRT(number of
;X rays in the peak + number in the background) = gain*SQRT(number of counts
;in the peak/gain + (number of counts in the background-offset)/gain) =
;SQRT(gain)*SQRT(number of counts in the peak+number of counts in the 
;background - offset)
;additionnaly, spots which are too close (less than 3*SQRT(2)*raster) may 
;end up on exactly the same position. This is not allowed.

variance = SQRT(gain)*SQRT(max_value-par.dark_current)
IF (MAX(smallbox) GE SQRT(2)*variance) AND (spots(i).ovlp EQ 0) THEN BEGIN
	matched(j)=spots(i)
	matched(j).bc_coeff= [0,n_back,bc_mean]
	matched(j).intp(0) = max_value
	matched(j).x = pix_sizeX*(maxpos_im2(0) + relX - centerX)
	matched(j).y = pix_sizeY*(maxpos_im2(1) + relY - centerY)
	IF max_value GE satvalue THEN BEGIN
		;PRINT, 'Saturated spot at :',i
		matched(j).close = 1
		number_sat = number_sat + 1
	ENDIF
	number_shifted = number_shifted + 1 
ENDIF ELSE BEGIN ;min_snr not good enough
	matched(j)=spots(i)
	matched(j).bc_coeff =  [0,n_back,bc_mean]
	matched(j).intp(0) = orig_value
ENDELSE ;min_snr not good enough

ENDIF ELSE BEGIN ; keyword 'change pred' not set
	matched(j)=spots(i)
	matched(j).bc_coeff =  [0,n_back,bc_mean]
	matched(j).intp(0) = orig_value
	IF max_value GE satvalue THEN BEGIN
		PRINT, 'Saturated spot at :',i
		matched(j).close = 1
		number_sat = number_sat + 1
	ENDIF
ENDELSE ; keyword 'change pred' not set

      xpred = spots(i).x
      ypred = spots(i).y
      xobs =  pix_sizeX*(maxpos_im2(0) + relX - centerX)
      yobs =  pix_sizeY*(maxpos_im2(1) + relY - centerY)
      sigma_b(j) = (xpred - xobs)^2 + (ypred - yobs)^2
      sigma(j) = (matched(j).x - xobs)^2 + (matched(j).y - yobs)^2
	image2(FIX(maxpos_im2(0))-bigbox_size.x:FIX(maxpos_im2(0))+bigbox_size.x,$
	       FIX(maxpos_im2(1))-bigbox_size.y:FIX(maxpos_im2(1))+bigbox_size.y) = background
	j = j + 1
 
  ENDELSE ;put the spot on the matching list

IF KEYWORD_SET(show) THEN BEGIN
PRINT,''
PRINT,'Spot #',j - 1
PRINT,'Max(bigbox) (not background subtracted) :',MAX(bigbox+background)
PRINT,'Max(smallbox) (not background subtracted) :',MAX(smallbox+smallback)
PRINT,'min_snr*Bc at max(smallbox) :',min_snr*(background(FLOAT(maxpos),IMAGINARY(maxpos))-par.dark_current)
IF MAX(searchbox) GT MAX(smallbox) THEN PRINT,'--> Waiting list #',k
SURFACE,bigbox
ok=''
READ,'bigbox OK ?',ok
SURFACE,background
ok=''
READ,'background OK ?',ok
SURFACE,smallbox
ok=''
READ,'small box OK ?',ok
SURFACE,searchbox
ok=''
READ,'searchbox OK ?',ok
ENDIF ;show

ENDIF ELSE BEGIN ; treat weak spots
IF KEYWORD_SET(show) THEN PRINT,'--> Bad SNR' ;the spot was not treated
weak(l) = spots(i)
weak(l).bc_coeff =  [0,n_back,bc_mean]
;set integrated intensity to 0 and variance to 1
weak(l).intp(0) = 0
weak(l).isigp(0) = 1

l = l+1
IF l GE number_weak THEN MESSAGE,'Too many weak spots : are the predicted spots OK ?'
ENDELSE ; treat weak spots


ENDFOR ; go through each spot


;clean the weak spots
IF l GE 1 THEN weak = weak(0:l-1) ELSE weak = 0

;How many spots on the waiting list ?
PRINT,'Number of ambiguous spots :',k+1
;we examine which of these waiting spots must be selected
recovered = 0
FOR i=0L,k DO BEGIN ;go through each ambiguous spot

IF KEYWORD_SET(CHANGE_PRED) THEN BEGIN ; change pred
;is the spot not ambiguous any more ?
 IF image2(FIX((waitpar(i).maxpos)(0)),FIX((waitpar(i).maxpos)(1))) EQ  waitpar(i).maxval1 THEN BEGIN ; spot not ambiguous

;the spot can be treated
;is max_value of sufficient signal to noise ?
;the expected variance is gain*SQRT(numberof X-rays) = gain*SQRT(number of
;X rays in the peak + number in the background) = gain*SQRT(number of counts
;in the peak/gain + (number of counts in the background-offset)/gain) =
;SQRT(gain)*SQRT(number of counts in the peak+number of counts in the 
;background - offset)
;additionnaly, spots which are too close (less than 3*SQRT(2)*raster) may 
;end up on exactly the same position. This is not allowed.

gain = par.gain*1.5418/waitspots(i).l

variance = SQRT(gain)*SQRT(waitpar(i).maxval1-par.dark_current)

IF (waitpar(i).maxval2 GE SQRT(2)*variance) AND (waitpar(i).flag EQ 0) AND (waitspots(i).ovlp EQ 0) THEN BEGIN
	matched(j) = waitspots(i)
	matched(j).bc_coeff = waitpar(i).bc_coeff
	matched(j).x = pix_sizeX*((waitpar(i).maxpos)(0) + relX - centerX)
	matched(j).y = pix_sizeY*((waitpar(i).maxpos)(1) + relY - centerY)
	matched(j).intp(0) = waitpar(i).maxval1
	IF waitpar(i).maxval1 GE satvalue THEN BEGIN
		PRINT, 'Saturated spot at :',j
		matched(j).close = 1
		number_sat = number_sat + 1
	ENDIF
	number_shifted = number_shifted + 1 

ENDIF ELSE BEGIN ;min_snr not OK : don't change anything !
	matched(j) = waitspots(i)
	matched(j).bc_coeff=waitpar(i).bc_coeff
	matched(j).intp(0) = waitpar(i).maxval0
	IF waitpar(i).maxval1 GE satvalue THEN BEGIN
		PRINT, 'Saturated spot at :',j
		matched(j).close = 1
		number_sat = number_sat + 1
	ENDIF
ENDELSE ; min_snr not OK

  image2(FIX((waitpar(i).maxpos)(0))-bigbox_size.x:FIX((waitpar(i).maxpos)(0))+bigbox_size.x, FIX((waitpar(i).maxpos)(1))-bigbox_size.y:FIX((waitpar(i).maxpos)(1))+bigbox_size.y) = waitpar(i).background

recovered = recovered + 1

ENDIF ELSE BEGIN ;spot is ambiguous : don't change anything !
	matched(j) = waitspots(i)
	matched(j).bc_coeff=waitpar(i).bc_coeff
	matched(j).intp(0) = waitpar(i).maxval0
	IF waitpar(i).maxval1 GE satvalue THEN BEGIN
		PRINT, 'Saturated spot at :',j
		matched(j).close = 1
		number_sat = number_sat + 1
	ENDIF
ENDELSE ;spot ambiguous

ENDIF ELSE BEGIN; keyword 'change pred' not set : don't change anything !

	matched(j) = waitspots(i)
	matched(j).bc_coeff=waitpar(i).bc_coeff
	matched(j).intp(0) = waitpar(i).maxval0
	IF waitpar(i).maxval1 GE satvalue THEN BEGIN
	PRINT, 'Saturated spot at :',j
	matched(j).close = 1
	number_sat = number_sat + 1
	ENDIF ; saturated

;is the spot not ambiguous any more ?
IF image2(FIX((waitpar(i).maxpos)(0)),FIX((waitpar(i).maxpos)(1))) EQ  waitpar(i).maxval1 THEN BEGIN recovered = recovered + 1
  image2(FIX((waitpar(i).maxpos)(0))-bigbox_size.x:FIX((waitpar(i).maxpos)(0))+bigbox_size.x, FIX((waitpar(i).maxpos)(1))-bigbox_size.y:FIX((waitpar(i).maxpos)(1))+bigbox_size.y) = waitpar(i).background
ENDIF ; spot not ambiguous

ENDELSE ; keyword 'change pred' not set.

	xobs = pix_sizeX*((waitpar(i).maxpos)(0) + relX - centerX)
	yobs = pix_sizeY*((waitpar(i).maxpos)(1) + relY - centerY)
	sigma_b(j) = (waitspots(i).x - xobs)^2 + (waitspots(i).y - yobs)^2
	sigma(j) = (matched(j).x - xobs)^2 + (matched(j).y - yobs)^2

j = j + 1
ENDFOR ; go through all ambiguous spots

PRINT,'Number of recovered spots :',recovered
PRINT,'Number of saturated spots :',number_sat
PRINT,'Number of weak spots :', l
PRINT,"Number of spots observed (not weak) : ",j
PRINT,"Number of spots shifted : ", number_shifted
IF KEYWORD_SET(show) THEN WSET,main_window
sigma = sigma(0:j-1)
mean = TOTAL(SQRT(sigma))/(SIZE(sigma))(1)
std = SQRT(TOTAL(sigma)/(SIZE(sigma))(1))
sigma_b = sigma_b(0:j-1)
mean_b = TOTAL(SQRT(sigma_b))/(SIZE(sigma_b))(1)
std_b = SQRT(TOTAL(sigma_b)/(SIZE(sigma_b))(1))
;std = STDEV(sigma,mean)

IF KEYWORD_SET(change_pred) THEN BEGIN
PRINT,'Average distance from predicted to observed [um, before refinement] :',mean_b*1000.
PRINT,'RMS distance from predicted to observed [um, before refinement] :',std_b*1000.
PRINT,'Average distance from predicted to observed [um, after refinement] :',mean*1000.
PRINT,'RMS distance from predicted to observed [um, after refinement] :',std*1000.
ENDIF ELSE BEGIN
PRINT,'Average distance from predicted to observed [um, no refined] :',mean_b*1000.
PRINT,'RMS distance from predicted to observed [um, not refined] :',std_b*1000.
ENDELSE
;tottime=SYSTIME(1)-tottime
RETURN,matched(0:j-1)
END
