FUNCTION INTEGRATE_PF4,spot_list,ref_profile,image,red,par,ref_distance,ref_ang_width,chi_square_test,peak_variance,success_flag,show_weak=show_weak,show_ovlp=show_ovlp,show_m=show_m,miller=miller,sym_rad=sym_rad
;+
; NAME:
;	INTEGRATE_PF4
;
; PURPOSE:
;	Profile fitting by standard Least Square method, with overlap
;	deconvolution. Optimization of fitting cutoff level.
;
; CATEGORY:
;	Laue processing.
;
; CALLING SEQUENCE:
;	integrated = INTEGRATE_PF4(spot_list,ref_profile,image,par, $
;			ref_distance,ref_ang_width,chi_square_test,$
;			peak_variance,success_flag,/show_weak,/show_ovlp,$
;			show_m,miller=miller,sym_rad=sym_rad)
;
; INPUTS:
;	spot_list : a structure containing the spot information
;	ref_profile : list of structure containing the learned profiles, 
;	and associated parameters
;	image : the original image
;	red : the rebinned version of image
;	par : the parameters of the Laue experiment
;	ref_distance : the maximum radial distance to look for a reference peak
;	   	from the position of the currently integrated peak [mm]
;	ref_ang_width : the maximum tengential distance to look for a 
;		reference peak from the position of the currently integrated 
;		peak [degrees]
;	show_weak : keyword set if the procedure is to be displayed for weak 
;			(non overlapped) peaks.
;	show_ovlp : keyword set if the procedure is to be displayed for 
;			overlapped peaks.
;	show_m : keyword set if one reflection specified by miller is to be
;		 checked
;	miller : INTARR[3] containing the Miller indices of the reflection 
;		to be studied if the keyword /show_m is set
;	sym_rad : set this keyword if the reference peak search is to be extented
;		to symmetrical sector if necessary
;
; OUTPUTS:
;	integrated : a FLTARR containing the integrated intensities
;	chi_square_test : [optionnal] gives an estimate of the goodness of fit
;	peak_variance : [optionnal] gives the expected variance of peaks
;	success_flag : [optionnal] returns 0 if integration successful
;		1, if problem with background, 2 if problem at integration
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	Its an unbiased profile fitting procedure. 
;	FIRST --> the spots are recognized as being overlapped or not.
;	SECOND --> CASE OF OVERLAPPED SPOTS
;	The spots involved in the overlap are obtained with the routine 
;	GET_OVLP_REG.PRO.
;	Each of these spots are background subtracted with the routine
;	BACK2D_POLY4.PRO.
;	Then a learned profile is obtained for each spot by calculating
;	a weighted average of the neighbouring reference profiles.
;	A LEAST_SQUARE correlation matrix is computed with the help
;	of the routine GET_CORREL2.PRO.
;	This matrix is inverted by the Cholewsky method. (NR_CHOLDC,NR_CHOLSL)
;	A chi-square test is evaluated for the integration.
;	THIRD --> CASE OF NON OVERLAPPED SPOTS
;	Each spot is background subtracted with the routine
;	BACK2D_POLY4.PRO.
;	Then a learned profile is obtained by calculating
;	a weighted average of the neighbouring reference profiles.
;	The integration is done by unbiased LEAST_SQUARE fit (J minimizes
;	the criterium Sum(J*Pi - Qi)^2, with Pi being the intensities,
;	of the learned profile, and Qi being the measured intensities)
;	over a limited region determined by the routine GET_FITLEVEL.PRO.
;	A chi-square test is evaluated for the integration.
;	The variances are calculated with the routine GET_PFVAR.PRO
;	If the /show_weak or /show_ovlp keywords are used, various information
;	 are printed,
;	and the profiles (learned, original and difference) are plotted.
;
; MODIFICATION HISTORY:
;	Dominique Bourgeois, June 95.
;-

;ON_ERROR,2                              ;Return to caller if an error occurs

;*************
;GET GENERAL PARAMETERS
;*************

;get the dimension and size of the spot_list
number_spots=(SIZE(spot_list))(1)

;get the dimension and size of the reference profile
sizearea_x = par.boxsize.x
sizearea_y = par.boxsize.y

;get pixel sizes and position of the image center
centerX = par.cenx - 1
centerY = par.ceny - 1
scaleY = par.yscale
pix_size = par.raster
pix_sizeX = 0.001*pix_size
pix_sizeY = 0.001*pix_size/scaleY


;create the array for the integrated intensities
integrated = FLTARR(number_spots)
chi_square_test = FLTARR(number_spots)
peak_variance = FLTARR(number_spots)
success_flag = FLTARR(number_spots)
ovlp_err = 0
adapted_profile = FLTARR(par.boxsize.x,par.boxsize.y)

IF KEYWORD_SET(show_weak) OR KEYWORD_SET(show_ovlp) THEN WIN,512,512


;if a particular reflexion is to be studied, is it overlapped or not

IF KEYWORD_SET(show_m) THEN BEGIN

w = WHERE((spot_list.m(0) EQ miller(0)) AND (spot_list.m(1) EQ miller(1)) AND (spot_list.m(2) EQ miller(2)),count)
IF count EQ 1 THEN BEGIN
PRINT,'Spot found !'
;is it overlapped ?
IF spot_list(w).ovlp GT 0 THEN BEGIN
PRINT,'Overlapped spot !'
first_ref = w
old_first_ref = w
;check all reflections connected to spot_list(w)
FOR loc_i=0,spot_list(w).ovlp-1 DO BEGIN
first_ref = WHERE(spot_list.id EQ spot_list(w).connect(loc_i),count)
IF count EQ 1 THEN first_ref = first_ref < old_first_ref 
ENDFOR
miller = spot_list(first_ref).m
;give a flag
show_m_flag = 1
ENDIF ELSE show_m_flag = 2

ENDIF ; (count = 1)
ENDIF ;(show_m)

;*******************
;START INTEGRATING
;*******************

;run the procedure for each spot
FOR i=0,number_spots-1 DO BEGIN ;(for_loop1)

IF KEYWORD_SET(show_m) THEN BEGIN

IF ((spot_list(i).m(0) EQ miller(0)) AND (spot_list(i).m(1) EQ miller(1)) AND (spot_list(i).m(2) EQ miller(2))) THEN BEGIN
IF show_m_flag EQ 1 THEN show_ovlp = 1 ELSE show_weak =1
ENDIF
ENDIF



;is it an overlapped spot ?
escape :
IF spot_list(i).ovlp GT 0 THEN BEGIN ;(if_loop1) yes it is overlapped

; ***********************
;START INTEGRATING OVERLAPPED SPOTS
; ***********************

;***
;PRINT INFO
;***
IF KEYWORD_SET(show_ovlp) THEN BEGIN
PRINT,''
PRINT,'Peak number :',i
PRINT,'Miller indices :',spot_list(i).m
PRINT,'Overlapping situation'
ENDIF

;***
;get the spots involved in the overlap.
;***
; ovlp_spots is a structure similar to spot_list, but containing the 
; spots involved in the overlap.
ovlp_spots = GET_OVLP_REG(spot_list,i,index_list)
num_ovlp = (SIZE(ovlp_spots))(1)

IF num_ovlp EQ 1 THEN BEGIN
PRINT,'No overlapped spot found, treating as non overlapped ...'

;still treat the spot as non overlapped but put the flag to 3 via ovlp_err
spot_list(i).ovlp = 0
ovlp_err = 3
GOTO, escape 
ENDIF ELSE BEGIN ; (else_loop1) escape if no overlap

;define spots caracteristiques
cara_spots = REPLICATE({xpos : 0, ypos : 0, area : FLTARR(sizearea_x,sizearea_y),indices : LONARR(sizearea_x*sizearea_y),adapted_profile : FLTARR(sizearea_x,sizearea_y), background : FLTARR(sizearea_x,sizearea_y), bc_coeff : FLTARR(3), flag : 0, count : 0, fit_level : 0.0, opt_peak_area : INTARR(sizearea_x*sizearea_y),peak_area : INTARR(sizearea_x*sizearea_y),peak_indices : LONARR(sizearea_x*sizearea_y),bg_indices : LONARR(sizearea_x*sizearea_y),bg_area : INTARR(sizearea_x*sizearea_y)}, num_ovlp)

;***
;START STUDY FOR EACH OVERLAPPED SPOT
;***
FOR k=0,num_ovlp-1 DO BEGIN ;(for_loop2)

;get the center [raster]
cara_spots(k).xpos = ROUND(ovlp_spots(k).x/pix_sizeX+centerX)
cara_spots(k).ypos = ROUND(ovlp_spots(k).y/pix_sizeY+centerY)

;get the corresponding area in image
cara_spots(k).area = image(cara_spots(k).xpos - sizearea_x/2 : $
    cara_spots(k).xpos + sizearea_x/2-1 , cara_spots(k).ypos - $
    sizearea_y/2:cara_spots(k).ypos + sizearea_y/2-1)

;****
;get a "learned profile" by a weighted average of the reference profiles
;****
;select the possible reference profiles :
radius_i = SQRT(ovlp_spots(k).x^2+ovlp_spots(k).y^2)
angle_i =  ATAN(ovlp_spots(k).y,ovlp_spots(k).x)*180/!Pi
r_max = radius_i + ref_distance/2.
r_min = (radius_i - ref_distance/2.)>0
phi_max = angle_i + ref_ang_width/2.
IF phi_max GT 180 THEN phi_max = phi_max - 360
phi_min = angle_i - ref_ang_width/2.
IF phi_min LT -180 THEN phi_min = phi_min + 360


IF phi_min LE phi_max THEN $
ref_index = WHERE( ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max AND $
		ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min AND $
		SQRT(ref_profile.x^2 + ref_profile.y^2) LE r_max AND $
		SQRT(ref_profile.x^2 + ref_profile.y^2) GT r_min,n_ref) ELSE $
ref_index = WHERE( ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max OR $
		ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min AND $
		SQRT(ref_profile.x^2 + ref_profile.y^2) LE r_max AND $
		SQRT(ref_profile.x^2 + ref_profile.y^2) GT r_min,n_ref)


IF n_ref LE 1 THEN BEGIN
;less than 2 reference peaks found ! Take twice full radial sector instead !

IF NOT KEYWORD_SET(sym_rad) THEN BEGIN
phi_max = angle_i + ref_ang_width
IF phi_max GT 180 THEN phi_max = phi_max - 360
phi_min = angle_i - ref_ang_width
IF phi_min LT -180 THEN phi_min = phi_min + 360

IF phi_min LE phi_max THEN $
ref_index = WHERE( ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max AND $
		ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min,n_ref) ELSE $
ref_index = WHERE( ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max OR $
		ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min,n_ref)

IF n_ref GT 0 THEN BEGIN
IF KEYWORD_SET(show_ovlp) THEN PRINT,'No close reference peaks ! Ref peaks taken from full radial sector !'
ENDIF ELSE MESSAGE,'Choose larger area for reference peak search !'
ENDIF ELSE BEGIN
;********************
;include symmetrical sector

IF angle_i GE 0 THEN angle_i2 = angle_i - 180 ELSE angle_i2 = angle_i + 180 

phi_max = angle_i + ref_ang_width
phi_max2 = angle_i2 + ref_ang_width
IF phi_max GT 180 THEN phi_max = phi_max - 360
IF phi_max2 GT 180 THEN phi_max2 = phi_max2 - 360

phi_min = angle_i - ref_ang_width
phi_min2 = angle_i2 - ref_ang_width
IF phi_min LT -180 THEN phi_min = phi_min + 360
IF phi_min2 LT -180 THEN phi_min2 = phi_min2 + 360

IF ((phi_min LE phi_max) AND (phi_min2 LE phi_max2)) THEN $
ref_index = WHERE((ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max AND $
		   ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min) OR $
		  (ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max2 AND $
                   ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min2),n_ref)

IF ((phi_min LE phi_max) AND (phi_min2 GT phi_max2)) THEN $
ref_index = WHERE((ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max AND $
		   ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min) OR $
		  (ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max2 OR $
                   ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min2),n_ref)

IF ((phi_min GT phi_max) AND (phi_min2 LE phi_max2)) THEN $
ref_index = WHERE((ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max OR $
		   ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min) OR $
		  (ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max2 AND $
                   ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min2),n_ref)

IF ((phi_min GT phi_max) AND (phi_min2 GT phi_max2)) THEN $
ref_index = WHERE((ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max OR $
		   ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min) OR $
		  (ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max2 OR $
                   ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min2),n_ref)


IF n_ref GT 0 THEN BEGIN
IF KEYWORD_SET(show_ovlp) THEN PRINT,'No close reference peaks ! Ref peaks taken from full radial sector ! (including symmetrical sector)'
ENDIF ELSE MESSAGE,'Choose larger area for reference peak search !'
ENDELSE
;********************
ENDIF

possible_ref = ref_profile(ref_index)

;Take the 5 highest/close to spot profiles if more than 5.
IF n_ref GT 5 THEN possible_ref = (possible_ref(REVERSE(SORT(possible_ref.intb(0)/SQRT((possible_ref.x-ovlp_spots(k).x)^2 + (possible_ref.y-ovlp_spots(k).y)^2)))))(0:(4 < (n_ref-1)))


IF MIN((possible_ref.x-ovlp_spots(k).x)^2 + (possible_ref.y-ovlp_spots(k).y)^2) EQ 0 THEN BEGIN
;We are in this case in the presence of one of the reference spots !
;This will actually never happen for overlapped spots !
index = WHERE(((possible_ref.x-ovlp_spots(k).x)^2 + (possible_ref.y-ovlp_spots(k).y)^2) EQ 0)
cara_spots(k).adapted_profile = possible_ref(index).sp -possible_ref(index).bc

ENDIF ELSE FOR index=0,(SIZE(possible_ref))(1)-1 DO cara_spots(k).adapted_profile = cara_spots(k).adapted_profile + (possible_ref(index).sp - possible_ref(index).bc)*1/SQRT((possible_ref(index).x-ovlp_spots(k).x)^2 + (possible_ref(index).y-ovlp_spots(k).y)^2)


;normalize the integrated intensity of the reference profile to 1.
cara_spots(k).adapted_profile = cara_spots(k).adapted_profile/TOTAL(cara_spots(k).adapted_profile)


;allowed indices for background calculation : at 0.1% of maximum or minimum level to get at least 20% of pixels.
scale_factor = 1
;REPEAT BEGIN
;cara_spots(k).bg_area(*)=-1
;cara_spots(k).bg_area = WHERE(cara_spots(k).adapted_profile LE MAX(cara_spots(k).adapted_profile)*0.01*scale_factor,ct)
;cara_spots(k).peak_area = WHERE(cara_spots(k).adapted_profile GT MAX(cara_spots(k).adapted_profile)*0.01*scale_factor)
;scale_factor = 2*scale_factor
;ENDREP UNTIL (ct GT 0.35*(SIZE(cara_spots(k).adapted_profile))(4))

;&&&NEW
maxsub = (WHERE(cara_spots(k).adapted_profile EQ MAX(cara_spots(k).adapted_profile)))(0)
maxpos = WHERESUB(maxsub,cara_spots(k).adapted_profile)


cutoff = GET_FITLEVEL(cara_spots(k).adapted_profile,ovlp_spots(k).intp(0),ovlp_spots(k).bc_coeff(1),ovlp_spots(k).bc_coeff(2),par.dark_current)

REPEAT BEGIN
allowed_indices = SEARCH2D(cara_spots(k).adapted_profile,maxpos(0),maxpos(1),MAX(cara_spots(k).adapted_profile)*cutoff/100.*scale_factor,MAX(cara_spots(k).adapted_profile))
ct = (SIZE(allowed_indices))(1)
scale_factor = 2*scale_factor
ENDREP UNTIL (ct LE 0.7*(SIZE(cara_spots(k).adapted_profile))(4))
tmp = cara_spots(k).adapted_profile
tmp(allowed_indices)=0
cara_spots(k).bg_area(*) = -1
cara_spots(k).bg_area = WHERE(tmp NE 0)
cara_spots(k).peak_area = allowed_indices
;&&&NEW



;pixel indices in image of peak_area
x_y_coord = WHERESUB(cara_spots(k).peak_area(WHERE(cara_spots(k).peak_area GT 0)),cara_spots(k).area)

cara_spots(k).peak_indices = cara_spots(k).xpos - sizearea_x/2 + FLOAT(x_y_coord) + par.rastx*(cara_spots(k).ypos - sizearea_y/2 + IMAGINARY(x_y_coord))

;pixel indices in image of bg_area
x_y_coord = WHERESUB(cara_spots(k).bg_area(WHERE(cara_spots(k).bg_area GT -1)),cara_spots(k).area)

cara_spots(k).bg_indices = cara_spots(k).xpos - sizearea_x/2 + FLOAT(x_y_coord) + par.rastx*(cara_spots(k).ypos - sizearea_y/2 + IMAGINARY(x_y_coord))

ENDFOR ; (for_loop2) run over k


FOR k=0,num_ovlp-1 DO BEGIN
FOR k2=0,num_ovlp-1 DO BEGIN
 IF MIN(ABS(ovlp_spots(k).connect - ovlp_spots(k2).id)) EQ 0 THEN BEGIN
      	w = WHERE_BGCORREL(cara_spots(k),cara_spots(k2),nu_pix)
	IF nu_pix NE 0 THEN BEGIN
	cara_spots(k).bg_area((WHERE(cara_spots(k).bg_area GT -1))(w)) = -1
	cara_spots(k).bg_indices((WHERE(cara_spots(k).bg_indices GT 0))(w)) = 0
	ENDIF
 ENDIF ; (there is correlation) 
ENDFOR ; (over k2)
ENDFOR ;(over k)


FOR k=0,num_ovlp-1 DO BEGIN ;(for_loop22)

;***
;subtract background 
;***
cara_spots(k).background =  BACK2D_POLY5(cara_spots(k).area,cara_spots(k).bg_area(WHERE(cara_spots(k).bg_area GT -1)),par.noise_percentage,par.gain,par.dark_current,flag,coeff,bc_mean,n_back,chi_square)

;dummy=INTARR(sizearea_x,sizearea_y)
;dummy(cara_spots(k).bg_area(WHERE(cara_spots(k).bg_area GT -1)))=1
;surface,area
;ok=''
;read,'ok ?',ok

cara_spots(k).flag = flag
cara_spots(k).bc_coeff = [0,n_back,bc_mean]
cara_spots(k).area = cara_spots(k).area - cara_spots(k).background
;the value of flag can be used to possibly remove the stuff with bad background

;optimization of fit_level by variance minimization
IF KEYWORD_SET(show_weak) THEN $
cara_spots(k).fit_level = GET_FITLEVEL(cara_spots(k).adapted_profile,ovlp_spots(k).intp(0),cara_spots(k).bc_coeff(1),cara_spots(k).bc_coeff(2),par.dark_current,/show) $
ELSE cara_spots(k).fit_level = GET_FITLEVEL(cara_spots(k).adapted_profile,ovlp_spots(k).intp(0),cara_spots(k).bc_coeff(1),cara_spots(k).bc_coeff(2),par.dark_current)

cara_spots(k).opt_peak_area = WHERE(cara_spots(k).adapted_profile GE MAX(cara_spots(k).adapted_profile)*cara_spots(k).fit_level/100.,ct)

cara_spots(k).count = ct

;pixel indices in image of opt_peak_area
x_y_coord = WHERESUB(cara_spots(k).opt_peak_area(WHERE(cara_spots(k).opt_peak_area GT 0)),cara_spots(k).area)

cara_spots(k).indices = cara_spots(k).xpos - sizearea_x/2 + FLOAT(x_y_coord) + par.rastx*(cara_spots(k).ypos - sizearea_y/2 + IMAGINARY(x_y_coord))

;***
;PRINT INFO
;***
IF KEYWORD_SET(show_ovlp) THEN BEGIN
center_area = WHERESUB(WHERE(cara_spots(k).area EQ MAX(cara_spots(k).area)), cara_spots(k).area)
center_adapted = WHERESUB(WHERE(cara_spots(k).adapted_profile EQ MAX(cara_spots(k).adapted_profile)), cara_spots(k).adapted_profile)
shift_area_adapted = center_area(0) - center_adapted(0)
PRINT,'Shift between peaks [raster], [x,y] for peak ',k,' :',shift_area_adapted(0)
ENDIF

ENDFOR ; (for_loop22) run over k


;***
;ARE THERE SPOTS TOO CLOSE TO BE DECONVOLUTED 
;***
w = WHERE(ovlp_spots.close EQ 2, num_too_close)
IF num_too_close GT 0 THEN BEGIN
;separate between spots too close to be deconvoluted and spots OK !
cara_too_close = cara_spots(w)
index_too_close = index_list(w)
ovlp_too_close = ovlp_spots(w)
integrated(index_too_close) = 0
peak_variance(index_too_close) = 1
w = WHERE(ovlp_spots.close NE 2,num_ovlp)
IF num_ovlp GT 0 THEN BEGIN
cara_spots = cara_spots(w)
ovlp_spots = ovlp_spots(w)
index_list = index_list(w)
ENDIF ELSE GOTO,escape2
IF KEYWORD_SET(show_ovlp) THEN BEGIN
PRINT,'Number of spots to deconvolute : ',num_ovlp
PRINT,'Number of spots too close to deconvolute : ',num_too_close
ENDIF ; (show)
FOR k=0,num_ovlp-1 DO BEGIN
FOR k2=0,num_too_close-1 DO BEGIN
 IF MIN(ABS(ovlp_spots(k).connect - ovlp_too_close(k2).id)) EQ 0 THEN BEGIN
      	w = WHERE_CORREL(cara_spots(k),cara_too_close(k2),nu_pix)
	IF nu_pix NE 0 THEN BEGIN
	cara_spots(k).opt_peak_area((WHERE(cara_spots(k).opt_peak_area GT 0))(w)) = 0
	cara_spots(k).indices((WHERE(cara_spots(k).indices GT 0))(w)) = 0
	cara_spots(k).count = cara_spots(k).count - nu_pix
	ENDIF
 ENDIF ; (there is correlation) 
ENDFOR ; (over k2)
ENDFOR ;(over k)

;Take out spots entirely eaten
w = WHERE(cara_spots.count EQ 0,num_eaten)
IF num_eaten GT 0 THEN BEGIN
;set to 0 the integrated intensity for the eaten spots
index_eaten = index_list(w)
integrated(index_eaten) = 0
peak_variance(index_eaten) = 1
;reduce the set of ovlp spots to deconvolute
w = WHERE(cara_spots.count GT 0,num_ovlp)
IF num_ovlp GT 0 THEN BEGIN
cara_spots = cara_spots(w)
ovlp_spots = ovlp_spots(w)
index_list = index_list(w)
ENDIF ELSE GOTO,escape2
ENDIF ; spots eaten

ENDIF



;******
;START CALCULATING INTEGRATED INTENSITIES (UNBIASED FITTING PROCEDURE ; WITH FIT_LEVEL CUTOFF)
;******

;calculate the b [num_ovlp] vector 
b_vect = FLTARR(num_ovlp)
FOR k=0,num_ovlp-1 DO b_vect(k) = TOTAL(cara_spots(k).area(cara_spots(k).opt_peak_area(WHERE(cara_spots(k).opt_peak_area GT 0)))*cara_spots(k).adapted_profile(cara_spots(k).opt_peak_area(WHERE(cara_spots(k).opt_peak_area GT 0))))

;***
;calculate the correlation matrix (dimensions = [num_ovlp,num_ovlp])
;***
corr_matrix=FLTARR(num_ovlp,num_ovlp)

;this matrix is symmetric, we only need to calculate the upper region

;in order to calculate later a chi_square test we need to know the number
;of pixels in the overlapped region, which are common to 2 peaks (this
;number is related to the degree of freedom). We ignore
;those common to 3 peaks, which may lead to a small error at the end
number_common_indices = 0

FOR row=0,num_ovlp-1 DO BEGIN ;(for_loop3)
 FOR column=0,row DO BEGIN ;(for_loop4)
 ;calculate corr_matrix(row,column)
 IF row EQ column THEN corr_matrix(row,column)= TOTAL(cara_spots(row).adapted_profile(cara_spots(row).opt_peak_area(WHERE(cara_spots(row).opt_peak_area GT 0)))^2) ELSE BEGIN
 ;cross term --> look for a correlation
 IF MIN(ABS(ovlp_spots(row).connect - ovlp_spots(column).id)) EQ 0 THEN BEGIN
 ;there is correlation : we look for the common pixels
 ;print,'Connecting id ',ovlp_spots(row).id,' with id ',ovlp_spots(column).id
 corr_matrix(row,column) = GET_CORREL2(cara_spots(row),cara_spots(column),nu_pix)
 number_common_indices = number_common_indices + nu_pix
 ENDIF 
 ENDELSE ;(cross term)
 ENDFOR ;(for_loop4)
ENDFOR ;(for_loop3)

;******
;MATRIX INVERSION (Cholevsky method) and calculation of integrated intensities.
;******

;print info
IF KEYWORD_SET(show_ovlp) THEN print,'corr_matrix :',corr_matrix

;///
;produce variance matrix
orig_corr_matrix = corr_matrix
;///

;check if more than one spot is integrated
IF (SIZE(index_list))(1) GT 1 THEN BEGIN

CATCH, Error_status	
IF Error_status NE 0 THEN BEGIN	;This statement begins the error handler.
  PRINT, 'Error message:', !ERR_STRING
  ;Handles the error
  ;if the error is due to a MESSAGE command then quit the program
  IF Error_status EQ -5 THEN RETURN,0
  ;if not fix the CHOLDC problem
  corr_matrix(*)=0
  FOR loc_index=0, num_ovlp-1 DO corr_matrix(loc_index,loc_index)=1  
  cara_spots.flag = 5
ENDIF

NR_CHOLDC,corr_matrix,p
integrated(index_list) = NR_CHOLSL(corr_matrix,p,b_vect)
ENDIF ELSE integrated(index_list(0)) = b_vect(0)/corr_matrix(0) 


;****
;PRINT AND PLOT INFO
;****
IF KEYWORD_SET(SHOW_OVLP) THEN BEGIN ;(if_loop2)
FOR loc_index=0,(SIZE(index_list))(1)-1 DO $
PRINT,'Miller indices for spot :',loc_index,':',spot_list(index_list(loc_index)).m
IF num_too_close GT 0 THEN BEGIN 
FOR loc_index=0,(SIZE(index_too_close))(1)-1 DO $
PRINT,'Miller indices for spot (too close to integrate):',loc_index,':',spot_list(index_too_close(loc_index)).m
ENDIF

WSET,!D.WINDOW-1
dummy=PIXLAUE2(spot_list(index_list),red,image,par,draw=8,color=2)
WSET,!D.WINDOW+1

;first get an image containing the whole of the overlapping region.
;get the xmax,xmin,ymax,ymin
xmin = par.rastx
ymin = par.rasty
xmax = 0
ymax = 0

FOR k=0,num_ovlp-1 DO BEGIN
xy_coord = WHERESUB(cara_spots(k).indices(WHERE(cara_spots(k).indices GT 0)),image)
x_coord = FLOAT(xy_coord)
y_coord = IMAGINARY(xy_coord)
xmin = xmin < MIN(x_coord)
ymin = ymin < MIN(y_coord)
xmax = xmax > MAX(x_coord)
ymax = ymax > MAX(y_coord)
ENDFOR

IF num_too_close GT 0 THEN BEGIN
FOR k=0,num_too_close-1 DO BEGIN
xy_coord = WHERESUB(cara_too_close(k).indices(WHERE(cara_too_close(k).indices GT 0)),image)
x_coord = FLOAT(xy_coord)
y_coord = IMAGINARY(xy_coord)
xmin = xmin < MIN(x_coord)
ymin = ymin < MIN(y_coord)
xmax = xmax > MAX(x_coord)
ymax = ymax > MAX(y_coord)
ENDFOR
ENDIF

;best is to get a square image
big_size = (ymax - ymin) > (xmax - xmin)
loc_centerX = (xmin + xmax)/2.
loc_centerY = (ymin + ymax)/2.
xmax = ROUND(loc_centerX + big_size/2.+1)
xmin = ROUND(loc_centerX - big_size/2.-1)
ymax = ROUND(loc_centerY + big_size/2.+1)
ymin = ROUND(loc_centerY - big_size/2.-1)

;indice of the left bottom pixel
min_indice = ymin*par.rastx + xmin

;this is the original (with background subtracted) image
ovlp_im=image(xmin:xmax,ymin:ymax) - BACK2D_POLY4(image(xmin:xmax,ymin:ymax),par.noise_percentage,par.gain,par.dark_current)

;we now want the fitted profile image : for that we need to play with the 
;field "indices", in order to shift them from "image" to "ovlp_im"
;the formula is given by :

FOR k=0,num_ovlp-1 DO BEGIN
cara_spots(k).indices(WHERE(cara_spots(k).indices GT 0)) = cara_spots(k).indices(WHERE(cara_spots(k).indices GT 0))-min_indice+(par.rastx-(xmax-xmin+1))*(ymin-cara_spots(k).indices(WHERE(cara_spots(k).indices GT 0))/par.rastx)
ENDFOR
;we can now build the image of learned profiles
profile_im=FLTARR(xmax-xmin+1,ymax-ymin+1)


FOR k=0,num_ovlp-1 DO BEGIN
profile_im(cara_spots(k).indices(WHERE(cara_spots(k).indices GT 0)))=profile_im(cara_spots(k).indices(WHERE(cara_spots(k).indices GT 0)))+integrated(index_list(k))*cara_spots(k).adapted_profile(cara_spots(k).opt_peak_area(WHERE(cara_spots(k).opt_peak_area GT 0)))

;we can now display
PRINT,'Position of maximum [x,y] for spot ',k,' (profile) : ',WHERESUB(WHERE(cara_spots(k).adapted_profile EQ MAX(cara_spots(k).adapted_profile)),cara_spots(k).adapted_profile)
ENDFOR
PRINT,'Fitted profile'
CONTOUR,profile_im,LEVELS=[(INDGEN(10)+1)*(0.01>MAX(profile_im))/10.],C_COLORS=!D.N_COLORS - 150
XSURFACE,profile_im

FOR k=0,num_ovlp-1 DO PRINT,'Position of maximum [x,y] for spot ',k,' (original) : ',WHERESUB(WHERE(cara_spots(k).area EQ MAX(cara_spots(k).area)),cara_spots(k).area)
PRINT,'Original profile'
CONTOUR,ovlp_im,LEVELS=[(INDGEN(10)+1)*MAX(ovlp_im)/10.],/NOERASE,C_COLORS=!D.N_COLORS - 60
XSURFACE,ovlp_im

PRINT,'Difference profile'
SURFACE,profile_im - ovlp_im
OK=''
READ,'OK ? [Y/N]',OK

PRINT,'Background calculations'
FOR k=0,num_ovlp-1 DO BEGIN
PRINT,'Spot # :',k
PRINT,'Backgroud for reference profile :'
SURFACE,cara_spots(k).adapted_profile < MAX(cara_spots(k).adapted_profile)/2.,ax=0
OK=''
READ,'OK ? [Y/N]',OK

PRINT,'Backgroud area for profile :'
dummy = INTARR(sizearea_x,sizearea_y)
dummy(cara_spots(k).bg_area(WHERE(cara_spots(k).bg_area GT -1)))=1
SURFACE,dummy
OK=''
READ,'OK ? [Y/N]',OK

PRINT,'Backgroud for spot :'
dummy = cara_spots(k).area
SURFACE,dummy < MAX(dummy)/2.,ax=0
OK=''
READ,'OK ? [Y/N]',OK

ENDFOR

FOR k=0,num_ovlp-1 DO BEGIN
PRINT,'Integrated intensity for spot ',k,' :',integrated(index_list(k))
PRINT,'Percentage of used profile [%] :',100*cara_spots(k).count/(SIZE(cara_spots(k).adapted_profile))(4)
ENDFOR

ENDIF ; (if_loop2) show


;***********
;estimate the goodness of fit (chi_square_test)
;***********
;the variance^2 has to be multiplied by par.gain (the variance is sigma = gain*
;SQRT(number of Xrays) = SQRT(gain)*SQRT(number of counts)
chi_square = 0
;the gain is proportionnal to energy and equal to par.gain at 20 keV
;the chi_square value is multiplied by 1/10. because experimentally, the
;chi_square test tends to undestimate the fit quality (and it has been checked
;that the esimated variances are not far from corresponding to truth)

;the chi_square must be calculated independantly for each spot if there is
;no correlation between them. If yes, a common chi_square value is given.

IF number_common_indices GT 0 THEN BEGIN

FOR k=0,num_ovlp-1 DO chi_square = chi_square + 1/10.*TOTAL((integrated(index_list(k))*cara_spots(k).adapted_profile(cara_spots(k).opt_peak_area(WHERE(cara_spots(k).opt_peak_area GT 0))) - cara_spots(k).area(cara_spots(k).opt_peak_area(WHERE(cara_spots(k).opt_peak_area GT 0))))^2/(par.gain*12.4/ovlp_spots(k).l/20.)/(1>(cara_spots(k).area(cara_spots(k).opt_peak_area(WHERE(cara_spots(k).opt_peak_area GT 0))) + (1+1/cara_spots(k).bc_coeff(1))*(cara_spots(k).bc_coeff(2) - par.dark_current))))

;we must count the number of pixels involved into the overlapping situation
;we withdraw num_ovlp at the end to account for the "num_ovlp" correlations
;count is the degree of freedom = number_pixels_involved - num_ovlp
count = TOTAL(cara_spots.count) - number_common_indices - num_ovlp

proba = 1 - CHI_SQR1(chi_square,count)

;the calculation should be rejected if chi_square_test is too low,
; ie proba < 0.001
IF proba LE 0.001 THEN cara_spots.flag = 2

chi_square_test(index_list) = proba
success_flag(index_list) = cara_spots.flag

IF KEYWORD_SET(show_ovlp) THEN BEGIN
PRINT,'Chi square test :', proba
PRINT,'Common chi square value :',chi_square
PRINT,'Value for which chi_square test = 0.001 :',CHI_SQR(0.001,count)

PRINT,'Flag [0 : OK ; 1 : bad background ; 2 : bad ii] :',MAX(cara_spots.flag)
ENDIF

;note the chi_square test is performed over the whole of the region.

ENDIF ELSE BEGIN

FOR k=0,num_ovlp-1 DO BEGIN
chi_square = 1/10.*TOTAL((integrated(index_list(k))*cara_spots(k).adapted_profile(cara_spots(k).opt_peak_area(WHERE(cara_spots(k).opt_peak_area GT 0))) - cara_spots(k).area(cara_spots(k).opt_peak_area(WHERE(cara_spots(k).opt_peak_area GT 0))))^2/(par.gain*12.4/ovlp_spots(k).l/20.)/(1>(cara_spots(k).area(cara_spots(k).opt_peak_area(WHERE(cara_spots(k).opt_peak_area GT 0))) + (1+1/cara_spots(k).bc_coeff(1))*(cara_spots(k).bc_coeff(2) - par.dark_current))))

count = cara_spots(k).count - 1
proba = 1 - CHI_SQR1(chi_square,count)

;the calculation should be rejected if chi_square_test is too low,
; ie proba < 0.001
IF proba LE 0.001 THEN cara_spots(k).flag = 2

chi_square_test(index_list(k)) = proba
success_flag(index_list(k)) = cara_spots(k).flag

IF KEYWORD_SET(show_ovlp) THEN BEGIN
PRINT,'Chi square test for peak ',k,':', proba
PRINT,'Chi square value :',chi_square
PRINT,'Value for which chi_square test = 0.001 :',CHI_SQR(0.001,count)

PRINT,'Flag [0 : OK ; 1 : bad background ; 2 : bad ii] :',cara_spots(k).flag
ENDIF

ENDFOR

ENDELSE


;***********
;estimate the variance
;***********
;the variance calculation considers only the region where the peaks seat.

;check if more than one spot is involved
IF (SIZE(index_list))(1) GT 1 THEN BEGIN
;symetrize the variance matrix
FOR loc_i = 0,num_ovlp-1 DO FOR loc_j = loc_i+1,num_ovlp-1 DO orig_corr_matrix(loc_i,loc_j) = orig_corr_matrix(loc_j,loc_i)


;get the variance of b_vect
covar_matrix = FLTARR(num_ovlp,num_ovlp)

FOR row=0,num_ovlp-1 DO BEGIN ;(for_loop3)
 FOR column=0,row DO BEGIN ;(for_loop4)
 ;calculate corr_matrix(row,column)
 IF row EQ column THEN covar_matrix(row,column) = GET_OVLPV2(cara_spots(row).adapted_profile,cara_spots(row).area + cara_spots(row).background,cara_spots(row).opt_peak_area,cara_spots(row).bc_coeff(1),cara_spots(row).bc_coeff(2),par.gain*12.4/ovlp_spots(row).l/20.,par.gain,par.dark_current) ELSE BEGIN 
 ;cross term --> look for a correlation
 IF MIN(ABS(ovlp_spots(row).connect - ovlp_spots(column).id)) EQ 0 THEN BEGIN
 ;there is correlation : we look for the common pixels
 ;print,'Connecting id ',ovlp_spots(row).id,' with id ',ovlp_spots(column).id
 covar_matrix(row,column) = GET_OVLPCOV2(cara_spots(row),cara_spots(column),ovlp_spots(row).l,ovlp_spots(column).l,par)
 ENDIF 
 ENDELSE ;(cross term)
 ENDFOR ;(for_loop4)
ENDFOR ;(for_loop3)

;print,'orig_corr_matrix'
;print,orig_corr_matrix


;symetrize the variance matrix
FOR loc_i = 0,num_ovlp-1 DO FOR loc_j = loc_i+1,num_ovlp-1 DO covar_matrix(loc_i,loc_j) = covar_matrix(loc_j,loc_i)

inv_orig_corr_matrix = INVERT(orig_corr_matrix)
;print,'a-1^2'
;print,inv_orig_corr_matrix^2

;print,'q_matrix'
;print,covar_matrix

ovlp_var_matrix = inv_orig_corr_matrix # covar_matrix # TRANSPOSE(inv_orig_corr_matrix)
;print,'var_vect'
;print,ovlp_var_matrix

FOR k=0,num_ovlp-1 DO peak_variance(index_list(k)) = SQRT(ovlp_var_matrix(k,k)>0)
ENDIF ELSE peak_variance(index_list(0)) = SQRT(GET_1OVLPVAR(cara_spots(0).adapted_profile,cara_spots(0).area+cara_spots(0).background,cara_spots(0).opt_peak_area,cara_spots(0).bc_coeff(1),cara_spots(0).bc_coeff(2),par.gain*12.4/ovlp_spots(0).l/20.,par.gain,par.dark_current))
; case of only one spot

;***
;PRINT INFO
;***
IF KEYWORD_SET(show_ovlp) THEN BEGIN
FOR k=0,num_ovlp-1 DO PRINT,'Peak variance (peak ',k,') :',peak_variance(index_list(k))
ENDIF

IF num_ovlp GE 10 THEN PRINT,FORMAT = '("o",i2,$)',num_ovlp ELSE PRINT,FORMAT = '("o",i1,$)',num_ovlp

ENDELSE ;  (else_loop1) for the case of already treated spots


; ***********************
; END OF OVERLAP CASE
; ***********************

ENDIF ELSE BEGIN ;(if_loop1, else_loop2)
IF spot_list(i).ovlp EQ 0 THEN BEGIN ;if the spot not already treated for
					;overlap problems
;***
;PRINT INFO
;***

IF KEYWORD_SET(show_weak) THEN BEGIN
WSET,!D.WINDOW-1
dummy=PIXLAUE2(spot_list(i),red,image,par,draw=8,color=2)
WSET,!D.WINDOW+1

PRINT,''
PRINT,'Peak number :',i
PRINT,'Miller indices :',spot_list(i).m
ENDIF

;***
;get the corresponding area in image
;***
xpos = ROUND(spot_list(i).x/pix_sizeX+centerX)
ypos = ROUND(spot_list(i).y/pix_sizeY+centerY)
area = image(xpos - sizearea_x/2:xpos + sizearea_x/2-1 , $
		ypos - sizearea_y/2:ypos + sizearea_y/2-1)


;************
;GET THE REFERENCE LEARNED PROFILE by a weighted average of the reference profiles
;************

;select the possible reference profiles :
radius_i = SQRT(spot_list(i).x^2+spot_list(i).y^2)
angle_i =  ATAN(spot_list(i).y,spot_list(i).x)*180/!Pi
r_max = radius_i + ref_distance/2.
r_min = (radius_i - ref_distance/2.)>0
phi_max = angle_i + ref_ang_width/2.
IF phi_max GT 180 THEN phi_max = phi_max - 360
phi_min = angle_i - ref_ang_width/2.
IF phi_min LT -180 THEN phi_min = phi_min + 360

IF phi_min LE phi_max THEN $
ref_index = WHERE( ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max AND $
		ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min AND $
		SQRT(ref_profile.x^2 + ref_profile.y^2) LE r_max AND $
		SQRT(ref_profile.x^2 + ref_profile.y^2) GT r_min,n_ref) ELSE $
ref_index = WHERE( ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max OR $
		ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min AND $
		SQRT(ref_profile.x^2 + ref_profile.y^2) LE r_max AND $
		SQRT(ref_profile.x^2 + ref_profile.y^2) GT r_min,n_ref)


IF n_ref LE 1 THEN BEGIN

IF NOT KEYWORD_SET(sym_rad) THEN BEGIN
phi_max = angle_i + ref_ang_width
IF phi_max GT 180 THEN phi_max = phi_max - 360
phi_min = angle_i - ref_ang_width
IF phi_min LT -180 THEN phi_min = phi_min + 360

IF phi_min LE phi_max THEN $
ref_index = WHERE( ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max AND $
		ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min,n_ref) ELSE $
ref_index = WHERE( ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max OR $
		ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min,n_ref)

IF n_ref GT 0 THEN BEGIN
IF KEYWORD_SET(show_weak) THEN PRINT,'No close reference peaks ! Ref peaks taken from full radial sector !'
ENDIF ELSE MESSAGE,'Choose larger area for reference peak search !'
ENDIF ELSE BEGIN
;********************
;include symmetrical sector

IF angle_i GE 0 THEN angle_i2 = angle_i - 180 ELSE angle_i2 = angle_i + 180 

phi_max = angle_i + ref_ang_width
phi_max2 = angle_i2 + ref_ang_width
IF phi_max GT 180 THEN phi_max = phi_max - 360
IF phi_max2 GT 180 THEN phi_max2 = phi_max2 - 360

phi_min = angle_i - ref_ang_width
phi_min2 = angle_i2 - ref_ang_width
IF phi_min LT -180 THEN phi_min = phi_min + 360
IF phi_min2 LT -180 THEN phi_min2 = phi_min2 + 360

IF ((phi_min LE phi_max) AND (phi_min2 LE phi_max2)) THEN $
ref_index = WHERE((ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max AND $
		   ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min) OR $
		  (ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max2 AND $
                   ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min2),n_ref)

IF ((phi_min LE phi_max) AND (phi_min2 GT phi_max2)) THEN $
ref_index = WHERE((ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max AND $
		   ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min) OR $
		  (ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max2 OR $
                   ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min2),n_ref)

IF ((phi_min GT phi_max) AND (phi_min2 LE phi_max2)) THEN $
ref_index = WHERE((ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max OR $
		   ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min) OR $
		  (ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max2 AND $
                   ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min2),n_ref)

IF ((phi_min GT phi_max) AND (phi_min2 GT phi_max2)) THEN $
ref_index = WHERE((ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max OR $
		   ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min) OR $
		  (ATAN(ref_profile.y,ref_profile.x)*180/!Pi LE phi_max2 OR $
                   ATAN(ref_profile.y,ref_profile.x)*180/!Pi GE phi_min2),n_ref)


IF n_ref GT 0 THEN BEGIN
IF KEYWORD_SET(show_weak) THEN PRINT,'No close reference peaks ! Ref peaks taken from full radial sector ! (including symmetrical sector)'
ENDIF ELSE MESSAGE,'Choose larger area for reference peak search !'
ENDELSE
;********************
ENDIF

possible_ref = ref_profile(ref_index)


;Take the 5 highest/close to spot profiles if more than 5.
IF n_ref GT 5 THEN possible_ref = (possible_ref(REVERSE(SORT(possible_ref.intb(0)/SQRT((possible_ref.x-spot_list(i).x)^2 + (possible_ref.y-spot_list(i).y)^2)))))(0:(4 < (n_ref-1)))


IF MIN((possible_ref.x-spot_list(i).x)^2 + (possible_ref.y-spot_list(i).y)^2) EQ 0 THEN BEGIN
;We are in this case in the presence of one of the reference spots !
;If spot list does not contain any of the reference spots, this should never
;happen !
index = WHERE(((possible_ref.x-spot_list(i).x)^2 + (possible_ref.y-spot_list(i).y)^2) EQ 0)
adapted_profile = possible_ref(index).sp -possible_ref(index).bc


ENDIF ELSE FOR index=0,(SIZE(possible_ref))(1)-1 DO adapted_profile = adapted_profile + (possible_ref(index).sp - possible_ref(index).bc)*1/SQRT((possible_ref(index).x-spot_list(i).x)^2 + (possible_ref(index).y-spot_list(i).y)^2)


;*******
;CALCULATION OF INTEGRATED INTENSITY (UNBIASED PROFILE FITTING)
;*******

;normalize the integrated intensity of the reference profile is to 1. It is the
;integrated intensity of the total profile which is set to 1, not just only the
;part under "opt_peak_area" (see after)
adapted_profile = adapted_profile/TOTAL(adapted_profile)

;allowed indices for background calculation : at 0.1% of maximum or minimum level to get at least 20% of pixels.
scale_factor = 1
;REPEAT BEGIN
;allowed_indices = WHERE(adapted_profile LE MAX(adapted_profile)*0.01*scale_factor,ct)
;scale_factor = 2*scale_factor
;ENDREP UNTIL (ct GT 0.3*(SIZE(adapted_profile))(4))

;&&&NEW
maxsub = (WHERE(adapted_profile EQ MAX(adapted_profile)))(0)
maxpos = WHERESUB(maxsub,adapted_profile)

cutoff = GET_FITLEVEL(adapted_profile,spot_list(i).intp(0),spot_list(i).bc_coeff(1),spot_list(i).bc_coeff(2),par.dark_current)

REPEAT BEGIN
allowed_indices = SEARCH2D(adapted_profile,maxpos(0),maxpos(1),MAX(adapted_profile)*cutoff/100.*scale_factor,MAX(adapted_profile))
ct = (SIZE(allowed_indices))(1)
scale_factor = 2*scale_factor
ENDREP UNTIL (ct LE 0.7*(SIZE(adapted_profile))(4))
tmp = adapted_profile
tmp(allowed_indices)=0
allowed_indices = WHERE(tmp NE 0)
;&&&NEW

;***
;subtract background
;***
background =  BACK2D_POLY5(area,allowed_indices,par.noise_percentage,par.gain,par.dark_current,flag,coeff,bc_mean,n_back,chi_square)
area = area - background
 ;the value of flag can be used to possibly remove the stuff with bad
 ; background

;optimization of fit_level by variance minimization
IF KEYWORD_SET(show_weak) THEN $
fit_level = GET_FITLEVEL(adapted_profile,spot_list(i).intp(0),n_back,bc_mean,par.dark_current,/show) $
ELSE  fit_level = GET_FITLEVEL(adapted_profile,spot_list(i).intp(0),n_back,bc_mean,par.dark_current)

opt_peak_area = WHERE(adapted_profile GE MAX(adapted_profile)*fit_level/100.,count)

integrated(i) = TOTAL(adapted_profile(opt_peak_area)*area(opt_peak_area))/TOTAL(adapted_profile(opt_peak_area)^2)


;weighted integrated intensity
;integrated(i) = TOTAL(adapted_profile(opt_peak_area)*area(opt_peak_area)/(gain*(1>(area(opt_peak_area) + (1+1/n_back)*(bc_mean - par.dark_current)))))/TOTAL(adapted_profile(opt_peak_area)^2/(gain*(1>(area(opt_peak_area) + (1+1/n_back)*(bc_mean - par.dark_current)))))


;***
;PRINT INFO
;***

IF KEYWORD_SET(SHOW_WEAK) THEN BEGIN
PRINT,'Number of reference peaks : ',(SIZE(possible_ref))(1)
PRINT,'Spot position [X,Y], [mm]:',spot_list(i).x,spot_list(i).y
PRINT,'X values [mm]:',possible_ref.x
PRINT,'Y values [mm]:',possible_ref.y
center_area = WHERESUB(WHERE(area EQ MAX(area)), area)
center_adapted = WHERESUB(WHERE(adapted_profile EQ MAX(adapted_profile)), adapted_profile)
shift_area_adapted = center_area(0) - center_adapted(0)
PRINT,'Shift between peaks :',shift_area_adapted
PRINT,'Fitted profile'
PRINT,'Percentage of used profile [%] :',100*count/(SIZE(adapted_profile))(4)
PRINT,'Position of maximum [x,y] :',WHERESUB(WHERE(adapted_profile EQ MAX(adapted_profile)),adapted_profile)
;SURFACE,integrated(i)*adapted_profile
temp = adapted_profile
temp(*) = 0
temp(opt_peak_area) = adapted_profile(opt_peak_area)
CONTOUR,temp,LEVELS=[(INDGEN(10)+1)*MAX(temp)/10.],C_COLORS=!D.N_COLORS - 150
OK=''
READ,'OK ? [Y/N]',OK

PRINT,'Original profile'
PRINT,'Position of maximum [x,y] :',WHERESUB(WHERE(area EQ MAX(area)),area)
PRINT,'Maximum :',MAX(area)
PRINT,'Maximum in kept region :',MAX(area(opt_peak_area))
CONTOUR,area,LEVELS=[(INDGEN(10)+1)*MAX(area)/10.],/NOERASE,C_COLORS=!D.N_COLORS - 60
OK=''
READ,'OK ? [Y/N]',OK

PRINT,'Kept region from Original profile'
temp(*) = 0
temp(opt_peak_area) = area(opt_peak_area)
CONTOUR,temp,LEVELS=[(INDGEN(10)+1)*MAX(temp)/10.],/NOERASE,C_COLORS=!D.N_COLORS - 190
OK=''
READ,'OK ? [Y/N]',OK

PRINT,'Difference profile'
PRINT,'Integrated intensity :',integrated(i)
SURFACE,integrated(i)*adapted_profile - area
OK=''
READ,'OK ? [Y/N]',OK

PRINT,'Background level for ref profile'
dum=FLTARR(2*par.boxsize.x,2*par.boxsize.y)
dum(FIX(par.boxsize.x/2.):FIX(par.boxsize.x/2.)+par.boxsize.x-1,FIX(par.boxsize.y/2.):FIX(par.boxsize.y/2.)+par.boxsize.y-1)=adapted_profile
SURFACE,dum<Max(dum)/2.,ax=0
OK=''
READ,'OK ? [Y/N]',OK

PRINT,'Backgroud area for profile :'
dummy = INTARR(sizearea_x,sizearea_y)
dummy(allowed_indices)=1
SURFACE,dummy
OK=''
READ,'OK ? [Y/N]',OK

PRINT,'Background level for area'
dum=FLTARR(2*par.boxsize.x,2*par.boxsize.y)
dum(FIX(par.boxsize.x/2.):FIX(par.boxsize.x/2.)+par.boxsize.x-1,FIX(par.boxsize.y/2.):FIX(par.boxsize.y/2.)+par.boxsize.y-1)=area
SURFACE,dum<Max(dum)/2.,ax=0
OK=''
READ,'OK ? [Y/N]',OK

ENDIF

;***
;get an estimate of the goodness of fit (chi_square_test)
;***

;the gain is (see below)
gain = par.gain*12.4/spot_list(i).l/20.


;the chi_square value is multiplied by 1/10. because experimentally, the
;chi_square test tends to undestimate the fit quality (and it has been checked
;that the esimated variances are not far from corresponding to truth)

chi_square = 1/10.*TOTAL((integrated(i)*adapted_profile(opt_peak_area) - area(opt_peak_area))^2/gain/(1>(area(opt_peak_area) + (1+1/n_back)*(bc_mean - par.dark_current))))

proba = 1 - CHI_SQR1(chi_square,count - 1)
; gammaq(degrees of freedom/2,chi_square/2) = 1 - CHI_SQR1(chi_square,degrees of freedom) (see Numerical recipes in C. p.525)


;the calculation should be rejected if chi_square_test is too low,
; ie proba < 0.001
IF proba LE 0.001 THEN flag = 2
chi_square_test(i) = proba
success_flag(i) = flag > ovlp_err
;reset ovlp_err
ovlp_err = 0

;***
;get an estimate of the variance
;***
;the gain is supposed to be proportionnal to X-ray energy
;par.gain is the gain at 20 KeV

;print,'Old variance:', SQRT(par.gain)*SQRT(TOTAL(area(opt_peak_area) + background(opt_peak_area) - par.dark_current))

peak_variance(i) = SQRT(GET_PFVAR(adapted_profile,area+background,fit_level,n_back,bc_mean,par.gain*12.4/spot_list(i).l/20.,par.gain,par.dark_current))


;***
;PRINT INFO
;***
IF KEYWORD_SET(show_weak) THEN BEGIN
PRINT,'Chi square test :', proba
PRINT,'Chi square value :',chi_square
PRINT,'Value for which chi_square test = 0.001 :',CHI_SQR(0.001,count-1)
PRINT,'Ratio [>1 good, < 1 bad] :',CHI_SQR(0.001,count-1)/chi_square

PRINT,'Flag [0 : OK ; 1 : bad background ; 2 : bad ii] :',flag
PRINT,'Peak variance :',peak_variance(i)
ENDIF

IF flag EQ 0 THEN PRINT,FORMAT = '($, "*")'
IF flag EQ 1 THEN PRINT,FORMAT = '($, "b")'
IF flag EQ 2 THEN PRINT,FORMAT = '($, "i")'
IF flag GT 2 THEN PRINT,FORMAT = '($, "s")'


ENDIF ; for the case of already treated spots
ENDELSE ; case for a non overlapped spot

IF KEYWORD_SET(show_m) THEN BEGIN
show_ovlp = 0
show_weak = 0
ENDIF

escape2:

ENDFOR  ;end of the main loop going on for each spot.

PRINT,''
PRINT,'  --> Number of spots integrated :',number_spots
PRINT,''
RETURN,integrated

END

