function BACK2D_POLY5,array, allowed_indices,percentage,bg_gain,dark_current,bad_flag,coeff,bc_mean,n_back,chi_square_test,noise_index,show=show,max_cycle=max_cycle
;+
; NAME:
;	BACK2D_POLY5
;
; PURPOSE:
;	Background removal by plane (polynomial) fitting
;
; CATEGORY:
;	Laue processing.
;
; CALLING SEQUENCE:
;	back = BACK2D_POLY5(array,allowed_indices,percentage,bg_gain,dark_current,[bad_flag,coeff,
; 				chi_square_test,bc_mean,n_back],/show)
;
; INPUTS:
;	array : the array to process
;	allowed_indices : indices in array not to consider in background determination
;	percentage : the initial percentage of the pixels to consider as noise. (the closer
;		this value to the real percentage, the faster the routine converges)
;	bg_gain : the gain of the detector at the wavelength of the mean background.
;	dark_current : the dark current of the detector (a constant offset)
;	show : keyword set to print info
;	max_cycle : set this keyword to the maximum number of cycles allowed
;			(default = 100)
;
; OUTPUTS:
;	bad_flag : (optionnal) this flag set to 1 if the chi_square test is not successfull
;	coeff : (optionnal) the coefficient of the background plane
;	chi_square_test : (optionnal) the chi_square test
;	bc_mean : [optional] the mean background
;	n_back : [optional] the number of pixels used for background 
;			subtraction
;	noise_index : [optional] the indices of the pixels selected for
;		background calculation
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	Taken from Rossman, J. App.
;	Cryst. (1979),12,225-238
;	The noise pixels which deviate too much from the fitted plane (more
;	than 1 times the variance) are taken away and the procedure
;	repeated until all noise pixels match the criteria.
;
; MODIFICATION HISTORY:
;	Dominique Bourgeois, June 95
;-

;ON_ERROR,2                              ;Return to caller if an error occurs
;get the dimension and size of the array
arr_size=SIZE(array)
xsize=arr_size(1)
ysize=arr_size(2)
totsize=arr_size(4)
bad_flag=0
x = INDGEN(xsize) & y = INDGEN(ysize)
x2 = x # REPLICATE(1,ysize) & y2 = REPLICATE(1,xsize) # y


;the number of pixels to be considered as background as a starting point is
number_back = FIX(percentage/100.*totsize)

;take the smaller pixels which are allowed, and which are not below the dark current
w = WHERE(array(allowed_indices) GT dark_current,count)
w2 = allowed_indices(w)
w3 = (SORT(array(w2)))(0:(count-1)<(number_back-1))
noise_index = w2(w3)

;the fitted polynom is
bc = FLTARR(xsize,ysize)

;a correction factor for the variance calculation is taken as :
;(because the higher the gain the more correlated are the noisy pixels,
;this is empirical ...: corr_fac = 1.5 at bg_gain =1 and 0.2 at bg_gain =50)
p=[0.010597419,0.69533171,-0.042181274,0.0012884429,-2.0547306e-05,1.6284568e-07,-5.0490175e-10]
;corr_fac = (-0.025*bg_gain^0.7 + 1.525)>0.1
corr_fac = POLY(bg_gain,p)/(SQRT(bg_gain)>1)

rejected = 1
number_cycle = 0

n_max_cycle = 100
IF KEYWORD_SET(max_cycle) THEN n_max_cycle = max_cycle

WHILE (rejected NE 0) AND (number_cycle LE n_max_cycle) DO BEGIN
number_cycle = number_cycle + 1
;the array of density of noise is given by :
noise_density = array(noise_index)
;the corresponding coordinates are :
coord = WHERESUB(noise_index,array)

;we define the matrix A by
a11 = TOTAL(FLOAT(coord)^2)
a12 = TOTAL(FLOAT(coord)*IMAGINARY(coord))
a13 = TOTAL(FLOAT(coord))

a21 = a12
a22 = TOTAL(IMAGINARY(coord)^2)
a23 = TOTAL(IMAGINARY(coord))

a31 = a13
a32 = a23
a33 = (SIZE(FLOAT(coord)))(1)

a=[[a11,a12,a13],$
   [a21,a22,a23],$
   [a31,a32,a33]]

;the B column matrix is then defined by :
b1 = TOTAL(noise_density*FLOAT(coord))
b2 = TOTAL(noise_density*IMAGINARY(coord))
b3 = TOTAL(noise_density)

b = [b1,b2,b3]

;we now apply the Cholevsky method

CATCH, Error_status	

IF Error_status NE 0 THEN BEGIN	;This statement begins the error handler.
  ;Handles the error
  a(*)=0
  a(0,0)=1 & a(1,1)=1 & a(2,2)=1
  bad_flag = 1
ENDIF

NR_CHOLDC,a,p
coeff = NR_CHOLSL(a,p,b)

;matrix inversion
;ai = INVERT(a,status)
;IF status EQ 1 THEN MESSAGE,'Couldn''t invert matrix ...'

;the polynomial coefficients are :
;coeff = ai # b

;the polynom itself is

bc = coeff(0)*x2 + coeff(1)*y2 + coeff(2)

;we iterate the procedure by selecting the pixels not deviating too much from
;the fitted plane.
;The variance is given by :

;we reject the noise pixels which deviate too much from the fitted plane
;The variance is corr_factor*SQRT(bg_gain*number of counts)
kept_index = WHERE(((array(allowed_indices)-bc) LE corr_fac*SQRT(bg_gain*array(allowed_indices))) AND (array(allowed_indices) GE dark_current))
kept_index = allowed_indices(kept_index)
;kept_index = WHERE(((array-bc) LE corr_fac*SQRT(bg_gain*bc)) AND (array GE dark_current))

rejected = MAX(kept_index NE noise_index)
;rejected = (SIZE(kept_index))(1) - (SIZE(noise_index))(1)
noise_index = kept_index

ENDWHILE

;now get the estimate of the fit:
chi_square = TOTAL(((bc-array)(noise_index))^2/bg_gain/((array(noise_index) - dark_current)>0.1))


;the probability to get a higher chi_square value is
IF (SIZE(noise_index))(1) LE 2*(xsize+ysize) THEN BEGIN
 chi_square_test = 0 
 bad_flag = 1
ENDIF ELSE BEGIN
chi_square_test = 1 - CHI_SQR1(chi_square,(SIZE(noise_index))(1) - 3)
;the background calculation should be rejected if chi_square_test is too low,
; ie chi_square_test < 0.1
IF chi_square_test LE 0.1 THEN bad_flag = 1 ELSE bad_flag = 0>bad_flag
ENDELSE

;check the uniformity of the distribution of the background pixels
;in array. The borders of array should have at least 15% of background
;pixels
w1=WHERE(FLOAT(coord) EQ (xsize-1),ct1)
w2=WHERE(IMAGINARY(coord) EQ (ysize-1),ct2)
w3=WHERE(FLOAT(coord) EQ 0,ct3)
w4=WHERE(IMAGINARY(coord) EQ 0,ct4)

IF (ct1/FLOAT(ysize) < ct2/FLOAT(xsize) < ct3/FLOAT(ysize) <ct4/FLOAT(xsize)) LE 0.15 THEN bad_flag = 1

;*****
;calculation of the background variance --> for the future !
;*****
;variance of the coefficients
;coeffv = FLTARR(3)
;the variance of the fitted polynom
;bcv = FLTARR(xsize,ysize)


;the variance of coeff are given by (assuming that all pixels are 
;independant ...)
;bvar11 = TOTAL((noise_density - dark_current)*FLOAT(coord)^2)
;bvar12 = TOTAL((noise_density - dark_current)*FLOAT(coord)*IMAGINARY(coord))
;bvar13 = TOTAL((noise_density - dark_current)*FLOAT(coord))

;bvar21 = bvar12
;bvar22 = TOTAL((noise_density - dark_current)*IMAGINARY(coord)^2)
;bvar23 = TOTAL((noise_density - dark_current)*IMAGINARY(coord))

;bvar31 = bvar13
;bvar32 = bvar23
;bvar33 = TOTAL((noise_density - dark_current))

;bvar =  bg_gain * 	[[bvar11,bvar12,bvar13],$
;  		[bvar21,bvar22,bvar23],$
;		[bvar31,bvar32,bvar33]]

;coeffvar = ai # bvar # TRANSPOSE(ai)
;coeffv(0) = coeffvar(0,0)
;coeffv(1) = coeffvar(1,1)
;coeffv(2) = coeffvar(2,2)

;FOR i=0,ysize-1 DO bcv(*,i) = (INDGEN(xsize))^2*coeffv(0) + coeffv(1)*i^2 + coeffv(2) + 2*INDGEN(xsize)*i*coeffvar(0,1) + 2*INDGEN(xsize)*coeffvar(0,2) + 2*i*coeffvar(1,2)

;chi_square = TOTAL(((bc-array)(noise_index))^2/(bcv(noise_index)>0.1))
;****


;number of rejected pixels :
rejected = totsize - (SIZE(noise_index))(1)
;number of pixels used for the calculation
n_back = (SIZE(noise_index))(1)
;mean background: 
dum=STDEV(bc,bc_mean)

IF KEYWORD_SET(show) THEN BEGIN
PRINT,'Chi square value :',chi_square
PRINT,'Probability to find a higher value [chi_square_test] : ', chi_square_test
PRINT,'Number of used pixels :',(SIZE(noise_index))(1)
PRINT,'Number of background pixels at max(x),max(y),min(x),min(y) :',ct1,ct2,ct3,ct4
PRINT,'Number of cycles :',number_cycle
PRINT,'Polynomial coeff. x.[1]+y[2]+[3] :',coeff(0),coeff(1),coeff(2)
;PRINT,'Polynomial coeff. SdtDev Sig[1], Sig[2], Sig[3] :',SQRT(coeffv(0)),SQRT(coeffv(1)),SQRT(coeffv(2))
PRINT,'Mean background value :',bc_mean
PRINT,'Number of rejected pixels [%] : ',(1-n_back/FLOAT(totsize))*100
IF bad_flag EQ 0 THEN PRINT,'******CORRECT BACKGROUND******' ELSE PRINT,'******BAD BACKGROUND******' 
ENDIF

;IF rejected/FLOAT(number_back) GE 0.5 THEN BEGIN
;IF KEYWORD_SET(show) THEN PRINT,'Bad background !'
;bad_flag = 1
;ENDIF


RETURN,bc
END




