function WEIGHT_DISTR_EXT, m, F_1, F_2, sigma=sigma, $
	deconvolute=deconvolute, $
	centric=centric, epsilon=epsilon, $
	par=par, no_bins=no_bins, count=count
;+
; NAME:
;	WEIGHT_DISTR_EXT
;
; PURPOSE:
; 	(ext=extended). New way of inputing centric and epsilon.
;
;	Calculate SIGMA_WILSON^2= "Sum(f^2)" = n<F^2>/epsilon
;	where SIGMA_WILSON is the parameter in the Wilson
;	distribution.
;	If both F_1 and F_2 are given then "n<(F_1-F_2)^2>/epsilon"
;	is calculated which is an estimate of SIGMA_WILSON^2 for the
;	difference. n is 2 for non-centric reflections and 1 for 
;	centric reflections which takes into account the fact that
;	F_1-F_2 is the measurement of the component of the vector
;	difference along F_1 (this is only true if DeltaF<<F).
;	If SIGMA is given then instead (only if F_1 and F_2 are given)
;	SIGMA_WILSON^2= n<(F_1-F_2)^2-SIGMA^2>/epsilon
;	where ((F_1-F_2)^2-SIGMA^2) is set to (F_1-F_2)^2/2 if less than
;	(F_1-F_2)^2/2. This is an estimation of the real difference taking
;	the measurement noise into account. No idea if it works!
;
;       THIS IS WRONG(?): (Actually SIGMA_D^2=SIGMA_WILSON^2 also in the 
;	  the difference case due to the factor n in the formula)
;	NOTE: SIGMA_D^2=2*SIGMA_WILSON^2 in the difference case where
;	SIGMA_D^2=SIGMA_N^2*(1-SIGMA_A^2). This is because WEIGHT_DISTR_EXT
;	estimates the rms value whereas SIGMA_D^2=2*SIGMA_G^2 where 
;	SIGMA_G is the distribution parameter in a Gaussian distribution.
;
; CATEGORY:
;	Crystallography
;
; CALLING SEQUENCE:
; 	result= WEIGHT_DISTR_EXT( m, F_1 [, F_2, sigma=sigma, $
;		deconvolute=deconvolute, $ 
;		centric=centric, special=special, epsilon=epsilon, $
;		par=par, no_bins=no_bins, count=count] )
;
; INPUTS:
;	M, F_1 and F_2 should contain the same number of reflections.
;	M: Miller indices of the input reflections. INTARR(3,*)
;	F_1,F_2: The structure factors. F_1 should always be given. If
;		F_2 is given then the SIGMA_WILSON for the difference is
;		calculated (otherwise for F_1 alone). 
;
; KEYWORDED PARAMETERS:
;	SIGMA: If SIGMA is given then the rms of SIGMA is also
;		returned as a function of resolution.
;		If DECONVOLUTE is set and F_1 and F_2 are given then
;		;SIGMA_WILSON^2= n<(F_1-F_2)^2-SIGMA^2>/epsilon
;		;where ((F_1-F_2)^2-SIGMA^2) is set to (F_1-F_2)^2/2 if less 
;		;than (F_1-F_2)^2/2.
;		SIGMA_WILSON^2= n(<(F_1-F_2)^2>-<2*SIGMA^2>)/epsilon but set to
;		                n<(F_1-F_2)^2>/(2*epsilon)
;		if <(F_1-F_2)^2>-<2*SIGMA^2> is less than <(F_1-F_2)^2>/2
;		(The factor two in front of SIGMA is there since <(F_1-F_2)^2>
;		 is two times the Gaussian parameter sigma and SIGMA is a
;		 Gaussian parameter (i.e. the true standard deviation).)
;	DECONVOLUTE: See SIGMA.
;	CENTRIC: An array BYTARR(7) containing information of whether the 
;		the different classes of reflections are centro-symmetric.
;		The seven different set of reflections are in order:
;		hkl,hk0,h0k,0kl,h00,0k0,00l
;		If the set is centric then the corresponding element should
;		be set to 1. 
;		Default: No centric sets.
;	EPSILON: BYTARR(7) in the same way as for CENTRIC but this 
;		array contains the value of epsilon.
;		Default: epsilon=1 for all sets.
;		Epsilon is the ratio <F^2>/Sum(f^2) (see Int 
;		Tables (1985), vol II, p355-6). Epsilon is thus not the same
;		as alpha in the above reference since alpha assumes that also
;		systematic absences are included in the average. 
;		The systematic absent reflections should not be included in 
;		the input (or alternatively alpha should be input instead
;		and the absences included (all of them, but what if data not
;		complete?); in this case systematic absences should off course
;		still be set to zero which is not done by the program).
;	PAR: The cell parameters. Structure containing the tags a,b,c,alpha, $
;		beta,gamma. Angles in degrees. Used for caclulating the d-
;		spacing to assign right bin for SIGMA_WILSON (see D_CALC). 
;		Defaults to cubic with a=50.0.
;	NO_BINS: The number of bins in which SIGMA_WILSON is calculated.
;		Default 10.
;
; OUTPUTS:
;	The result is DBLARR(3,no_bins) where RESULT(0,*) is the value
;	of 1/d^2 at the top of the range of the bin and RESULT(1,*) is 
;	the value of SIGMA_WILSON in the bin. RESULT(2,*) is the 
;	rms of SIGMA in the bin.
;	COUNT: A vector of same length as SIGMA_WILSON containing the number
;		of reflections used for determining the value in the 
;		corresponding bin.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	
;
; MODIFICATION HISTORY:
;	Thomas Ursby, Feb 1996
;	April, 1996
;-

IF (N_ELEMENTS(no_bins) eq 0) THEN no_bins=10
str1=F_1
IF (N_ELEMENTS(F_2) eq 0) THEN type='normal' ELSE type='diff'
IF (type eq 'diff') THEN BEGIN
  str2=F_2
  IF ((N_ELEMENTS(sigma) gt 0) and (N_ELEMENTS(deconvolute) gt 0)) $
	THEN sd=sigma $
  ELSE sd=BYTARR((size(F_1))(1))
ENDIF
IF (N_ELEMENTS(sigma) eq 0) THEN sigma=BYTARR((size(F_1))(1))
classes= [[1,1,1],[1,1,0],[1,0,1],[0,1,1],[1,0,0],[0,1,0],[0,0,1]]
IF (N_ELEMENTS(centric) eq 0) THEN centric=BYTARR(7) 
IF (N_ELEMENTS(epsilon) eq 0) THEN epsilon=BYTARR(7)+1

; Special set of reflections:
FOR i=0,6 DO BEGIN
  IF (epsilon(i) gt 1) THEN BEGIN 
    ind= WHERE(((classes(0,i) xor (abs(m(0)) gt 0))+ $
	(classes(1,i) xor (abs(m(1)) gt 0))+ $
	(classes(2,i) xor (abs(m(2)) gt 0))) eq 0, ct)
    IF (ct gt 0) THEN str1(ind)= str1(ind)/sqrt(epsilon(i))
    IF (ct gt 0) THEN sd(ind)= sd(ind)/sqrt(epsilon(i))
    IF ((ct gt 0) and (type eq 'diff')) THEN $
	str2(ind)= str2(ind)/sqrt(epsilon(i))
  ENDIF
ENDFOR

; Centric reflections:
IF (type eq 'diff') THEN BEGIN
  str1= sqrt(2.0)*str1
  str2= sqrt(2.0)*str2
  sd= sqrt(2.0)*sd
  FOR i=0,6 DO BEGIN
    IF (centric(i) gt 0) THEN BEGIN 
      ind= WHERE(((classes(0,i) xor (abs(m(0)) gt 0))+ $
	(classes(1,i) xor (abs(m(1)) gt 0))+ $
	(classes(2,i) xor (abs(m(2)) gt 0))) eq 0, ct)
      IF (ct gt 0) THEN str1(ind)= str1(ind)/sqrt(2.0)
      IF (ct gt 0) THEN str2(ind)= str2(ind)/sqrt(2.0)
      IF (ct gt 0) THEN sd(ind)= sd(ind)/sqrt(2.0)
    ENDIF
  ENDFOR
ENDIF

dstar2= 1/d_calc(m,par=par)^2
binsize= MAX(dstar2)/no_bins
sigma_w= DBLARR(3, no_bins)
count= INTARR(no_bins)

CASE type OF
  'normal': BEGIN
      FOR i=0,no_bins-1 DO BEGIN 
        ind= WHERE((dstar2 gt i*binsize) and (dstar2 le (i+1)*binsize), ct)
        IF (ct gt 0) THEN BEGIN
          sigma_w(0,i)= (i+1)*binsize
          sigma_w(1,i)= sqrt(total(str1(ind)^2)/ct)
          sigma_w(2,i)= sqrt(total(sigma(ind)^2)/ct)
	  count(i)= ct
        ENDIF
      ENDFOR
    END
  'diff': BEGIN
      FOR i=0,no_bins-1 DO BEGIN 
        ind= WHERE((dstar2 gt i*binsize) and (dstar2 le (i+1)*binsize), ct)
        IF (ct gt 0) THEN BEGIN
          sigma_w(0,i)= (i+1)*binsize
;          sigma_w(1,i)= sqrt( $
;		total(((str1(ind)-str2(ind))^2-sd(ind)^2) $
;			> ((str1(ind)-str2(ind))^2/2)) $
;		/ct)
          rms_sd = sqrt(total(sd(ind)^2)/ct)
          sigma_w(1,i)= sqrt(total((str1(ind)-str2(ind))^2)/ct)
          sigma_w(1,i)= sqrt((sigma_w(1,i)^2-2*rms_sd^2) > (sigma_w(1,i)^2/2))
          sigma_w(2,i)= sqrt(total(sigma(ind)^2)/ct)
	  count(i)= ct
        ENDIF
      ENDFOR
    END
ENDCASE

RETURN, sigma_w

END
