function RFACTOR_I, arr1, arr2, w=w, type=type, ibinsize=ibinsize, $
		intbins=intbins, cutoff=cutoff, print=print
;+
; NAME:
;	RFACTOR_I
;
; PURPOSE:
;	Calculate R-factors in intensity-bins. Normally called from
;	RFACTOR. 
;
; CATEGORY:
;	Crystallography, data analysis.
;
; CALLING SEQUENCE:
; 	rfactor = RFACTOR_I(arr1, arr2 [, w=w, type=type, intbins=intbins, $
;			ibinsize=ibinsize, cutoff=cutoff, print=print] )
;
; INPUTS:
;	ARR1,ARR2: The arrays containing the data to be scaled.
;
; KEYWORDED PARAMETERS:
;	W: Weighting. Array of the same dimension as ARR1 and ARR2.
;		The weighting is used for the
;		R-factor calculations if weighting is requested.
;	TYPE: Indicates what type of R-factor (default 1):
;		1: R= Sum(abs(arr1-arr2))/Sum(arr1)
;		2: R= Sum(abs(arr1-arr2))/Sum((arr1+arr2)/2)
;		3: R= Sum(w*abs(arr1-arr2))/ Sum(w*arr1)
;		4: R= Sum(w*abs(arr1-arr2))/ Sum(w*(arr1+arr2)/2)
;		5: R= Sum(w*(arr1-arr2)^2) / Sum(w*arr1^2)
;		6: R= Sum(w*(arr1-arr2)^2) / Sum(w*(arr1+arr2)^2/4)
;	CUTOFF: The first bin starts at CUTOFF. Default 0.0.
;		Only reflections which are greater than cutoff (both
;		arrays) are used. 
;	INTBINS: The R-factor is calculated in intensity bins
;		with CUTOFF as the start of the first bin and IBINSIZE as
;		the size of the bins.  INTBINS indicates how many bins.
;		The bins refer to arr1. Default 1.
;	IBINSIZE: Size of the bins (see INTBINS). Default 100. The last bin
;		includes everything above CUTOFF+(INTBINS-1)*IBINSIZE.
;	PRINT: If set then the table of calculated R-factors is printed. 
;
; OUTPUTS:
;	Returns the calculated R-factor (on the whole data set). 
;	If the PRINT keyword is set the R-factor is also printed
;	to standard output.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	Straightforward
;
; MODIFICATION HISTORY:
;	Thomas Ursby, 1995
;-

ON_ERROR, 2

IF NOT KEYWORD_SET(w) THEN w=1.0
IF NOT KEYWORD_SET(intbins) THEN intbins=1
IF NOT KEYWORD_SET(type) THEN type=1
IF NOT KEYWORD_SET(cutoff) THEN cutoff=0.0
IF NOT KEYWORD_SET(ibinsize) THEN ibinsize=100.0 ELSE ibinsize=float(ibinsize)
r=dblarr(intbins)
racc=dblarr(intbins)
no = intarr(intbins)
noacc = intarr(intbins)

; Calculate R-factors in intensity bins:
; The order is "reversed" to get Racc going from stronger
; to weaker reflections. 
; Last bin includes the reflections above CUTOFF+(INTBINS-1)*IBINSIZE:
i=intbins-1
ind= WHERE(arr1 ge (cutoff+i*ibinsize), count)
IF (count ne 0) THEN BEGIN
  no(i) =(size(ind))(1)
  CASE (type) OF
    1: r(i) = TOTAL(abs(arr1(ind)-arr2(ind))) / TOTAL(arr1(ind))
    2: r(i) = TOTAL(abs(arr1(ind)-arr2(ind))) / $
		(TOTAL(arr1(ind)+arr2(ind))/2)
    3: r(i) = TOTAL(w*abs(arr1(ind)-arr2(ind))) / TOTAL(w*arr1(ind))
    4: r(i) = TOTAL(w*abs(arr1(ind)-arr2(ind))) / $
		(TOTAL(w*(arr1(ind)+arr2(ind)))/2)
    5: r(i) = TOTAL(w*(arr1(ind)-arr2(ind))^2) / TOTAL(w*arr1(ind)^2)
    6: r(i) = TOTAL(w*(arr1(ind)-arr2(ind))^2) / $
		(TOTAL(w*(arr1(ind)+arr2(ind))^2)/4)
    7:  BEGIN
		tmp_arr=arr1(ind)
		SORT_MILLER,tmp_arr
		num=0. & denum=0.
;		h=0 & k=0 & l=0
		next_i=0L
		s_tmp_arr=(SIZE(tmp_arr))(1)
		FOR loc_i = 0,s_tmp_arr-1 DO BEGIN
		 REPEAT BEGIN
		  next_i = (next_i+1)<(s_tmp_arr-1)
		 ENDREP UNTIL (tmp_arr(next_i).m(0) NE tmp_arr(next_i-1).m(0)) OR (tmp_arr(next_i).m(1) NE tmp_arr(next_i-1).m(1)) OR (tmp_arr(next_i).m(2) NE tmp_arr(next_i-1).m(2)) OR (next_i EQ s_tmp_arr-1)
		 imean = TOTAL(tmp_arr(loc_i:next_i).entry(2))
		 num=num+TOTAL(ABS(imean-tmp_arr(loc_i:next_i).entry(2)))
		 denum=denum+TOTAL(tmp_arr(loc_i:next_i).entry(2))
		 loc_i=next_i
		ENDFOR
		r(i) = num/denum
	END
  ENDCASE
ENDIF $
ELSE BEGIN
  r(i)=0.0
  no(i)=0
ENDELSE
indacc= WHERE(arr1 gt (cutoff+i*ibinsize), count)
IF (count ne 0) THEN BEGIN
  noacc(i) =(size(indacc))(1)
  CASE (type) OF
    1: racc(i) = TOTAL(abs(arr1(indacc)-arr2(indacc))) / $
			TOTAL(arr1(indacc))
    2: racc(i) = TOTAL(abs(arr1(indacc)-arr2(indacc))) / $
			(TOTAL(arr1(indacc)+arr2(indacc))/2)
    3: racc(i) = TOTAL(w*abs(arr1(indacc)-arr2(indacc))) / $
			TOTAL(w*arr1(indacc))
    4: racc(i) = TOTAL(w*abs(arr1(indacc)-arr2(indacc))) / $
			(TOTAL(w*(arr1(indacc)+arr2(indacc)))/2)
    5: racc(i) = TOTAL(w*(arr1(indacc)-arr2(indacc))^2) / $
			TOTAL(w*arr1(indacc)^2)
    6: racc(i) = TOTAL(w*(arr1(indacc)-arr2(indacc))^2) / $
			(TOTAL(w*(arr1(indacc)+arr2(indacc))^2)/4)
  ENDCASE
ENDIF $
ELSE BEGIN
  racc(i)=0.0
  noacc(i)=0.0
ENDELSE
FOR i=intbins-2,0,-1 DO BEGIN
  ind= WHERE((arr1 ge (cutoff+i*ibinsize)) and $
		(arr1 lt (cutoff+(i+1)*ibinsize)), count)
  IF (count ne 0) THEN BEGIN
    no(i) =(size(ind))(1)
    CASE (type) OF
      1: r(i) = TOTAL(abs(arr1(ind)-arr2(ind))) / TOTAL(arr1(ind))
      2: r(i) = TOTAL(abs(arr1(ind)-arr2(ind))) / $
		(TOTAL(arr1(ind)+arr2(ind))/2)
      3: r(i) = TOTAL(w*abs(arr1(ind)-arr2(ind))) / TOTAL(w*arr1(ind))
      4: r(i) = TOTAL(w*abs(arr1(ind)-arr2(ind))) / $
		(TOTAL(w*(arr1(ind)+arr2(ind)))/2)
      5: r(i) = TOTAL(w*(arr1(ind)-arr2(ind))^2) / TOTAL(w*arr1(ind)^2)
      6: r(i) = TOTAL(w*(arr1(ind)-arr2(ind))^2) / $
		(TOTAL(w*(arr1(ind)+arr2(ind))^2)/4)
      ELSE: MESSAGE,'Illegal type'
    ENDCASE
  ENDIF $
  ELSE BEGIN
    r(i)= 0.0
    no(i)=0
  ENDELSE 
  indacc= WHERE(arr1 gt (cutoff+i*ibinsize), count)
  IF (count ne 0) THEN BEGIN
    noacc(i) =(size(indacc))(1)
    CASE (type) OF
      1: racc(i) = TOTAL(abs(arr1(indacc)-arr2(indacc))) / $
			TOTAL(arr1(indacc))
      2: racc(i) = TOTAL(abs(arr1(indacc)-arr2(indacc))) / $
		(TOTAL(arr1(indacc)+arr2(indacc))/2)
      3: racc(i) = TOTAL(w*abs(arr1(indacc)-arr2(indacc))) / $
			TOTAL(w*arr1(indacc))
      4: racc(i) = TOTAL(w*abs(arr1(indacc)-arr2(indacc))) / $
			(TOTAL(w*(arr1(indacc)+arr2(indacc)))/2)
      5: racc(i) = TOTAL(w*(arr1(indacc)-arr2(indacc))^2) / $
			TOTAL(w*arr1(indacc)^2)
      6: racc(i) = TOTAL(w*(arr1(indacc)-arr2(indacc))^2) / $
			(TOTAL(w*(arr1(indacc)+arr2(indacc))^2)/4)
      ELSE: MESSAGE,'Illegal type'
    ENDCASE
  ENDIF $
  ELSE BEGIN
    racc(i)= racc(i+1) 
    noacc(i)= noacc(i+1)
  ENDELSE
ENDFOR

; Print R-factors
IF KEYWORD_SET(print) THEN BEGIN
  PRINT, '       Imin           Imax             R          No         Racc       No,acc'
  i=intbins-1
  PRINT,cutoff+i*ibinsize, '-', max(arr1), ' :', r(i), $
		no(i), racc(i), noacc(i)
  FOR i=intbins-2,0,-1 DO PRINT,cutoff+i*ibinsize, '-', $
		cutoff+(i+1)*ibinsize,' :', r(i), no(i), racc(i), noacc(i)
  CASE (type) OF
    1: PRINT, 'R= Sum(abs(arr1-arr2))/Sum(arr1)'
    2: PRINT, 'R= Sum(abs(arr1-arr2))/Sum((arr1+arr2)/2)'
    3: PRINT, 'R= Sum(w*abs(arr1-arr2))/ Sum(w*arr1)'
    4: PRINT, 'R= Sum(w*abs(arr1-arr2))/ Sum(w*(arr1+arr2)/2)'
    5: PRINT, 'R= Sum(w*(arr1-arr2)^2) / Sum(w*arr1^2)'
    6: PRINT, 'R= Sum(w*(arr1-arr2)^2) / Sum(w*(arr1+arr2)^2/4)'
  ENDCASE
ENDIF

RETURN, racc(0)
 
END
