function RFACTOR, arr1, arr2, type=type, scales=scales, $
		w=w, k=k, b=b, ssq=ssq, $
		multiplicity=multiplicity, cutoff=cutoff, $
		intbins=intbins, ibinsize=ibinsize, sbins=sbins, $
		lambda=lambda, lbins=lbins, print=print
;+
; NAME:
;	RFACTOR
;
; PURPOSE:
;	Calculate R-factors.
;
; CATEGORY:
;	Crystallography, data analysis.
;
; CALLING SEQUENCE:
; 	rfactor = RFACTOR(arr1, arr2 [, type=type, scales=scales, $
;		w=w, k=k, b=b, ssq=ssq, $ 
;		multiplicity=multiplicity, cutoff=cutoff, $
;		intbins=intbins, ibinsize=ibinsize, sbins=sbins, $
;		lambda=lambda, lbins=lbins, print=print] )
;
; INPUTS:
;	ARR1,ARR2: The arrays containing the data to be scaled.
;
; KEYWORDED PARAMETERS:
;	W: Weighting. Array of the same dimension as ARR1 and ARR2.
;		ARR1 and ARR1 are multplied by Sqrt(W) during the scaling
;		(<=> Weight=W). The same weighting is used for the
;		R-factor calculations if weighting is requested.
;	TYPE: Indicates what type of R-factor (default 1):
;		1: R= Sum(abs(arr1-arr2))  / Sum(arr1)
;		2: R= Sum(abs(arr1-arr2))  / Sum((arr1+arr2)/2)
;		3: R= Sum(w*abs(arr1-arr2))/ Sum(w*arr1)
;		4: R= Sum(w*abs(arr1-arr2))/ Sum(w*(arr1+arr2)/2)
;		5: R= Sum(w*(arr1-arr2)^2) / Sum(w*arr1^2)
;		6: R= Sum(w*(arr1-arr2)^2) / Sum(w*(arr1+arr2)^2/4)
;	SCALES: If set then arr1 and arr2 are scaled together first
;		with a global scalefactor. 
;	      0=No scaling
;	      If bit 0 (1) is set (K scaling) 
;	      If bit 1 (2) is set (K&B scaling)
;	      If bit 2 (4) is set then the scale is applied to ARR1 
; 		(but only to those reflections that were used to calculate 
;		the scale, see MULTIPLICITY and CUTOFF). NOTE: ARR1 will 
;		only be affected if it is passed to RFACTOR as a variable 
;		and not as a subscripted array, structure etc. See IDL 
;		Reference Manual p. 10-5.
;	      If bit 3 (8) is set then weighting is applied (if W is given).
;	      If bit 4 (16) is set then the scaling is done using the values
;		given with the keywords K and B.
;	K,B: Scale factors, see SCALES.
;	SSQ: The square of s=d*/2=1/2d. If SCALES=2 or SBINS is set this 
;		must be given. Array of same size as the data. 
;	MULTIPLICITY: Should be of same dimensions as the arrays.
;		If set then only those reflections with mul=1 are
;		used.
;	CUTOFF: Only reflections which are greater than cutoff (both
;		arrays) are used. Default 0.0. This cut-off is applied
;		before scaling.
;	INTBINS: If set then the R-factor is calculated in intensity bins
;		with CUTOFF as the start of the first bin and IBINSIZE as
;		the size of the bins.  INTBINS indicates how many bins.
;		The bins refer to arr1 after it has been scaled.
;	IBINSIZE: Size of the bins (see INTBINS). Default 100. The last bin
;		includes everything above CUTOFF+(INTBINS-1)*IBINSIZE.
;	SBINS:  If set then the R-factor is calculated in bins of s^2
;		where s=d*/2=1/2d. SBINS indicates how many bins.
;		The range is s=[0,max(s^2)]
;	LAMBDA: Array containing the lambda values for the reflections.
;		Must be given if LBINS is set.
;	LBINS: If set then the R-factor is calculated in bins of LAMBDA
;		LBINS indicates how many bins. The range is lambda=
;		[min(lambda),max(lambda)]. LAMBDA must be provided.
;	PRINT: If set then the calculated R-factor is printed as well
;		how many reflections were used, the total number of input
;		reflections and the scale factor. If SBINS or LBINS are
;		set then these tables will be printed anyway.
;
; OUTPUTS:
;	Returns the calculated R-factor (on the whole data set). 
;	If the PRINT keyword is set the R-factor is also printed
;	to standard output (as tables if INTBINS and/or SBINS
;	are set).
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	ARR1 is scaled if bit 3 of the keyword SCALES is set.
;	NOTE: ARR1 will only be affected if it is
;	passed to RFACTOR as a variable and not as a subscripted
;	array, structure etc. See IDL Reference Manual p. 10-5.
;
; RESTRICTIONS:
;	If the keyword SCALES is set RFACTOR calls a function SCALE that 
;	should return a scale factor either as an array with one number 
;	for each reflection or as a single number.
; 	The call is:
;	  scales= SCALE(arr1,arr2,type=type,ssq=ssq, k=k, b=b) 
;	where K and B is output interpreted as the scale factors
;	K and B (only used if SCALES returns an array).
;
; PROCEDURE:
;	Straightforward
;
; MODIFICATION HISTORY:
;	Thomas Ursby, 1995
;-

ON_ERROR, 2

IF NOT KEYWORD_SET(k) THEN k=1.0
IF NOT KEYWORD_SET(b) THEN b=0.0
IF NOT KEYWORD_SET(cutoff) THEN cutoff=0.0
IF KEYWORD_SET(multiplicity) THEN ind_input=WHERE((arr1 gt cutoff) $
	  and (arr2 gt cutoff) and (multiplicity eq 1)) $
ELSE ind_input=WHERE((arr1 gt cutoff) and (arr2 gt cutoff)) 
apply=0
IF KEYWORD_SET(scales) THEN BEGIN
  IF ((scales and 2) eq 2) THEN sctype='blsq' ELSE sctype='lsq' 
  IF ((scales and 4) eq 4) THEN apply=1 ELSE apply=0 
  IF ((scales and 8) eq 8) THEN scw=w  
  IF ((scales and 16) eq 16) THEN sctype='kb'  
  IF KEYWORD_SET(ssq) THEN ssq2=ssq(ind_input)
  scales= SCALE(arr1(ind_input),arr2(ind_input), w=scw, $
		type=sctype, ssq=ssq2, k=k, b=b) 
ENDIF $
ELSE scales=1.0
i1= scales*arr1(ind_input) & i2= arr2(ind_input)
IF (apply) THEN arr1(ind_input)= i1 

; Calculate R-factors in intensity bins:
IF NOT KEYWORD_SET(intbins) THEN intbins=1
r= RFACTOR_I(i1, i2, w=w, type=type, ibinsize=ibinsize, $
		intbins=intbins, cutoff=cutoff, print=print)

; Calculate R-factors in s^2-bins:
IF KEYWORD_SET(sbins) THEN RFACTOR_S, i1, i2, ssq(ind_input), $
			w=w, sbins=sbins, type=type

; Calculate R-factors in lambda-bins:
IF KEYWORD_SET(lbins) THEN RFACTOR_L, i1, i2, lambda(ind_input), $
			w=w, lbins=lbins, type=type

IF KEYWORD_SET(print) THEN BEGIN
  IF ((size(scales))(0) eq 0) THEN PRINT, 'Scale factor:',scales $
  ELSE PRINT, 'Scale factor:', k, ' B-factor:', b 
  PRINT, 'Total reflections:', (size(arr1))(1), '   Used:', (size(i1))(1) 
ENDIF

RETURN, r
 
END
