function READ_SCALA, filename, par, logfilm, logi
;+
; NAME:
;	READ_SCALA
;
; PURPOSE:
;
; 	Read in an (ASCII) logfile from SCALA output.
;	Prints out global statistical factors (Rsym, completeness, redundancy ...)
;
; CATEGORY:
;	File managing
;
; CALLING SEQUENCE:
;	logres = READ_SCALA(filename,[par],[logfilm],[logi]) ;
;
; INPUTS:
;	filename : the path name of the file to process (with extension)
;
; OUTPUTS:
;	logres : the structure array containing the informations in resolution ranges
;	[logfilm] : [optional] the structure array containing the informations in images
;	[logi] : [optional] the structure array containing the informations in intensity bins
;	[par] : optionnal output overall info. 
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	Par -->
;		nbatch 	: The number of batches or images
;		nmeasin : The number of measurements read in
;		nuniq   : The number of unique measurements read in
;		nout    : The number of unique measurements written out
;		noutres : The number of measurements out of the resolution limits
;		isigi 	: The mean I/Sigma(I) over the data set
;		red	: The mean redundancy over the data set
;		r	: The global Rsym over the data set
;		ranom	: The global anomalous Rsym over the data set
;		rmeas	: The global Rmeas over the data set (a weighted Rsym)
;		pcv	: The global PCV factor over the data set (a weighted Rsym, takes into account redundancy)
;		comp	: The global completeness over the data set
;		nrej	: The total number of rejected measurements
;		nmeas	: The total number of USED MEASURED intensities
;		nref	: The total number of USED UNIQUE reflections
;		cell 	: The cell parameters
;		imean 	: The mean integrated intensity
;
;	Logres -->
;		s	: = 1/hres^2
;		hres	: The higher resolution
;		r	: The Rsym in the resolution range
;		rsym	: The Rsym in the resolution range
;		rmeas	: The Rmeas (weighted R) in the resolution range
;		pcv	: The PCV (weighted R) in the resolution range
;		rcum	: The cumulated Rsym in the resolution range
;		ranom	: The anomalous Rsym in the resolution range
;		rfull	: The Rsym for the fullies in the resolution range
;		isigi	: The I/SIGI in the resolution range
;		frcbias	: The fractional bias in the resolution range
;		comp	: The completeness in the resolution range
;		nmeas	: The number of mesurements in the resolution range
;		nref	: The number of unique reflections measured in the resolution range
;		red	: The redundancy in the resolution range
;		imean   : The mean intensity in the resolution range
;
;	Logi   -->
;		imax	: The higher intensity
;		r	: The Rmerge in the intensity range
;		isigi	: The I/SIGI in the intensity range
;		sigma	: The sigma in the intensity range
;		frcbias	: The fractional bias in the intensity range
;		nmeas	: The number of measurements in the intensity range
;		nref	: The number of unique reflections in the intensity range
;		
;	Logfilm -->
;		batch	: The image or batch number
;		scale	: The isotropic scale factor for the image
;		bfactor	: The isotropic B factor for the image
;		npr	: The number of partial reflections in the image
;		nfr	: The number of full reflections in the image
;		nr	: The number of measured reflections in the image
;		nbrej	: Measurements rejected for reasons other than statistical
;		chisqr	: The Chi square value in the image
;		r	: The Rmerge in the image
;		isigi	: The I/SIGI in the image
;		imean	: The mean intensity in the image
;		sigma	: The mean sigma in the image
;		sigma	: The mean sigma in the image
;		meanphi	: The mean oscillation angle in the image
;
; MODIFICATION HISTORY:
;	D.Bourgeois, Aout 1996.
;-

;on_error,2                              ;Return to caller if an error occurs


;open the SCALA  logfile 
dum = FINDFILE(filename,count=count)
IF count EQ 1 THEN PRINT,'Reading ',filename ELSE BEGIN 
PRINT,'File not found : ',filename
GOTO,escape
ENDELSE

;Define the structures :
maxrange = 100
maxfilm = 1000
maxi = 100

par = { $
		nbatch 	: 0  ,$
		nmeasin : 0L  ,$
		nuniq   : 0L  ,$
		nout    : 0L  ,$
		noutres : 0L  ,$
		isigi 	: 0. ,$
		red	: 0. ,$
		r	: 0. ,$ 
		ranom	: 0. ,$ 
		rmeas	: 0. ,$ 
		rsym	: 0. ,$ 
		pcv	: 0. ,$ 
		comp	: 0. ,$ 
		nref	: 0L ,$ 
		nrej	: 0L ,$ 
		nmeas	: 0L ,$ 
		imean	: 0. ,$ 
		cell 	: {a:0.,b:0.,c:0.,alpha:0.,beta:0.,gamma:0.} $ 
	} 


onerange =	{ $
		s	: 0., $
		hres	: 0., $
		r	: 0., $
		rsym	: 0., $
		rmeas	: 0., $
		rcum	: 0., $
		ranom	: 0., $
		rfull	: 0., $
		pcv	: 0., $
		isigi	: 0., $
		frcbias	: 0., $
		comp	: 0., $
		nmeas	: 0L, $
		nref	: 0L, $
		red	: 0., $
		imean	: 0. $
		}

logres = REPLICATE(onerange,maxrange) 

onei =	{ $
		imax	: 0., $
		r	: 0., $
		isigi	: 0., $
		sigma	: 0., $
		frcbias	: 0., $
		nmeas	: 0L, $
		nref	: 0L $
		}

logi = REPLICATE(onei,maxi) 

onefilm = 	{ $
		batch	: 0 , $
		scale 	: 0., $
		bfactor : 0., $
		npr	: 0L, $
		nfr	: 0L, $
		nr	: 0L, $
		nbrej	: 0L, $
		chisqr	: 0., $
		isigi	: 0., $
		imean	: 0., $
		sigma	: 0., $
		r	: 0., $
		meanphi	: 0. $  
		}

logfilm = REPLICATE(onefilm,maxfilm) 


line = STRING(90)
cycle = 0
filmindex = 0
resindex = 0

OPENR,unit,filename,/GET_LUN

;find the parameters :

WHILE (NOT EOF(unit)) DO BEGIN
  READF, unit, line

 ;SCALA : get the number of film packs
 IF (STRMID(line, 0, 20) EQ ' * Number of Batches') THEN BEGIN
  READS,line,FORMAT='(22X,I4)',nbatch
  par.nbatch = nbatch
 ENDIF

 ;SCALA : get the number of reflection read in :
 IF (STRMID(line, 0, 28) EQ ' Number of observations read') THEN BEGIN
  IF par.nmeasin EQ 0 THEN BEGIN 
   READS,line,FORMAT='(52X,I8)',nmeasin
   par.nmeasin = nmeasin
  ENDIF
 ENDIF

 ;SCALA : get the number of unique reflections read in :
 IF (STRMID(line, 0, 34) EQ ' Number of unique reflections read') THEN BEGIN
  IF par.nuniq EQ 0 THEN BEGIN 
   READS,line,FORMAT='(52X,I8)',nuniq
   par.nuniq = nuniq
  ENDIF
 ENDIF

 ;SCALA : get the number of unique reflections output :
 IF (STRMID(line, 0, 30) EQ ' Number of observations output') THEN BEGIN
  IF par.nout EQ 0 THEN BEGIN 
   READS,line,FORMAT='(52X,I8)',nout
   par.nout = nout
  ENDIF
 ENDIF

 ;SCALA : get the number of measurements rejected :
 IF (STRMID(line, 0, 28) EQ ' Number of outliers rejected') THEN BEGIN
  IF par.nrej EQ 0 THEN BEGIN 
   READS,line,FORMAT='(52X,I8)',nrej
   par.nrej = nrej
  ENDIF
 ENDIF

 ;SCALA : get the number of measurements outside resolution limit
 IF (STRMID(line, 0, 49) EQ ' Number of observations outside resolution limits') THEN BEGIN
  IF par.noutres EQ 0 THEN BEGIN
   READS,line,FORMAT='(52X,I8)',noutres
   par.noutres = noutres
  ENDIF
 ENDIF

 ;SCALA : get the cell dimensions
 IF (STRMID(line, 0, 20) EQ ' * Cell Dimensions :') THEN BEGIN
  READF,unit,line
  READF,unit,line
  READS,line,FORMAT='(6F10.2)',a,b,c,alpha,beta,gamma
  par.cell.a = a & par.cell.b = b & par.cell.c = c
  par.cell.alpha = alpha & par.cell.beta = beta & par.cell.gamma = gamma
 ENDIF

;SCALA : get B-factor and scale and chi_square and number_of_ref and number_of_rejected by film against batch.
IF (STRMID(line, 0, 36) EQ ' $TABLE: >>> Scales v rotation range') THEN BEGIN
 filmindex = 0
 REPEAT READF,unit,line UNTIL STRMID(line,0,14) EQ '    N  Run.Rot'
 READF,unit,line

 WHILE(STRLEN(line) NE 0 AND STRMID(line,0,3) NE ' $$') DO BEGIN
  READS,line,FORMAT='(I5,F6.2,6X,F8.2,4X,I4,F10.3,2F10.4,2X,I8,2X,I8)',n,runrot,meanphi,batch,bfactor,scale,chisqr,nr,nbrej
 logfilm(filmindex).scale = scale
 logfilm(filmindex).bfactor = bfactor
 logfilm(filmindex).chisqr = chisqr
 logfilm(filmindex).nr = nr
 logfilm(filmindex).nbrej = nbrej
 logfilm(filmindex).meanphi = meanphi
 logfilm(filmindex).batch = batch
 filmindex = filmindex + 1
 READF,unit,line

 ENDWHILE
ENDIF

;SCALA : Get Imean, Sigma, NFR, NPR, SIGMO, Imean/SIGMO Rfactor NR NBREJ NBRJM, against batch
IF (STRMID(line, 0, 32) EQ ' $TABLE : Analysis against Batch') THEN BEGIN
 filmindex = 0
 REPEAT READF,unit,line UNTIL STRMID(line,0,3) EQ ' $$'
 READF,unit,line
 WHILE(STRLEN(line) NE 0 AND STRMID(line,0,3) NE ' $$') DO BEGIN
  READS,line,FORMAT='(I6,4X,-I4,3X,F9.1,1X,F9.1,2I7,F10.1,F9.2,3X,F8.3,3I7)',n,batch,imean,sigma,nfr,npr,sigmo,isigi,rfactor,nr,nbrej,nbrjm
 logfilm(filmindex).imean = imean
 logfilm(filmindex).sigma = sigma
 logfilm(filmindex).nfr = nfr
 logfilm(filmindex).npr = npr
 logfilm(filmindex).isigi = isigi
 logfilm(filmindex).r = 100*rfactor
 logfilm(filmindex).nr = nr
 logfilm(filmindex).nbrej = nbrej

 filmindex = filmindex + 1
 READF,unit,line
 ENDWHILE
ENDIF

;SCALA get completeness, multiplicity, rfactors by shells
IF (STRMID(line, 0, 56) EQ ' $TABLE: Completeness, multiplicity, Rmeas v. resolution') THEN BEGIN

 REPEAT READF,unit,line UNTIL STRMID(line,0,3) EQ ' $$'
 READF,unit,line
 resindex = 0
WHILE(STRMID(line,0,3) NE ' $$') DO BEGIN
 READS,line,FORMAT='(I4,F7.3,F8.2,I9,I8,I7,5F8.1,5F8.3)',n,s,hres,nmeas,nref,ncent,comp,cumcomp,red,anomcmpl,anomfrc,rmeas,rmeaso,rsym,pcv,pcvo
; READS,line,FORMAT='(I4,F7.3,F7.2,I9,I8,I7,5F7.1,5F8.3)',n,s,hres,nmeas,nref,ncent,comp,cumcomp,red,anomcmpl,anomfrc,rmeas,rmeaso,rsym,pcv,pcvo
 logres(resindex).s = s
 logres(resindex).hres = hres
 logres(resindex).nref = nref
 logres(resindex).nmeas = nmeas
 logres(resindex).comp = comp
 logres(resindex).red = red
 logres(resindex).rsym = 100*rsym
 logres(resindex).pcv = 100*pcvo
 logres(resindex).rmeas = 100*rmeaso
 resindex = resindex + 1
 READF,unit,line

ENDWHILE

REPEAT BEGIN
 READF,unit,line 
;ENDREP UNTIL (STRMID(line,0,17) EQ '          Overall')
ENDREP UNTIL (STRMID(line,0,18) EQ '           Overall')
READS,line,FORMAT='(19X,I9,I8,I7,5F8.1,5F8.3)',nmeas,nref,ncent,comp,cumcomp,red,anomcmpl,anomfrc,rmeas,rmeaso,rsym,pcv,pcvo
;READS,line,FORMAT='(18X,I9,I8,I7,5F7.1,5F8.3)',nmeas,nref,ncent,comp,cumcomp,red,anomcmpl,anomfrc,rmeas,rmeaso,rsym,pcv,pcvo
par.comp = comp
par.red = red
;par.nfull = nfull
par.nref = nref
par.nmeas=nmeas
par.rmeas=100*rmeaso
par.rsym=100*rsym
par.pcv=100*pcvo

ENDIF

; SCALA get final R-factor by intensity
IF (STRMID(line, 0, 35) EQ ' $TABLE: Analysis against intensity') THEN BEGIN

REPEAT READF,unit,line UNTIL STRMID(line,0,3) EQ ' $$'
REPEAT READF,unit,line UNTIL STRMID(line,0,3) EQ ' $$'
READF,unit,line
iindex = 0

WHILE(STRMID(line,0,3) NE ' $$') DO BEGIN

 READS,line,FORMAT='(F9.0,3F7.3,I7,F9.0,3F8.1,I9,2I7,2X,F8.3,I7,3X,F8.1)',imax,r,rfull,ranom,nanom,av_i,sigma,isigi,sd,nmeas,nref,ncent,frcbias,nbias,abias
 logi(iindex).imax = imax
 logi(iindex).r = 100*r
 logi(iindex).nmeas = nmeas
 logi(iindex).nref = nref
 logi(iindex).isigi = isigi
 logi(iindex).sigma = sigma
 logi(iindex).frcbias = frcbias
 iindex = iindex + 1
 READF,unit,line
ENDWHILE
maxiindex = iindex

REPEAT BEGIN
 READF,unit,line 
ENDREP UNTIL (STRMID(line,0,8) EQ ' Overall')
READS,line,FORMAT='(9X,3F7.3,I7,F9.0,3F8.1,I9,2I7,2X,F8.3,I7,3X,F8.1)',r,rfull,ranom,nanom,imean,sigma,isigi,sd,nmeas,nref,ncent,frcbias,nbias,abias

par.r = 100*r
par.ranom = 100*ranom
par.imean = imean
par.isigi = isigi

ENDIF

; SCALA : get final R-factor by shell (rfac, rfull, rcum, ranom, imean, sigma, isigi, sd, imean_sd, nmeas, nref, frcbias, nbias
;IF (STRMID(line, 0, 37) EQ ' $TABLE: Analysis against resolution:') THEN BEGIN
IF (STRMID(line, 0, 36) EQ ' $TABLE: Analysis against resolution') THEN BEGIN
 resindex = 0
 REPEAT READF,unit,line UNTIL STRMID(line,0,3) EQ ' $$'
 READF,unit,line
 
 WHILE(STRLEN(line) NE 0 AND STRMID(line,0,3) NE ' $$') DO BEGIN
  READS,line,FORMAT='(10X,F6.2,F6.2,3F7.3, I7,F9.0,F8.1,F6.1,F7.0,F5.1,1X,2I7,I6,F8.3,I7)',hres,rfactor,rfull,rcum,ranom,nanom,imean,sigma,isigi,sd,imean_sd,nmeas,nref,ncent,frcbias,nbias
  logres(resindex).r = 100*rfactor
  logres(resindex).rcum = 100*rcum
  logres(resindex).ranom = 100*ranom
  logres(resindex).rfull = 100*rfull
  logres(resindex).imean = imean
  logres(resindex).isigi = isigi
  logres(resindex).nmeas = nmeas
  logres(resindex).frcbias = frcbias
  resindex = resindex + 1
  READF,unit,line
ENDWHILE
maxresindex = resindex
ENDIF


ENDWHILE

FREE_LUN, unit

logfilm = logfilm(0:nbatch-1)
logres = logres(0:maxresindex-1)
;the r_factors are inverted (lres to hres) 
logi = logi(0:maxiindex-1)

PRINT,''
PRINT,'&&&&&&& CELL PARAMETERS &&&&&&&'
PRINT,''
PRINT,'Cell parameters : ',par.cell.a,par.cell.b,par.cell.c,par.cell.alpha,par.cell.beta,par.cell.gamma

PRINT,''
PRINT,'&&&&&&& NUMBER OF REFLECTIONS &&&&&&&'
PRINT,''
PRINT,'Number of packs : ',par.nbatch
PRINT,'Number of measurements read in : ',par.nmeasin
PRINT,'Number of unique measurements read in : ',par.nuniq
PRINT,'Total number of measurements rejected : ',par.nrej
PRINT,'Total number of reflections written out : ', par.nout
PRINT,''
PRINT,'&&&&&&& FINAL PARAMETERS &&&&&&&'
PRINT,''
PRINT,'Average redundancy : ', TOTAL(logres.nmeas)/TOTAL(logres.nref)
PRINT,'Mean <I/Sigma> : ',par.isigi
PRINT,'Mean <I> : ',par.imean
PRINT,'Overall completeness [%] : ',par.comp
PRINT,'Merging Rsym-Factor [SUM ( ABS(I - <I>)) / SUM (I)] [%] : ',par.rsym
PRINT,'Weighted Rmeas-Factor w[SUM ( ABS(I - <I>)) / SUM (I)] [%] : ',par.rmeas
PRINT,'PCV R-Factor [%] : ',par.pcv

RETURN,logres

escape :
END







