FUNCTION READ_MTZ, name, labels, columns=columns, number=number, keap=keap, format=format, bad_ref=bad_ref
;+
; NAME:
;	READ_MTZ
;
; PURPOSE:
;	Reads in a MTZ-file by running mtzdmp (the set-up for ccp4
;	must be called with "ccp4") or by reading a MTZ-dump file.
;
; CATEGORY:
;	Crystallography, Input
;
; CALLING SEQUENCE:
;   Result = READ_MTZ(name [,labels,columns=columns,number=number,/keap, format=format, bad_ref=bad_ref])
;
; INPUTS:
;	NAME: String containing the name of the mtz/mtz-dump file.
;		If the name ends with ".mtz" then mtzdump is first run 
;		(via mtzdmp) to create the mtz-dump file (created as 
;		"file.dmp" where "file" is NAME with "mtz" removed.)
;		Unless the keyword KEAP is set the created dump file
;		is removed.
;
; KEYWORDED PARAMETERS:
;	COLUMNS: The number of columns in the file (after the three
;		first containing the Miller indices) that you want to
;		read (but there might be more columns in the file).
;		Default to 'Number of Columns'-3 as read from the file
;		header. 
;	NUMBER: If given only the NUMBER first reflections are read.
;		Otherwise all reflections are read.
;	KEAP: If NAME is a MTZ-file and KEAP is set then the created
;		MTZ-dump file is not removed.
;	FORMAT : The format can be specified if necessary.
;	BAD_REF : contains the reflections marked  'MNF', 'na', '****' or '?'
;
; OUTPUTS:
;	READ_MTZ returns an array of structures with the reflection info
;	from the file. The structure has two elements:
;		M: An integer array with three elements (hkl).
;		ENTRY: An double array with COLUMNS elements.
;	LABELS: An array with the name of the columns.
;		E.g. LABELS(0) is the name of M(0) and LABELS(3) is 
;		the name of ENTRY(0). 
;		Note: Labels include h,k,l !!
;
; SIDE EFFECTS:
;	None.
;
; RETRICTIONS:
;	Reads a dumpfile from a MTZ-file created by MTZDUMP
;	which contains hkl and at least COLUMNS columns with (real) numbers.
;	The format created by MTZDUMP varies slightly and the routine might
;	file. This version assumes that the reflection data starts after
;	a line beginning with " FORMAT".
;	If the name end with '.mtz' then 'mtzdmp' is called. Before calling
;	mtzdmp the command 'ccp4' is given in the subshell so this must be
;	the setup for CCP4 (or do something harmless if CCP4 is already
;	setup).
;	The number of columns (not counting h,k,l) must be at least 1. 
;
; MODIFICATION HISTORY:
;       Written by Thomas Ursby, 1994
;       04-07-1995 Fixed reading of more than 3+6 columns. 
;	17-07-1995 Added call to MTZDMP. 
;	23-10-1995 Added LABELS.
;	04-09-1996 Fixed reading of more than one line of LABELS.
;	07-01-2000 Add reading bad_ref
;-

;on_error,2                      ;Return to caller if an error occurs

dummy='' & dummy2='' & mtz=0
name1=name

IF (STRMID(name,STRLEN(name)-4,4) eq '.mtz') THEN BEGIN
  PRINT,'NEED CCP4 VERSION 3.2'
  PRINT, 'Running MTZDMP'
  mtz=1
  name1= STRMID(name,0,STRLEN(name)-4)+'.dmp'
  dummyarr=FINDFILE(name, count=count)
  IF (count eq 0) THEN MESSAGE, "Couldn't find the MTZ file."
  WHICH, 'read_mtz', file=file, /QUIET
  SPAWN, STRMID(file, 0, RSTRPOS(file, '.'))+'.com '+name+' '+name1
ENDIF 
dummyarr=FINDFILE(name1, count=count)
IF (count eq 0) THEN MESSAGE, "Couldn't find the MTZDUMP file."
OPENR, file, name1, /GET_LUN

; Find and read the number of columns:
searchstr = ' * Number of Columns ='
start= 0
length = STRLEN(searchstr)
WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
  READF, file, dummy
ENDWHILE
tot_columns=0L
READS, STRMID(dummy,length,50), tot_columns
IF NOT KEYWORD_SET(columns) THEN columns=tot_columns-3

; Find and read the number of reflections:
searchstr = ' * Number of Reflections ='
start= 0
length = STRLEN(searchstr)
WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
  READF, file, dummy
ENDWHILE
nb_ref=0L
READS, STRMID(dummy,length,50), nb_ref

; Find and read the column labels:
searchstr = ' * Column Labels :'
start= 0
length = STRLEN(searchstr)
WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
  READF, file, dummy
ENDWHILE
READF, file, dummy
READF, file, dummy ; First line with labels. More lines? :
dummy2= 'something'
WHILE (dummy2 ne '') DO BEGIN
    READF,file,dummy2
    dummy= dummy + dummy2
ENDWHILE
PRINT, 'Input column labels (from file):' 
PRINT, STRTRIM(dummy,2)
labels= STRARR(columns+3)
colname=''
FOR i=0,columns+1 DO BEGIN
  dummy= STRTRIM(dummy,2)
  READS, STRMID(dummy,0,STRPOS(dummy,' ')), colname
  labels(i)=colname
  dummy= STRTRIM(STRMID(dummy,STRPOS(dummy,' '),STRLEN(dummy)),2)
ENDFOR
IF (STRPOS(dummy,' ') ne -1) THEN dummy=STRMID(dummy,0,STRPOS(dummy,' '))
READS, dummy, colname
labels(columns+2)=colname
PRINT, 'Output column labels (output from function):' 
PRINT, labels
PRINT, 'Output 3 (hkl) + ',columns, ' columns.'
 
; Find the start of the reflection data:
;searchstr = ' FORMAT'
searchstr = ' LIST OF REFLECTIONS' ;CCP4 vsn 3.2
start= 0
length = STRLEN(searchstr)
WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
  READF, file, dummy
ENDWHILE

   READF, file, dummy ;CCP4 vsn 3.2
;  READF, file, dummy ;CCP4 vsn 3.1

IF KEYWORD_SET(number) THEN nb_read=(number<nb_ref)>0 ELSE nb_read=nb_ref
oneref= {m:intarr(3), entry:dblarr(columns)}
onebad_ref= {m:intarr(3), entry:dblarr(columns)}
ref= REPLICATE(oneref,nb_read) 
bad_ref= REPLICATE(onebad_ref,nb_read) 
i=0L
j=0L 
cr = STRING("15b)

REPEAT READF,file,dummy UNTIL STRLEN(STRTRIM(dummy)) GT 0
col_read=8
WHILE ((NOT EOF(file)) AND (NOT (STRMID(dummy,0,8) EQ "  Normal")) AND $
	(NOT (STRMID(dummy,4,13) EQ "SUMMARY_BEGIN")) AND $
      	(NOT (STRMID(dummy,0,8) EQ " MTZDUMP")) AND $
      	(NOT (STRMID(dummy,4,10) EQ "FONT COLOR")) AND (i lt nb_read)) DO BEGIN

  IF i/100 EQ i/100.0 THEN PRINT,FORMAT="($,I5,' ie : ',F3.0,'%',A)",i,100.*i/(FLOAT(nb_ref)>1),cr
  ; If more than 3+6 columns then the MTZ-dump file is divided into several
  ; lines, each following line having six columns:
  ; (But if 3+6 columns there is one empty line between each reflection,
  ;  what about 3+12 columns etc?)
  WHILE (col_read lt tot_columns) DO BEGIN
    READF,file,dummy2
    dummy= dummy + dummy2
    col_read=col_read+6
  ENDWHILE
  IF STRPOS(dummy,'MNF') EQ -1 AND STRPOS(dummy,'na') EQ -1 AND STRPOS(dummy,'?') EQ -1 AND STRPOS(dummy,'****') EQ -1 THEN BEGIN
   IF N_ELEMENTS(format) EQ 0 THEN READS,dummy,oneref ELSE $
	READS,dummy,oneref,FORMAT=format
   ref(i)=oneref
   i = i + 1
  ENDIF ELSE BEGIN
   REPEAT BEGIN
    IF STRPOS(dummy,'MNF') NE -1 THEN dummy=STR_REPLACE(dummy,'MNF  ','-9999',/NO_COMPRESS)
    IF STRPOS(dummy,'na') NE -1 THEN dummy=STR_REPLACE(dummy,' na  ','-9999',/NO_COMPRESS)
    IF STRPOS(dummy,'?') NE -1 THEN dummy=STR_REPLACE(dummy,'  ?  ','-9999',/NO_COMPRESS)
    IF STRPOS(dummy,'****') NE -1 THEN dummy=STR_REPLACE(dummy,'****','-9999',/NO_COMPRESS)
   ENDREP UNTIL (STRPOS(dummy,'MNF') EQ -1 AND STRPOS(dummy,'na') EQ -1 AND STRPOS(dummy,'?') EQ -1 AND STRPOS(dummy,'****') EQ -1)
   IF N_ELEMENTS(format) EQ 0 THEN READS,dummy,onebad_ref ELSE $
	READS,dummy,onebad_ref,FORMAT=format
   bad_ref(j)=onebad_ref
   j = j + 1
  ENDELSE
  REPEAT READF,file,dummy UNTIL STRLEN(STRTRIM(dummy)) GT 0
  col_read=8
ENDWHILE

FREE_LUN, file

PRINT, 'Number of reflections according to header:',nb_ref
PRINT, 'Number of reflections marked ''MNF'', ''na'', ''****'' or ''?'': ',nb_ref-i
PRINT, 'Number of reflections read               :',i

IF (mtz and not KEYWORD_SET(keap)) THEN $
	SPAWN, '/bin/rm '+name1

IF j GT 0 THEN bad_ref=bad_ref(0:j-1) ELSE bad_ref=0
IF i GT 0 THEN RETURN, ref(0:i-1) ELSE RETURN,0


END 

