function READ_AGRO, filename, par, logfilm, logi
;+
; NAME:
;	READ_AGRO
;
; PURPOSE:
;
; 	Read in an (ASCII) logfile from AGROVATA output.
;	Prints out global statistical factors (Rsym, completeness, redundancy ...)
;
; CATEGORY:
;	File managing
;
; CALLING SEQUENCE:
;	logres = READ_AGRO(filename,[par],[logfilm],[logi]) ;
;
; INPUTS:
;	filename : the path name of the file to process (with extension)
;
; OUTPUTS:
;	logres : the structure array containing the informations in resolution ranges
;	[logfilm] : [optional] the structure array containing the informations in images
;	[logi] : [optional] the structure array containing the informations in intensity bins
;	[par] : optionnal output overall info. 
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	Par -->
;		npacks 	: The number of packs or images
;		isigi 	: The mean I/Sigma(I) over the data set
;		red	: The mean redundancy over the data set
;		r	: The global Rmerge over the data set
;		comp	: The global completeness over the data set
;		nrej	: The total number of rejected measurements
;		nmeas	: The total number of MEASURED intensities
;		nref	: The total number of USED reflections
;		nfull	: The total number of "reconstituted" full observations
;		cell 	: The cell parameters
;		imean 	: The mean integrated intensity
;
;	Logres -->
;		s	: = 1/hres^2
;		hres	: The higher resolution
;		r	: The Rmerge in the resolution range
;		comp	: The completeness in the resolution range
;		nmeas	: The number of mesurements in the resolution range
;		nref	: The number of reflections measured in the resolution range
;		red	: The redundancy in the resolution range
;		imean   : The mean intensity in the resolution range
;
;	Logi   -->
;		imax	: The higher intensity
;		r	: The Rmerge in the intensity range
;		nmeas	: The number of mesurements in the intensity range
;		nref	: The number of reflections measured in the intensity range
;		
;	Logfilm -->
;		film	: The image number
;		scale	: The isotropic scale factor for the image
;		bfactor	: The isotropic B factor for the image
;		partial	: The number of partial reflections in the image
;		full	: The number of full reflections in the image
;		nrej	: Measurements rejected for reasons other than statistical
;		chis	: The Chi square value in the image
;		r	: The Rmerge in the image
;
; MODIFICATION HISTORY:
;	D.Bourgeois, Aout 1996.
;-

;on_error,2                              ;Return to caller if an error occurs


;open the AGROVATA  logfile 
dum = FINDFILE(filename,count=count)
IF count EQ 1 THEN PRINT,'Reading ',filename ELSE BEGIN 
PRINT,'File not found : ',filename
GOTO,escape
ENDELSE

;Define the structures :
maxrange = 100
maxfilm = 1000
maxi = 100

par = { $
		npacks 	: 0  ,$
		isigi 	: 0. ,$
		red	: 0. ,$
		r	: 0. ,$ 
		comp	: 0. ,$ 
		nref	: 0L ,$ 
		nrej	: 0L ,$ 
		nmeas	: 0L ,$ 
		nfull	: 0L ,$ 
		imean	: 0. ,$ 
		cell 	: {a:0.,b:0.,c:0.,alpha:0.,beta:0.,gamma:0.} $ 
	} 


onerange =	{ $
		s	: 0., $
		hres	: 0., $
		r	: 0., $
		comp	: 0., $
		nmeas	: 0L, $
		nref	: 0L, $
		red	: 0., $
		imean	: 0. $
		}

logres = REPLICATE(onerange,maxrange) 

onei =	{ $
		imax	: 0., $
		r	: 0., $
		nmeas	: 0L, $
		nref	: 0L $
		}

logi = REPLICATE(onei,maxi) 

onefilm = 	{ $
		film	: 0 , $
		scale 	: 0., $
		bfactor : 0., $
		partial : 0L, $
		full	: 0L, $
		chis	: 0., $
		nrej	: 0., $
		r	: 0.  $
		}

logfilm = REPLICATE(onefilm,maxfilm) 



line = STRING(90)
cycle = 0
filmindex = 0
resindex = 0

OPENR,unit,filename,/GET_LUN

;find the parameters :

WHILE (NOT EOF(unit)) DO BEGIN
  READF, unit, line

;get the number of film packs
IF (STRMID(line, 0, 20) EQ ' * Number of Batches') THEN BEGIN
READS,line,FORMAT='(22X,I4)',npacks
par.npacks = npacks
ENDIF

;get the number of reflection read in :
IF (STRMID(line, 0, 24) EQ ' * Number of Reflections') THEN BEGIN
IF par.nmeas EQ 0 THEN BEGIN 
READS,line,FORMAT='(26X,I7)',nmeas
par.nmeas = nmeas
ENDIF
ENDIF

;get the number of measurements rejected :
IF (STRMID(line, 0, 40) EQ '   and Number of measurements rejected =') THEN BEGIN
READS,line,FORMAT='(41X,I8)',nrej
par.nrej = nrej
ENDIF

IF (STRMID(line, 0, 20) EQ ' * Cell Dimensions :') THEN BEGIN
READF,unit,line
READF,unit,line
READS,line,FORMAT='(1X,6F8.2)',a,b,c,alpha,beta,gamma
par.cell.a = a & par.cell.b = b & par.cell.c = c
par.cell.alpha = alpha & par.cell.beta = beta & par.cell.gamma = gamma
ENDIF

;get number of partial and full reflections
IF (STRMID(line, 0, 33) EQ ' $TABLE : Analysis against Batch:') THEN BEGIN
filmindex = 0
REPEAT READF,unit,line UNTIL STRMID(line,0,3) EQ ' $$'
READF,unit,line

WHILE(STRMID(line,0,3) NE ' $$') DO BEGIN
 READS,line,FORMAT='(1X,I3,2X,I3,3X,F12.2,F10.2,2F9.1,4I7,2F10.2,5X,F7.3,1X,3I6)',batch,no,imean,sigma,biasfull,biaspar,n_full,n_par,unmixed_full,unmixed_par,scatter_from_imean,ratio,r,number_in_r,number_rejected,number_invalid

logfilm(filmindex).film = no
;logfilm(filmindex).scale = scale
;logfilm(filmindex).bfactor = bfactor
logfilm(filmindex).nrej = number_rejected
logfilm(filmindex).partial = n_par
logfilm(filmindex).full = n_full
logfilm(filmindex).r = r*100

filmindex = filmindex + 1
READF,unit,line

ENDWHILE
ENDIF


;get B-factor and scale by film
IF (STRMID(line, 0, 36) EQ ' $TABLE: Batch scales and B-factors:') THEN BEGIN
filmindex = 0
REPEAT READF,unit,line UNTIL STRMID(line,0,3) EQ ' $$'
READF,unit,line

WHILE(STRMID(line,0,3) NE ' $$') DO BEGIN
 READS,line,FORMAT='(1X,2I6,2F8.3,2F8.1)',batch,no,scale,sd_scale,bfactor,sd_b

logfilm(filmindex).scale = scale
logfilm(filmindex).bfactor = bfactor

filmindex = filmindex + 1
READF,unit,line

ENDWHILE
ENDIF

;get observation redundancies by shells
IF (STRMID(line, 0, 36) EQ ' $TABLE: Completeness & multiplicity') THEN BEGIN
resindex = 0
READF,unit,line
READF,unit,line
READF,unit,line
READF,unit,line
READF,unit,line

WHILE(STRMID(line,0,3) NE ' $$') DO BEGIN
 READS,line,FORMAT='(5X,F6.3,4X,F6.2,2X,I7,3X,I6,4X,F5.1,4X,F5.1)',s,hres,nmeas,nref,comp,red
logres(resindex).s = s
logres(resindex).hres = hres
logres(resindex).nref = nref
logres(resindex).nmeas = nmeas
logres(resindex).comp = comp
logres(resindex).red = red
resindex = resindex + 1
READF,unit,line

ENDWHILE

READF,unit,line
READF,unit,line
READS,line,FORMAT='(23X,I7,2X,I7,4X,F5.1,4X,F5.1)',nfull,nref,comp,red
par.comp = comp
par.red = red
par.nfull = nfull
par.nref = nref


ENDIF

;get final R-factor by intensity
IF (STRMID(line, 0, 36) EQ ' $TABLE: Analysis against intensity:') THEN BEGIN

REPEAT READF,unit,line UNTIL STRMID(line,0,3) EQ ' $$'
REPEAT READF,unit,line UNTIL STRMID(line,0,3) EQ ' $$'
READF,unit,line
iindex = 0

WHILE(STRMID(line,0,3) NE ' $$') DO BEGIN

 READS,line,FORMAT='(F7.0,3X,F6.3,3X,F6.3,I7,F9.0,3F8.1,F9.0,I7)',imax,r,ranom,nanom,av_i,sigma,ioversig,sd,nmeas,nref
 logi(iindex).imax = imax
 logi(iindex).r = 100*r
 logi(iindex).nmeas = nmeas
 logi(iindex).nref = nref
 iindex = iindex + 1
READF,unit,line

ENDWHILE
maxiindex = iindex

READF,unit,line
READF,unit,line

 READS,line,FORMAT='(10X,F6.3,F9.3,I7,F9.0,3F8.1,F9.0,2I7)',r,ranom,nanom,imean,sigma,isigi,sd,nmeas,nref,ncent

par.r = 100*r
par.imean = imean
par.isigi = isigi

ENDIF

;get final R-factor by shell
IF (STRMID(line, 0, 36) EQ ' $TABLE: Analysis against resolution') THEN BEGIN

READF,unit,line
READF,unit,line
READF,unit,line
READF,unit,line
READF,unit,line

resindex = 0

WHILE(STRMID(line,0,3) NE ' $$') DO BEGIN

 READS,line,FORMAT='(12X,F6.2,3X,F6.3,3X,F6.3,16X,F9.0)',hres,r,rcum,imean
 logres(resindex).r = 100*r
 logres(resindex).imean = imean
 resindex = resindex + 1
READF,unit,line

ENDWHILE
maxresindex = resindex
ENDIF


ENDWHILE

FREE_LUN, unit

logfilm = logfilm(0:npacks-1)
logres = logres(0:maxresindex-1)
;the r_factors are inverted (lres to hres) 
logi = logi(0:maxiindex-1)

PRINT,''
PRINT,'&&&&&&& CELL PARAMETERS &&&&&&&'
PRINT,''
PRINT,'Cell parameters : ',par.cell.a,par.cell.b,par.cell.c,par.cell.alpha,par.cell.beta,par.cell.gamma

PRINT,''
PRINT,'&&&&&&& NUMBER OF REFLECTIONS &&&&&&&'
PRINT,''
PRINT,'Number of packs : ',par.npacks
PRINT,'Number of measurements read in : ',par.nmeas
PRINT,'Total number of measurements rejected : ',par.nrej
PRINT,'Number of reconstituted "full" measurements : ',par.nfull
PRINT,'Total number of reflections written out : ', par.nref
PRINT,''
PRINT,'&&&&&&& FINAL PARAMETERS &&&&&&&'
PRINT,''
PRINT,'Average redundancy : ', TOTAL(logres.nmeas)/TOTAL(logres.nref)
PRINT,'Mean <I/Sigma> : ',par.isigi
PRINT,'Mean <I> : ',par.imean
PRINT,'Overall completeness [%] : ',par.comp
PRINT,'Merging R-Factor [SUM ( ABS(I - <I>)) / SUM (I)] [%] : ',par.r

RETURN,logres

escape :
END







