pro LAUENORM, logfilename, integfilename, outfilename, $
		frames_per_pack=frames_per_pack, film_no=film_no, $
		scale=scale, pack_kb=pack_kb, k=k, b=b, inscale=inscale, $
		not_read_log=not_read_log, $
		inpolynom=inpolynom, inpolorder=inpolorder, $
		inranges=inranges, inlmin=inlmin, inlmax=inlmax, $
		logfit=logfit, sigrej=sigrej, $
		tau=tau, pntgrp=pntgrp, geasc=geasc, update=update, $
		append=append, norm_plot=norm_plot,$
		set_neg_to_zero=set_neg_to_zero
;+
; NAME:
;	LAUENORM
;
; PURPOSE:
;	Apply scaling as determined by the Daresbury program Lauenorm.
;	Reads the normalisation polynomial and the scale factors (K and
;	B) from a Lauenorm logfile (see READ_LN) and applies these to
;	the singles in a INTEG file (output from the DB integration routine).
;	Reflections outside the wavelength limits in Lauenorm are removed
;	as are reflections with bad-integration flag, sigma equals to zero
;	and I/Sigma<SIGREJ if SIGREJ is given.
;	The tags INTP(FILM_NO) and ISIGP(FILM_NO) are scaled in the same way.
;	The result is output in a ASCII file with format as shown below.
;	The indices are changed to the asymmetric unit (according to the
;	given PNTGRP; see EQUIV_HKL.PRO).
;	GEASC (or GE1) files can also be input.
;
; CATEGORY:
;	Crystallography
;
; CALLING SEQUENCE:
;	LAUENORM, logfilename, integfilename, outfilename, $
;		frames_per_pack=frames_per_pack, film_no=film_no, $
;		scale=scale, pack_kb=pack_kb, k=k, b=b, inscale=inscale, $
;		not_read_log=not_read_log, $
;		inpolynom=inpolynom, inpolorder=inpolorder, $
;		inranges=inranges, inlmin=inlmin, inlmax=inlmax, $
;		logfit=logfit, sigrej=sigrej, $
;		tau=tau, pntgrp=pntgrp, geasc=geasc, update=update, $
;		append=append, /norm_plot, /set_neg_to_zero
;
; INPUTS:
;	LOGFILENAME: Name of the logfile from Lauenorm where the normalisation
;		polynomial and the film pack scale factors should be read.
;		Also handles multiple wavelength ranges.
;	INTEGFILENAME: Name of the file(s) as output from the integration
;		routine. If an array of strings then each string in the array
;		is taken as a filename and the procedure is applied to each.
;		If FRAMES_PER_PACK is given and greater than 1 then file
;		1,...,FRAMES_PER_PACK are considered to be the frames of
;		pack 1 etc. The different frames will be added before
;		scaling and output. The different frames are then output
;		to the same file so the array (if an array) OUTFILENAME
;		should not be of the same size as INTEGFILENAME but
;		of size (size of INTEGFILENAME)/FRAMES_PER_PACK.
;		The whole filename should be given except when GEASC is set
;		in which case the trailing ".geasc" (or ".ge1"/".ge2") should
;		be left out.
;		The first frame in each pack must be present but if the 
;		consecutive frames are not found they are simply skipped
;		(and the sigmas adjusted accordingly).
;	OUTFILENAME: Name of the ASCII output file. If INTEGFILENAME
;		is an array and OUTFILENAME is a scalar string then the
;		output is appended to the one file OUTFILENAME. See OUTPUTS
;		below.
;
; KEYWORDED PARAMETERS:
;	FRAMES_PER_PACK: Number of frames per pack. See INTEGFILENAME.
;		If set and more than 1 then the different frames are
;		added up before any scaling is done. The sigma of the
;		sum is calculated as S=frames/sqrt(1/s1^2+1/s2^2+...).
;		Only those reflections appearing in all frames are kept.
;	FILM_NO: Number between 1 and 6 that specifies which film in 
;		the INTP and ISIGP arrays to use. Default 1.
;     If SCALE is not set and K,B not given the different packs are not
;     scaled at all. Either set SCALE to take scaling from the Lauenorm
;     logfile or set K,B to explicitly give the scale factors. It is 
;     recommended not to scale here but scale the output with for example
;     ROTAVATA. 
;	SCALE: If set then scaling is done using a scale factor and an
;		isotropic B-factor (see PACK_KB) read from the Lauenorm
;		logfile (unless K and B set). It is useful to not set SCALE
;		if the Lauenorm logfile doesn't contain K,B-scaling since
;		READ_LN will crash if it is asked to read the scale factors
;		and only K-scaling is done for exampel. 
;	PACK_KB: If set then this is an integer or an array of integers 
;		containing the number of the film pack in the Lauenorm
;		logfile whose K,B-values should be used. If not given
;		then the first film pack is used for the first pack and
;		the second film pack for the second etc. Not used if K and
;		B is given.
;	K,B: The scale factor K and B-factor for the scaling. If an array
;		then K(0),B(0) is used for the first file etc. 
;	INSCALE: An extra scale factor that is applied to intensities and
;		sigmas (scalar).
;	NOT_READ_LOG: If set then the Lauenorm logfile is not read.
;	INPOLYNOM: Input polynomial. If given then this is used instead of
;		the polynomial read from the Lauenorm logfile. 
;		ARR(ranges,max(polorder)+1)
;	INPOLORDER: The order of the polynomial(s). Otherwise it
;		is read from the Lauenorm logfile. ARR(ranges)
;	INRANGES: Number of ranges for the polynomials. Otherwise it
;		is read from the Lauenorm logfile.
;	INLMIN: If given then this will be used as minimum lambda. 
;		Otherwise it is read from the Lauenorm logfile. ARR(ranges)
;	INLMAX: If given then this will be used as maximum lambda. 
;		Otherwise it is read from the Lauenorm logfile. ARR(ranges)
;	LOGFIT: If set then the polynomial will be interpreted as a
;		fit to the logarithm of scale factor in each bin
;		(i.e. a fit to alog(SCALE) instead of SCALE). Use with
;		INPOLYNOM since this does not exist in Lauenorm.
;	SIGREJ: If given the measurements with I/Sigma < SIGREJ are rejected
;		on input.
;	TAU: The vertical polarisation (where vertical is along the Lauenorm 
;		y-coordinate). Default 1.0.
;	PNTGRP: The point group number (see EQUIV_HKL). Used for changing
;		indices to asymmetric unit. Default 1. 
;	GEASC: If set then the input files are supposed to be GEASC (or 
;		ge1/ge2) files instead of INTEG files.
;	UPDATE: If set then the reflection list will be updated with 
;		UPDATE_PRED. This updates l,l2 and tth if the rotation
;		matrix has changed after the prediction was made. Note that
;		this is normally made in REFINE_COORD if refinement is done.
;	APPEND: If set then the output is appended to the file OUTFILENAME.
;		Note that if OUTFILENAME is a scalar string and there are
;		several files the output will be appended to this one file
;		(even the first output).
;	NORM_PLOT: If set then the output format will be:
;		"(3(I4),F10.3,F8.3,2F8.4)" with "h,k,l, I, sigma, scalefactor, lambda"
;		I is LP-corrected but not normalised and where scalefactor
;		is the normalisation factor. This file can be input to
;		NORM_PLOT.PRO for a "normalisation scatter plot". Not
;		that the keywords K and B might need to be set (with
;		for example output from Rotavata/Scala already determined
;		with the normal output from this program, i.e. not with
;		the NORM_PLOT option).
;	SET_NEG_TO_ZERO : If set, the negative intensities will be set to
;		zero (actually, 0.0001). Seems to be preferable, despite
;		truncate theory !
;
; OUTPUTS:
;	A file with "0.0, h,k,l, F^2, sig(F^2), pack_no" in the format
;	"(21X,I4,4X,3(I4),8X,2(F10.3),I4)" and where pack_no starts with 1.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;
; PROCEDURE:
;	Straightforward
;
; MODIFICATION HISTORY:
;	Thomas Ursby, November 1995
;	22/7/96. Added posibility to input polynomial and to have
;	"logarithmic fit". Fixed bug with several ranges (couldn't
;	have worked before).
;
;-

IF (N_ELEMENTS(tau) eq 0) THEN tau=1.0
IF (N_ELEMENTS(pntgrp) eq 0) THEN pntgrp=1
; GEASC or INTEG?
IF (N_ELEMENTS(geasc) eq 0) THEN filetype='integ' ELSE filetype='geasc'
IF (N_ELEMENTS(film_no) eq 0) THEN film_no=0 ELSE film_no=film_no-1

; How many files?
sz= size(integfilename)
IF (sz(0) eq 0) THEN no_files=1 ELSE no_files=sz(1)
IF NOT KEYWORD_SET(frames_per_pack) THEN frames_per_pack=1
no_packs= no_files/frames_per_pack

; Append all output to one file?
outfile= outfilename
IF ((no_packs gt 1) and ((size(outfilename))(0) eq 0)) THEN BEGIN
  append=1
  outfile= STRARR(no_packs)+outfilename
ENDIF ELSE IF (N_ELEMENTS(append) eq 0) THEN append=0 

; Read the Lauenorm logfile:
IF (not KEYWORD_SET(not_read_log)) THEN $
  READ_LN, logfilename, bins, lambda, pol, ranges, polorder, lmin, lmax, $
	scale=scale, k=k0, b=b0

; Use K,B from logfile?
IF (N_ELEMENTS(k0) eq 0) THEN k0=1.0
IF (N_ELEMENTS(b0) eq 0) THEN b0=0.0
IF (N_ELEMENTS(k) eq 0) THEN k=k0
IF (N_ELEMENTS(b) eq 0) THEN b=b0
; If using one K,B set for many input files then we have to make an array:
IF (N_ELEMENTS(k) eq 1) THEN k=FLTARR(no_packs)+k
IF (N_ELEMENTS(b) eq 1) THEN b=FLTARR(no_packs)+b
IF (N_ELEMENTS(inscale) eq 0) THEN inscale=1.0
IF (N_ELEMENTS(inpolynom) ne 0) THEN pol=inpolynom
IF (N_ELEMENTS(inpolorder) ne 0) THEN polorder=inpolorder
IF (N_ELEMENTS(inranges) ne 0) THEN ranges=inranges
IF (N_ELEMENTS(inlmin) ne 0) THEN lmin=inlmin
IF (N_ELEMENTS(inlmax) ne 0) THEN lmax=inlmax
IF (N_ELEMENTS(sigrej) eq 0) THEN sigrej=-10.0^10

; Make an array of the pack index:
IF (N_ELEMENTS(pack_kb) gt 0) THEN BEGIN
  IF ((size(pack_kb))(0) eq 0) THEN pack_kb=INTARR(no_packs)+pack_kb
ENDIF ELSE pack_kb=INDGEN(no_packs)

; Check dimensions:
szk=(size(k))(1) & szb=(size(b))(1) & szp=(size(pack_kb))(1)
IF ((szk ne szb) or (szk lt szp) or (szk lt no_packs)) THEN $
	MESSAGE,'Wrong dimensions of K,B or PACK_KB'

IF (not KEYWORD_SET(norm_plot)) THEN format='(21X,I4,4X,3(I4),8X,2(F10.3),I4)' $
ELSE format='(3(I4),F10.3,F8.3,2F8.4)'

FOR i=0,no_packs-1 DO BEGIN
  PRINT, 'File: ',  integfilename(frames_per_pack*i)
  PRINT, '  K: ', k(pack_kb(i)), ' B: ', b(pack_kb(i))
  CASE (filetype) OF
    'geasc': BEGIN
	dummy=FINDFILE(integfilename(frames_per_pack*i)+ $
		'.geasc', count=count)
	IF (count eq 0) THEN SPAWN, $
	  	'source /users/a/bourgeoi/ccp4/scripts/get_geasc.scr '+ $
		integfilename(frames_per_pack*i)
	dummy=FINDFILE(integfilename(frames_per_pack*i)+ $
		'.geasc', count=count)
	IF (count eq 0) THEN $
		MESSAGE, "Couldn't find GEASC- or GE1/GE2-files."
	integ = REFLIST(integfilename(frames_per_pack*i),par)
        PRINT, '  Reflections read:', (size(integ))(1)
      END	
    'integ': BEGIN
	RESTORE, integfilename(frames_per_pack*i)
        PRINT, '  Reflections read:', (size(integ))(1)
;	integ=integ(WHERE((integ.flag eq 0) OR (integ.flag EQ 2) OR (integ.flag EQ 4)))
	integ=integ(WHERE((integ.flag eq 0) OR (integ.flag EQ 2)))
;	PRINT, '  Number of selected spots with flag 0[good], 2[bad fit] or 4[weak] : ',(SIZE(integ))(1)
	PRINT, '  Number of selected spots with flag 0[good], 2[bad fit] : ',(SIZE(integ))(1)
      END
    ELSE:
  ENDCASE
  integ.intp(0)= integ.intp(film_no) & integ.isigp(0)= integ.isigp(film_no)
  IF (N_ELEMENTS(update) ne 0) THEN integ= UPDATE_PRED(integ,par) ; DOES NOT WORK ?
  ;integ= integ(WHERE(integ.isigp(0) gt 0.0))
  integ.isigp(0) = integ.isigp(0) > 1
  PRINT, '  Number of selected spots with SIGI > 0 : ',(SIZE(integ))(1)
  integ= integ(WHERE(integ.intp(0) gt (sigrej*integ.isigp(0))))
  PRINT, '  Number of selected spots above I/SIGI cutoff : ',(SIZE(integ))(1)
  integ.isigp(0)=1/integ.isigp(0)^2
  frames=1
  FOR l=1,frames_per_pack-1 DO BEGIN
    PRINT, 'File: ',  integfilename(frames_per_pack*i+l)
    integt=integ & found=1
    CASE (filetype) OF
      'geasc': BEGIN
	  dummy=FINDFILE(integfilename(frames_per_pack*i+l)+ $
		'.geasc', count=count)
	  IF (count eq 0) THEN SPAWN, $
	  	'source /users/a/bourgeoi/ccp4/scripts/get_geasc.scr '+ $
		integfilename(frames_per_pack*i+l)
	  dummy=FINDFILE(integfilename(frames_per_pack*i+l)+ $
		'.geasc', count=count)
	  IF (count eq 1) THEN BEGIN
            integ = REFLIST(integfilename(frames_per_pack*i+l),par)
            PRINT, '  Reflections read:', (size(integ))(1)
	  ENDIF ELSE found=0
        END	
      'integ': BEGIN
	  dummy=FINDFILE(integfilename(frames_per_pack*i+l), count=count)
	  IF (count eq 1) THEN BEGIN
	    RESTORE, integfilename(frames_per_pack*i+l)
            PRINT, '  Reflections read:', (size(integ))(1)
;	    integ=integ(WHERE((integ.flag eq 0) OR (integ.flag eq 2) OR (integ.flag EQ 4)))
	    integ=integ(WHERE((integ.flag eq 0) OR (integ.flag eq 2)))
          ENDIF ELSE found=0
        END
      ELSE:
    ENDCASE
    IF (found) THEN BEGIN
      integ.intp(0)= integ.intp(film_no) & integ.isigp(0)= integ.isigp(film_no)
      IF (N_ELEMENTS(update) ne 0) THEN integ= UPDATE_PRED(integ,par)
      integ= integ(WHERE(integ.isigp(0) gt 0.0))
      integ= integ(WHERE(integ.intp(0) gt (sigrej*integ.isigp(0))))
      integ= INTERSEC(integ,integt,ct,integt)
      integ.intp(0)= integ.intp(0)+integt.intp(0)
      integ.isigp(0)= 1/integ.isigp(0)^2 + integt.isigp(0)
      frames= frames+1
    ENDIF
  ENDFOR
  integ.isigp(0)=frames/sqrt(integ.isigp(0))
  FOR l=0,ranges-1 DO BEGIN ; Loop over the different l-ranges
    ; Select singles, lambda within [lmin,lmax]:
    PRINT, '  Selecting singles within wavelength range no ' + $
	STRCOMPRESS(STRING(l+1),/REMOVE_ALL)
    integt= integ(WHERE((integ.mul eq 1) and (integ.l ge lmin(l)) and $
		(integ.l le lmax(l))))
    cos_2rho= (integt.y^2-integt.x^2)/(integt.y^2+integt.x^2)
    tth= !PI/180.0*integt.tth
    polfactor= 0.5*(1.0+cos(tth)^2-tau*cos_2rho*sin(tth)^2)

    ; the expression for polfactor takes into account the fact that in Lauegen, the top of the
    ; image is one the right side (ie the equivalent of tau = -1 for full polarization)

    lp=polfactor/(sin(tth/2)^2)

    IF KEYWORD_SET(set_neg_to_zero) THEN BEGIN
	PRINT,'Setting negative intensities to 0.0001 ...'
	integt.intp(0) = integt.intp(0) > 0.00001
    ENDIF

    IF (not KEYWORD_SET(logfit)) THEN BEGIN
      integt.intp(0)=  integt.intp(0) *POLY(integt.l,pol(l,0:polorder(l)))*k(pack_kb(i))*$
	exp(-2*b(pack_kb(i))/(2*integt.res)^2)*inscale/lp
      integt.isigp(0)= integt.isigp(0)*POLY(integt.l,pol(l,0:polorder(l)))*k(pack_kb(i))*$
	exp(-2*b(pack_kb(i))/(2*integt.res)^2)*inscale/lp
      integt.intb(0)= POLY(integt.l,pol(l,0:polorder(l)))
    ENDIF ELSE BEGIN
      integt.intp(0)=  integt.intp(0) *exp(POLY(integt.l,pol(l,0:polorder(l))))*k(pack_kb(i))*$
	exp(-2*b(pack_kb(i))/(2*integt.res)^2)*inscale/lp
      integt.isigp(0)= integt.isigp(0)*exp(POLY(integt.l,pol(l,0:polorder(l))))*k(pack_kb(i))*$
	exp(-2*b(pack_kb(i))/(2*integt.res)^2)*inscale/lp
      integt.intb(0)= exp(POLY(integt.l,pol(l,0:polorder(l))))
    ENDELSE
    mnew= integt.m & equiv_hkl, pntgrp, mnew & integt.m=mnew
    PRINT, '  Number of reflections output : ', (size(integt))(1)
    PRINT, '  Output to : ', outfile(i)
    OPENW, file, outfile(i), /GET_LUN, APPEND=append
    IF (not KEYWORD_SET(norm_plot)) THEN $
	FOR j=0,(size(integt))(1)-1 DO PRINTF, file, format=format, $
		0.0, integt(j).m, integt(j).intp(0), integt(j).isigp(0), i+1 $
    ELSE FOR j=0,(size(integt))(1)-1 DO PRINTF, file, format=format, $
		integt(j).m, integt(j).intp(0)/integt(j).intb(0), $
		integt(j).isigp(0)/integt(j).intb(0), $
		integt(j).intb(0), integ(j).l
    FREE_LUN, file
    append=1 ; If several ranges then the output should now be appended.
  ENDFOR
ENDFOR

END