;function GET_FIT,x, y, XRANGE=xrange, YRANGE=yrange, FIT_TYPE=fit_type, $
;                START_ESTIMATE=start_estimate,FIXED=fixed, END_ESTIMATE=end_estimate, $
;                 CHI_SQUARE=chi_square, N_CYCLES=n_cycles, $
;                 SIGMA=sigma, $
;                 GET_PLOT=get_plot, PLOT_RESIDUAL=plot_residual
;
;
;+
; NAME:
;	GET_FIT
;
; PURPOSE:
;
;	Returns a fitting of an experimental data-set
;
; CATEGORY:
;	Data processing
;
; CALLING SEQUENCE:
;	y_fit = GET_FIT(x, y, XRANGE=xrange, YRANGE=yrange,FIT_TYPE=fit_type, $
;                START_ESTIMATE=start_estimate,FIXED=fixed, END_ESTIMATE=end_estimate, $
;                GET_PLOT=get_plot, PLOT_RESIDUAL=plot_residual
;                  CHI_SQUARE=chi_square, N_CYCLES=n_cycles, SIGMA=sigma)
;
; INPUTS:
;	x : the x-values of the experimental data set 
;	y : the y-values of the experimental data set 
;       XRANGE : the range of x-values over which the fit is performed
;       YRANGE : the range of y-values for plotting
;       FIT_TYPE : the type of fitted function. Only some functions
;                  are programed
;       START_ESTIMATE : the starting values of the parameters
;       describing the fitting function
;       FIXED: those parameters that are fixed (array of same size as START_ESTIMATES with 1/0 values for fixed/not fixed
;       END_ESTIMATE : the ending values of the parameters
;       describing the fitting function
;       CHI_SQUARE : A chi-square quality factor for the fit.
;       N_CYCLES : the number of cycles in the fit 
;       SIGMA: standard deviations of the output vector
;       GET_PLOT : plot the experimental data together with the fitting
;       data
;       PLOT_RESIDUAL: overplots the residual.
;
; OUTPUTS:
;	yfit : the y-values of the fitting data-set
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	Uses the function CURVEFIT from the IDL library.
;
; MODIFICATION HISTORY:
;	D.Bourgeois, November 2004.
;-
;Fit a function of the form f(x) = a0 * exp(-[(x-a1)/a2]^2) +  a3 + a4*x to
;	sample pairs contained in x and y.
pro single_gauss,x,a,f,pder

 RESTORE,'variables.idl'
 local_a = start_estimate
 local_a(w_variables) = a 

 n = n_elements(local_a)
 g1x = exp(-(x-local_a(1))^2/local_a(2)^2)

 CASE n OF
  3: f= local_a(0) * g1x 
  4: f= local_a(0) * g1x + local_a(3)
  5: f= local_a(0) * g1x + local_a(3) + local_a(4)*x	 
  6: f= local_a(0) * g1x + local_a(3) + local_a(4)*x + local_a(5)*x^2	 
 ENDCASE


  PDER = FLTARR(N_ELEMENTS(X),n) ;YES, MAKE ARRAY.
  PDER[*,0] = g1x
  PDER[*,1] = [2 * local_a(0) * (x-local_a(1)) / local_a(2)^2 * g1x]
  PDER[*,2] = [2 * local_a(0) * (x-local_a(1))^2 / local_a(2)^3 * g1x]
    IF n GT 3 THEN PDER[*,3] = 1.
    IF n GT 4 then PDER[*,4] = x
    IF n GT 5 then PDER[*,5] = x^2

;only select the running variables
 pder = pder(*,w_variables)
 RETURN
END

;Fit a function of the form f(x) = a0 * exp(-[(x-a1)/a2]^2) + a3 * exp(-[(x-a4)/a5]^2) + a6 + a7*x to
;	sample pairs contained in x and y.
pro double_gauss,x,a,f,pder

 RESTORE,'variables.idl'
 local_a = start_estimate
 local_a(w_variables) = a 
 n = n_elements(local_a)
 g1x = exp(-(x-local_a(1))^2/local_a(2)^2)
 g2x = exp(-(x-local_a(4))^2/local_a(5)^2)

 CASE n OF
  6: f= local_a(0) * g1x + local_a(3) * g2x 
  7: f= local_a(0) * g1x + local_a(3) * g2x + local_a(6)
  8: f= local_a(0) * g1x + local_a(3) * g2x + local_a(6) + local_a(7)*x	 
  9: f= local_a(0) * g1x + local_a(3) * g2x + local_a(6) + local_a(7)*x	 + local_a(8)*x^2
ENDCASE

  PDER = FLTARR(N_ELEMENTS(X),n) ;YES, MAKE ARRAY.
  PDER[*,0] = g1x
  PDER[*,1] = [2 * local_a(0) * (x-local_a(1)) / local_a(2)^2 * g1x]
  PDER[*,2] = [2 * local_a(0) * (x-local_a(1))^2 / local_a(2)^3 * g1x]
  PDER[*,3] = g2x
  PDER[*,4] = [2 * local_a(3) * (x-local_a(4)) / local_a(5)^2 * g2x]
  PDER[*,5] = [2 * local_a(3) * (x-local_a(4))^2 / local_a(5)^3 * g2x]
    IF n GT 6 THEN PDER[*,6] = 1.
    IF n GT 7 then PDER[*,7] = x
    IF n GT 8 then PDER[*,7] = x^2

;only select the running variables
 pder = pder(*,w_variables)
 RETURN

END


;Fit a function of the form f(x) = a * exp(b*x) + c * exp(d*x) +e to
;	sample pairs contained in x and y.
pro double_exp,x,a,f,pder
 RESTORE,'variables.idl'
 local_a = start_estimate
 local_a(w_variables) = a 

 bx = exp(local_a(1) * x)
 dx = exp(local_a(3) * x)
 f= local_a(0)*bx + local_a(2)*dx +local_a(4)		;Evaluate the function

 pder= [[bx], [local_a(0) * x * bx], [dx], [local_a(2) * x * dx], [replicate(1.0, N_ELEMENTS(x))]]

;only select the running variables
 pder = pder(*,w_variables)

end

;Fit a function of the form f(x) = a * [exp(b*x) -  exp(c*x)] + d to
;	sample pairs contained in x and y.
pro inter_exp,x,a,f,pder
 RESTORE,'variables.idl'
 local_a = start_estimate
 local_a(w_variables) = a 

 bx = exp(local_a(1) * x)
 cx = exp(local_a(2) * x)
 f= local_a(0)*(bx - cx) + local_a(3)		;Evaluate the function
 pder= [[bx-cx], [local_a(0) * x * bx], [-local_a(0) * x * cx],  [replicate(1.0, N_ELEMENTS(x))]]

;only select the running variables
 pder = pder(*,w_variables)

end


function GET_FIT,x, y, XRANGE=xrange, YRANGE=yrange, FIT_TYPE=fit_type, $
                START_ESTIMATE=start_estimate,FIXED=fixed,END_ESTIMATE=end_estimate, $
                 CHI_SQUARE=chi_square, N_CYCLES=n_cycles, SIGMA=sigma, $
                 GET_PLOT=get_plot, PLOT_RESIDUAL=plot_residual, GUESS_ESTIMATES=guess_estimates

 CASE FIT_TYPE OF
  'Single_Gauss' : BEGIN
      IF N_ELEMENTS(start_estimate) LT 3 THEN MESSAGE,'Number of initial parameters is wrong !'

	    END
  'Double_Gauss' : BEGIN
      IF N_ELEMENTS(start_estimate) LT 6 THEN MESSAGE,'Number of initial parameters is wrong !'
	   END
  'inter_exp' : BEGIN
      IF N_ELEMENTS(start_estimate) NE 4 THEN MESSAGE,'Number of initial parameters is wrong !'
	   END
  'double_exp' : BEGIN
      IF N_ELEMENTS(start_estimate) NE 5 THEN MESSAGE,'Number of initial parameters is wrong !'
	   END
   ELSE : MESSAGE,'Fitting function Unprogramed  !'
 ENDCASE

w1 = MIN(WHERE(x GE xrange(0)))
w2 = MAX(WHERE(x LE xrange(1)))
x2=x(w1:w2)
y2=y(w1:w2)
;w=1.0/y			;Weights
w=replicate(1.0, N_ELEMENTS(y2))
n=N_ELEMENTS(y2)

nt=N_ELEMENTS(start_estimate)
IF N_ELEMENTS(fixed) EQ 0 THEN fixed = INTARR(nt)

IF guess_estimates THEN BEGIN ; only works for Single_Gauss (taken from routine Gaussfit.pro)
         dbl = SIZE(y2, /TNAME) eq 'DOUBLE'
         if (nt gt 3) then begin
            ; For a Gaussian + polynomial, we need to subtract off either
            ; a constant or a straight line to get good estimates.
            ; NOTE: Because a Gaussian and a quadratic can be highly correlated,
            ; we do not want to subtract off the quadratic term.
            c = POLY_FIT(x2, y2, (nt eq 4) ? 0 : 1, yf)
            yd = y2 - yf
        endif else begin
            ; Just fitting a Gaussian. Don't need to subtract off anything.
            yd = y2
            c = 0d
        endelse

        if not dbl then begin    ;If Y is not double, use float
            yd = float(yd)
            c = float(c)
        endif

        ;x2,y2 and subscript of extrema
        ymax=max(yd, imax)
        xmax=x2[imax]
        ymin=min(yd, imin)
        xmin=x2[imin]
        if abs(ymax) gt abs(ymin) then i0=imax else i0=imin ;emiss or absorp?
        i0 = i0 > 1 < (n-2)     ;never take edges
        dy=yd[i0]           ;diff between extreme and mean
        del = dy/exp(1.)        ;1/e value
        i=0
        while ((i0+i+1) lt n) and $ ;guess at 1/2 width.
        ((i0-i) gt 0) and $
        (abs(yd[i0+i]) gt abs(del)) and $
        (abs(yd[i0-i]) gt abs(del)) do i=i+1
        a = [yd[i0], x2[i0], abs(x2[i0]-x2[i0+i])]
        if nt gt 3 then a = [a, c[0]] ; estimate for constant term
        if nt gt 4 then a = [a, c[1]] ; estimate for linear term
        if nt gt 5 then a = [a, 0.]   ; assume zero for quadratic estimate
        start_estimate = FLOAT(a) 
   ENDIF ELSE BEGIN
        start_estimate = FLOAT(start_estimate)
    ENDELSE

w_variables = WHERE(fixed EQ 0, n_variables)
w_fixed = WHERE(fixed EQ 1, n_fixed)
IF n_variables EQ 0 THEN MESSAGE, 'All parameters are fixed: cannot fit !'
estimate_used = start_estimate(w_variables)
SAVE,start_estimate, w_variables, w_fixed, file='variables.idl'

;this procedure is to guess the starting estimates; see gaussfit.pro

; ymax=max(y2, imax)
; xmax=x2[imax]
 ;ymin=min(y2, imin)
 ;xmin=x2[imin]
 ;if abs(ymax) gt abs(ymin) then i0=imax else i0=imin ;emiss or absorp?
 ;i0 = i0 > 1 < (n-2)     ;never take edges
 ;dy=y2[i0]           ;diff between extreme and mean
 ;del = dy/exp(1.)        ;1/e value
 ;i=0
 ;while ((i0+i+1) lt n) and $ ;guess at 1/2 width.
 ;((i0-i) gt 0) and $
 ;(abs(y2[i0+i]) gt abs(del)) and $
 ;(abs(y2[i0-i]) gt abs(del)) do i=i+1
 ; f_estimate = [y2[i0], x2[i0], abs(x2[i0]-x2[i0+i])]
;ENDIF ELSE f_estimate = FLOAT(start_estimate)

yfit=CURVEFIT(x2,y2,w,estimate_used,sigma,function_name=FIT_TYPE,CHI2=chi_square, ITER=n_cycles)

IF chi_square LT 10000.0 THEN BEGIN
 end_estimate = start_estimate
 end_estimate(w_variables) = estimate_used

ENDIF ELSE PRINT,'Chi_square value is too high; aborting fit ! '

PRINT,'Starting Function parameters: ',start_estimate
PRINT, 'New Function parameters: ', end_estimate
PRINT,'Standard deviations: ',sigma
PRINT,'**************************************'
PRINT,'Chi square value: ',chi_square
PRINT,'**************************************'
PRINT,'Number of iterations performed: ', n_cycles
PRINT,'**************************************'

IF KEYWORD_SET(get_plot) THEN BEGIN
 PLOT,x2,y2, YRANGE=yrange, XRANGE=xrange, CHARSIZE=1.5, THICK=2.0, XTITLE = 'Wavelength [nm]', YTITLE='Data [AU]'
 OPLOT,x2,yfit
 IF KEYWORD_SET(plot_residual) THEN BEGIN
  OPLOT, x2, yfit-y2, LINESTYLE=1
 ENDIF
ENDIF

y_fit = FLTARR(N_ELEMENTS(y))
y_fit(w1:w2)=yfit
RETURN, y_fit

END
