function D_CALC, m, par=par
;+
; NAME:
;	D_CALC
;
; PURPOSE:
;	Calculate the d-spacing given the crystal parameters and 
;	the Miller indices.
;
; CATEGORY:
;	Crystallography
;
; CALLING SEQUENCE:
; 	d_spacing = D_CALC(m [, par=par])
;
; INPUTS:
;	M: The Miller indices (array with three elements) or array
;		with dimensions (3,n) where n is the number of Miller
;		indices. In the latter case an array (dimensions 1,n) of 
;		d-spacing is returned.
;
; KEYWORDED PARAMETERS:
;	PAR: The cell parameters. Structure containing the tags
;		a,b,c,alpha,beta,gamma. Angles in degrees.
;		Defaults to cubic with a=50.0
;
; OUTPUTS:
;	The d-spacing.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	Uses the formula for triclinic crystal system.
;
; MODIFICATION HISTORY:
;	Thomas Ursby, 1995
;
;-

ON_ERROR,2  

IF ((size(m))(0) eq 1) THEN BEGIN
  h=m(0) & k=m(1) & l=m(2) 
ENDIF ELSE BEGIN
  h=INTARR((size(m))(2))+m(0,*) 
  k=INTARR((size(m))(2))+m(1,*) 
  l=INTARR((size(m))(2))+m(2,*)
ENDELSE

IF NOT KEYWORD_SET(par) THEN BEGIN
  a=50.0
  b=50.0
  c=50.0
  alpha=!pi/2
  beta =!pi/2
  gamma=!pi/2
ENDIF ELSE BEGIN
  a=par.a & b=par.b & c=par.c
  alpha= !pi/180.0*par.alpha
  beta = !pi/180.0*par.beta
  gamma= !pi/180.0*par.gamma
ENDELSE

f1 = 1 - cos(alpha)^2 - cos(beta)^2 - cos(gamma)^2 + 2*cos(alpha)*cos(beta)*cos(gamma)
t1 = (h/a*sin(alpha))^2 + (k/b*sin(beta))^2 + (l/c*sin(gamma))^2
t2 = 2*k*l/(b*c)*(cos(beta)*cos(gamma)-cos(alpha))
t3 = 2*l*h/(c*a)*(cos(gamma)*cos(alpha)-cos(beta))
t4 = 2*h*k/(a*b)*(cos(alpha)*cos(beta)-cos(gamma))

RETURN, sqrt(f1/(t1+t2+t3+t4))

END