; $Id: search2d.pro,v 1.7 1998/07/15 18:59:02 scottm Exp $

; Copyright (c) 1992-1998, Research Systems, Inc. All rights reserved.
;	Unauthorized reproduction prohibited.
;
;+
; NAME:
;	SEARCH2D
;
; PURPOSE:
;	This function finds "objects" or regions of similar data
;       values within a 2-D array of data. Given a starting location
;       and a range of values to search for, SEARCH2D will find all
;       the cells within the array that are within the specified range
;       of values, and have some path of connectivity through these cells
;       to the starting location. In addition to searching for cells
;       within a global range of data values, SEARCH2D can also search
;       for adjacent cells whose values deviate from their neighbors within
;       specified tolerances. See the procedure "SEARCH3D" for the
;       three dimensional case.
;
;       This function returns a list of the array subscripts that define
;       the selected object or region.
;
; CATEGORY:
;       Data subsetting.
;	Image manipulation.
;
; CALLING SEQUENCE:
;       Region = SEARCH2D(Array, Xpos, Ypos, Min_val, Max_val)
;
; INPUTS:
;       Array:      The 2-D array of data to search.
;                   Data type : Any 2-D array except string or structure.
;       Xpos:       The X coordinate (first subscript into the 2-D Array)
;                   of the starting cell for the search.
;                   Data type : Long.
;       Ypos:       The Y coordinate (second subscript into the 2-D Array)
;                   of the starting cell for the search.
;                   Data type : Long.
;       Min_val:    The minimum data value to search for. All cells that
;                   are connected to the starting cell, and have a value
;                   greater than or equal to Min_val and less that or equal
;                   to Max_val, will be considered part of the "object".
;       Max_val:    The maximum data value to search for.
;
; KEYWORD PARAMETERS:
;       DECREASE:   If the DECREASE or INCREASE keywords are specified,
;                   then SEARCH2D creates an internal copy of Array.
;                   This internal copy is then processed to enhance the
;                   object edges by using an algorithm similar to the
;                   "SOBEL" edge enhancement process. Any adjacent
;                   cells will be found if their corresponding data value
;                   in the edge enhanced array is greater than DECREASE and
;                   less than INCREASE. In any case, the adjacent cells
;                   will NEVER be selected if their data value is not
;                   between Min_val and Max_val.
;                   The default is 0.0 if INCREASE is specified.
;                   Otherwise, the default is no edge checking.
;                   Data type : Int or Float (usually less than zero).
;       INCREASE:   The maximum value in the edge enhanced array for
;                   a cell to be considered part of the selected object.
;                   Some savings in execution time and memory usage result
;                   when DECREASE and INCREASE are omitted.
;                   See DECREASE above.
;                   The default is 0.0 if DECREASE is specified.
;                   Otherwise, the default is no edge checking.
;                   Data type : Int or Float (usually greater than zero).
;       LPF_BAND:   This keyword indicates what (if any) Low Pass Filtering
;                   is performed on the edge enhanced array before the
;                   search begins. If LPF_BAND is set to 3 or higher
;                   then the edge enhanced array will be smoothed using
;                   LPF_BAND as the width of the smoothing window.
;                   If LPF_BAND is less than 3 then no smoothing is
;                   performed. This keyword only has effect when the
;                   DECREASE or INCREASE keywords are supplied.
;                   See DECREASE above.
;                   The default is zero (no smoothing).
;                   Data type : Int.
;       DIAGONAL:   Normally, cells are considered adjacent only when
;                   squares surrounding the cells share a common edge.
;                   If a non-zero value is passed to DIAGONAL then
;                   SEARCH2D will also locate cells meeting the search
;                   criteria whose surrounding squares share a common
;                   corner. Specifying diagonal search mode requires
;                   more memory and execution time.
;                   The default is no diagonal searching.
;                   Data type : int
;
; OUTPUTS:
;       This function returns a list of the indices into the 2-D array
;       that are part of the located object or region. This list is
;       returned as a LONARR(n) where n is the number of cells found.
;
;       If the returned array of indices is called Region, and the
;       size of the 2-D array of data is size_x by size_y, then the
;       actual X and Y indices can be obtained by using the following
;       algorithm :
;
;          index_y = Region / size_x
;          index_x = Region - (index_y * size_x)
;
;       The object within the 2-D Array could then be subscripted as :
;
;          Array(Region)
;       OR
;          Array(index_x, index_y)
;
; EXAMPLE:
;       Find all the indices corresponding to an object contained in a
;       2-D array of data.
;
;       ; Create some data.
;          img = FLTARR(512, 512)
;          img(3:503, 9:488) = 0.7
;          img(37:455, 18:438) = 0.5
;          img(144:388, 90:400) = 0.7
;          img(200:301, 1:255) = 1.0
;          img(155:193, 333:387) = 0.3
;
;       ; Display the image.
;          TVSCL, img
;
;       ; Search for an object starting at (175, 300) whose data values
;       ; are between (0.6) and (0.8).
;          Region = SEARCH2D(img, 175, 300, 0.6, 0.8, /DIAGONAL)
;
;       ; Scale the background cells into the range 0 to 127.
;          img = BYTSCL(img, TOP=127B)
;
;       ; Highlight the object region by setting it to 255.
;          img(Region) = 255B
;
;       ; Display the array with the highlighted object in it.
;          TVSCL, img
;
; MODIFICATION HISTORY:
;       Written by:     Daniel Carr. Thu Sep  3 15:36:17 MDT 1992
;-

FUNCTION search_pass, indices=indices, array, tolerance=tolerance

sx=(SIZE(array))(1) & sy=(SIZE(array))(2)
IF N_ELEMENTS(indices) EQ 0 THEN indices=INDGEN(sx*sy)
IF N_ELEMENTS(tolerance) EQ 0 THEN tolerance=0
s=(SIZE(indices))(1)
IF s LE 1 THEN RETURN,0

pass_indices=indices & pass_indices(*)=0


FOR i=0,s-1 DO BEGIN
 xy=WHERESUB(indices(i),array)
; x=xy(0) & y=xy(1) & val=array(x,y)*(1-tolerance/100.)
 x=xy(0) & y=xy(1) & val=array(x,y)*(1+tolerance/100.)
 x1=(x-1)>0 & y1=(y-1)>0
 x2=(x+1)<(sx-1) & y2=(y+1)<(sy-1)
 
 IF ((array(x1,y) GT val) AND (array(x2,y) GT val)) OR $
    ((array(x,y1) GT val) AND (array(x,y2) GT val)) OR $
    ((array(x1,y1) GT val) AND (array(x2,y2) GT val)) OR $
    ((array(x1,y2) GT val) AND (array(x2,y1) GT val)) THEN pass_indices(i)=indices(i)


ENDFOR

w=WHERE(pass_indices GT 0,ct_w)
IF ct_w GT 0 THEN pass_indices=pass_indices(w) ELSE pass_indices=0
RETURN,pass_indices
END




