FUNCTION GET_ADAPTED_PROFILE,x,y,xr_fl,yr_fl,ref_profile,ref_distance,ref_ang_width,at=at,sq=sq,sym_rad=sym_rad,show=show,area=area
;+
; NAME:
;	GET_ADAPTED_PROFILE
;
; PURPOSE:
;	Builds a reference profile by weighted averaging, from a list of available profiles.
;
; CATEGORY:
;	Data processing.
;
; CALLING SEQUENCE:
;	adapted_profile = GET_ADAPTED_PROFILE( $
;				x,y,$
;				xr_fl,yr_fl,$
;				ref_profile,$
;				ref_distance,ref_ang_width,$
;				at=at,sq=sq,$
;				sym_rad=sym_rad,$
;				area=area, /SHOW)
;
;
; INPUTS:
;	x,y : the coordinates of the spot to be profile fitted [mm]
;	xr_fl,yr_fl : the coordinates of the spot to be profile fitted [raster units]
;	ref_profile : list of structure containing the learned profiles, 
;	and associated parameters
;	ref_distance : the maximum radial distance to look for a reference peak
;	   	from the position of the currently integrated peak [mm]
;	ref_ang_width : the maximum tengential distance to look for a 
;		reference peak from the position of the currently integrated 
;		peak [degrees]
;	at : the angular position of the reference profiles
;	sq : the radial position of the reference profiles
;	sym_rad : set this keyword if the reference peak search is to be extented
;		to symmetrical sector if necessary
;	area : use this with the SHOW keyword to see the effect of interpolation
;	show: set this keyword to see what's happening.
;
; OUTPUTS:
;	
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	The reference profile is built from a set of selected profiles which fullfill the
;	following conditions :
;	--> if there are at least two profiles in the sector around the spot to profile fitted, then
;		keep only these two
;	--> if there are less than two, and the SYM_RAD keyword is set, then look
;		over the full radial, twice tangential sector AND look also in the
;		symetrical sector.
;	--> if there less than two profiles and SYM_RAD keyword is not set, then look
;		over the full radial, twice tangential sector.
;	--> if still nothing, exit with an error message.
;
; MODIFICATION HISTORY:
;	Dominique Bourgeois, August 97.
;-

;********************
;select a first set of possible reference profiles :
;********************
radius_i = SQRT(x^2+y^2)
angle_i =  ATAN(y,x)*180/!Pi
r_max = radius_i + ref_distance/2.
r_min = (radius_i - ref_distance/2.)>0
phi_max = angle_i + ref_ang_width/2.
IF phi_max GT 180 THEN phi_max = phi_max - 360
phi_min = angle_i - ref_ang_width/2.
IF phi_min LT -180 THEN phi_min = phi_min + 360

IF N_ELEMENTS(at) EQ 0 THEN at = ATAN(ref_profile.y,ref_profile.x)*180/!Pi
IF N_ELEMENTS(sq) EQ 0 THEN sq = SQRT(ref_profile.x^2 + ref_profile.y^2)

IF phi_min LE phi_max THEN $
 ref_index = WHERE( at LE phi_max AND $
		at GE phi_min AND $
	 	sq LE r_max AND $
		sq GT r_min,n_ref) $
ELSE $
 ref_index = WHERE( at LE phi_max OR $
		at GE phi_min AND $
		sq LE r_max AND $
		sq GT r_min,n_ref)


;********************
;handle cases where insufficient number of reference peaks.
;********************

IF n_ref LE 1 THEN BEGIN

 IF NOT KEYWORD_SET(sym_rad) THEN BEGIN
 ;********************
 ;do not include symmetrical sector
 ;********************

  ;take twice the angular width 
  phi_max = angle_i + ref_ang_width
  IF phi_max GT 180 THEN phi_max = phi_max - 360
  phi_min = angle_i - ref_ang_width
  IF phi_min LT -180 THEN phi_min = phi_min + 360

  ;do not consider radial limits
  IF phi_min LE phi_max THEN $
   ref_index = WHERE( at LE phi_max AND $
		at GE phi_min,n_ref) $
  ELSE $
   ref_index = WHERE( at LE phi_max OR $
		at GE phi_min,n_ref)

  IF n_ref GT 0 THEN BEGIN
   IF KEYWORD_SET(show) THEN PRINT,'No close reference peaks ! Ref peaks taken from full radial sector !'
  ENDIF ELSE MESSAGE,'Choose larger area for reference peak search !'
 ENDIF ELSE BEGIN

 ;********************
 ;include symmetrical sector
 ;********************

  ;take twice the angular width 
  IF angle_i GE 0 THEN angle_i2 = angle_i - 180 ELSE angle_i2 = angle_i + 180 
  phi_max = angle_i + ref_ang_width
  phi_max2 = angle_i2 + ref_ang_width
  IF phi_max GT 180 THEN phi_max = phi_max - 360
  IF phi_max2 GT 180 THEN phi_max2 = phi_max2 - 360
  phi_min = angle_i - ref_ang_width
  phi_min2 = angle_i2 - ref_ang_width
  IF phi_min LT -180 THEN phi_min = phi_min + 360
  IF phi_min2 LT -180 THEN phi_min2 = phi_min2 + 360

  ;do not consider radial limits
  IF ((phi_min LE phi_max) AND (phi_min2 LE phi_max2)) THEN $
   ref_index = WHERE((at LE phi_max AND $
		   at GE phi_min) OR $
		  (at LE phi_max2 AND $
                   at GE phi_min2),n_ref)

  IF ((phi_min LE phi_max) AND (phi_min2 GT phi_max2)) THEN $
   ref_index = WHERE((at LE phi_max AND $
		   at GE phi_min) OR $
		  (at LE phi_max2 OR $
                   at GE phi_min2),n_ref)

  IF ((phi_min GT phi_max) AND (phi_min2 LE phi_max2)) THEN $
   ref_index = WHERE((at LE phi_max OR $
		   at GE phi_min) OR $
		  (at LE phi_max2 AND $
                   at GE phi_min2),n_ref)

  IF ((phi_min GT phi_max) AND (phi_min2 GT phi_max2)) THEN $
   ref_index = WHERE((at LE phi_max OR $
		   at GE phi_min) OR $
		  (at LE phi_max2 OR $
                   at GE phi_min2),n_ref)


  IF n_ref GT 0 THEN BEGIN
   IF KEYWORD_SET(show_weak) THEN PRINT,'No close reference peaks ! Ref peaks taken from full radial sector ! (including symmetrical sector)'
  ENDIF ELSE MESSAGE,'Choose larger area for reference peak search !'
 ENDELSE
;********************
ENDIF

possible_ref = ref_profile(ref_index)

IF MIN((possible_ref.x-x)^2 + (possible_ref.y-y)^2) EQ 0 THEN BEGIN
 ;We are in this case in the presence of one of the reference spots !
 ;If spot list does not contain any of the reference spots, this should never
 ;happen !
 MESSAGE,'Trying to auto-fit a spot. This does not make sense !'
ENDIF ELSE BEGIN
	adapted_profile =(possible_ref(0).sp)*1/SQRT((possible_ref(0).x-x)^2 +(possible_ref(0).y-y)^2)
	 IF n_ref GT 1 THEN FOR index=1,n_ref-1 DO $
	  adapted_profile = adapted_profile + (possible_ref(index).sp)*1/SQRT((possible_ref(index).x-x)^2 + (possible_ref(index).y-y)^2)
ENDELSE

;***
;INTERPOLATE THE PROFILE
;***

IF KEYWORD_SET(show) AND N_ELEMENTS(area) NE 0 THEN BEGIN
 adapted_profile2 = adapted_profile/TOTAL(adapted_profile)
 PRINT,'REF_PROFILE + SPOT (Before interpolation) : '
 temp = adapted_profile2
 CONTOUR,temp,LEVELS=[(INDGEN(10)+1)*MAX(temp)/10.],C_COLORS=!D.N_COLORS - 150
 CONTOUR,area,LEVELS=[(INDGEN(10)+1)*MAX(area)/10.],/NOERASE,C_COLORS=!D.N_COLORS - 60

 OK=''
 READ,'OK ? [Y/N]',OK
ENDIF

 adapted_profile = GET_INTERP_PROFILE(adapted_profile,xr_fl,yr_fl)

 ;***
 ;normalize the integrated intensity of the reference profile to 1. 
 ;***
 adapted_profile = adapted_profile/TOTAL(adapted_profile)

IF KEYWORD_SET(show) THEN BEGIN
 IF N_ELEMENTS(area) NE 0 THEN BEGIN
  PRINT,'REF_PROFILE + SPOT (After interpolation) : '
  temp = adapted_profile
  CONTOUR,temp,LEVELS=[(INDGEN(10)+1)*MAX(temp)/10.],C_COLORS=!D.N_COLORS - 150
  CONTOUR,area,LEVELS=[(INDGEN(10)+1)*MAX(area)/10.],/NOERASE,C_COLORS=!D.N_COLORS - 60

  OK=''
  READ,'OK ? [Y/N]',OK
 ENDIF
 ;***
 ;PRINT INFO
 ;***
 PRINT,'Number of reference peaks : ',(SIZE(possible_ref))(1)
 PRINT,'Spot position [X,Y], [mm]:',x,y
 PRINT,'X values [mm]:',possible_ref.x
 PRINT,'Y values [mm]:',possible_ref.y
 PRINT,'Position of maximum [x,y] :',WHERESUB(WHERE(adapted_profile EQ MAX(adapted_profile)),adapted_profile)

ENDIF

RETURN,adapted_profile

END





