;+ 
;
;
;      ============================= XPLOT  ======================
;
; 
; XPLOT is a general package for creating X-Y plots that also allows for
; sophisticated data visualization, processing, manipulation and analysis.
; Some of the XPLOT features are: 
;
;  1) Direct loading of data from multicolumn ASCII files. Multiple sets of
; data can be accessed from several files and merged together. Interactive
; action on data columns with the creation of new data from scratch or by
; manipulating existing data 
;  2) Access to SPEC data files 
;  3) Direct printing from Unix workstations and PCs, and creation of
; PostScript and encapsulated PostScript files 
;  4) Linear and logarithmic X- and Y-axes. Apply and change symbols, line
; styles, colors, thickness, fonts etc. 
;  5) Label, annotate, add legends, titles, etc. to the plots. Zoom, shift,
; switch columns, and over-plot data. 
;  6) Save and restore plots and attributes in a XPLOT backup file. 
;  7) Data analysis: extrema, moments, integrals, derivatives, CDFs,
; smoothing, user-defined operations, interactive Fourier filtering,
; interpolations, convolutions and correlations. It also includes several
; fitting routines: linear regression, polynomial fit, Gaussian fit and
; nonlinear fit (the gradient-expansion algorithm or the Powell method) with an
; on-line user-defined function. 
;
;
; XPLOT also includes some packages for specific data analysis. At present
; the XAID package for XAFS (X-ray Absorption Fine Structure)
; analysis is available. New applications and interfaces for data
; processing applied to different synchrotron techniques are under
; development, and they will be integrated in future releases.
;
;
; On line help is available in most of the cases. A Web Xplot page and
; documentation can be found at:
; http://www.esrf.fr/computing/scientific/xop/xplot
;
;
; Overplotting with Xplot
; =======================
;
; One of the most important features of Xplot is the capability to display
; multi-line graphs with data coming fro different origins. Data in Xplot
; is structured in the so-called sets. 
; 
; A set is an array or matrix containing N points (or rows) and M columns.
; 
; Xplot stores the data in multiple sets, and each set may hold a
; different number of points or columns. Xplot allows multi-line plots, in
; which the data for each line is a couple of columns from a given set.
; Xplot options Calculations and File/Load Data File... allows to create
; new columns in defined sets and new sets from files or from operations
; between existing sets. 
; 
; You can change the X and Y columns for the plot in the current set of
; data by just typing the column numbers in the bottom of the Xplot
; window. It is very important to enter <Enter> after modifying any text
; widget in Xplot. That permits Xplot to understand you have finished with
; the editing, and process the event.
;
; If you want to do a plot with multiple lines, first load the data set
; with the first line. Then use the "Save Plot" button to store this set
; as a "saved set". At this moment, the saved set is copied to the
; current set, and both will be displayed. Then, you can change columns
; of even load a new data set using File/Load menu and the new data set
; will be overplotted on the saved one. Repeat this process as many times
; as you want. Once a set is saved, the Xplot controls do do affect it.
; The window created by pressing the "Plot Mgr" (Plot manager) button
; permits to delete save sets, or to promote a saved set to the current
; set.
;
;
; Xplot has been written by:
;	Manuel Sanchez del Rio
;	European Synchrotron Radiation Facility
;	BP 200
;	38043 Grenoble-Cedex 9
;	Email: srio@esrf.fr
;
;-

;+
;      ================= USE OF XPLOT FROM THE IDL PROMPT  ==========
;
;
;
;	NAME:
;		XPLOT
;	PURPOSE:
;		to plot and interactive manipulate data
;	CALLING SEQUENCE:
;		XPLOT [,indata] [,keywords]
;	OPTIONAL INPUT PARAMETERS:
;	indata can be one of the following:
;		1) filename: a string containing the file name to be
;		plotted or an idl variable [fltarr(n_points,n_columns]
;		with data. The file must be ascii data arranged in 
;		columns. Lines starting with characters [except . + -]
;		are comments and not considered for the plot.
;		2) an idl variable containing the numerical data
;		set (a fltarr(n_columns,n_points)
;		3) x,y two arrays with the  abscissas and ordinate
;		arrays.
;		4) h an IDL handler pointing to a set or to a number
;		of sets.
;		
;	KEYWORD PARAMETERS:
;		GROUP = The widget ID of the widget that calls Xplot.
;		When this ID is specified, a death of the caller results 
;		in a death of Xplot.
;		WINDOW_SIZE = a 2-dim vector containing the width and
;		height of the graphical window (in pixels).
;		XCOL = for multicolumn entries, the column number to be
;		plotted in the X axis (default=1)
;		YCOL = for multicolumn entries, the column number to be
;		plotted in the Y axis (default=last column)
;		WTITLE = title for the main window
;		TITLE = title for the plot
;		XTITLE = abscissas title. In the case that xtitle is
;			set as XTITLE='-1', then xplot will place
;			the text of COLTITLES corresponding to the 
;			plotted column.
;		YTITLE = ordinates title. In the case that ytitle is
;                       set as YTITLE='-1', then xplot will place
;                       the text of COLTITLES corresponding to the
;                       plotted column.
;		COLTITLES = string array with column labels.
;		LEGEND = A legend call to the LEGEND procedure by 
;			 F K Knight (knight@ll.mit.edu)
;		STATE = a xplot state structure (only for internal 
;			use of Xplot (when creating clones))
;		PARENT = The wiggler id of the main Xplot window 
;			created. This is useful to pass this id 
;			as parent of other widgets that must die when
;			the present XPlot window dies.
;		[XY]LOG = when set to 1, initializes [XY] logaritmic axis.
;		SPEC = input, initialize xplot with a SPEC file. Input
;			may be a file name (string) or a spec handler (str).
;			Example, IDL> xplor,spec='file.dat'
;               XRANGE=[xmin,xmax] range of the x variable for the plot
;               YRANGE=[ymin,ymax]              y
;
;
;	OUTPUTS:
;		Open a widget utility and present a graphic.
;	COMMON BLOCKS:
;		
;	SIDE EFFECTS:
;		If not active, starts Xmanager
;	RESTRICTIONS:
;		Unknown.
;	PROCEDURE:
;		Uses the PLOTFILE routine for producing the plot
;	KNOWN BUGS:
;
;		When starting xplot with coltitles keyword, the "clone"
;		  option does not work.
;		[xy]title='-1' does not work properly with spec/mca data.
;	
;		Using Edit/Data and removing data causes xplot to
;		crash when SPEC data is used or "-1" is used in titles.
;		The automatic selection in Edit/Data does not work for
;		data set in which the abscissas are reversed.
;
;		The Save/Restore backup option is not very robust. It may 
;		give problems using complicated plots. It does not work with 
;		SPEC files. It may present incompatibilities between different
;		xplot versions. 
;
;	MODIFICATION HISTORY:
;       by  Manuel Sanchez del Rio. ESRF. December 1993
;		94-01-21 MSR improves the evaluate option
;		94-02-02 MSR add style options in LIMITS menu
;		94-02-18 MSR removes common blocks and allow
;			multiple copies of the widget
;		94-05-19 MSR fix a bug in the print buffer managment
;			found by R. Dejus.
;		94-06-27 MSR adds the /Calculations/correlation function 
;			option.
;		94-11-17 MSR adds Tools/Xplot and Tools/IDL_command options
;		95-02-11 MSR adds Calculations/Fitting option, check 
;			existence of input files.
;		95-03-08 MSR: updating of fourierfilter: we pass now also 
;			the abscissas values.
;		95-08-08 MSR: adds LEGEND keyword.
;
;		95-09-14 MSR Releases Xplot version 2.0
;		Nov 1995  version 2.1 major upgrades:
;			- Integration and interfacing of spec_acces
;			- COLTITLES keyword
;			- Options Polynomial and Gaussian fit integrated
;			in a single one.
;			- new interface of non-linear fit.
;		96-01-18 srio@esrf.fr changes xtext call by xedit from RJD.
;		96-01-19 srio@esrf.fr adds options [xy]title='-1'
;		96-01-22 srio@esrf.fr adds the possibility to edit column
;			titles for SPEC files, and number the columns.
;		96-04-12 srio@esrf.fr adds /xlog and /ylog kws. Change 
;			fieldlen in settitles.
;		96-05-29 srio@esrf.fr xplot/title: when using spec files, 
;			the symbol #x (i.e. #S) is interpreted as its 
;			corresponding value in the spec scan header.
;		96-07-09 srio@esrf.fr Adds SPEC keyword.
;		96-08-13 srio@esrf.fr changes "'" by '"' in xplot_plot
;		96-08-20 srio@esrf.fr makes the main window resizable,
;			and removes the ZoomSel button (now automatic selection
;			of zooming area by dragging the mouse with the left 
;			button down). Do not write intermediate files
;			for view/showData. Introduce xhelp.
;		96-12-04 srio@esrf.fr add "catch" in xplot_event to
;			avoid crassing the application when there is an error.
;		97-01-15 srio@esrf.fr complete the list of spec functions in
;			xplot_compile().
;		97-02-17 srio@esrf.fr debugging: log->lin change when saving, 
;			dynamic_resize of column widgets, correct cursor 
;			read-out when switching between Xplot windows. Add 
;			path to pickfile and uses xhelp for help.  Version 2.2
;		97-06-27 srio@esrf.fr added View/Mesh option. version 2.2.
;			Use copy_structure to restore backup file (assures 
;			compatibility between save files from different xplot
;			versions).
;		97-09-07 srio@esrf.fr added non-interactive use of xplot. 
;			Version 2.4
;		97-09-30 srio@esrf.fr substitutes read_ascii to rascii
;		97-10-27 srio@esrf.fr version 2.5, with lots of modifications:
;			New zoom rubberband code. Re-load file. Apply
;			button in many windows. Changed many calculations 
;			windows (smooth, convolution, etc), non-linaer fit 
;			with fixed parameters, new Plot Manager button 
;			obsoleting the Set Current one, fixed many bugs 
;			regarding zoom limits and resizing, etc, etc.
;		97-11-10 srio@esrf.fr adds Edit/Data... menu using xtable.
;		97-12-03 srio@esrf.fr adds xplot_cleardata() and fix
;			a bug with non-interactice use of plotmgr. V 2.5.1.
;			Adds xplot_setlimits().
;		98-02-18 srio@esrf.fr adds NOREFRESH keyword to 
;			 xplot_[setlimits,loadfile,changecol,exebuffer,
;			errorbars,addcol,plotmgr] routines.
;			Fix a bug (not displaying error bars in saved
;			sets.)
;		98-03-25 srio@esrf.fr small changes to allow starting
;			xplot when spec_access is not available.
;		98-04-07 srio@esrf.fr removed /no_copy kw in main 
;			widget_draw definition (problems in w95).
;		98-09-18 srio@esrf.fr Avoid changing columns when reloading.
;			Improved SPEC window.  Avoid changing columns
;			after column operations. Corrected bug in
;			displaying calculations,FWHM.../Max x.
;			Improved save/asciioption. Improved oper w columns.
;			Correct managment of ![x,y].style. Many other things.
;			Version 2.6.Beta1.
;		98-09-25 xplot_mesh and xplot_shift added
;		98-09-30 bug when selecting columns after adding cols fixed.
;			Use of psysvar when overplotting after fit.
;			Shortened "command" in xplot_plot
;			Other cosmetics. 
;		98-09-30 reintroduced changes for PSD (done in revision 1.35 
;			but lost). Version 2.6Beta3.
;		98-11-16 fixed a bug when applying oper w cols and the 
;			coltitles kw is set. Shorts the number of
;			characters in the column pulldown menus. 
;			Version 2.6Beta4.
;		98-11-20 fixed a bug when save/restore backup file.
;			Added "add to file" for spec files. Version 2.6Beta5.
;		99-01-08 substituted cw_clr_index by cw_clr_index1 to allow
;			entering the index by simple typing.
;
;		99-01-13 Removed calls to xaid.
;		99-03-12 Added non-modal window for setting limits. Help
;			menu updated. Pemoved xplot_compile. Added
;			option to Wrile/Load qattributes from file.
;			Aded hardware and TT fonts. Version 2.7B1
;		00-03-21 Improved Calculation|Interpolation/Spline option 
;		01-01-22 Improving Edit|Cut
;		01-01-24 Allows printing a zoomed image. Improves
;			operations with sets...
;			Fixed bugs in Edit|Columns (with columns labels and 
;			SPEC)
;		02-03-08 srio@esrf.fr adds AutoReLoadFile option. The autoreload
;			if the file has been modified works only for idl >= 5.5
;		02-03-13 srio@esrf.fr changed PolyFitW by Poly_Fit
;		02-05-27 srio@esrf.fr Bug fixed: Added /SORT kw to 
;			Int_Tabulated calculations (before it returned wrong 
;			values when the array was sorted in descendent order)
;		02-06-20 srio@esrf.fr manages the spec keywords with
;			more than one character in the xplot title label.
;		04-03-10 srio@esrf.fr adds "calculations/list of points"
;-
;

;+
;
;      ================= NON-INTERACTIVE MANIPULATION OF XPLOT ==========
;
;	Since Xplot version 2.4, the Xplot internals are being modified
;	to allow also non-interactive manipulation of Xplot.
;	This is very useful when using Xplot from other widget applications
;	and one wants to prepare sophisticated plots with options not
;	availables as keywords in the main Xplot call. Typical examples
;	are to overplot several plots, to add commands in the EXEbuffer,
;	or to display error bars.
;	The access is done by a set of routines with general syntaxis:
;
;	xplot_<name>, parent [,other_inputs, keywords]
;
;	where <name> indicates the action to be performed (i.e. loadfile,
;	quit, etc.), parent is the widget id of the manin Xplot window to 
;	be manipulated and the other information  is passed using the
;	other inputs and keywords.
;
;	EXAPLE:
;		; you can execute these commands as an example
;		tmp = fltarr(5,100)
;		tmp(0,*) = findgen(100)
;		tmp(1,*) = tmp(0,*)^2     ; first curve
;		tmp(2,*) = tmp(0,*)^2+100 ; second curve
;		tmp(3,*) = tmp(0,*)^2-500 ; bottom of error bars
;		tmp(4,*) = tmp(0,*)^2+500 ; top of error bars
;		xplot,parent=p,/no_block  ; starts xplot, retrieved parent id
;		xplot_loadfile,p,tmp      ; load the data in xplot
;		xplot_changecol,p,ycol=3  ; set Y as column 3
;		xplot_controls_action,p,linestyle=1 ; set dotted line
;		xplot_savecurrent,p       ; save this plot
;		xplot_changecol,p,ycol=2  ; overplot column 2
;		xplot_controls_action,p,linestyle='0' ; overplotted line to solid
;		xplot_errorbars,p,4,5     ; plot error bars
;		xplot_exebuffer,p,setbuffer=$ ; add an annotation
;			"xyouts,.2,.7,'Example of non-interactive label',/norm"
;
;	Available routines are:
;		xplot_quit
;		xplot_cleardata
;		xplot_setlimits
;		xplot_loadfile
;		xplot_savecurrent
;		xplot_changecol
;		xplot_exebuffer
;		xplot_errorbars
;		xplot_addcol
;		xplot_plotmgr
;		xplot_saveascii
;		xplot_mesh
;		xplot_shift
;		xplot_controls_action
;		xplot_reloadfile
;		xplot_autoreloadfile
;		xplot_getcurrentdata
;
;	The routines help id listed below.
;-
;
;=========================================================================
;
function xplot_version
return,(xplot_defaults('version')).version
;return,'2.4'
end
;
;=========================================================================
;
function xplot_text,text
;
; returns a text array with particular contents
;
case strupcase(text) of
  'ANNOTATE': text = ['This Xplot options calls the IDL utility ANNOTATE.',$
			'The ANNOTATE widget allows you to interactively ',$
			'annotate images and plots with text and drawings.',$
			' ',$
			'The ANNOTATE utility is *NOT* integrated in Xplot.',$
			'This means that the objects overplotted with ',$
			'ANNOTATE onto your Xplot graphic will not be passed',$
			'to XPlot, and will be lost when closing the ANNOTATE',$
			'window. The Xplot graphic modified by ANNOTATE',$
			'can only be printed from the ',$
			'ANNOTATE/File/Write PostScript/Everything option to',$
			'a PostScript file containing the bitmap of the',$
			'display, not the PostScript objects and lines, as',$
			'Xplot does. In any case, this application may be ',$
			'useful sometimes, and that is why we are here.']
  'HELPABOUT': text = [' ',' ', $
		  '                    Xplot                   ',$
		  ' ',$
		  '                     by                     ',$
		  ' ',$
		  '          Manuel Sanchez del Rio            ',$
		  ' ',' ', $
		  '   European Synchrotron Radiation Facility  ', $
		  '      BP 220, F-38043 Grenoble Cedex 9      ', $
		  ' ', $
		  '            email: srio@esrf.fr             ', $
		  '            Tel: +33-476882513              ', $
		  '            Fax: +33-476882160              ', $
		  ' ',' ']
  'LIMITS': text = [' ', $
	'Enter here the [xy]min, [xy]max and [xy]style. The values for',$
	'minima and maxima entered here will be applied to the graph. If',$
	'the minimum and the maximum vales are the same, the program',$
	'ignores the values and applies the default interval range. The',$
	'default interval range is calculated taking into account only',$
	'the current set, and not the saved sets. That is why sometimes the',$
	'data of the saved sets may be hidden, and one has to change manually',$
	'the limits to give a suitable interval that includes them. ',$
	' ']
  'TITLES': text = [' ', $
  'Place in these boxes the titles for the plot.',$
  ' ',$
  ' ',$
  'When you start XPLOT with the keyword "COLTITLES" keyword, or when', $
  'you use SPEC files, then a text in the corresponding [xy]title equal ',$
  "to '-1' means that you will get as [xy]titles those defined in the",$
  "'Column Titles...' option.", $
  ' ',$
  'With SPEC files, for all the title entries, the symbol #x (i.e. #S for',$
  'scan number or #F for file name) is substituted by its corresponding',$
  'value in the scan header.',$
  'In the case of #S two possibilities are allowed:',$
  '  i)  #S (uppercase) displays scan number and title',$
  '  ii) #s (lowercase) displays only the title','', $
  '',$
  'Embedded Formatting Commands',$
  '============================',$
  'When you use vector, TrueType, and some device fonts, text strings can ',$
  'include embedded formatting commands that facilitate subscripting, ',$
  'superscripting, and equation formatting. ',$
  'See documentation in the Xplot/File/Print Plot window.' ]
  'MESH': text = [' ',' ', $
     'The mesh option permits to display a multiple data set in a 3D view. ',$
     ' ',$
     'A multiple data set is defined as a single data set with different',$
     'plots one after the other. All individual plots must have the same',$
     'number of points. The position where a plot ends and the next plot starts',$
     'can be defined by the line position (or the number of points of the',$
     'individual plots) of by an additional label column. The label column',$
     'has a different value for each individual plot.',$
     ' ',$
     'Three kind of MESH view are possible: lines, surface and contour.',$
     ' ',$
     'When the MESH option is set in XPLOT other XPLOT functions (like',$
     'Save Current, Plot symbols, etc.) do not work with it. The attributes',$
     "to the MESH option are given in the 'Additional keywords' box.",$
     ' ',$
     'Example: Consider two curves with points (x,y)=(1,1), (2,6) and (3,3) ',$
     'for the first one and (1,3), (2,7) and (3,0) for the second one. ',$
     'The first curve is labelled with the value 27 and the second is labelled ',$
     'with 60.  The data set may be (col1=x, col2=label and col3=y) :',$
     '1  1 27',$
     '2  6 27',$
     '3  3 27',$
     '1  3 60',$
     '2  7 60',$
     '3  0 60',$
     ' ',$
     'This format is also presented in the SPEC mesh scans.',$
     "If you load this set in Xplot, the 'split plot' can either be set by",$
     " 'each 3 points' or 'using a label column 3'. A negative value of the column",$
     '(i.e. col=-3) indicates that the plots are labelled with their label values,',$
     'otherwise Xplot labels them as 0,1,2,...',$
     ' ',$
     'Typical examples of accepted keywords are:',$
     'For the lines/surface kind: AZ(rotation angle around Z), ZAXIS (Z axis ',$
     'position),ZRANGE ([Zmin,Zmax] limits) and ZLOG (=1 for log Z scale).',$
     '(Note: These keywords are the only ones accepted if you run idl 4.x)',$
     'For contour mesh the NLEVELS kwyword can be used. See the IDL users',$
     'documentation for a complete list of accepted keywords. ' ]
  'COLUMNOPER': text = [' ',$
   "The 'Operation With Columns' option allows the user to ", $
   'calculate a new column data in the current set as',$
   'a function of the existing ones.', '   ', $
   'The expression for the new column must be written as a function of the',$
   'existing columns, which are named col<n>, being <n> the column number ',$
   '(starting by one). The expressions are stored in a buffer in order to',$
   'facilitate their reuse in other runs. ',' ',$
   'The resulting column can either: ', $
   '    1) Be appended to the set, thus substituting the ', $
   '       current set by a new one with one more. ',$
   '    2) Substitute an existing column in the set.',$
   '    3) Export the result to a new Xplot window.' ]
  'SMOOTH': text = [' ',$
	'This option smooths the Y data of the current graph.',$
	' ',$
	'Two methods are available: ',$
	'  Boxcar average: the results are a current set smoothed with a ',$
	'                 boxcar average of the specified width.',$
	'  Least Squares: the results are a current set smoothed applying',$
	'                 a least squares smoothing polynomial filter',$
	'                 also called Savitzky-Golay smoothing filter.',$
	'   ', $
	'It uses the SMOOTH function of IDL for the Boxcar method and',$
	'the POLY_SMOOTH routine of Frank Varosi for the Savitzky-Golay',$
	'one.',' ']
  'OPERWITHSETS': text = [' ',$
      "The 'operations with sets' option allows the user to create", $
      'a new set of data called C (with abscissas Cx, and ', $
      'ordinates Cy) which is calculated as a function of two', $
      'existing sets: A (usually the current displayed data, ', $
      'with abscissas Ax and ordinates Ay) and an optional', $
      'set B read from a 2-column file.', $
      '   ',$
      'The user must provide the evaluation expression written', $
      'using IDL syntax, using Ax,Ay,Bx,By and Cx as variables.', $
      '   ',$
      'In the case that the two sets A and B have different', $
      'abscissas, the set B is substituted by an inter-', $
      'polated one with new abscissas as Ax.', $
      '   ',$
      'Examples:',$
      '1) This option can also be used to create new data, for', $
      '   example abscissas Cx=findgen(100) and ordinates Cy=Cx^2',$
      '2) Background removal: when removing a beckground, it is often',$
      '   needed to perform a fit of the data EXCLUDING THE PEAKS.',$
      '   For creating a new data set without the peaks (for example',$
      '   using ordinate values smaller that 500) one can write:',$
      '   Cx=Ax[where(Ay LT 500)] and  Cy=Ay[where(Ay LT 500)] ',$
      '   example abscissas Cx=findgen(100) and ordinates Cy=Cx^2',$
      '   Now, the new data set can be fitted with any function, and ',$
      '   the resulting fitting coefficients can be used to create the',$
      '   ordinate values over the full abscissas range (using the option',$
      '   "operation with columns") and then remove this background.' ]
  'CDF': text = [' ',$
        'Each value of the CDF (Cumulative Distribution Function) is the',$
	'integral of the original data from minus infinity to the current',$
	'abscissas value. ',' ',$
	'Two integration methods are possible: ',$
	'1) Simple addition of the ordinates values y[i] multiplied by its',$
	'   step value x[i]-x[i-1]',$
	'2) Five-point Newton-Cotes method (with int_tabulated IDL function']
  'PSD': text = [' ',$
	'This option computes the power-spectral-density function from',$
        'the current profile data.',$
	'Inputs: ',$
        ' fmin and fmax: minimun and maximum frequencies. These values ',$
	'   specify the min and max spatial frequencies to be considered. ',$
	'   The default is applied when both values are equal (f min=f max).',$
	'   The default goes from 1/(length) to ',$
	'   1/(2*interval) (i.e., the Nyquist frequency), where length is the ',$
	'   length of the scan, and interval is the spacing between points.',$
        ' output frequencies: set to compute the psd function for positive',$
	'   frequencies only.',' ',$
        'RESTRICTION: The X values must be equally spaced.',' ',$
        'PROCEDURE: S=Length*ABS(FFT(Y),-1)^2',' ',$
        'METHOD: PSD is calculated using the prof2psd function, courtesy of ',$
	' David L. Windt, Bell Laboratories (windt@bell-labs.com) ']
  'FITPOLGAUSS': text = [' ',$
        'The Polynomial/Gaussian fit option of XPlot permits to perform', $
	'different kind of linear, polynomial and gaussian fitting.',$
	'The different options refer to different IDL routines.', $
	'  ',$
	'Please read the documentation in the Help/Routines for',$
	'detailed references of the method used for each fit.', $
	'  ',$
	'A correspondence between Xplot polynomial fit options and ',$
	'the IDL routines is the following: ', $
	'  ',$
	'  1) Chi-Square Linear Regression                       - LINFIT', $
	'  2) Robust Estimation Linear Regression                - LADFIT', $
	'  3) Least Square Polynomial Fit                        - POLY_FIT', $
	'  4) Singular Value Decomposition Polynomial Fit        - SVDFIT', $
	'  5) Gaussian + 2nd degree polynomial                   - GAUSSFIT']
  'EXEBUFFEREDIT': text = [' ',$
		'This is an advanced option which permits ', $
		'to create and modify the Xplot internal buffer ',$
		'where the Legend and Label commands are stored.',$
		'This buffer is a set of IDL commands executed ',$
		'right after the plot. It is used to store labels ',$
		'annotations etc.',$
		'  ',$
		'It is also used for modifying the legend/labels ',$
		'(created with ethe Edit/Legend and Edit/Label options)',$
		'after being saved. Its use requires a small knowdledge ',$
		'of IDL syntax.', $
		'It is not recommended for the general user.', $
		'   ']
  'SAVEASCII': text = [' ',$
	'This option permits to write a new ASCII file containing the ',$
	'numerical data of the current set. Just enter the file name',$
	'where to write the data. ',$
	' ',$
	'You can save only the two columns of current data in the current',$
	'set (default option), meaning the X and Y column currently',$
	'displayed, or all the columns present in the current set, thus',$
	'a multi column file.',$
	' ',$
	'A question mark (?) in the file entry will start a browser to define',$
	'the file name. ',$
	'If the selected file name already exists, then a confirmation window',$
	'will appear.',$
	' ',$
	'HEADERS',$
	'The created file can optionally have a header with the SPEC style:',$
	'  #F <file name>',$
	'  #S 1 <comment written by xplot>',$
	'  #N <number of columns>',$
	'  #L <column names>',$
	'This file can be read with xplot.',' ',$
	'FORMATS',$
	'Each number of the data set will be written to the file with the ',$
	'format specified defined in the format entry. (i.e. the record',$
	'format will be (ncol(defined_format)), being ncol the number of',$
	'columns to be written and defined_format the format entered.)',$
	'Few of the format codes are the following:',$
	'  E[w.q] or E[w.dEe]: Transfer floating-point data using scientific ',$
	'       (exponential) notation.',$
	'  F[w.q] : Transfer floating-point data.',$
	'  G[w.d] or [n]G[w.dEe] : Use F or E format depending on the ',$
	'       magnitude of the value being processed.',$
	'       [w.d] is an optional width specification (1 <= w <= 256, ',$
	'         1 <= d < w). The variable w specifies the number of ',$
	'         characters in the external field. For the F, and E format',$
	'         codes, d specifies the number of positions after the',$
	'         decimal point. For the G format code, d specifies the ',$
	'         number of significant digits displayed.',$
	'       [e] is an optional width (1 <= e <= 256) specifying the width',$
	'         of exponent part of the field. ',$
	'  I[w.m] : Transfer integer data.	',$
	'  O[w.m] : Transfer octal data.	',$
	'  Z[w.m] : Transfer Hexadecimal data.',$
	'       [w] is an optional integer value specifying the width of the',$
	'         field in characters. ',$
	'       [m] specifies the minimum number of nonblank digits required',$
	'         The field is zero-filled on the left if necessary. If m is',$
	'          omitted or zero, the external field is blank filled.',$
	'For more information on formats, consult the IDL manual or help.',$
	' ',$
	'This option is also very useful to create temporal files, which can',$
	'be re-read in other Xplot window (because it is possible to create',$
	'many Xplot windows, holding simultaneously different data sets), ',$
	'or used for Operations with sets... (in the Calculations menu.)']
  'COLTITLES': text = [' ',$
	'This option permits to edit the column titles displayed in the ',$
	'droplist at the bottom of the Xplot window.']
  'SHIFT': text = [' ',$
	'This option permits to shift horizontally or vertically the current',$
	"displayed X/Y line of a given quantity measured in plot's ",$
	'coordinates.',$
	' ',$
	'This option is very useful when one has to present a multiple set',$
	'of lines (for example spectra) which change a little from one to',$
	'another, and we are not interested in the absolute coordinates.',$
	' ',$
	'Thus, it permits to shift the plot line without changing the data',$
	'or creating a new set of data. ',' ']
  'ERRORBARS': text = [' ',$
	'This option displays error bars in your plot. For that purpose the',$
	'upper values and bottom values of the error bars to be displayed ',$
	'have to be defined as columns in the current set. This can be done,',$
	'for example, by using the Calculations/Operations with columns...',$
	'option.']
  'DRAWDIM': text = [' ',$
	'This option permits to define the dimension (in pixels) of the',$
	'Xplot graph window. The Horizontal/Verical ratio is conserved',$
	'in the plotting option.',' ']
  'SECONDAXIS': text = [' ',$
	'This option permits to draw a second X axis (on the top of the',$
	'data window) and/or Y axis (on the right). Give the minimum and',$
	'maximum values for the axis. These values are completely',$
	'disconnected from the data of the graph, thus are not changed when',$
	'graph limits are modified. ',$
	'The same minimum and maximum values means that the second axis ',$
	'will not be plotted. ',$
	' ',$
	'Advanced feature: If your graph does not look very nice after the',$
	'insertion of the second axis, use the Edit/System Variables/',$
	'!p.position to reduce/center your graph.']
  'INTERPOLATION': text = [' ',$
	'This options creates a new Xplot window with interpolated data.',$
	' ',$
	'This option is normally used to create a new set of data with the',$
	'same look than the original, but with a different number of points.',$
	' ',$
	'Three possible interpolations are possible: ',$
	'  Linear Interpolation (using INTERPOL from IDL)',$
	'  Spline interpolation (using SPLINE from IDL). ',$
	'                    This option asks for the TENSION value. The ',$
	'                     default value is 1.0. If TENSION is close to',$
	'                     0, (e.g., .01), then effectively there is a',$
	'                     cubic spline fit. If TENSION is large, (e.g.,',$
	'                     greater than 10), then the fit will be like a',$
	'                     polynomial interpolation.',$
	'  Parametric Spline (using SPLINE_P from IDL). Here the number',$
	'                    of points of the result is defined by the',$
	'                    program itself.',' ']
  'PEAKFINDER': text = [' ',$
	'This option finds the peaks of a 1-D data. ',$
	'',$
	'In case that the routine does not provide realistic results',$
	'it is recommended to make interpolation and/or smoothing before',$
	'searching the peaks.',$
	'',$
	'brief description of the method used:',$
	'',$
	'A peak is a local maximum  in your data. A local maximum ',$
	'has derivative zero. I calculate the derivative of the y',$
	'data (using IDLs Deriv) and I consider "a peak" any ',$
	'value which derivative is positive (or zero) and the ',$
	'point after has a negative derivative. All peaks are returned.',$
	'',$
	'However, real data contains noise and the returned peaks',$
	'are much more than the "intrinsic", "reasonable" or supposely',$
	'"real" peaks. The routine performs some kind of evaluation',$
	'of the peaks to see is the peaks is "good" or not. For that',$
	'purpose we assign a weight (col 3) value to each peak. The larger',$
	'is the weight, the peak is more "important". ',$
	'',$
	'How to get the weight: looking at the derivetive of some',$
	'data one realizes that the in the neighbourhood of the ',$
	'"important" peaks the absolute values of the derivative are',$
	'larger. In addition, an "important" peak has meny points ',$
	'with positive derivtive to its left and many points with ',$
	'negative derivative to its right. We evaluate, for each peak,',$
	'the number of points that have positive derivative at its ',$
	'left plus the number of points with negative derivative',$
	'at its right. This is the "peak width".',$
	'The sum of the absolute values of the derivatives for all',$
	'the points inside the "peak width" constitutes the "peak',$
	'weight". The "normalized peak weight" (norm-w) is the peak weight',$
	'divided by the maximum peak weight. ',$
	'',$
	'One can reduce the number of peaks setting a "cutoff" value,',$
	'i.e., considering only the peaks which normalized weight',$
	'is larger than this cutoff value. ',$
	'What we do next is calculate the number of peaks as a function',$
	'of the cutoff value. This information is optionally displayed ',$
	'if the "plot number of peaks vs cutoff" option is set.',$
	'From this plot one can estimate the number of peaks that ',$
	'are "reasonable". I obtain this when the plot "stabilizes",',$
	'i.e, I get the same number of points for a wide internal of',$
	'cutoff values. The difference between minimum and maximum cutoff ',$
	'values that provide with the "stabilized" number of points is',$
	'an estimator of the confidence of the "resonable" of "stabilized" ',$
	'number of peaks.',$
	' ',$
	'For more information see the doc in routine PeakFinder.']

  'CONVOLUTION': text = [' ',$
	'This option permits to calculate convolution and correlation between',$
	'two data sets, A (the current set) and B.',$
	'B can be one of the following cases:',$
	'  B=A autoconvolution/autocorrelation cases.',$
	'  B   from an external file.',$
	'  B   a gaussian, lorentzian or a liner combination of both',$
	'     (the so-called pseudo-voigt function:',$
	'      v(x)=r*L(x)+(1-r)G(x), with r the ratio or linear coefficient',$
	'       between Lorentzian and Gaussian).',$
	' ',$
	' The Convolution and correelation are calculated from:',$
	' CONV(x) = integral( A(y) B(x-y) dy)',$
	' CORR(x) = integral( A(y) B(y-x) dy)',$
	' Of course, CONV(x) = CORR(x) if B(y) is symmetrical.',$
	' ',$
	'The prigram interpolate both curves (using INTERPOL) to the',$
	'biggest interval and smallest step between both abscissas and',$
	'convolute/correlate them by using CONV or INFL routines).',$
	'The result may be normalized. The program uses the FFT method',$
	'to perform the calculations.',$
	' ',$
	'The available controls are: ',$
	' Method: to use either convolution or correlation.',$
	' Type: Select to conv/corr A with itself (auto), with another',$
	' function from a file or with a pseudo-voigt function.',$
	' Normalization: Divide the result by integral(A) (to the first',$
	'                set), by integral(B) (to the second set), or by',$
	'                integral(A)*integral(B) (to both sets).',$
	' Voigt ration: when tyle="Gaussian and/or Lorentzian" this ',$
	'               permits to give the Gaussian and Lorentzian',$
	'               contriburions to the pseudo-voigt function (1=',$
	'               pure Lorentzian, 0=pure Gaussian).',$
	' Voigt Full Width at Half Maximum: The width for the voigt',$
	'               function (when selected).',$
	' Overplot original data: Overplots the original data on the',$
	'               top of the convoluted data.',$
	' Write output to a file: if Yes, give the name in the next entry.',$
	' Output file name: enter the nema for a 2-column file with the ',$
	'                   results.',$
	' Calculation by: The calculations may be done using either the Fast ',$
	'                 Fourier transform algorithm or by numerical ',$
	'                 integration. The FFT is much faster and should be',$
	'                 preferred.',$
	' ']
  else: text = ''
endcase
return,text
end ; xplot_text
;
;=========================================================================
;
FUNCTION xplot_coltitles_short,coltitles,Max=cmax,Get_Maximum=gmax
on_error,2
IF NOT(Keyword_Set(cmax)) THEN cmax=23

IF Keyword_Set(gmax) THEN RETURN,cmax

output = coltitles
FOR i=0L,N_Elements(output)-1 DO BEGIN
  IF StrLen(output[i]) GT cmax THEN BEGIN
    output[i]=StrMid(output[i],0,(cmax-3)/2)+'...'+$
      StrMid(output[i],StrLen(output[i])-(cmax-3)/2,(cmax-3)/2)
  ENDIF
ENDFOR

RETURN,output
END ; xplot_coltitles_short
;
;=========================================================================
;
function xplot_readspec, state, scan_index, mca, oldcol=col

catch, error_status
if error_status ne 0 then begin
   ; error is managed by the caller
   ;message,/info,'error caught: '+!error_state.msg
   ;itmp = Dialog_Message(/Error,$
   ;	'XPLOT_READSPEC: error caught: '+!error_state.msg)
   catch, /cancel
   on_error,2
   RETURN,0
endif

   handle_value, state.spec.handle, hspec
   if not mca then begin
      data = spec_data(hspec, scan_index, /index, /float)
      ncol = n_elements(data(*,0))
      labels = spec_labels(hspec, scan_index, /index)
      if ncol ne n_elements(labels) then begin
         tmp = widget_message(DIALOG_PARENT=state.wids.main,$
		'Error reading data from scan #'+ $
                 strcompress(scan,/REM)+'.',/ERROR)
         return, 0
      endif
; srio adds:
      ;labels = strcompress((sindgen(ncol+1))(1:ncol) ) + ': '+labels
      tmplabels = labels(0)
	for i=1,n_elements(labels)-1 do tmplabels = tmplabels+'!'+labels(i)
      state.spec.labels = tmplabels

      xcol = 1
      ycol = ncol
      if keyword_set(col) then begin
         xcol = state.xpdt.Xcol
         ycol = state.xpdt.Ycol
	 if xcol GT Ncol or xcol LT 1 then xcol = 1
	 if ycol GT Ncol or ycol LT 1 then ycol = ncol
         ;if state.xpdt.Xcol le ncol and state.xpdt.Ycol le ncol then begin
         ;   xcol = state.xpdt.Xcol
         ;   ycol = state.xpdt.Ycol
         ;endif
      endif
   endif else begin
      data = spec_data(hspec, scan_index, /index, /float, /mcdata, $
                                           channels=chann, calibrated=energy)
      data = transpose(temporary(data))
      ncol = n_elements(data(*,0))
      nch = n_elements(chann)
      if n_elements(data(0,*)) ne nch then begin
         tmp = widget_message(DIALOG_PARENT=state.wids.main,$
		'Error reading MCA data from scan #'+ $
                strcompress(scan,/REM)+'.',/ERROR)
         return, 0
      endif
      data = [reform(data,ncol,nch), reform(chann,1,nch), reform(energy,1,nch)]
      widget_control,state.wids.xcolmca,get_value = aux
      xcol = ncol+1+aux
      ycol = 1
      if keyword_set(col) then begin
         if state.xpdt.Ycol le ncol then begin
            ycol = state.xpdt.Ycol
         endif
      endif
   endelse

   handle_value, state.spec.handle, hspec, /set
   state.spec.scan = scan_index
   state.spec.mca = mca

   if not mca then begin
      tmplabels = strcompress((sindgen(n_elements(labels)+ $
	1))(1:n_elements(labels)) ) + ': '+Xplot_ColTitles_Short(labels)
      widget_control,state.wids.xcolSPEC, SET_VALUE=tmplabels, $
	SET_DROPLIST_SELECT = xcol-1
      widget_control,state.wids.ycolSPEC,SET_VALUE=tmplabels, $
        SET_DROPLIST_SELECT = ycol-1
   endif else begin
      widget_control,state.wids.ycolmca,SET_VALUE=strcompress(ycol,/REM)
   endelse
   widget_control,state.wids.ncol,SET_VALUE=strcompress(ncol,/REM)

   state.xpdt.Xcol = xcol
   state.xpdt.Ycol = ycol
   state.xpdt.shift=[0.,0.]
   state.xpdt.ErrPlot=[0,0]

   return, data
end
;
;=========================================================================
;
pro get_fileinfo, state, spechandle, file
   on_error, 2
   file='unknown'
   list = spec_access(spechandle,/list,/compress,/str, $
                        filename=file,mcdata=state.mca, no_empty=(1-state.mca))
   ; coommented by srio 97/02/13. Useless?
   ;spawn, 'file '+file, aux, /sh
   widget_control, state.file, set_value=file
   if list eq '' then list = '<none>'
   widget_control, state.list, set_value=list
   return
end
;
;=========================================================================
;
pro xplot_specscan_init, state, error_status
   catch, error_status
   if error_status ne 0 then begin
      catch, /cancel
      tmp = widget_message(DIALOG_PARENT=state.wids.main,$
	"Can't find SPEC file: "+file, /ERROR)
      goto, out
   endif

   handle_value, state.handle, spechandle, /no_copy

   scan_index = spec_scan(spechandle, 'current',/return_index)
   state.scanidx = scan_index
   scan = spec_scan(spechandle, scan_index, /index)
   widget_control, state.field, set_value=scan
   widget_control, state.current, set_value='Current scan :'+string(scan)

   get_fileinfo, state, spechandle, file

Out:
   handle_value, state.handle, spechandle, /set, /no_copy
end
;
;=========================================================================
;
pro xplot_specscan_event, event
   widget_control, event.top, get_uvalue=state
   if event.id eq event.top then begin
      xplot_specscan_init, state, error
      widget_control, event.top, set_uvalue=state
      if error then widget_control, event.top, /destroy
      return
   endif

   handle_value, state.handle, spechandle, /no_copy
   widget_control, event.id, get_uvalue=evuval

   case evuval of
      'ACCEPT': begin
         widget_control, state.field, get_value=scan
         scan=strtrim(scan(0),2)
         index = -1
         if strpos(scan, '-') eq -1 then index = (spec_scan(spechandle, scan, $
                                  /return_index, /no_range,  $
                                  mcdata=state.mca, no_empty=(1-state.mca)))(0)
         if index(0) gt 0 then begin
            state.scanidx = index(0)
            dummyevent={NEW_SCAN_EVENT, ID:state.butid, TOP:0L, $
                        HANDLER:0L, FHANDLE:0L, $
			NEWSCAN:state.scanidx, MCA:state.mca}
            widget_control, state.butid, send_event=dummyevent
            widget_control, state.current, set_value='Current scan :' + $
                            string(spec_scan(spechandle, index, /index))
         endif else begin
            msg='Scan "'+scan+'" is not valid.'
            tmp = widget_message(DIALOG_PARENT=event.top,$
		msg, /error, dialog_parent=event.top)
         endelse
      end

      'CLOSE': 

      'DATATYPE': begin
         widget_control, event.id, get_value=aux
         state.mca = aux
         get_fileinfo, state, spechandle
      end
   endcase

   handle_value, state.handle, spechandle, /set, /no_copy
   widget_control, event.top, set_uvalue=state
   if evuval eq 'CLOSE' then widget_control, event.top, /destroy
end
;
;=========================================================================
;
function xplot_specscan, handle, butid, mca, group_leader=group,$
   no_block=no_block
   MainBase = widget_base(title='Scan Selection', mbar=menubar, $
                   column=1,uvalue=0)
      FileName = widget_label(MainBase)
      ScanCurrent = widget_label(MainBase)
      tmp0 = widget_base(MainBase, column=1, /frame)
         ScanList = widget_label(tmp0)
         tmp1 = widget_base(tmp0, row=1)
            ScanField = cw_field(tmp1, title='Scan id.', uvalue='ACCEPT', $
                                           xsize=5, /return_events)
            DataType = cw_bgroup(tmp1, ['Counters','MCA'],/exclusive, $
                               /row, set_value=mca, uvalue='DATATYPE', /frame)
         tmp1 = widget_base(tmp0, row=1)
            AcceptButt = widget_button(tmp1, value='Load', uvalue='ACCEPT')
            CloseButt = widget_button(tmp1, value='Close', uvalue='CLOSE')

   state={butid:butid, field:ScanField, file:FileName, current:ScanCurrent, $
          list:ScanList, handle:handle, scanidx:0, mca:mca}
   xplot_specscan_init, state, error
   if error eq 0 then begin
      widget_control, MainBase, /realize, set_uvalue=state
      if keyword_set(no_block) then $
      xmanager, 'xplot_specscan', MainBase, group_leader=group, /no_block else $ 
      xmanager, 'xplot_specscan', MainBase, group_leader=group
   endif else begin
      widget_control, MainBase, /destroy
   endelse
   return, MainBase
end
;
;=========================================================================
;
Function xplot_spectitle,title,spechandle,specscan
;
; Process a title for spec. It will substitute the characters
; #S, #D, etc by their values in the corresponding header.
;

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XPLOT_SPECTITLE: error caught: '+!error_state.msg)
  catch, /cancel
  return,title
endif
if n_elements(specscan) LE 1 then specscan =spec_scan(spechandle, 'current')

; find number of # characters in the title

i=0
cnt=0
pos = -1
while (i ne -1) do begin
  i = strpos(title,'#',i)
  if (i ne -1) then begin 
    cnt = cnt+1
    i = i + 1
    if n_elements(pos) EQ 1 then begin
       if (pos EQ -1) then pos = i-1 else pos=[pos,i-1]
    endif else  pos=[pos,i]
  endif
endwhile 
if cnt EQ 0 then return,title

newtitle=title
for i=0,cnt-1 do begin
  ; compute ii, the number of charactes of the kw (e.g., #S = 1, #UT0 = 3)
  istop = 0
  ii=-1
  WHILE istop NE 1 DO BEGIN
    ii = ii+1
    IF (byte(title))[pos[i]+ii] EQ byte(' ') THEN istop=1
    IF pos[i]+ii+1 EQ StrLen(title) THEN istop=1
  ENDWHILE
  s_from = strmid(title,pos(i),II+1)
  ; keyword ALL added srio@esrf.fr 97/08/29
  s_to = spec_headers(spechandle,specscan,strupcase(s_from),/ALL)
  IF N_Elements(s_to) GT 1 THEN s_to = s_to[N_Elements(s_to)-1]
  ; allows #s (not display the scan number) and #S
  if strmid(s_from,1,1) EQ 's' then begin
    tmp = strpos(s_to,' ',3)
    if tmp[0] NE -1 then s_to = strmid(s_to,tmp[0],strlen(s_to)-2)
  endif else begin
    s_to = strmid(s_to,ii+1,strlen(s_to)-2)
  endelse
  newtitle=strsubstitute(newtitle,s_from, s_to)
endfor
return,newtitle
end


;
;=========================================================================
;

PRO Xplot_WriteAttributes,state,file=file,Group=group

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!error_state.msg
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'XPLOT_WRITEATTRIBUTES: error caught: '+!error_state.msg)
   catch, /cancel
   on_error,2
   IF N_Elements(unit) NE 0 THEN Free_Lun,unit
   RETURN
endif

names = ['strcontrols','xpdt_titles', 'xpdt_limits','xpdt.xcol','xpdt.ycol']
  ;,'xpdt.mesh','xpevaluate','xpconvolution','saveascii']

names_desc = 'control panel|Edit/Titles|Edit/Set Limits and Styles'+$
   '|X column value|Y column value'
  ;'|View/Mesh|Calculations/Oper with sets|Calculation/Convolution'+$
  ;'|File/Save as new ASCII file'

desc = [ $
  '1, BASE,, ROW', $
  '0, BUTTON, Accept, QUIT', $
  '2, BUTTON, Cancel, QUIT,TAG=CANCEL', $
;  '0, LABEL, Saving XPlot current attributes to a file, CENTER', $
  '1, BASE,, COLUMN, FRAME', $
  '0, BUTTON, '+names_desc+', LABEL_TOP=Select items to save:,COLUMN, TAG=names,SET_VALUE=1', $
  '2, BUTTON, No|Yes, ROW,  EXCLUSIVE,LABEL_TOP=Set as default for current session:,TAG=default, SET_VALUE=0', $
  '0, TEXT,?, LABEL_TOP=Enter File name (? for launching Browser):, WIDTH=12,TAG=fname, COLUMN' ] 

a = CW_FORM(desc, /COLUMN, Title='Save XPlot attributes to a file')
IF a.cancel EQ 1 THEN RETURN ; Cancel pressed

tmpstr=0
file = a.fname

IF StrCompress(file,/Rem) EQ '' OR StrCompress(file,/Rem) EQ '?' THEN BEGIN
  back:
  file = Dialog_PickFile(Dialog_Parent=group, /Write, File='xplot.att', $
	Filter='*.att')
  IF StrCompress(file,/Rem) EQ '' THEN BEGIN
    itmp = Dialog_Message(Dialog_Parent=group,/Info,$
	'Operation aborted by request of user.')
    RETURN
  ENDIF
  IF CheckFile(file) EQ 1 THEN BEGIN
    itmp = Dialog_Message(Dialog_Parent=group,/Question,$
	['File '+file+' exists.','Overwrite it?'])
    IF itmp EQ 'No' THEN GoTo,back
  ENDIF
ENDIF

OpenW,unit,file,/Get_Lun

PrintF,unit,';'
PrintF,unit,'; File created by Xplot on '+SysTime()
PrintF,unit,';'
PrintF,unit,'; File containing the XOP/Xplot attributes. Each block'
PrintF,unit,'; of data corresponds to the attribute parameters available'
PrintF,unit,'; from a specific window in Xplot. The fields can be edited'
PrintF,unit,'; or removed. In the case of fields consisting in an array'
PrintF,unit,'; of strings (a droplist widget in teh corresponding window),'
PrintF,unit,'; the first field indicate the selection. For example: '
PrintF,unit,"; xpdt.mesh.FLAG=['0','NO','YES'] means 'NO' selected"
PrintF,unit,"; xpdt.mesh.FLAG=['1','NO','YES'] means 'YES' selected"
PrintF,unit,';'
;
; be careful with this comment as "xpdt.xcol" and "xpdt.ycol" are
; counted when executing this file...
;
PrintF,unit,'; (Please do dot comment xpdt.xcol and xpdt.ycol sections nor'
PrintF,unit,'; remove their comments. If not wanted, remove completely the '
PrintF,unit,'; sections.)'
PrintF,unit,';'
FOR inames=0,N_Elements(names)-1 DO BEGIN
  IF a.names[inames] EQ 1 THEN BEGIN
    PrintF,unit,';'
    PrintF,unit,'; Block: '+names[inames]
    PrintF,unit,';'
    IF N_elements(state) EQ 0 THEN BEGIN
      IF names[inames] EQ 'xpdt.mesh' THEN BEGIN
        tmpstr={junk:{l:0}}
      ENDIF ELSE BEGIN
        tmpstr = xplot_defaults(names[inames])
      ENDELSE
    ENDIF ELSE BEGIN
      command = 'tmpstr = state.'+names[inames]
      itmp = execute(command)
    ENDELSE
  
    IF Type(tmpstr) EQ 8 THEN BEGIN
      tnames = Tag_Names(tmpstr)
      FOR i=0,N_tags(tmpstr)-1 DO BEGIN
        tmp = tmpstr.(i)
        IF Type (tmp) NE 8 THEN BEGIN
          txt=StrLowCase(names[inames]+'.'+tnames[i])+'='+Vect2String(tmp)
          PrintF,unit,txt
        ENDIF
      ENDFOR
    ENDIF
    ; 
    ; other things to do
    ;
    CASE inames OF 
      0: BEGIN  ; controls
	PrintF,unit,'xpdt.connect = '+Vect2String(tmpstr.connect)
	PrintF,unit,'xpdt.psym = '+Vect2String(tmpstr.symbol )
        PrintF,unit,'xpdt.linestyle = '+Vect2String(tmpstr.LineStyle )
	temp = tmpstr.Axis
        if temp eq 0 then PrintF,unit,'xpdt.kind = 1 '
        if temp eq 1 then PrintF,unit,'xpdt.kind = 3 '
        if temp eq 2 then PrintF,unit,'xpdt.kind = 4 '
        if temp eq 3 then PrintF,unit,'xpdt.kind = 5 '

	temp = tmpstr.Ticks
        if temp eq 0 then PrintF,unit,"xpdt.sysvar[1] = '!p.ticklen = 0.02' "
        if temp eq 1 then PrintF,unit,"xpdt.sysvar[1] = '!p.ticklen = -0.02' "
        if temp eq 2 then PrintF,unit,"xpdt.sysvar[1] = '!p.ticklen = 0.5'"
	PrintF,unit,'xpdt.Thick = '+Vect2String(tmpstr.Thickness )

	temp = tmpstr.CharSize 
	PrintF,unit,"xpdt.sysvar[0]= '!p.charsize="+strcompress(temp,/rem)+"'"
	temp = tmpstr.BackGround 
	PrintF,unit,"xpdt.sysvar[6]= '!p.background="+strcompress(temp,/rem)+"'"
	temp = tmpstr.Color 
        PrintF,unit,"xpdt.sysvar[7]= '!p.color="+strcompress(temp,/rem)+"'"

	IF tmpstr.font EQ 16 THEN $
	  PrintF,unit,"xpdt.sysvar[8] = '!p.font = 0' " ELSE BEGIN
	  IF tmpstr.font LT 16 THEN $
	    PrintF,unit,"xpdt.sysvar[8] = '!p.font = -1' " ELSE $
	    PrintF,unit,"xpdt.sysvar[8] = '!p.font =  1' " 
	ENDELSE

;	PrintF,unit,'!p.background = '+Vect2String(tmpstr.BackGround )
;	PrintF,unit,'!p.color = '+Vect2String(tmpstr.Axiscolor )
;	PrintF,unit,'!p.color = '+Vect2String(tmpstr.Color )
;	PrintF,unit,"xyouts,'!"+strcompress(3+ fix(tmpstr.Font),/REM)+"'"

	END
      2: BEGIN  ; limits
	PrintF,unit,"xpdt.sysvar[2] = '!x.range = [" + $
	  strcompress(tmpstr.Xmin,/rem)+','+ $
	  strcompress(tmpstr.Xmax,/rem)+"]'"
	PrintF,unit,"xpdt.sysvar[3] = '!y.range = [" + $
	  strcompress(tmpstr.Ymin,/rem)+','+ $
	  strcompress(tmpstr.Ymax,/rem)+"]'"

	xtmp = [ tmpstr.xstyle0[0],tmpstr.xstyle1[0],tmpstr.xstyle2[0],$
	  tmpstr.xstyle3[0],tmpstr.xstyle4[0] ]
	xval = 0
	for i=0,n_elements(xtmp)-1 do xval=xval+long(xtmp[i])*(2^i)

	ytmp = [ tmpstr.ystyle0[0],tmpstr.ystyle1[0],tmpstr.ystyle2[0],$
	  tmpstr.ystyle3[0],tmpstr.ystyle4[0] ]
	yval = 0
	for i=0,n_elements(ytmp)-1 do yval=yval+long(ytmp[i])*(2^i)

	PrintF,unit,"xpdt.sysvar(4) = '!x.style = "+strcompress(xval,/rem)+"'"
	PrintF,unit,"xpdt.sysvar(5) = '!y.style = "+strcompress(yval,/rem)+"'"
	END
      3: BEGIN  ; X column
	PrintF,unit,"xpdt.xcol = "+strcompress(tmpstr,/Rem)
	END
      4: BEGIN  ; Y column
	PrintF,unit,"xpdt.ycol = "+strcompress(tmpstr,/Rem)
	END
      else: BEGIN
	END
    ENDCASE
  ENDIF
ENDFOR
Free_Lun,unit
Message,/Info,'File '+file+' written to disk.'

IF a.default EQ 1 THEN BEGIN
  Xop_SetEnv,'XPLOT_DEFAULTS='+file
  itmp = Dialog_Message(/Info,Dialog_parent=group, $
    ['The attributes in the save file: ',file,$
     'will be used as default in the current XOP/Xplot session.',$
     '',$
     '(Done by setting the XPLOT_DEFAULTS environment. To unset it,',$
     'use Xplot File/Xplot Attributes File/Unset Session Default.)' ])
ENDIF

END ; Xplot_WriteAttributes
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_QUIT 
;	PURPOSE: Quits the Xplot window and destroy the internal handlers. 
;	CALLING SEQUENCE: XPLOT_QUIT,parent 
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;	MODIFICATION HISTORY:
;		97-09-11 srio@esrf.fr initial version from xplot code
;
;-
PRO xplot_quit,wid

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XPLOT_QUIT: error caught: '+!error_state.msg)
  catch, /cancel
  if n_elements(state) NE 0 then widget_control, wid, SET_UVALUE = state, $
    /NO_COPY
  return
endif
widget_control, wid, GET_UVALUE = state, /NO_COPY

widget_control,state.wids.draw,get_value=windownum
wset,windownum
xyouts,'!3'
widget_control,wid,/DESTROY
;for i=0,n_elements(state.xpdt.sysvar)-1 do $
;	tmp = execute(state.xpdt.sysvar_old(i))

;kill handles if window has not been cloned or is not a clone
if state.cloned EQ 0 then begin
  if keyword_set(state.dthbuffer) then $
    for i=0,n_elements(state.dthbuffer)-1 do $
      if handle_info(state.dthbuffer(i)) then $
      handle_free,state.dthbuffer(i)
  if keyword_set(state.dthcurrent) then $
    if handle_info(state.dthcurrent) then $
    handle_free,state.dthcurrent
  if handle_info(state.spec.handle) then $
    handle_free,state.spec.handle
endif
end ; xplot_quit

;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_CLEARDATA 
;	PURPOSE: clears graphs in xplot
;	CALLING SEQUENCE: XPLOT_CLEARDATA,parent 
;	INPUTS:
;	KEYWORD PARAMETERS:
;		CONFIRM = When set, display a confirmation window.
;	MODIFICATION HISTORY:
;		97-12-03 srio@esrf.fr initial version from xplot code
;
;-
PRO XPLOT_CLEARDATA,id,confirm=confirm

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XPLOT_CLEARDATA: error caught: '+!error_state.msg)
  catch, /cancel
  goto,out
endif
;

widget_control, id , GET_UVALUE = state, /NO_COPY

if keyword_set(confirm) then begin
  tmp = widget_message(dialog_parent=id,$
    /CANCEL,/DEFAULT_CANCEL,$
    ['This options deletes all your Xplot graph and ',$
    'initialize command.','Please confirm:'])
  if tmp EQ 'Cancel' then goto,out
endif
;
;kill handles if window has not been cloned or is not a clone
;
if state.cloned EQ 0 then begin
  if keyword_set(state.dthbuffer) then $
    for i=0,n_elements(state.dthbuffer)-1 do $
    if handle_info(state.dthbuffer(i)) then $
    handle_free,state.dthbuffer(i)
  if keyword_set(state.dthcurrent) then $
    if handle_info(state.dthcurrent) then $
    handle_free,state.dthcurrent
  ;if handle_info(state.spec.handle) then $
  ;  handle_free,state.spec.handle
endif
;
; cleans the buffer
;
xpstr_update,state,/nobuffer,/nodthcurrent,/nodthbuffer,$
  /noexebuffer
;
; reset some valuse
;
state.xpdt.shift=[0.,0.]
state.xpdt.ErrPlot=[0,0]

tmpin = xplot_defaults('xpdt_titles')
tmpout = state.xpdt_titles
copy_structure,tmpin,tmpout
state.xpdt_titles = tmpout

tmpin = xplot_defaults('xpdt_limits')
tmpout = state.xpdt_limits
copy_structure,tmpin,tmpout

state.xpdt_limits = tmpout
tmpin = xplot_defaults('xpdt')
tmpout = state.xpdt
copy_structure,tmpin,tmpout
state.xpdt= tmpout

tmpin = xplot_defaults('strcontrols')
tmpout = state.strcontrols
copy_structure,tmpin,tmpout
state.strcontrols= tmpout

; added srio 2002/03/08
state.fileinfo.name='<none>'
state.fileinfo.dir=''
state.fileinfo.mtime=long64(0)

; 
; erase drawing window
;
erase

;
; reset labels
;
widget_control,state.wids.ncol,SET_VALUE= '0'
widget_control,state.wids.xcol,SET_VALUE= '0'
widget_control,state.wids.ycol,SET_VALUE= '0'
widget_control,state.wids.saved,SET_VALUE= '0'
;
; close control panel if open
;
if widget_info(state.wids.controls(0),/valid_id) then begin
  widget_control,state.wids.showcontrols, $
  set_value='Show Controls'
  widget_control,state.wids.controls(0),/DESTROY
endif
;
; set off spec
;
if state.spec.on then begin
  widget_control,state.spec.scanbase,/destroy, bad_id=bad
  state.spec.on = 0
  xplot_mappanels,state.wids,state.spec
endif
;
; reset message and window title
;
widget_control,state.wids.message,set_value=$
'Xplot window and data cleaned up.'
Widget_Control,state.wids.main,TLB_Set_Title= 'Xplot '+xplot_version()

out:
widget_control, id , SET_UVALUE = state, /NO_COPY
end ;xplot_cleardata

;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_SETLIMITS
;	PURPOSE: Sets the xrange and/or yrange to xplot.
;	CALLING SEQUENCE: XPLOT_SETLIMITS,parent
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named 
;		variable)
;
;	KEYWORD PARAMETERS:
;		XRANGE = [xmin,xmax] The extrema in X
;		YRANGE = [ymin,ymax] The extrema in Y
;		XSTYLE = The XSTYLE Graphic Keyword (see IDL doc)
;		YSTYLE = The YSTYLE Graphic Keyword (see IDL doc)
;		NOREFRESH set this keyword to avoid refreshing the graph
;			after the loading process. Useful when loading
;			many curves.
;
;	MODIFICATION HISTORY:
;		97-12-03 srio@esrf.fr initial version from xplot code.
;		99-03-10 srio@esrf.fr adds xplot_setlimits_interactive
;			for non-modal interactie change of limits.
;
;-

;
;=========================================================================
;
FUNCTION Xplot_SetLimits_UpdateXplotState,state,xpdt_limits, $
  Group=group
;
; updates the xplot state by using a modified xpdt_limits structure
;
catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!error_state.msg
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'XPLOT_SETLIMITS_UPDATEXPLOTSTATE: error caught: '+!error_state.msg)
   catch, /cancel
   on_error,2
   RETURN,0
endif

stateNew = state

stateNew.xpdt_limits = xpdt_limits

stateNew.xpdt.sysvar(2) = '!x.range = [' + $
  strcompress(xpdt_limits.Xmin,/rem)+','+ $
  strcompress(xpdt_limits.Xmax,/rem)+']'
stateNew.xpdt.sysvar(3) = '!y.range = [' + $
  strcompress(xpdt_limits.Ymin,/rem)+','+ $
  strcompress(xpdt_limits.Ymax,/rem)+']'

xtmp = [ xpdt_limits.xstyle0[0],xpdt_limits.xstyle1[0],xpdt_limits.xstyle2[0],$
  xpdt_limits.xstyle3[0],xpdt_limits.xstyle4[0] ]
xval = 0
for i=0,n_elements(xtmp)-1 do xval=xval+long(xtmp[i])*(2^i)

ytmp = [ xpdt_limits.ystyle0[0],xpdt_limits.ystyle1[0],xpdt_limits.ystyle2[0],$
  xpdt_limits.ystyle3[0],xpdt_limits.ystyle4[0] ]
yval = 0
for i=0,n_elements(ytmp)-1 do yval=yval+long(ytmp[i])*(2^i)

stateNew.xpdt.sysvar(4) = '!x.style = '+strcompress(xval,/rem)
stateNew.xpdt.sysvar(5) = '!y.style = '+strcompress(yval,/rem)

RETURN,stateNew
END ; Xplot_SetLimits_UpdateXplotState

;
;=========================================================================
;
PRO Xplot_SetLimits_Interactive_Event,event

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error,Dialog_Parent=event.top,$
    'XPLOT_SETLIMITS_INTERACTIVE_EVENT: '+'error caught: '+!error_state.msg)
  Catch, /Cancel
  GoTo,out
ENDIF


action = ''
CASE Tag_Names(event,/Structure_Name) OF
  'WIDGET_BUTTON':BEGIN
    val = ''
    Widget_Control,event.id,Get_value=val
    CASE val OF
      'Dismiss': BEGIN
	Widget_Control,event.top,/Destroy
	RETURN
	END
      'Apply': BEGIN
	action = 'update'
	END
      'Help': BEGIN
	XDisplayFile1,Text=XPlot_Text('LIMITS'),Group=event.top,$
	  Title='Help: Xplot: Set Limits'
	RETURN
	END
       else:
    ENDCASE
    END
  else:action='update'
ENDCASE

stateid = Widget_Info(event.handler,/Child)
Widget_control, stateid,  Get_uValue=state,/No_copy

IF action EQ 'update' THEN BEGIN
  tmp=0
  FOR i=0,N_Elements(state.widsArr)-1 DO BEGIN
    Widget_Control,state.widsArr[i],Get_Value=tmp
    str = state.str
    myField=state.str.(i)
    myField[0]=tmp[0]
    state.str.(i)=myField
  ENDFOR
  ; 
  ; Update Xplot Structure 
  ;
  IF Widget_Info(state.XplotId ,/Valid_Id) THEN BEGIN
    Widget_control, state.XplotId , GET_UVALUE = xplotState, /NO_COPY
    tmp = Xplot_SetLimits_UpdateXplotState(xplotState,state.str)
    IF Type(tmp) NE 8 THEN BEGIN
      itmp = Dialog_Message(Dialog_Parent=event.top,/Error,$
	'XPLOT_SETLIMITS_INTERACTIVE_EVENT: Error updating the xplot state')
      GoTo,out
    ENDIF ELSE xplotState=tmp
    Xplot_Plot,xplotState
    Widget_Control, state.XplotId , SET_UVALUE = xplotState, /NO_COPY
  ENDIF
ENDIF

out:
IF Type(stateid) EQ 3 AND Type(state) EQ 8 THEN $
  Widget_control, stateid,  Set_uValue=state,/No_copy

END ; Xplot_SetLimits_Interactive_Event

;
;============================================================================
;
PRO Xplot_SetLimits_Interactive, id , Group=group, No_Block=no_Block

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XPLOT_SETLIMITS_INTERACTIVE: error caught: '+!error_state.msg)
  catch, /cancel
  RETURN
endif

;
; create structure with data
;
IF N_Params() EQ 1 THEN BEGIN
  Widget_control, id , GET_UVALUE = state, /NO_COPY
  str0 = state.xpdt_limits
  Widget_control, id , SET_UVALUE = state, /NO_COPY
ENDIF ELSE BEGIN
  str0 = Xplot_Defaults('xpdt_limits')
ENDELSE

wTop = Widget_Base(Title='Xplot limits',/Col) 
wtmp = Widget_Base(wTop) ; to store state

wBaseButtons = Widget_Base(wTop,/Row)
  wtmp = Widget_Button(wBaseButtons,Value='Dismiss')
  wtmp = Widget_Button(wBaseButtons,Value='Apply')
  wtmp = Widget_Button(wBaseButtons,Value='Help')


wBaseXY = Widget_Base(wTop,/ROW)

widsArr =  LonArr(14)
wBaseX = Widget_Base(wBaseXY,/Col,/Frame)
   wtmp=Widget_Label(wBaseX,Value='X axis')
   widsArr[0]=CW_Field(wBaseX,Value=str0.xmin,/Floating,Row=1,Title='min',$
	/Return_Events)
   widsArr[1]=CW_Field(wBaseX,Value=str0.xmax,/Floating,Row=1,Title='max',$
	/Return_Events)
   widsArr[2]=CW_DropList(wBaseX,Value=str0.xStyle0,Title='Boundaries        ')
   widsArr[3]=CW_DropList(wBaseX,Value=str0.xStyle1,Title='Extended range    ')
   widsArr[4]=CW_DropList(wBaseX,Value=str0.xStyle2,Title='Axis & annotations')
   widsArr[5]=CW_DropList(wBaseX,Value=str0.xStyle3,Title='Box               ')
   widsArr[6]=CW_DropList(wBaseX,Value=str0.xStyle4,Title='Origin ')

wBaseY = Widget_Base(wBaseXY,/Col,/Frame)
   wtmp=Widget_Label(wBaseY,Value='Y axis')
   widsArr[7]=CW_Field(wBaseY,Value=str0.Ymin,/Floating,Row=1,Title='min',$
	/Return_Events)
   widsArr[8]=CW_Field(wBaseY,Value=str0.Ymax,/Floating,Row=1,Title='max',$
	/Return_Events)
   widsArr[9]=CW_DropList(wBaseY,Value=str0.YStyle0,Title='Boundaries         ')
   widsArr[10]=CW_DropList(wBaseY,Value=str0.YStyle1,Title='Extended range    ')
   widsArr[11]=CW_DropList(wBaseY,Value=str0.YStyle2,Title='Axis & annotations')
   widsArr[12]=CW_DropList(wBaseY,Value=str0.YStyle3,Title='Box               ')
   widsArr[13]=CW_DropList(wBaseY,Value=str0.YStyle4,Title='Origin ')


Widget_Control,wTop,/Realize 

IF N_Elements(id) EQ 0 THEN id=0L

state = {widsArr:widsArr, str:str0, XplotId:id }

Widget_control,Widget_Info(wTop,/CHILD),set_uValue=state,/No_copy

XManager, 'Xplot_SetLimits_Interactive', wTop, GROUP_LEADER=group, $
  No_Block=no_block

END ; Xplot_SetLimits_Interactive

;
;=========================================================================
;
PRO xplot_setlimits,id,xrange=xrange,yrange=yrange,xstyle=xstyle,$
	ystyle=ystyle,norefresh=norefresh

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XPLOT_SETLIMITS: error caught: '+!error_state.msg)
  catch, /cancel
  goto,out
endif
interactive=1
if keyword_set(xrange) then interactive = 0
if keyword_set(yrange) then interactive = 0
if keyword_set(xstyle) then interactive = 0
if keyword_set(ystyle) then interactive = 0
;
IF interactive  THEN BEGIN
  Xplot_SetLimits_Interactive,id
  RETURN
ENDIF

widget_control, id , GET_UVALUE = state, /NO_COPY
action=''
if keyword_set(xrange) then begin
	state.xpdt_limits.Xmin = xrange(0)
	state.xpdt_limits.Xmax = xrange(1)
	state.xpdt.sysvar(2) = '!x.range = [' + $
		strcompress(state.xpdt_limits.Xmin,/rem)+','+ $
		strcompress(state.xpdt_limits.Xmax,/rem)+']'
endif
if keyword_set(yrange) then begin
	state.xpdt_limits.Ymin = Yrange(0)
	state.xpdt_limits.Ymax = Yrange(1)
	state.xpdt.sysvar(3) = '!Y.range = [' + $
		strcompress(state.xpdt_limits.Ymin,/rem)+','+ $
		strcompress(state.xpdt_limits.Ymax,/rem)+']'
endif
if keyword_set(xstyle) then begin
	state.xpdt.sysvar(4)='!x.style = '+strcompress(xstyle,/rem)
	tmp = reverse(long(ess_base1(xstyle,2,/return_array)))
	tmp1 = intarr(5)
	tmp1[0:(n_elements(tmp)-1)]=tmp
	tmp1=strcompress(tmp1,/rem)
	state.xpdt_limits.Xstyle0[0]=tmp1[0]
	state.xpdt_limits.Xstyle1[0]=tmp1[1]
	state.xpdt_limits.Xstyle2[0]=tmp1[2]
	state.xpdt_limits.Xstyle3[0]=tmp1[3]
	state.xpdt_limits.Xstyle4[0]=tmp1[4]
endif
if keyword_set(ystyle) then begin
	state.xpdt.sysvar(5)='!y.style = '+strcompress(ystyle,/rem)
	tmp = reverse(long(ess_base1(ystyle,2,/return_array)))
	tmp1 = intarr(5)
	tmp1[0:(n_elements(tmp)-1)]=tmp
	tmp1=strcompress(tmp1,/rem)
	state.xpdt_limits.Ystyle0[0]=tmp1[0]
	state.xpdt_limits.Ystyle1[0]=tmp1[1]
	state.xpdt_limits.Ystyle2[0]=tmp1[2]
	state.xpdt_limits.Ystyle3[0]=tmp1[3]
	state.xpdt_limits.Ystyle4[0]=tmp1[4]
endif
state.xpdt.zoom_on=0
if not(keyword_set(norefresh)) then Xplot_plot,state

out:
widget_control, id , SET_UVALUE = state, /NO_COPY

END ; Xplot_SetLimits
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_LOADFILE 
;	PURPOSE: load a file or variable in xplot
;	CALLING SEQUENCE: XPLOT_LOADFILE,parent [,input,spec=spec]
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;
;		INPUT  it can either be:
;		   1) a string with the ASCII file name
;		   2) an idl variable with a multi-column set (i.e.  fltarr(2,100) )
;	KEYWORD PARAMETERS:
;		SPEC set this keyword to a SPEC file name. In this case the
;		file is loaded and interpreted as SPEC file. IN this case, 
;		"INPUT" is not used.
;		PLOTNAME set this keyword to a string that identifies the 
;			plot (to be used by plotmgr).
;		NOREFRESH set this keyword to avoid refreshing the graph
;			after the loading process. Useful when loading
;			many curves.
;		SCANID When using SPEC input, set this keyword to the 
;			scan id to be loaded (e.g., 'CURRENT').  Default='LAST'
;		WTTILE Title for the main window bar
;
;	MODIFICATION HISTORY:
;		97-09-11 srio@esrf.fr initial version from xplot code
;		97-12-03 srio@esrf.fr adds plotname keyword.
;		98-02-17 srio@esrf.fr adds norefresh keyword. Xplot v 2.5.2
;		01-02-23 srio@esrf.fr adds wTitle keyword. 
;
;-
PRO XPLOT_LOADFILE,id,input,spec=spec,plotname=plotname,norefresh=norefresh,$
  ScanId=scanId,Wtitle=wTitle

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  if !error_state.msg EQ $
   "Attempt to call undefined procedure/function: 'SPEC_ACCESS'." then begin
    spec_access_flag = -1
    txt='Could not check for SPEC file. Input taken as standard' 
    message,/info,txt
    itmp=Dialog_Message(/Info, txt)
    spec_loaded=0 
  endif else spec_loaded=1
  if spec_loaded EQ 0 then goto,CONTINUE1
  itmp=Dialog_Message(/Error, Dialog_Parent=id,$
    'XPLOT_LOADFILE: error caught: '+!error_state.msg)
  catch, /cancel
  goto,out
endif
;
widget_control, id , GET_UVALUE = state, /NO_COPY
IF NOT(Keyword_Set(wTitle)) THEN $
  Widget_Control,state.wids.main,TLB_Set_Title='Xplot '+xplot_version()

input_type = 'none'
state.plotname_current='<Variable>'
if n_params() GE 2 then begin
  if type(input) EQ 7 then begin
    file = input
    if checkfile(file) EQ 0 then begin
      itmp = widget_message(DIALOG_PARENT=state.wids.main,$
	'File not found: '+file,/error)
      goto,out
    endif else input_type = 'file'
    state.plotname_current=file
    state.fileinf.name=file
    IF Float(!version.release) GE 5.5 THEN state.fileinf.mtime = (File_Info(file)).mtime
  endif  else begin
    input_type = 'variable'
    spec_access_flag = 0
  endelse
endif
if keyword_set(spec) then begin
  ntype = type(spec)
  if ntype EQ 7 then begin
    spec_access_flag=spec_access(hspec,spec) 
    state.plotname_current=spec
    state.fileinf.name=spec
    IF Float(!version.release) GE 5.5 THEN state.fileinf.mtime = (File_Info(spec)).mtime
    IF NOT(Keyword_Set(wTitle)) THEN $
      Widget_Control,state.wids.main,TLB_Set_Title= $
      spec+' - '+'Xplot '+xplot_version()
  endif else begin
    hspec=spec
    spec_access_flag = spec_access(hspec)
    state.plotname_current='<SpecHandle>'
  endelse
  input_type = 'variable'
endif

path = ''
if input_type EQ 'none' then begin
		if state.fileinf.dir EQ '' then cd,current=oldpath else $
		  oldpath=state.fileinf.dir 
		file = Dialog_pickfile(GROUP=id,/NOCONF, /MUST_EXIST, $
                       TITLE='Select a data file',GET_PATH=path,PATH=oldpath)
		if file eq '' then GOTO,out
		input_type = 'file'
		state.plotname_current=file
endif

; commented by now
;		; Windows check *MSR 97/01/13*
;		if sdep() ne 'WINDOWS' then begin
;                  ; Check if ASCII
;                  spawn, 'file ' + file, aux, /sh
;		  ; add 'link' to allow soft links in SUN. MSR 27/11/95
;                  if ( (strpos(aux(0), 'text') eq -1) AND $ 
;                       (strpos(aux(0), 'link') eq -1) ) then begin
;                    tmp=widget_message('"'+file+'" is not an ASCII file.',/ERR)
;                    goto,OUT
;                  endif
;		endif
if input_type EQ 'file' then begin
                widget_control,/HOURGLASS
		state.fileinf.name=file
                IF Float(!version.release) GE 5.5 THEN  $
			state.fileinf.mtime = (File_Info(file)).mtime
		state.fileinf.dir=path
                handle_value, state.spec.handle, hspec
		; If spec_access.pro is not somewhere in the directories
		; listed in !path, then suppose ASCII input.
		; This allows to run xplot without needing to install
		; spec_access.
		; MSR 97/02/10 This section is commented because if spec_acess 
		; is precompiled with xop.xdr, this option does not work. 
		;
		;if sdep() EQ 'WINDOWS' then tmpf='SPEC_A~1.DAT' else $
		;	tmpf='spec_access.pro'
		;if filepath1(tmpf) NE '' then $
		;  spec_access_flag = spec_access(hspec, file) else $
		;  spec_access_flag = -1
		spec_access_flag = spec_access(hspec, file) 
endif
CONTINUE1:
if input_type EQ 'file' or input_type EQ 'variable' then begin
                if spec_access_flag le 0 then begin  ; ASCII input
		   if input_type EQ 'file' then begin
		     tmp = rascii(file)
		     if n_elements(tmp) eq 1 then begin
		      tmp = widget_message(DIALOG_PARENT=state.wids.main,$
			'Error reading ASCII file.',/ERROR)
		      goto,OUT
		     endif
		   IF NOT(Keyword_Set(wTitle)) THEN $
		        Widget_Control,state.wids.main,TLB_Set_Title= $
			file+' - '+'Xplot '+xplot_version()
		   endif else begin
		     tmp = input
		     if n_elements(tmp) eq 1 then begin
		      tmp = widget_message(DIALOG_PARENT=state.wids.main,$
			'Wrong input to xplot.',/ERROR)
		      goto,OUT
		     endif
		   endelse
                   state.spec.on = 0
		   Ncol = n_elements(tmp(*,0))
		   xcol = state.xpdt.Xcol
		   ycol = state.xpdt.Ycol
		   if xcol GT Ncol or xcol LT 1 then begin
			xcol = 1
			state.xpdt.Xcol = xcol
		   endif
		   if ycol GT Ncol or ycol LT 1 then begin
			ycol = Ncol
			state.xpdt.Ycol = ycol
		   endif
		   ;state.xpdt.Xcol = 1
		   ;state.xpdt.Ycol = Ncol
		   if widget_info(state.wids.xcol,/NAME) EQ 'DROPLIST' $
		     then begin
		     ; column names only changed if Ncol is changed
                     IF N_Elements(state.coltitles) NE Ncol THEN BEGIN
		     coltitles = 'Col: '+$
			strcompress((sindgen(Ncol+1))(1:Ncol),/rem)
		     xpstr_update,state,coltitles=coltitles
		     widget_control,state.wids.xcol,set_value=coltitles,$
			set_droplist_select=xcol-1
			;set_droplist_select=0
		     widget_control,state.wids.ycol,set_value=coltitles,$
			set_droplist_select=ycol-1
			;set_droplist_select=ncol-1
                     ENDIF
		   endif else begin
		     widget_control,state.wids.xcol,SET_VALUE=$
                                                     strcompress(xcol,/REM)
		     widget_control,state.wids.ycol,SET_VALUE=$
                                                     strcompress(ycol,/REM)
		   endelse
                   widget_control,state.spec.scanbase,/destroy, bad_id=bad
		   state.xpdt.shift=[0.,0.]
		   state.xpdt.ErrPlot=[0,0]
		   widget_control,state.wids.ncol,SET_VALUE= $
			                              strcompress(Ncol,/REM)
                   xplot_mappanels, state.wids, state.spec
		   if input_type EQ 'file' then $
			widget_control,state.wids.message,set_value=$
			'ASCII data file ('+file+ $
			') read succesfully' else $
			widget_control,state.wids.message,set_value=$
			'ASCII data received.'
		endif else begin     ; spec input
                   mca = 0
                   ;fscan = (spec_scan(hspec, '-', /no_empty, /return_index))(0)
                   ;fscan=(spec_scan(hspec, 'LAST', /no_empty,/return_index))(0)
		   IF N_Elements(scanId) EQ 0 THEN scanId='LAST'
                   fscan=(spec_scan(hspec, scanId, /no_empty,/return_index))(0)
                   if fscan eq -1 then begin
                      ;fscan = (spec_scan(hspec, '-', /mcdata,/return_index))(0)
                      fscan=(spec_scan(hspec, 'LAST', /mcdata,/return_index))(0)
                      if fscan eq -1 then begin
			 scanList = Spec_Access(hspec,/List)
			 scanList = Spec_Scan(hspec,ScanList,/No_Empty,$
				/Return_Index)
			 fscan = scanList(N_elements(scanList)-1)
			 ;
			 ; in the case that the last scan contains no data
			 ; then load a previous one
			 ;
                         if fscan[0] eq -1 then begin
		            tmp = $
			    Dialog_Message(DIALOG_PARENT=state.wids.main,$
			    'XPLOT_LOADFILE: No data found in SPEC scan ('+$
			    StrCompress(fscan,/Remove_All)+')',/ERROR)
		           goto,OUT
			 endif
                      endif
                      ;mca = 1
                   endif
                   handle_value, state.spec.handle, hspec, /set
                   tmp = xplot_readspec(state, fscan, mca, /OLDCOL)
		   ; if error, check only mca scans
		   IF N_Elements(tmp) EQ 1 AND tmp[0] EQ 0 THEN BEGIN
		     mca=1
                     tmp = xplot_readspec(state, fscan, mca, /OLDCOL)
		     IF N_Elements(tmp) EQ 1 AND tmp[0] EQ 0 THEN BEGIN
		       Message,'Scan contains no data.'
		     ENDIF
		   ENDIF
		   if n_elements(tmp) eq 1 then begin
		      goto,OUT
		   endif
                   state.spec.on = 1
		   widget_control,state.wids.message,set_value=$
			'SPEC data file ('+hspec.specfile+$
			') read succesfully'
		   IF NOT(Keyword_Set(wTitle)) THEN $
		        Widget_Control,state.wids.main,TLB_Set_Title= $
			hspec.specfile+' - '+'Xplot '+xplot_version()
		  ; set default titles to #S -1 -1
		  if strcompress(state.xpdt_titles.title,/rem) EQ '' then $
			state.xpdt_titles.title='#S'
		  if strcompress(state.xpdt_titles.xtitle,/rem) EQ '' then $
			state.xpdt_titles.xtitle='-1'
		  if strcompress(state.xpdt_titles.ytitle,/rem) EQ '' then $
			state.xpdt_titles.ytitle='-1'
                end 
endif
		IF Keyword_Set(wTitle) THEN $
		        Widget_Control,state.wids.main,TLB_Set_Title=wTitle
		if keyword_set(state.dthcurrent) then $
			handle_free,state.dthcurrent
		state.dthcurrent = handle_create(value=tmp)

                xplot_mappanels, state.wids, state.spec
		; reinitialize spec pannel
                widget_control,state.spec.scanbase,bad_id=tmp, send_event=$
                   {ID:state.spec.scanbase, TOP:0L, HANDLER:0L}
		; reset mesh flag
		;state.xpdt.mesh.flag(0)='0'
		if not(keyword_set(norefresh)) then xplot_plot,state
out:
if keyword_set(plotname) then state.plotname_current = plotname
widget_control, id , SET_UVALUE = state, /NO_COPY
end ;xplot_loadfile
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_SAVECURRENT 
;	PURPOSE: to save the current displayed plot in xplot (same
;		effect as pressing the "Save Current" button on the xplot
;		window)
;	CALLING SEQUENCE: XPLOT_SAVECURRENT,parent 
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;
;	MODIFICATION HISTORY:
;		97-09-11 srio@esrf.fr initial version from xplot code
;
;-
PRO xplot_savecurrent,wid_top

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XPLOT_SAVECURRENT: error caught: '+!error_state.msg)
  catch, /cancel
  goto,out
endif

widget_control, wid_top , GET_UVALUE = state, /NO_COPY

if strcompress(state.xpdt.mesh.flag(0),/rem) EQ '1' then begin
  itmp = widget_message(dialog_parent=state.wids.main, /error,$
    'SAVE option not allowed when'+$
    ' View/Mesh option is selected. Not saved.')
  goto,out
endif
xplot_plot,state,/SAVE
widget_control,state.wids.saved,get_value=tmp
widget_control,state.wids.saved,set_value= $
strcompress(long(tmp)+1,/rem)
widget_control,state.wids.Message,set_value= $
'Current set saved. Use "Plot Mgr" button to'+$
' delete or move saved plots.'
widget_control,state.wids.plotmgr,/SENSITIVE
;commented to avoid bug when saving.
;state.xpdt.Kind = 2

out:
widget_control, wid_top , SET_UVALUE = state, /NO_COPY
end ; xplot_savecurrent
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_CHANGECOL 
;	PURPOSE: to change the X or Y displayed column in xplot
;		It works aso with SPEC files, including MCA. In the last
;		case the (X) column for Channels is ncol+1 and for Energy
;		is ncol+2, being ncol the number of channels.
;	CALLING SEQUENCE: XPLOT_CHANGECOL,parent ,XCOL=xcol, YCOL=ycol
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;
;	KEYWORD PARAMETERS [USE AT LEAST ONE]:
;		XCOL The column index for X (starting from col 1, not 0)
;		YCOL The column index for Y (starting from col 1, not 0)
;		NOREFRESH set this keyword to avoid refreshing the graph
;			after the loading process. Useful when loading
;			many curves.
;
;	MODIFICATION HISTORY:
;		97-09-11 srio@esrf.fr initial version from xplot code
;
;-
PRO xplot_changecol,input,xcol=pxcol,ycol=pycol,norefresh=norefresh

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XPLOT_CHANGECOL: error caught: '+!error_state.msg)
  catch, /cancel
  goto,out
endif

eventUVal=0
case type(input) of
8: begin
   event = input
   widget_control, event.top , GET_UVALUE = state, /NO_COPY
   Widget_Control, event.id, get_UValue=eventUVal
   wid_top=event.top
   end
3: begin
   widget_control, input, GET_UVALUE = state, /NO_COPY
   wid_top=input
   eventuval = ''
   if keyword_set(PXCOL) then begin
	eventUVal=[eventUVal,'XCOL']
	XCOL = PXCOL
   endif
   if keyword_set(PYCOL) then begin
	eventUVal=[eventUVal,'YCOL']
	YCOL = PYCOL
   endif
   my_value = 0
   if n_elements(eventuval) GE 2 then $
     eventuval=eventuval(1:n_elements(eventuval)-1)
   end
else: begin
   message,/info,'Usage: xplot_changecol, event (or parent_id)'
   return
   end
endcase

widget_control,state.wids.ncol,GET_VALUE=temp2
ncol = fix(temp2(0))

for i=0,n_elements(eventUVal)-1 do begin

  event_tmp = eventUVal(i)
  if type(input) EQ 3 then begin
     command = 'my_value = '+event_tmp
     itmp = execute(command)
  endif

  case event_tmp of
	'XCOL':	BEGIN
		if not(keyword_set(pxcol)) then begin
                  if state.spec.on then begin
                   if state.spec.mca then begin
                      widget_control,event.id,GET_VALUE=temp1
                      xcol = ncol + 1 + temp1
                   endif else begin
                      xcol = event.index + 1
                   endelse
                  endif else begin
                   if widget_info(state.wids.xcol,/NAME) EQ 'TEXT' then begin
			widget_control,event.id,GET_VALUE=temp1
		        xcol = fix(temp1(0))
		   endif else begin
		        xcol = fix(event.index)+1
		   endelse
                  endelse
		endif 
		if not state.spec.mca and xcol gt ncol then begin
		  tmp = widget_message(DIALOG_PARENT=state.wids.main,$
		  'Error selecting column.',/ERROR)
		  if not(keyword_set(PXCOL)) then $
                  widget_control,event.id,SET_VALUE= $
			strcompress(state.xpdt.Xcol,/rem)
		  goto,out
		endif else begin
                  state.xpdt.Xcol = xcol
		  if keyword_set(pxcol) then begin
                    if state.spec.on then begin
                     if state.spec.mca then begin
                        w_tmp = state.wids.xcolmca
                     endif else begin
                        w_tmp = state.wids.xcolspec
                     endelse
                    endif else w_tmp = state.wids.xcol
		    case widget_info(w_tmp,/NAME) of
		      'TEXT':widget_control,w_tmp,SET_VALUE=strcompress(xcol,/rem)
		      'DROPLIST': widget_control,w_tmp,set_droplist_select=xcol-1
		      'BASE': widget_control,w_tmp,set_value=xcol-1-ncol
		      else: goto,out
		    endcase
		  endif
		endelse
		END
	'YCOL':	BEGIN
		if not(keyword_set(pycol)) then begin
                  if state.spec.on and not state.spec.mca then begin
                   ycol = event.index + 1
                  endif else begin
                   if widget_info(state.wids.ycol,/NAME) EQ 'TEXT' OR $
		   (state.spec.mca EQ 1) then begin
			widget_control,event.id,GET_VALUE=temp1
		        ycol = fix(temp1(0))
		   endif else begin
		        ycol = fix(event.index)+1
		   endelse
                   ;widget_control,event.id,GET_VALUE=temp1
		   ;ycol = fix(temp1(0))
                  endelse
		endif
		if ycol gt ncol or ycol le 0 then begin
		  tmp = widget_message(DIALOG_PARENT=state.wids.main,$
		  'Error selecting column.',/ERROR)
		  if not(keyword_set(PYCOL)) then $
                  widget_control,event.id,SET_VALUE= $
			strcompress(state.xpdt.Ycol,/rem)
		  goto,out
		endif else begin
                  state.xpdt.Ycol = ycol
		  if keyword_set(pycol) then begin
                    if state.spec.on then begin
                     if state.spec.mca then begin
                        w_tmp = state.wids.ycolmca
                     endif else begin
                        w_tmp = state.wids.ycolspec
                     endelse
                    endif else w_tmp = state.wids.ycol
		    case widget_info(w_tmp,/NAME) of
		      'TEXT':widget_control,w_tmp,SET_VALUE=strcompress(ycol,/rem)
		      'DROPLIST': widget_control,w_tmp,set_droplist_select=ycol-1
		      else: goto,out
		    endcase
		  endif
		endelse
		END
  else:
  endcase
endfor
if not(keyword_set(norefresh)) then xplot_plot,state
out:
widget_control, wid_top , SET_UVALUE = state, /NO_COPY
end ; xplot_changecol
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_EXEBUFFER 
;	PURPOSE: to set a buffer or to add commands to the buffer
;		The EXEbuffer is a set of IDL commands that are executed
;		after the plot. It is used to make labels, annotations, etc.
;	CALLING SEQUENCE: XPLOT_EXEBUFFER,parent, ADDBUFFER=addbuffer, $
;		SETBUFFER=setbuffer 
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;
;	KEYWORD PARAMETERS [MUST USE ONLY ONE (MUTUALLY EXCLUSIVE)]:
;		ADDBUFFER A string array with the commands to be added to the buffer
;
;		SETBUFFER A string array with the commands to be set to the buffer.
;			If the buffer contained commands, they are deleted before.
;		NOREFRESH set this keyword to avoid refreshing the graph
;			after the loading process. Useful when loading
;			many curves.
;
;	MODIFICATION HISTORY:
;		97-09-11 srio@esrf.fr initial version from xplot code
;
;-
PRO xplot_exebuffer,input,setbuffer=setbuffer,addbuffer=addbuffer,$
  norefresh=norefresh

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XPLOT_EXEBUFFER: error caught: '+!error_state.msg)
  catch, /cancel
  goto,out
endif

eventUVal=' '
case type(input) of
8: begin
   event = input
   widget_control, event.top , GET_UVALUE = state, /NO_COPY
   Widget_Control, event.id, get_UValue=eventUVal
   wid_top=event.top
   end
3: begin
   if keyword_set(setbuffer) then begin
     eventUVal = 'SETBUFFER'
     buffer = setbuffer
   endif
   if keyword_set(addbuffer) then begin
     eventUVal  = 'ADDBUFFER'
     buffer = addbuffer
   endif
   if buffer(0) EQ '' then begin
     message,/info,'Empty buffer'
     return
   endif
   if not(keyword_set(setbuffer)) AND not(keyword_set(addbuffer)) then begin
     message,/info,$
	'Usage: xplot_exebuffer,wid_top,setbuffer=setbuffer,addbuffer=addbuffer'
     message,/info,'       One keyword MUST be used'
     return
   endif
   widget_control, input, GET_UVALUE = state, /NO_COPY
   wid_top=input
   end
else: begin
   message,/info,'Usage: xplot_exebuffer, event (or parent_id)'
   return
   end
endcase

case eventUVal of
	'SETBUFFER': xpstr_update,state,EXEBUFFER=buffer
	'ADDBUFFER':	BEGIN
		tmp = state.exebuffer
		if tmp(0) NE '' then newbuffer=[tmp,buffer] else $
			newbuffer=buffer
                xpstr_update,state,EXEBUFFER=newbuffer
		END
	'EXEBUFFERCLEAN':	BEGIN
		tmp = widget_message(DIALOG_PARENT=wid_top, /CANCEL,$
		/DEFAULT_CANCEL, $
		  ['This option deletes all Legens, Labels',$
		  'and any other command stored in the buffer and',$
		  'present in the graph. Please comfirm'])
		if tmp EQ 'Cancel' then goto,out
		xpstr_update,state,/NOEXEBUFFER
		END
	'EXEBUFFEREDIT':	BEGIN
		tmp = state.exebuffer
		if type(tmp(0)) EQ 2 then begin
		  itmp = widget_message(DIALOG_PARENT=wid_top,/CANCEL,$
			/DEFAULT_CANCEL,['The buffer is empty. ',$
			'Do you want to create it?'])
		  if itmp EQ 'Cancel' then goto,out
		  tmp = $
		    '; please place below IDL commands to be executed after the plot'
		endif
		XDisplayFile1,text=tmp,group=wid_top,/MODAL,$
		  title='Editing EXE buffer',help=xplot_text('EXEBUFFEREDIT'),$
		  action=action
		if action EQ 'DONT' then goto,out
		if tmp(0) EQ '' then xpstr_update,state,/NOEXEBUFFER else $
			xpstr_update,state,EXEBUFFER=tmp
		xpstr_update,state,EXEBUFFER=tmp
		END
	'SETLABELS':	BEGIN
		buffer = 0
		xxyouts,BUFFER=buffer,GROUP=event.top
		if not(keyword_set(buffer)) then goto,out
		if keyword_set(state.exebuffer) then $
			exebuffer =  [state.exebuffer,buffer] else $
			exebuffer = buffer
		xpstr_update,state,EXEBUFFER=exebuffer
		END
	'SETLEGEND':	BEGIN
		command = 0
		xlegend,COMMAND=command,GROUP=event.top
		if command EQ '' then goto,out
		if keyword_set(state.exebuffer) then $
			buffer =  [state.exebuffer,command] else $
			buffer = command
		xpstr_update,state,EXEBUFFER=buffer
		END
	else:
ENDCASE


if not(keyword_set(norefresh)) then xplot_plot,state
out:
widget_control, wid_top , SET_UVALUE = state, /NO_COPY
end ; xplot_exebuffer
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_ERRORBARS 
;	PURPOSE: create error bars on a given plot
;	CALLING SEQUENCE: XPLOT_ERRORBARS,parent, col1, col2
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named 
;			variable)
;		COL1 the column containing the bottom limit of the error bars
;		COL2 the column containing the upper limit of the error bars
;		NOREFRESH set this keyword to avoid refreshing the graph
;			after the loading process. Useful when loading
;			many curves.
;	MODIFICATION HISTORY:
;		97-09-11 srio@esrf.fr initial version from xplot code
;
;-
PRO xplot_errorbars,input,col1,col2,norefresh=norefresh

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XPLOT_ERRORBARS: error caught: '+!error_state.msg)
  catch, /cancel
  goto,out
endif

eventUVal=' '
case type(input) of
8: begin
   event = input
   widget_control, event.top , GET_UVALUE = state, /NO_COPY
   Widget_Control, event.id, get_UValue=eventUVal
   wid_top=event.top
   end
3: begin
   if n_params() NE 3 then begin
     message,/info,'Usage: xplot_errorbars,wid_top,col1,col2'
     return
   endif
   widget_control, input, GET_UVALUE = state, /NO_COPY
   wid_top=input
   end
else: begin
   message,/info,'Usage: xplot_errorbars, event (or parent_id)'
   return
   end
endcase


if eventuval EQ 'ERRPLOT' then BEGIN
		tmp = { Bottom:state.xpdt.ErrPlot(0), $
			Top:state.xpdt.ErrPlot(1) }
		XscrMenu,tmp,GROUP=event.top,/NOTYPE,ACTION=action, $
		  WTITLE='Xplot Error bar columns', Titles = $
		  ['Col with bottom data error:','Col with top data error:'],$
		  HELP=xplot_text('ERRORBARS')
		if action eq 'DONT' then goto,out
		  state.xpdt.ErrPlot = [tmp.bottom,tmp.top]
ENDIF ELSE state.xpdt.ErrPlot = [col1,col2]
if not(keyword_set(norefresh)) then xplot_plot,state
out:
widget_control, wid_top , SET_UVALUE = state, /NO_COPY
end ; xplot_errorbars
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_ADDCOL 
;	PURPOSE: load a numerical array and include it as new column in
;		Xplot.
;	CALLING SEQUENCE: XPLOT_ADDCOL,parent,input
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named 
;		variable)
;
;		INPUT  an array with the new column.
;
;	KEYWORD PARAMETERS:
;
;	MODIFICATION HISTORY:
;		97-10-21 srio@esrf.fr initial version slightly modifying
;			the old xplot_addcol.
;
;-
PRO xplot_addcol,id,yfit,norefresh=norefresh,group=group

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error,Dialog_Parent=group, $
    'XPLOT_ADDCOL: error caught: '+!error_state.msg)
  catch, /cancel
  goto,out
endif
if not(keyword_set(group)) then group=id
;
widget_control, id , GET_UVALUE = state, /NO_COPY
			plotfile,state.dthcurrent,tmpset,/NODATA
			ncol = n_elements(tmpset(*,0))
			npts = n_elements(tmpset(0,*))
			newset = fltarr(ncol+1,npts)
nyfit=n_elements(yfit)
if nyfit NE npts then begin
  itmp = Dialog_Message(/QUESTION,dialog_parent=group,$
	['Error in XPLOT_ADDCOL: The column to be added has a different',$
	'number of points than the existing columns.',$
	'This operation may introduce an unrealistic shift of the data.',$
	'Resize and continue?'])
  if itmp EQ 'No' then goto,out
  if nyfit GT npts then yfit = reform(yfit(0:npts-1)) else begin
     yfitnew = fltarr(npts)
     yfitnew(0:nyfit-1) = yfit
     yfit = reform(yfitnew)
  endelse
endif

			; if spec/mca insert instead appending...
			if state.spec.on AND state.spec.mca then begin
			  ycol = ncol-1
			  newset(0:ycol-2,*) = tmpset(0:ycol-2,*)
			  newset(ycol-1,*) = yfit
			  newset(ycol:ncol,*) = tmpset(ncol-2:ncol-1,*)
			  ncol = ncol-2
			  ncoladd='+2'
			endif else begin
			  newset(0:ncol-1,*) = tmpset
			  newset(ncol,*) = yfit
			  ncoladd=''
			  ycol = ncol+1
			endelse

			widget_control,state.wids.ncol,set_value = $
			  strcompress(ncol+1,/rem)+ncoladd

			if widget_info(state.wids.xcol,/NAME) EQ 'TEXT' $
			then begin
			; commented to avoid changing column 98/09/30
			;  widget_control,state.wids.ycol,set_value = $
			;  strcompress(ycol,/rem)
			endif else begin
			  tmp  = [state.coltitles, $
			   strcompress(n_elements(state.coltitles)+1, $
			   /rem)+': Fit']
			  xpstr_update,state,coltitles=tmp
			  widget_control,state.wids.xcol,set_value=tmp
			  widget_control,state.wids.ycol,set_value=tmp, $
			    set_droplist_select=ycol-1
			endelse


			;state.xpdt.Ycol = ycol
			handle_free,state.dthcurrent
			h1 = handle_create(value=newset)
			state.dthcurrent = h1

			; spec setup
			if state.spec.on EQ 1 then begin
			  labels = state.spec.labels+'!Fit'
			  state.spec.labels = labels
			  tmp = strparse(labels,'!',labelslist)
			  labelslist=strcompress((sindgen(n_elements( $
			    labelslist)+1))(1:n_elements(labelslist)) ) + $
			    ': '+labelslist
			  if tmp EQ -1 then begin
			    tmp = widget_message(dialog_parent=state.wids.main,$
				'Error (column separator)',/ERROR)
			    return
			  endif
			  if state.spec.mca then begin
			    ; commented to avoid changing column 98/09/30
      			    ;;widget_control,state.wids.ycolMCA,SET_VALUE=$
			    ;;	strcompress(ycol,/rem)
			  endif else begin
      			    widget_control,state.wids.xcolSPEC,SET_VALUE=$
				labelslist
			    ; changed to avoid changing column 98/09/30
      			    widget_control,state.wids.ycolSPEC,SET_VALUE=$
				labelslist, $ ;;, SET_DROPLIST_SELECT = ncol
				SET_DROPLIST_SELECT = state.xpdt.Ycol-1
			  endelse
			endif


			widget_control,state.wids.message,set_value = $
			  'Result of the fit has been placed'+$
			  ' in column '+strcompress(ycol,/rem)
			if not(keyword_set(norefresh)) then xplot_plot,state

out:
widget_control, id , SET_UVALUE = state, /NO_COPY
end ; xplot_addcol
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_PLOTMGR
;	PURPOSE: manage the saved set (delete or move to current).
;	CALLING SEQUENCE: XPLOT_PLOTMGR,parent [,selected ]
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;
;		SELECTED = the saved set number (starting from 1) to be
;		edited (moved or deleted).
;	KEYWORD PARAMETERS 
;		[THE FOLLOWING ONES ARE EXCLUSIVE. AT LEAST ONE IS MANDATORY]:
;		DELETE_SEL will delete the selected saved set.
;		DELETE_ALL will delete all the saved sets.
;		MOVE will move the selected saved set to the current set.
;		[GENERAL KWYWORDS]:
;		NOREFRESH set this keyword to avoid refreshing the graph
;			after the loading process. Useful when loading
;			many curves.
;
;	MODIFICATION HISTORY:
;		97-10-24 srio@esrf.fr initial version from xplot code
;
;-
;=========================================================================
PRO xplot_plotmgr_win_event,event

widget_control,event.id,get_uvalue=uval
widget_control,event.top,get_uvalue=state,/no_copy

case uval of
 'CANCEL': begin
	widget_control,state.wids.action,get_uval=tmp
	tmp.action='CANCEL'
	widget_control,state.wids.action,set_uval=tmp
	widget_control,event.top,/destroy
	return
	end
 'ACCEPT':begin
	widget_control,state.wids.action,get_uval=actionstr
	widget_control,state.wids.buttons,get_value=tmp
	case tmp of
	  0:begin
		if state.selected EQ 0 then goto,accept_error
		actionstr.action='DELETE_SEL'
		end
	  1:actionstr.action='DELETE_CUR'
	  2:actionstr.action='DELETE_ALL'
	  3:begin
		if state.selected EQ 0 then goto,accept_error
		actionstr.action='MOVE'
		end
	  else:actionstr.action='CANCEL'
	endcase
	actionstr.sel=state.selected
	widget_control,state.wids.action,set_uval=actionstr
	widget_control,event.top,/destroy
	return
	accept_error:
		itmp=widget_message(/ERROR,'Plase select a set',$
			dialog_parent=event.top)
	end
 'SELECT':begin
	state.selected=event.index+1
	end
 'HELP':begin
	  infotext = [' ',$
	   'The Plot Manager permits to modify the saved data sets in Xplot.', $
	   ' ',$
	   'It will either delete set(s) or promote a saved set to the ',$
	   'current set.',$
	   ' ',$
	   'The sets are named either with the file name from which the ',$
	   'data was loaded, or with a name given by the user (with',$
	   'Edit/PlotNames Menu). In addition, there is a unique identifier ',$
	   'for each set (The id number).']
	XDisplayFile1,text=infotext,group=event.top,title=$
	'Help on Plot Manager',/NoMenubar
	end
  else:
endcase

out:
widget_control,event.top,set_uvalue=state,/no_copy
end ; xplot_plotmgr_win_event
;=========================================================================
PRO xplot_plotmgr_win,buffer,current,$
	group=group,action=action,selected=selected
;
;
wbase = widget_base(title='Plot Manager',/column) ;,GROUP=group,/MODAL)
wAction = widget_base(UVALUE={action:'CANCEL',sel:0}) ; to store the action

wbase0 = widget_base(wbase,/row)
 wtmp = widget_button(wbase0,VALUE='Accept',UVALUE='ACCEPT')
 wtmp = widget_button(wbase0,VALUE='Cancel',UVALUE='CANCEL')
 wtmp = widget_button(wbase0,VALUE='Help',UVALUE='HELP')

wbase1 = widget_base(wbase,/col,/frame)
wtmp = widget_label(wbase1,VALUE='The following sets are saved:')
wlist = widget_list(wbase1,VALUE=buffer,UVALUE='SELECT',XSIZE=30,$
  YSIZE=n_elements(buffer)<6)
wtmp = widget_label(wbase1,VALUE=strcompress(current,/rem)+$
	' is the current set.',/ALIGN_LEFT)
;wtmp = widget_TEXT(wbase1,VALUE=current,XSIZE=30)

wbase2 = widget_base(wbase,/COL,/frame)
wtmp = widget_label(wbase2,VALUE='Actions: ')
wbuttons = cw_bgroup(wbase2,['Delete selected set',$
'Delete current set','Delete all saved sets',$
'Move selected set to current set'],$
UVALUE='',$
/EXCLUSIVE,/NO_RELEASE,SET_VALUE=0,COLUMN=2)

wids = {buttons:wbuttons, action:wAction}
state={ wids:wids, selected:0L}
widget_control,wbase,set_uvalue=state
widget_control,wbase,/REALIZE
xmanager,'xplot_plotmgr_win',wbase,GROUP_LEADER=group,/modal
;
; retrieve the result
straction=0
widget_control,wAction,get_uvalue=straction
widget_control,wAction,/destroy
action=straction.action
selected=straction.sel
end ; xplot_plotmgr_win
;=========================================================================
;
pro xplot_plotmgr,id,nn_sel,DELETE_SEL=delete_sel,DELETE_ALL=delete_all,$
	MOVE=move,NOREFRESH=norefresh
;
catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XPLOT_PLOTMGR: error caught: '+!error_state.msg)
  catch, /cancel
  goto,out
endif

interactive=1
if keyword_set(DELETE_SEL) then begin
  if n_elements(nn_sel) EQ 0 then begin
     message,/info,'Usage: xplot_plotmgr,id,nn_sel,/DELETE_SEL'
     return
  endif
  action='DELETE_SEL'
  interactive=0
endif
if keyword_set(MOVE) then begin
  if n_elements(nn_sel) EQ 0 then begin
     message,/info,'Usage: xplot_plotmgr,id,nn_sel,/MOVE'
     return
  endif
  action='MOVE'
  interactive=0
endif
if keyword_set(DELETE_ALL) then begin
  action='DELETE_ALL'
  interactive=0
endif
;
widget_control, id , GET_UVALUE = state, /NO_COPY

;if n_elements(nn_sel) EQ 0 then begin
if interactive then begin
  action='' & nn_sel=0

  buffer = strcompress(state.plotname_buffer,/rem)
  buffer = buffer+'  (id: '+strcompress(state.dthbuffer,/rem)+')'
  current = state.plotname_current+$
	' (id: '+strcompress(state.dthcurrent,/rem)+')'

  xplot_plotmgr_win,buffer,current,Group=id,action=action,$
	selected=nn_sel
  ;if action EQ 'CANCEL' then goto,OUT
  ;if  nn_sel GT n_elements(state.buffer) then begin
  ;    tmp = widget_message(dialog_parent=event.top,$
  ;     'There are not so many saved sets.',/ERROR)
  ;     goto,out
  ;endif
endif
if action EQ 'DELETE_CUR' then begin
  action='MOVE'
  nn_sel = n_elements(state.buffer)
endif

case action of 
  'CANCEL': goto,OUT
  'DELETE_SEL': begin
		;
		; removing from buffer
		;
		hh = state.dthbuffer(nn_sel-1)
		nsaved = n_elements(state.buffer)
		if nsaved EQ 1 then begin
		  xpstr_update,state,/nobuffer,/nodthbuffer,/noplotname_buffer
		endif else begin
		  index = lindgen(nsaved)
		  newindex = where( index NE nn_sel-1)
		  buffer = state.buffer( newindex)
		  dthbuffer = state.dthbuffer( newindex )
		  plotname_buffer = state.plotname_buffer( newindex )
		  xpstr_update,state,buffer = buffer, $
			dthbuffer = dthbuffer,plotname_buffer = plotname_buffer
		endelse
		if n_elements(nsaved) EQ 1 then $  ; now zero
			widget_control,state.wids.plotmgr,sensitive=1
		widget_control,state.wids.saved, set_value= $
			strcompress(nsaved-1,/rem)
		handle_free,hh
	end
  'DELETE_ALL': begin
		for i=0,n_elements(state.buffer)-1 do $
		 if Ptr_Valid(state.dthbuffer(i)) then handle_free,state.dthbuffer(i)
		xpstr_update,state,/nobuffer,/nodthbuffer,/noplotname_buffer
		widget_control,state.wids.plotmgr,sensitive=0
		widget_control,state.wids.saved, set_value='0'
	end
  'MOVE': begin
		;print,'<><> selected buffer: ',state.buffer(long(outtext)-1)
		;nlist = strparse(state.buffer(long(outtext)-1),',',list)
		handle_free,state.dthcurrent
		nlist = strparse(state.buffer(nn_sel-1),',',list)
		; 
		; execute buffer
		;
		;print,'<><> executing : ','hh = '+list(1)
		tmp = execute( 'hh = '+list(1))
		for i=2,7 do begin
		;print,'<><> executing : ',list(i)
			tmp = execute( list(i) )
		endfor
		;print,'<><> executing : ',list(8)+','+list(9)
; the problematic part about ShiftVal ...
		tmp = execute( list(8)+','+list(9) )
		;
		; saving values
		;
		state.xpdt.xcol = xcol
		state.xpdt.ycol = ycol
		state.xpdt.psym = psym
		state.xpdt.linestyle = linestyle
		state.xpdt.thick = thick
		state.xpdt.sysvar(7) = '!p.color = '+strcompress(color,/rem)
		state.xpdt.shift = shiftval

		;
		; removing from buffer
		;
		nsaved = n_elements(state.buffer)
		if nsaved EQ 1 then begin
		  xpstr_update,state,/nobuffer,/nodthbuffer,/noplotname_buffer,$
			dthcurrent = hh 
		endif else begin
		  index = lindgen(nsaved)
		  newindex = where( index NE nn_sel-1)
		  buffer = state.buffer( newindex)
		  dthbuffer = state.dthbuffer( newindex )
		  plotname_buffer = state.plotname_buffer( newindex )
		  xpstr_update,state,buffer = buffer, $
			dthbuffer = dthbuffer, dthcurrent = hh, $
			plotname_buffer = plotname_buffer
		endelse
		;
		; close control panel and update its values
		;
                if widget_info(state.wids.controls(0),/valid_id) then begin
                     widget_control,state.wids.showcontrols, $
				set_value='Show Controls'
                     widget_control,state.wids.controls(0),/DESTROY
                endif
		state.strcontrols.symbol = psym
		state.strcontrols.linestyle = linestyle
		state.strcontrols.thickness = thick
		state.strcontrols.color = color
		;
		; updating widgets
		;
		if nsaved EQ 1 then $
			widget_control,state.wids.plotmgr,sensitive=0
		plotfile,hh,tmp,/nodata
		ncol = n_elements(tmp(*,0))
		
		if widget_info(state.wids.xcol,/NAME) EQ 'DROPLIST' then begin
		  coltitles = (sindgen(Ncol+1))(1:Ncol)
		  widget_control,state.wids.xcol, set_value = coltitles, $
			set_droplist_select = xcol-1
		  widget_control,state.wids.ycol, set_value = coltitles, $
			set_droplist_select = ycol-1
		endif else begin
		  widget_control,state.wids.xcol, set_value = $
			strcompress(xcol,/rem)
		  widget_control,state.wids.ycol, set_value = $
			strcompress(ycol,/rem)
		endelse
		widget_control,state.wids.ncol, set_value= $
			strcompress(ncol,/rem)
		widget_control,state.wids.saved, set_value= $
			strcompress(nsaved-1,/rem)
		widget_control,state.wids.message, set_value= ''
		; switch out spec (if selected)
		if state.spec.on EQ 1 then begin
		  widget_control,state.spec.scanbase,/destroy, bad_id=bad
		  state.spec.on = 0
		  xplot_mappanels, state.wids, state.spec
		endif
	end
endcase
; re-plot
if not(keyword_set(norefresh)) then xplot_plot,state

out:
widget_control, id , SET_UVALUE = state, /NO_COPY
end ;xplot_plotmgr
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_SAVEASCII 
;	PURPOSE: save to an ascii file the current set
;	CALLING SEQUENCE: XPLOT_SAVEASCII,parent 
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;
;	MODIFICATION HISTORY:
;		98-08-25 srio@esrf.fr initial version from xplot code
;
;-
PRO xplot_saveascii,wid_top
common xplot_saveascii,ffile

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XPLOT_SAVEASCII: error caught: '+!error_state.msg)
  catch, /cancel
  ffile = ffile_saved
  goto,out
endif

widget_control, wid_top , GET_UVALUE = state, /NO_COPY


if not(keyword_set(state.dthcurrent)) then begin
  tmp = widget_message(dialog_parent=event.top,'Nothing to save',/ERROR)
  goto,OUT
endif
plotfile,state.dthcurrent,/nodata,tmpset_pre
xcol = state.xpdt.xcol
ycol = state.xpdt.ycol
tmpset = tmpset_pre([xcol-1,ycol-1],*)
;
if n_elements(ffile) EQ 0 then ffile = xplot_defaults('SAVEASCII')
ffile_saved = ffile

;ffile = { ffile:'?', save: ['0','Current data','All columns'] }

TITLES=['Output file name:','Data to save:',$
  'Include header (SPEC type)','Comment line','Format']
FLAGS = ['1','1','1','w(2) EQ 1','1']

XscrMenu,ffile,TITLES=titles, FLAGS=flags, /NOTYPE,WTITLE='Save as...',$
 GROUP=wid_top,ACTION=action, /INTER,FIELDLEN=60,HELP=xplot_text('SAVEASCII')

ffile_old=''
if action eq 'DONT' then goto,out
if strcompress(ffile.ffile,/rem) eq '?' then begin
  ffile_old='?'
  file = Dialog_Pickfile(Dialog_Parent=wid_top,/Write,$
	Group=wid_top,File='xplot.dat')
  if file EQ '' then goto,out
  ffile.ffile=file
endif
if checkfile(ffile.ffile) EQ 1 then begin
  tmp = widget_message(dialog_parent=wid_top,/CANCEL,/DEFAULT_CANCEL,$
   ['File '+ffile.ffile+' already exists.', 'Overwrite it?'])
  if tmp EQ 'Cancel' then goto,out
endif

openw,Unit,ffile.ffile,/GET_LUN
if strcompress(ffile.header[0],/rem) EQ '1' then begin
  ncol = n_elements(tmpset_pre[*,0])
  printf,Unit,'#F '+ffile.ffile
  if strcompress(ffile.comment,/rem) NE '' then $
    printf,Unit,'#C '+ffile.comment
  printf,Unit,'#S 1 File written by xplot on '+systime()
  ; number of columns
  if ( strcompress(ffile.save(0),/REM) eq '0') then $
    printf,Unit,'#N 2' else $ 
    printf,Unit,'#N '+strcompress(n_elements(tmpset_pre[*,0]),/rem)
  ; column labels
  if state.spec.on EQ 1 then begin 
    labels = state.spec.labels 
    junk = strparse(labels,'!',tmp)
    if junk EQ -1 then begin
      junk = widget_message(dialog_parent=event.top,$
        'Error (column separator)',/ERROR)
      goto,out
    endif
  endif else begin
    if not(keyword_set(state.coltitles)) then begin
      tmp = strcompress( indgen(ncol)+1 ,/rem)
      tmp = 'Column_'+tmp
    endif else begin
      tmp = StrCompress(state.coltitles)
    endelse
  endelse
  if (strcompress(ffile.save(0),/REM) eq '0') then tmp=tmp([xcol-1,ycol-1])
  tt = '#L  '
  for ii=0,N_elements(tmp)-1 do tt = tt +'  '+tmp[ii]
  printf,Unit,tt
endif

if ( strcompress(ffile.save(0),/REM) eq '0') then begin
  ;format = '(2(G0.6," "))'
  format = '(2'+ffile.format+')'
  printf,Unit,tmpset,format=format
endif else begin
  npoints = n_elements(tmpset_pre(0,*))
  ncol = n_elements(tmpset_pre(*,0))
  ;format = '('+string(ncol)+'(G0.6," "))'
  format = '('+string(ncol)+ffile.format+')'
  printf,Unit,format=format,tmpset_pre
endelse
free_lun,Unit

widget_control,state.wids.message,set_value = $
  'File '+ffile.ffile+' written to disk.'
if ffile_old EQ '?' then ffile.ffile='?'

out:
widget_control, wid_top , SET_UVALUE = state, /NO_COPY
end ; xplot_saveascii
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_MESH 
;	PURPOSE: displays mesh data
;	CALLING SEQUENCE: XPLOT_MESH,parent 
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;
;	KEYWORD PARAMETERS [USE AT LEAST ONE]:
;		FLAG 0 displays data in usual mode (no-mesh)
;		     1 displays data in mesh mode
;		KIND 0 show mesh plot with lines
;		     1 shadow mesh plot with a surface
;		     2 shadow mesh plot with contour curves
;		     3 shadow mesh plot with a shaded surface
;		POINTS set this keyword to the number of points of each individual 
;			curve in the total data
;		COL set this keyword to a column indicating containing the index 
;			to recognise individual curves
;			(the COL and POINTS keyword are exclusive. If both are set,
;			only COL is used.)
;		KEYWORDS string with extra keywords to be passed to the graphical
;			routines.
;
;	MODIFICATION HISTORY:
;		98-09-24 srio@esrf.fr initial version from xplot code
;
;-
PRO xplot_mesh,input,FLAG=flag, KIND=kind, POINTS=points, COL=col, KEYWORDS=keywords

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XPLOT_MESH: error caught: '+!error_state.msg)
  catch, /cancel
  goto,out
endif

eventUVal=0
interactive=0
case type(input) of
8: begin
   event = input
   widget_control, event.top , GET_UVALUE = state, /NO_COPY
   Widget_Control, event.id, get_UValue=eventUVal
   wid_top=event.top
   interactive=1
   end
3: begin
   widget_control, input, GET_UVALUE = state, /NO_COPY
   wid_top=input
   eventuval = ''
   end
else: begin
   message,/info,'Usage: xplot_mesh, event (or parent_id)'
   return
   end
endcase



widget_control,state.wids.saved,get_value=tmp
if strcompress(tmp,/rem) NE '0' then begin
  itmp = widget_message(dialog_parent=event.top, $
    /error,'MESH option not available with Saved sets.')
    goto,out
endif
tmp = state.xpdt.mesh
if interactive then begin
  XscrMenu,tmp,GROUP=event.top,/NOTYPE,ACTION=action, $
    WTITLE='Mesh values',FIELDLEN=40,/INTERP,/APPLY_BUTTON,$
    TITLES=['Mesh plot:','Kind','Split plots','N for splitting',$
    'Column with mesh label values',$
    'Additional keywords'],HELP=xplot_text('MESH'), $
    FLAGS=['1','w(0) EQ 1','w(0) EQ 1','w(0) EQ 1 and w(2) EQ 0',$
    'w(0) EQ 1 and w(2) EQ 1','w(0) EQ 1']
  if action eq 'DONT' then goto,out
endif else begin
  action=''
  if n_elements(flag) NE 0 then tmp.flag[0]=strcompress(flag,/rem)
  if n_elements(kind) NE 0 then tmp.kind[0]=strcompress(kind,/rem)
  if n_elements(points) NE 0 then begin
    tmp.split[0]='0'
    tmp.points=points
  endif
  if n_elements(col) NE 0 then begin
    tmp.split[0]='1'
    tmp.col=col
  endif
  if n_elements(keywords) NE 0 then tmp.keywords=keywords
endelse
state.xpdt.mesh = tmp
xplot_plot,state
; autocall xplot_event when Apply button is pressed
if action eq 'DO_APPLY' then begin
  widget_control, event.top, SET_UVALUE = state, /NO_COPY
  xplot_event,event
  return
endif

out:
widget_control, wid_top , SET_UVALUE = state, /NO_COPY
end ; xplot_mesh
;
;=========================================================================
;

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_SHIFT 
;	PURPOSE: adds a horizontal and vertical shift to the current set
;	CALLING SEQUENCE: XPLOT_SHIFT,parent [,openwindow,set_value=set_value]
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;
;	KEYWORD PARAMETERS:
;		OPENWINDOW when set, opens a widget window for interactive 
;		manipulation
;		SET_VALUE a 2-element array [h,v] with the shift to be applied
;		to the horizontal (h) and vertical (v) axes.
;
;	MODIFICATION HISTORY:
;		98-09-17 srio@esrf.fr initial version.
;
;-
PRO xplot_shift_event,event
;
catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!error_state.msg
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'XPLOT_SHIFT: error caught: '+!error_state.msg)
   catch, /cancel
   on_error,2
   goto,out
endif
;
;
Widget_Control, event.id, get_UValue=eventUValue
Widget_Control, event.id, get_Value=Value
 
if n_elements(eventuvalue) EQ 0 then eventuvalue = ''
if not(keyword_set(eventuvalue)) then eventuvalue = ''
 
stateid = Widget_Info(event.handler,/Child)
Widget_Control, stateid, get_UValue=state, /No_Copy
 
iapply = 0
Case eventUValue OF
    'DISMISS': BEGIN
	Widget_Control,event.handler,/destroy
	END
    'RESET': BEGIN
	widget_control,state.wids.a,set_value=0.0
	widget_control,state.wids.b,set_value=0.0
	iapply = 1
	END
    'HELP': BEGIN
        XDisplayFile1,TEXT=xplot_text('SHIFT'),GROUP=event.top,$
        TITLE='Help on xplot shift'
	END
    'APPLY': iapply=1
    'MOVE': BEGIN
	iapply = 1
	step=0
	tmp=0.0
	widget_control,state.wids.step,get_value=step
	widget_control,event.id,get_value=tmp
	case tmp of
	  ' Up ': BEGIN
		isign = 1.0
		wid =  state.wids.b
		END
	  ' << ': BEGIN
		isign = -1.0
		wid =  state.wids.a
		END
	  ' >> ': BEGIN
		isign = 1.0
		wid =  state.wids.a
		END
	  'Down': BEGIN
		isign = -1.0
		wid =  state.wids.b
		END
	endcase
	widget_control,wid,get_value=tmp
	widget_control,wid,set_value=( tmp + isign*step )
	
	END
	else:
endcase

if iapply then begin
  a=0 & b=0 
  widget_control,state.wids.a,get_value=a
  widget_control,state.wids.b,get_value=b
 shift=[a,b]
 widget_control, state.wids.xplot , GET_UVALUE = xplotstate,/No_copy
 xplotstate.xpdt.shift = shift
 xplot_plot,xplotstate
 widget_control, state.wids.xplot , SET_UVALUE = xplotstate,/No_copy
endif

out:
if Widget_Info(stateid,/Valid_ID) then $
  Widget_Control, stateid, set_UValue=state, /No_Copy
end  ; xplot_shift_event
;
;=========================================================================
;
PRO xplot_shift, id, input, openwindow=openwindow

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!error_state.msg
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'XPLOT_SHIFT: error caught: '+!error_state.msg)
   if n_elements(xplotstate) NE 0 then $
	widget_control, id , SET_UVALUE = xplotstate,/no_copy
   catch, /cancel
   on_error,2
   return
endif

if not(keyword_set(openwindow)) then goto,out

if xregistered('xplot_shift') then begin
  itmp = Dialog_Message(Dialog_Parent=id, 'A "shift plot" window '+$
    'is already open. Open a new one?',/Question,/Default_No)
  if itmp EQ 'No' then return
endif

;
; define widgets
;
wbase=widget_base(/col,title='shift data')

wtmp = widget_base(wbase) ; to store state

if keyword_set(input) then value=input else $
  value=[0.0,0.0]
;
; main buttons
;
wtmp0 = widget_base(wbase,/Row,/Frame)
  wtmp = widget_button(wtmp0,Value='Dismiss',UValue='DISMISS')
  wtmp = widget_button(wtmp0,Value='Apply',UValue='APPLY')
  wtmp = widget_button(wtmp0,Value='Reset',UValue='RESET')
  wtmp = widget_button(wtmp0,Value='Help',UValue='HELP')
wtmp0 = widget_base(wbase,/Column,/Frame)
  wtmp1 = widget_base(wtmp0,/Row) 
    wa = cw_field(wtmp1,Title='X shift: ',Value=value[0],UValue='APPLY',$
	/Float,XSize=6,/Return_Event)
    wb = cw_field(wtmp1,Title='Y shift: ',Value=value[1],UValue='APPLY',$
	/Float,XSize=6,/Return_Event)

wtmp0 = widget_base(wbase,/Column,/Frame)
  wtmp = widget_label(wtmp0, Value='Change shift values: ')
  wstep = cw_field(wtmp0,Title='Moving step: ',Value='1.0',UValue='STEP',$
	/Float,XSize=8,/Return_Event)
  wtmp1 = widget_base(wtmp0)
  wtmp = widget_button(wtmp1,Value=' Up ',UValue='MOVE',XSize=60,XOffSet=35)
  wtmp1 = widget_base(wtmp0,/Row)
  wtmp = widget_button(wtmp1,Value=' << ',UValue='MOVE',XSize=60)
  wtmp = widget_button(wtmp1,Value=' >> ',UValue='MOVE',XSize=60)
  wtmp1 = widget_base(wtmp0)
  wtmp = widget_button(wtmp1,Value='Down',UValue='MOVE',XSize=60,XOffSet=35)

if n_elements(id) EQ 0 then id=0L
wids = {a:wa, b:wb, step:wstep, xplot:id}
state = {wids:wids}

;
;
widget_control,Widget_Info(wbase,/Child),set_uvalue=state,/no_copy

widget_control,wbase,/realize

; no_block?
widget_control, id , GET_UVALUE = xplotstate,/no_copy
no_block = xplotstate.no_block
widget_control, id , SET_UVALUE = xplotstate,/no_copy

xmanager,'xplot_shift',wbase,GROUP=id, No_Block=no_block

out:
if keyword_set(input) then begin
 if n_elements(input) ne 2 then return
 if widget_info(id,/valid_id) ne 1 then return
 widget_control, id , GET_UVALUE = xplotstate,/no_copy
 xplotstate.xpdt.shift = input
 xplot_plot,xplotstate
 widget_control, id , SET_UVALUE = xplotstate,/no_copy
endif
end ; xplot_shift
;
;============== END OF ROUTINES FOR NON-INTERACTIVE CALLS. ===================
;
;
;=========================================================================
;
pro xplot_mappanels, wids, spec
;
; Maps the column selection panels.
;
   if spec.on then begin
      widget_control,wids.asciipanel,map=0
      widget_control,wids.specpanel,map=1
      if spec.mca then begin
         widget_control,wids.normalpanel,map=0
         widget_control,wids.mcapanel,map=1
      endif else begin
         widget_control,wids.mcapanel,map=0
         widget_control,wids.normalpanel,map=1
      endelse
   endif else begin
      widget_control,wids.asciipanel,map=1
      widget_control,wids.specpanel,map=0
   endelse
end
;
;=========================================================================
;
FUNCTION xplot_fitpolgauss,set,OUTTEXT=outtext,EXTRAP=extrap,OPLOT=oplot,$
  GROUP=GROUP, PSYSVAR=psysvar
;
; drives the Calculations/Fitting/Polynomial/Gaussian option
; returns an array with the fitted point coordinates
;
; outtext: information text
; oplot: if set, overplots the fitting value. If set to a numerical 
;	value, tthen the number of overplotted points is set to its value.
; extrap: the resulting returned value is calculated from the 
;	abscissas points extrap(0,*)
;
on_error,2

back:
widget_control,sensitive=0
itmp = wmenu2(Dialog_Parent=group,$
       [' 1) Chi-Square Linear Regression', $
	' 2) Robust Estimation Linear Regrassion', $
	' 3) Least Square Polynomial Fit', $
	' 4) Singular Value Decomposition Polynomial Fit', $
	' 5) Gaussian + polynomial background', $
	' 6) Help',$
	' 7) Cancel'])
widget_control,sensitive=1


setx = reform(set(0,*))
sety = reform(set(1,*))

case  itmp  of
  0:BEGIN
	sdev = Replicate(1.0,n_elements(setx))
	Result = LinFit(setx,sety,CHISQR=chisqr,PROB=prob,SDEV=sdev,$
		SIG_AB=sig_ab)
	Equation = 'x*Result(1)+Result(0)'
	outtext = [ $
	'   Linear Regression (LINFIT) results ', $
	'   Fit result : yfit = A + Bx ', $
	'             A: '+strcompress(Result(0),/rem), $
	'             B: '+strcompress(Result(1),/rem), $
	'        ChiSqr: '+strcompress(ChiSqr,/rem), $
	'          Prob: '+strcompress(Prob,/rem), $
	'      Sigma(A): '+strcompress(sig_ab(0),/rem), $
	'      Sigma(B): '+strcompress(sig_ab(1),/rem)]

	for i=0,n_elements(outtext)-1 do print,outtext(i)
        yfit = Result(1)*setx+Result(0)
    END
  1:BEGIN
	Result = LadFit(setx,sety)
	Equation = 'x*Result(1)+Result(0)'
	outtext = [ $
	'   Linear Regression (LADFIT) results ', $
	'   Fit result : yfit = A + Bx ', $
	'             A: '+strcompress(Result(0),/rem), $
	'             B: '+strcompress(Result(1),/rem)]

	for i=0,n_elements(outtext)-1 do print,outtext(i)
        yfit = Result(1)*setx+Result(0)
    END
  2:BEGIN
	strtmp = { n:1 }
	XscrMenu,strtmp,/NoType,Titles='Polynomial Degree',Action=action, $
	Dialog_Parent=group,WTitle='Fit...'
	W = Replicate(1.0,n_elements(setx))
	if action ne 'DO' then return,0
	deg = strtmp.n
	;Result = PolyFitW(setx,sety,W,strtmp.n,yfit,yband,sigma,a)
	Result = Poly_Fit(setx,sety,strtmp.n,YFit=yfit,Sigma=sigma,Covar=a)
	Equation = ' X^'+strcompress(deg,/rem) + $
		'* Result['+strcompress(deg,/rem)+ ']' 
	for i=deg-1,0,-1 do Equation = Equation+ $
		' + X^'+strcompress(i,/rem)+ $
		' * Result['+strcompress(i,/rem)+']'
	eq_text = strsubstitute(equation,'Result','A')
	
        atext = strarr(strtmp.n+1)
	for i=0,strtmp.n do atext(i)=vect2string(a(*,i))
	outtext = [ $
        '   Polynomial fit (POLY_FIT) Results:',' ',$
	'   Equation: '+eq_text,$
	'   Fit coefficients  : A = '+vect2string(result),$
	'   1-sigma error estimates of A = '+vect2string(sigma),$
	'   Covarianve matrix : ',atext]

	for i=0,n_elements(outtext)-1 do print,outtext(i)
    END
  3:BEGIN
        strtmp = { n:1 }
        XscrMenu,strtmp,/NoType,Titles='Polynomial Degree',Action=action, $
	Dialog_Parent=group,WTitle='Fit...'
        W = Replicate(1.0,n_elements(setx))
	Yfit = W & ChiSq = 0  &  Covar = 0  &  Variance = 0
        if action ne 'DO' then return,0
        Result = SVDFit(setx,sety,strtmp.n+1,WEIGHT=W, YFIT=yfit, $
	ChiSq=ChiSq,COVAR=covar,VARIANCE=variance)
	deg = strtmp.n
	Equation = ' X^'+strcompress(deg,/rem) + $
		'* Result['+strcompress(deg,/rem)+ ']' 
	for i=deg-1,0,-1 do Equation = Equation+ $
		' + X^'+strcompress(i,/rem)+ $
		' * Result['+strcompress(i,/rem)+']'
	eq_text = strsubstitute(equation,'Result','A')
        atext = strarr(strtmp.n+1)
        for i=0,strtmp.n do atext(i)=vect2string(covar(*,i))
        outtext = [ $
        '   Polynomial fit (SVDFIT) Results: ',' ',$
	'   Equation: '+eq_text,$
        '   Fit coefficients  : A = '+vect2string(result),$
        '             Variance: '+vect2string(Variance),$
        '               ChiSq : '+strcompress(ChiSq,/rem),$
        '   Correlation matrix: ',atext]

        for i=0,n_elements(outtext)-1 do print,outtext(i)
    END
;  4:BEGIN
;	yfit  = 0
;	yfit = Gauss_Fit(setx,sety,Result)
;	Equation = 'Result(0)*EXP(-((x-Result(1))/Result(2))^2/2) + Result(3)'
;	outtext = [ $
;	'   Gaussina Fit (GAUSS_FIT) results ', $
;	'   Fit result :  F(x) = A0*EXP(-z^2/2) + A3, z=(x-A1)/A2', $
;	'            A0: '+strcompress(Result(0),/rem), $
;	'            A1: '+strcompress(Result(1),/rem), $
;	'            A2: '+strcompress(Result(2),/rem), $
;	'            A3: '+strcompress(Result(3),/rem)]
;
;	for i=0,n_elements(outtext)-1 do print,outtext(i)
;    END
  4:BEGIN
	widget_control,sensitive=0
	itmp2 = wmenu2(Dialog_Parent=group,$
	[' 1) Gaussian only: F(x) = A0*EXP(-z^2/2) ',$
	 ' 2) Gaussian + cte term A0*EXP(-z^2/2) + A3 ',$
	 ' 3) Gaussian + 1st degree pol: A0*EXP(-z^2/2) + A3 + A4*x',$
	 ' 4) Gaussian + 2nd degree pol: A0*EXP(-z^2/2) + A3 + A4*x + A5*x^2',$
	 ' 5) Cancel'])
	widget_control,sensitive=1
	IF itmp2 EQ 4 THEN RETURN,0
	CASE itmp2 OF
	0: nTerms = 3
	1: nTerms = 4
	2: nTerms = 5
	3: nTerms = 6
	else: nTerms = 3
	ENDCASE

	yfit  = 0
	yfit = GaussFit(setx,sety,Result,NTerms=nTerms)
	CASE nTerms OF
	4: BEGIN
		addeq = ' + Result(3) '
		eqtxt = 'F(x) = A0*EXP(-z^2/2) + A3 '
		addout  = ['            A3: '+strcompress(Result(3),/rem)]
	   END
	5: BEGIN
		addeq = ' + Result(3) + x*Result(4) '
		eqtxt = 'F(x) = A0*EXP(-z^2/2) + A3 + A4*x '
		addout  = ['            A3: '+strcompress(Result(3),/rem), $
	'            A4: '+strcompress(Result(4),/rem)]
	   END
	6: BEGIN
		addeq = ' + Result(3) + x*Result(4) + x^2 * Result(5)'
		eqtxt = 'F(x) = A0*EXP(-z^2/2) + A3 + A4*x + A5*x^2'
		addout  = ['            A3: '+strcompress(Result(3),/rem), $
	'            A4: '+strcompress(Result(4),/rem), $
	'            A5: '+strcompress(Result(5),/rem)]
	   END
	else: BEGIN
		addeq = ''
		eqtxt = 'F(x) = A0*EXP(-z^2/2) '
		addout = ''
	   END
	ENDCASE
	Equation = 'Result(0)*EXP(-((x-Result(1))/Result(2))^2/2) '+ $
		addeq
		;' + Result(3) + x*Result(4) + x^2 * Result(5)'
	outtext = [ $
	'   Gaussina Fit (GAUSS_FIT) results ', $
	'   Fit result :  '+eqtxt, $
	'                 z=(x-A1)/A2', $
	'            A0: '+strcompress(Result(0),/rem), $
	'            A1: '+strcompress(Result(1),/rem), $
	'            A2: '+strcompress(Result(2),/rem), $
	addout]

	for i=0,n_elements(outtext)-1 do print,outtext(i)
    END
  5: begin
	XDisplayFile1,TEXT=xplot_text('FITPOLGAUSS'),GROUP=group,$
	TITLE='Help on gaussian and polynomial fit',/NomenuBar
	goto,back
	end
  6: return,0
  else:
endcase

if keyword_set(oplot) then begin
  if oplot EQ 1 then x=setx else x = dindgen(oplot)/(oplot-1)* $
    (setx(n_elements(setx)-1)-setx(0)) + setx(0)
  yyfit = 0
  command = 'yyfit = '+Equation
  tmp = execute(command)
  if keyword_set(psysvar) then begin
    p_old=!p
    x_old=!x
    y_old=!y
    !p=psysvar.p
    !x=psysvar.x
    !y=psysvar.y
  endif
  oplot,x,yyfit,psym=-4
  if keyword_set(psysvar) then begin
    !p=p_old
    !x=x_old
    !y=y_old
  endif
endif

if keyword_set(extrap) then begin
  x=double(extrap(0,*)) 
  yfit = 0
  command = 'yfit = '+Equation
  tmp = execute(command)
endif

outtext = [outtext, ' ', $
  'Minimum y value for the fit: '+strcompress(min(yfit,max=yfitmax),/rem), $
  'Maximum y value for the fit: '+strcompress(yfitmax,/rem)]
return,yfit
end ; xplot_fitpolgauss
;
;=========================================================================
;
PRO xplot_columnoperpanel_Event, event
common xplot_columnoper,str,commandhistory,commandhistoryi

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error,Dialog_Parent=event.top,$
    'XPLOT_COLUMNOPERPANEL: error caught: '+!error_state.msg)
  catch, /cancel
  return
endif
widget_control,widget_info(event.handler,/child),get_uvalue=state
Widget_Control, event.id, Get_UValue=uval
CASE (uval) OF
  'ACCEPT': begin
	tmp=0
	widget_control,state.wids.expression,get_value=tmp
	if tmp[0] NE commandhistory[0] then $
	  commandhistory = [tmp[0],commandhistory]
	if n_elements(commandhistory) gt 50 then $
	  commandhistory=commandhistory[0:49]
	commandhistoryi = 0
	str.command=tmp[0]
	widget_control,state.wids.resultto,get_value=tmp
	str.resultto=tmp[0]
	widget_control,state.wids.over,get_value=tmp
	str.outcol=tmp[0]
	widget_control,event.id,get_value=tmp
	if tmp[0] EQ 'Apply' then str.action='DO_APPLY' else str.action='DO'
	widget_control,/destroy,event.top
	end
  'CANCEL': begin
	str.action = 'DONT'
	widget_control,/destroy,event.top
	end
  'HELP': xdisplayfile1,text=xplot_text('COLUMNOPER'),group=event.top, $
		/NoMenuBar
  'UP': begin
	commandhistoryi = commandhistoryi + 1
	commandhistoryi = commandhistoryi < (n_elements(commandhistory)-1) > 0
	widget_control,state.wids.expression,set_value=$
	  commandhistory[commandhistoryi]
	end
  'DOWN': begin
	commandhistoryi = commandhistoryi - 1
	commandhistoryi = commandhistoryi < (n_elements(commandhistory)-1) > 0
	widget_control,state.wids.expression,set_value=$
	  commandhistory[commandhistoryi]
	end
  'GET': begin
	tmp = xlist1(commandhistory,/wait,group_leader=event.handler)
	tmpi = where(commandhistory EQ tmp)
	if tmpi[0] EQ -1 then return
	commandhistoryi=tmpi[0]
	widget_control,state.wids.expression,set_value=$
	  commandhistory[commandhistoryi]
	end
  'CLEAN': widget_control,state.wids.expression,set_value=''
  'RESULTTO': begin
	tmp=''
	widget_control,state.wids.resultto,get_value=tmp
	if strcompress(tmp[0],/rem) EQ '1' then $
	  widget_control,state.wids.over,map=1 else $
	  widget_control,state.wids.over,map=0
	end
  ELSE:
ENDCASE
END ; xplot_columnoperpanel_event
;
;=========================================================================
;
PRO xplot_columnoperpanel, GROUP=group
common xplot_columnoper,str,commandhistory,commandhistoryi

if type(commandhistory) NE 7 then commandhistory = ['col1*2','col1-min(col1)']
commandhistoryi=-1

if keyword_set(group) then begin
  info=WIDGET_INFO(group, /GEOM)
  tlb2=widget_base(/col, /MODAL, GROUP=group, TITLE='Operation with columns', $
  XOFFSET=info.xoffset+50) 
endif else begin
  tlb2=widget_base(/col, TITLE='Operation with columns')
endelse
tmp = widget_base(tlb2,/row)
  wtmp=widget_button(tmp, Value='Accept', UValue='ACCEPT')
  wtmp=widget_button(tmp, Value='Cancel', UValue='CANCEL')
  wtmp=widget_button(tmp, Value='Apply', UValue='ACCEPT')
  wtmp=widget_button(tmp, Value='Help',UValue='HELP')
tmp = widget_base(tlb2,/col,/Frame)
  wtmp = widget_label(tmp,Value='Expression for new column:',/align_left)
  wexpression = widget_text(tmp,Value=str.command,xsize=60,/edit,$
	uvalue='ACCEPT')
  tmp2 = widget_base(tmp,/row)
  wtmp = widget_label(tmp2,Value='Expression history:',/align_left)
  wtmp=widget_button(tmp2, Value='Up',UValue='UP')
  wtmp=widget_button(tmp2, Value='Down',UValue='DOWN')
  wtmp=widget_button(tmp2, Value='Get...',UValue='GET')
  wtmp=widget_button(tmp2, Value='Clean',UValue='CLEAN')
  
tmp = widget_base(tlb2,/row,/Frame)
  txt = ['0','Append new column','Overwrite existing column','New Xplot window']
  wtmp = widget_label(tmp,Value='Result to:',/align_left)
  wresultto = cw_droplist(tmp, Value=txt, UValue='RESULTTO')

wover = cw_field(tlb2,Value=str.outcol,xsize=5,/integer,$
   title='Column to overwrite:',/frame,uvalue='ACCEPT',/return_events)

widget_control,wresultto,set_value=str.resultto
if strcompress(str.resultto,/rem) EQ '1' then $
  widget_control,wover,map=1 else $
  widget_control,wover,map=0

wids = {expression:wexpression, resultto:wresultto, over:wover}
state = { wids:wids}
widget_control,widget_info(tlb2,/child),set_uvalue=state
Widget_Control, tlb2, /Realize

XManager, 'xplot_columnoperpanel', tlb2

END ; xplot_columnoperpanel
;
;=========================================================================
;
PRO xplot_columnoper,set, group=group, resultcololumn = resultcolumn,$
  no_block=no_block,action=action
common xplot_columnoper,str,commandhistory,commandhistoryi
;
; drives the Calculations/Operations with columns option
;
;
;
; Inputs: set: data set to be oprated.
;	  str: structure with the window data.(since 98/08/27, placed in common)
;	  group: id of parent widget.
; Outputs: set: modified data set
;	   resultcolumn (keyword): modified column number. Returns -1
;	   if data is sent to a new Xplot window.
;
; Allow operations between columns. The resulting set overwrites
; the input set.
; MSR/95/08/29
; srio@esrf.fr Added no_block kw. Added common block.
;
;
on_error,2
;
; info window
;
col1 = 0  &  col2 = 0  &  col3 = 0  &  col4 = 0  &  col5 = 0
col6 = 0  &  col7 = 0  &  col8 = 0  &  col9 = 0  &  col10 = 0
col11 = 0  &  col12 = 0  &  col13 = 0  &  col14 = 0  &  col15 = 0
col16 = 0  &  col17 = 0  &  col18 = 0  &  col19 = 0  &  col20 = 0
col21 = 0  &  col22 = 0  &  col23 = 0  &  col24 = 0  &  col25 = 0
col26 = 0  &  col27 = 0  &  col28 = 0  &  col29 = 0  &  col30 = 0
col31 = 0  &  col32 = 0  &  col33 = 0  &  col34 = 0  &  col35 = 0
col36 = 0  &  col37 = 0  &  col38 = 0  &  col39 = 0  &  col40 = 0
col41 = 0  &  col42 = 0  &  col43 = 0  &  col44 = 0  &  col45 = 0
col46 = 0  &  col47 = 0  &  col48 = 0  &  col49 = 0  &  col50 = 0

ncol = n_elements(set(*,0))
if ncol gt 50 then begin
  txt = ['Operation with columns are only allowed', $
	'with set holding up to 50 columns.', $
	'The current set has '+strcompress(ncol,/rem)+' columns.',$
	'Problems could appear.']
  tmp = widget_message(dialog_parent=group,txt)
  ;return
end
;
; to evaluate an expression [Calculations Menu]
;
if type(str) NE 8 then begin
  ;str = xplot_defaults('XPcolumnoper')
  str = {  command:'col2*col1', $
           resultto:'0',$
           outcol:2,$
	   action:'DONT'}
endif

xplot_columnoperpanel,group=group
if str.action EQ 'DONT' then return
;
; start operations
;
; load data 
;
for i=1,ncol do begin
   command = 'col'+strcompress(i,/rem)+' = set('+$
	strcompress(i-1,/rem)+',*) '
   ;print,command
   tmp = execute(command)
endfor

  if not(keyword_set(set)) then begin
    ;xtext,TEXT=['Empty data set','Returning'],GROUP=group
    tmp = widget_message(dialog_parent=group,$
	['Empty data set','Returning'],/ERROR)
    ;str = str_old
    return
  endif

;
; evaluate the expression
;
command = 'result = '+str.command
tmp = execute(command)
if tmp ne 1 then begin
  tmp = widget_message(dialog_parent=group,$
	'Operation expression not understood',/ERROR)
  ;str = str_old
  return
endif   
;
; rearrange the result
;
;if strcompress(str.resultto(0),/rem) EQ '0' then begin
case  strcompress(str.resultto(0),/rem) of
  '0':BEGIN
      resultcolumn = ncol+1
      newset = fltarr(ncol+1,n_elements(col1))
      newset(0:ncol-1,*) = set
      command = 'newset('+strcompress(ncol,/rem)+',*) = result'
      ;print,command
      tmp = execute(command)
      END
  '1':BEGIN
      newset = set
      resultcolumn = fix(str.outcol)
      command = 'newset('+strcompress(str.outcol-1,/rem)+',*) = result'
      ;print,command
      tmp = execute(command)
      END
  '2':BEGIN
      newset = set
      xplot,reform(col1),reform(result),wtitle= $
	"Xplot 'Calculations/Operations with sets' results",no_block=no_block
      END
  else:
endcase
;
;
set = newset
action = str.action
;
end ; xplot_columnoper
;
;=========================================================================
;
PRO XPLOT_CALCULATIONS,event
;
; Event handler whefor the Calculations menu (Except operations with
; columns)
; State is passed from the event handler because it has been extracted
; by using /NO_COPY
; state is not passed MSR 97/09/08
; added common block to remember the user inputs (smooth).
;
;
common xplot_calculations1, smoothstr, interstr, cdfstr, createstr, $
  psdstr, peakfinderstr, histogramstr
catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error,Dialog_Parent=event.top,$
    'XPLOT_CALCULATIONS: error caught: '+!error_state.msg)
  catch, /cancel
  if n_elements(state) NE 0 then goto,out
  return
endif
widget_control, event.top , GET_UVALUE = state, /NO_COPY
widget_control, event.id, get_value=eventval
if not(keyword_set(state.dthcurrent)) then begin
  ;it is allowed to call Operations with sets... without
  ;data!!
  if eventval EQ 'Operations with sets...' then goto,EVALUATE
  if eventval EQ 'Create data set...' then goto,EVALUATE2

  tmp = Dialog_Message(DIALOG_PARENT=event.top,/Error, $
	'XPLOT_CALCULATIONS: No data in current set. Aborted.')
  goto,OUT
endif

plotfile,state.dthcurrent,/nodata,tmpset_pre1

tmpset1 = tmpset_pre1([state.xpdt.xcol-1,state.xpdt.ycol-1],*)
xmin = state.xpdt_limits.xmin
xmax = state.xpdt_limits.xmax
if (xmin NE xmax) then begin
  cutset,tmpset1,tmpset,xrange=[xmin,xmax]
  cutset,tmpset_pre1,tmpset_pre,xrange=[xmin,xmax],CutCol=state.xpdt.xcol-1
  ttext = $
  'Calculations have been performed in'+$
  ' the X interval ('+strcompress(xmin,/rem)+','+ $
  strcompress(xmax,/rem)+')'
endif else begin
  tmpset = tmpset1
  tmpset_pre = tmpset_pre1
  ttext = ''
endelse
widget_control,state.wids.message,set_value= ttext
                
if n_elements(tmpset EQ 1) then plotfile,tmpset,tmpset
CASE eventval OF
 'Width/Integral/MinMax':	begin
	nn_val = n_elements(tmpset(0,*))
	; this is needed to assure that the crosses plotted by
	; getfwhm will be on the right place.
	p_old=!p
	x_old=!x
	y_old=!y
	!p=state.psysvar.p
	!x=state.psysvar.x
	!y=state.psysvar.y
	fwhm_val = GETFWHM(tmpset,/PLOT,Group=event.top)
	!p=p_old
	!x=x_old
	!y=y_old
	;Check duplicate values.
	X = reform(tmpset[0,*])
	TypeX = SIZE(X)
	if TypeX[TypeX[0]+2] ne N_ELEMENTS(UNIQ(X[SORT(X)])) then begin
	  itmp = Widget_Message(Dialog_Parent= event.top,$
		['The abscissas array contains duplicated points.',$
		 'The value of the integral is affected'])
	  int_val = INTEGSET(tmpset)
	endif else begin
	  int_val = INT_TABULATED(tmpset(0,*),tmpset(1,*),/Sort)
	endelse

	max_val = max(tmpset(1,*),max_i)
	min_val = min(tmpset(1,*),min_i)
	max_val_x = max(tmpset(0,*),max_x_i)
	min_val_x = min(tmpset(0,*),min_x_i)
	text= [ ttext,' ', ' ',$
	'number of points:    '+strcompress(nn_val,/RE), $
	'FWHM value is:    '+strcompress(fwhm_val,/RE), $
	'Integral is  :    '+strcompress(int_val,/RE), $
	' ',$
	'Maximum y (Peak) at point '+strcompress(1+max_i,/re)+' :  ('+$
	  strcompress(tmpset(0,max_i),/RE)+','+strcompress(max_val,/RE)+')', $
	'Minimum y at point '+strcompress(1+min_i,/RE)+' : ('+ $
	  strcompress(tmpset(0,min_i),/RE)+','+strcompress(min_val,/RE)+')', $
	'Maximum x is at point '+strcompress(1+max_x_i,/RE)+' : ('+ $
	  strcompress(tmpset(0,max_x_i),/RE)+','+$
          strcompress(tmpset(1,max_x_i),/RE)+ $
	')',	$
	 'Minimum x is at point '+strcompress(1+min_x_i,/RE)+' : ('+ $
	   strcompress(tmpset(0,min_x_i),/RE)+','+$
           strcompress(tmpset(1,min_x_i),/RE)+$
	')', $
	' ',' ', $
	'Do you want to write few of these values in the plot legend? ']

	tmp = Dialog_Message(text,DIALOG_PARENT=event.top,$
		/QUESTION,/DEFAULT_NO)
	if tmp EQ 'No' then goto,out
	legendtxt = 'legend,'+ $
	  '['+"'"+'FWHM value is: '+strcompress(fwhm_val,/RE)+"',"+ $
	  "'"+'Integral is  : '+strcompress(int_val,/RE)+"',"+ $
	  "'"+'Peak at  : ('+strcompress(tmpset(0,max_i),/RE)+','+$
	  strcompress(max_val,/RE)+')'+"']"
	if keyword_set(state.exebuffer) then $
		exebuffer  = [state.exebuffer,legendtxt] else $
		exebuffer  = legendtxt
	xpstr_update,state,EXEBUFFER=exebuffer
	xplot_plot,state
	end
 'Moments':        begin
	npoints = n_elements(tmpset(0,*))
	xmom = moment(tmpset(0,*),mdev=xmdev,sdev=xsdev)
	ymom = moment(tmpset(1,*),mdev=ymdev,sdev=ysdev)
	text= [ ttext,' ', ' ',$
	  'Number of data points: '+strcompress(npoints,/rem), $
	  '  ', $
	  'Values for the abscissas values are: ', $
	  '    Mean:       '+strcompress(xmom(0),/rem), $
	  '    Variance:   '+strcompress(xmom(1),/rem), $
	  '    Skewness:   '+strcompress(xmom(2),/rem), $
	  '    Kurtosis:   '+strcompress(xmom(3),/rem), $
	  '    Mean Absolute dev:   '+strcompress(xmdev,/rem), $
	  '    Standard Deviation:  '+strcompress(xsdev,/rem), $
	  '  ', $
	  'Values for the ordinates values are: ', $
	  '    Mean:       '+strcompress(ymom(0),/rem), $
	  '    Variance:   '+strcompress(ymom(1),/rem), $
	  '    Skewness:   '+strcompress(ymom(2),/rem), $
	  '    Kurtosis:   '+strcompress(ymom(3),/rem), $
	  '    Mean Absolute dev:   '+strcompress(ymdev,/rem), $
	  '    Standard Deviation:  '+strcompress(ysdev,/rem)]
	;ShowInfo1,TITLE='Xplot Calculations/Moments', $
	;  GROUP=event.top,INFOTEXT=text, WIDTH=$
	;  max(strlen(text)), HEIGH=n_elements(text)
	Xdisplayfile1,TEXT=text,Title='Xplot Calculations/Moments results.'
	end
 'List of points...':        begin
	itmp = Dialog_Message(/Question,['This option allows to get a list of points from the ', $
                                     'current graphic by clicking with the mouse. The results',$
                                     'are sent to a new window.  ',$
                                     ' ',$
                                     'CLICK WITH **RIGHT** BUTTON TO GET A POINT', $
                                     'click with **CENTER** button to get the last point (end the list)', $
                                     ' ',$
                                     'Do you want to proceed? ',$
                                     '    Yes=start new list of points', $
                                     '    No=Cancel action and eventually end an ongoing list of points.'], $
				     Dialog_Parent=event.top, Title='List of points')
	IF itmp EQ 'No' THEN BEGIN
	  state.calcstr.getListOfPoints=0
	ENDIF ELSE BEGIN
	  state.calcstr.getListOfPoints=1
	  XDisplayFile1,Text=[';Get a list of points using the mouse (right button)', $
                              ';Get the last point by clicking the central button',$
                              ';Edit the list (if wanted) and use File->Save as to write a file'], $
		Title='List of points'
	ENDELSE
	end
 'Create data set...':		begin
	EVALUATE2:
	if n_elements(createstr) EQ 0 then begin
	  createstr = { xmin:'0.0', xmax:'2*!pi', n:100, y:'sin(x)'}
	endif
	action=0
	XscrMenu,createstr,/NoType,/Interp, Titles=['Minimum x','Maximum x',$
	 'Number of points','Expression for f(x)'],$
	  flags=replicate('1',4),ACTION=action,WTITLE='Create Data Set', $
	  FieldLen=60,Dialog_Parent=event.top
	if action eq 'DONT' then goto,out
	xmin = 0.0 & xmax = 0.0
	itmp = Execute('xmin = '+createstr.xmin)
	IF itmp NE 1 THEN BEGIN
	  itmp = Dialog_Message(Dialog_Parent=event.top,/Error,$
		'Error executing: xmin = '+createstr.xmin)
	  GoTo,out
	ENDIF
	itmp = Execute('xmax = '+createstr.xmax)
	IF itmp NE 1 THEN BEGIN
	  itmp = Dialog_Message(Dialog_Parent=event.top,/Error,$
		'Error executing: xmax = '+createstr.xmax)
	  GoTo,out
	ENDIF
	x = findgen(createstr.n)/float(createstr.n-1)*(xmax-xmin)
	x = x + xmin
	y = 0
	itmp = execute('y='+createstr.y)
	if itmp NE 1 then begin
		  itmp = widget_message(/ERROR,dialog_parent=event.top,$
		  ['Expression not understood: y='+createstr.y,'Aborting'])
		  goto,out
	endif 
	tmpset = fltarr(2,createstr.n)
	tmpset[0,*] = reform(x)
	tmpset[1,*] = reform(y)
	xplot,tmpset,wtitle= 'New data set: f(x)='+createstr.y,$
		NO_BLOCK=state.no_block
	end

 'New window with current set':		begin
	nCol = N_Elements(tmpset_pre[*,0])
	IF nCol GT 2 THEN BEGIN
	  itmp = Dialog_Message(Dialog_Parent=event.top,/Question,$
	  ['Current set has '+StrCompress(nCol,/Rem)+' columns',$
	   'You can export all columns or only the displayed X and Y columns.',$
	   'Do you want to export ALL columns?'])
	   IF itmp EQ 'Yes' THEN BEGIN
	     XPlot,tmpset_pre,No_Block=state.no_block, $
		xCol=state.xpdt.xcol, yCol=state.xpdt.ycol
	   ENDIF ELSE XPlot,tmpset,No_Block=state.no_block 
	ENDIF ELSE XPlot,tmpset,No_Block=state.no_block
	end
 'Operations with sets...':		begin
	EVALUATE:
	tmp = state.calcstr.xpevaluate
	xplot_evaluate,tmpset,eval_str= tmp,group=event.top,$
	no_block=state.no_block
	state.calcstr.xpevaluate = tmp
	end
 'Fourier Filtering...':	FOURIERFILTER,GROUP=event.top, $
	DATA=tmpset(1,*),XDATA=tmpset(0,*),NO_BLOCK=state.no_block
 'Derivative':		begin
	der = tmpset
	der(1,*) = Deriv(tmpset(0,*),tmpset(1,*))
	xplot,der,wtitle='Xplot Calculations/Derivative',$
	  NO_BLOCK=state.no_block
	end
 'CDF...':		begin
	tmpout = tmpset
	tmpout[1,*]=0

	if n_elements(cdfstr) EQ 0 then begin
	  cdfstr = { method:['0','Integral by simple addition',$
		'Integral by 5-point Newton-Cotes']}
	endif
	action=0
	XscrMenu,cdfstr,/NoType,/Interp, Titles=['Integration method'],$
	  flags=['1'],HELP=xplot_text('CDF'),ACTION=action,WTITLE='CDF',$
	  Dialog_Parent=event.top
	if action eq 'DONT' then goto,out
	nelem = n_elements(tmpset(0,*))
	Widget_Control,/HourGlass
	case strcompress(cdfstr.method[0],/Rem) of
	'0': begin
		steps = tmpset[0,0:nelem-2]
		steps = [0,reform(steps)]
		steps[0] = steps[1]
		steps = tmpset[0,*] - steps
	 	for i=1L,nelem-1 do tmpout(1,i)= $
		total(tmpset(1,0:i)*steps[0:i])
		end
	'1': begin
	 	for i=1L,nelem-1 do tmpout(1,i)= $
		int_tabulated(tmpset(0,0:i),tmpset(1,0:i),/Sort)
		end
	else:
	endcase

	xplot,tmpout,wtitle='Xplot Calculations/CDF/'+$
		cdfstr.method[fix(cdfstr.method[0])],NO_BLOCK=state.no_block
	end

 'Power Spectral Density...':		begin
	x = reform(tmpset[0,*])
	y = reform(tmpset[1,*])
	nx=n_elements(x)
	diff = x[1]-x[0]
	diff = (x[1:nx-1]-x[0:nx-2]) - diff
	fdiff = where( abs(diff) GT 1e-4)
	if fdiff[0] NE -1 then begin
	  itmp = Dialog_Message(Dialog_Parent=event.top,/Question, $
	     ['Abscissas array must be equally spaced.',$
	     'Calculation may fail.','Do you want to continue?'])
	  if itmp EQ 'No' then goto,out
	endif

	if n_elements(psdstr) EQ 0 then begin
	  ;psdstr = {window:['0','None','Hanning','Kaiser-Bessel'], $
	  psdstr = { fmin:0.0, fmax:0.0, $
             positive_only:['1','All frequencies','Only positive frequencies'] }
	endif
	titles=['f min','f max','output frequencies']
	flags=replicate('1',n_elements(titles))
	action=0
	XscrMenu,psdstr,/NoType,/Interp, Titles=titles,$
	  flags=flags,HELP=xplot_text('PSD'),ACTION=action,WTITLE='PSD',$
	  group=event.top,Dialog_Parent=event.top
	if action eq 'DONT' then goto,out

        if psdstr.fmin NE psdstr.fmax then $
	  range=[psdstr.fmin,psdstr.fmax] else range=0
	f=0
        s=prof2psd(x,y,f=f,range=range, $ 
	  positive_only=fix(psdstr.positive_only[0]) )
	if fix(psdstr.positive_only[0]) EQ 0 then xlog=0 else xlog=1
	xplot,f,s,xlog=xlog,/ylog, $
	  wtitle='Xplot/Calculations/PSD result',no_block=state.no_block, $
	  xtitle='Frequency in units of 1/[X]',ytitle='PSD in units of [Y]^3'
	end

 'Smooth...':		begin
	tmpout = tmpset
	if n_elements(smoothstr) EQ 0 then begin
	  smoothstr = { method:['1','Boxcar average','Least Squares'],$
	  width:5,edge:['0','No','Yes'],degree:2 }
	endif
	XscrMenu,smoothstr,/NoType,/Interp, $
	Titles=['Smoothing method','Width of smoothing window:', $
	 'Edge Truncation: ','Polynomial degree'],Action=action, $
	  help=xplot_text('SMOOTH'),wtitle='smooth',$
	  flags=['1','1','w(0) EQ 0','w(0) EQ 1'], $
	  Dialog_Parent=event.top
	if action eq 'DONT' then goto,out
	;tmpout = tmpset
	width = float(smoothstr.width)
	case strcompress(smoothstr.method(0),/rem) of
	  '0': begin ; BOXCAR
		;message,/info,'Using SMOOTH routine'
		wtitle='Xplot Calculations/Smooth(Boscar),width: '+$
		strcompress(width,/rem)
		edge = fix(smoothstr.edge(0))
		tmpout(1,*) = Smooth(tmpset(1,*),width,edge=edge)
	 	end
	  '1': begin ; LESAT-SQUARE
		;message,/info,'Using POLY_SMOOTH routine'
		degree = smoothstr.degree
		wtitle='Xplot Calculations/Smooth(LeastSquares),width: '+$
		strcompress(width,/rem)+', degree: '+$
		strcompress(degree,/rem)
		if width LE degree then begin
		  itmp = widget_message(/ERROR,dialog_parent=event.top,$
		  'Width must be LARGER than DEGREE. Aborting.')
		  goto,out
		endif
		tmpout(1,*) = Poly_Smooth(reform(tmpset(1,*)),width,$
			degree=degree)
	 	end
	endcase
	xplot,tmpout,wtitle=wtitle,NO_BLOCK=state.no_block
	end
 'Interpolation/Spline...': begin
	if n_elements(interstr) EQ 0 then begin
	  interstr = { kind:['0','Linear','Spline','Parametric spline'], $
	  npts:500,sigma:1.0,overplot:['0','No','Yes'] }
	endif
	titles=['Interpolation: ','Number of points of new set:', $
          'Spline tension','Overplot original data:']
	flags=['1','w(0) NE 2','w(0) EQ 1','1']
	XscrMenu,interstr,/NoType,/Interp,Titles=titles, $ 
	  Wtitle='Interpolation/Spline',Action=action,$
	  FLAGS=flags, HELP=xplot_text('INTERPOLATION'), $
	  Dialog_Parent=event.top
	if action eq 'DONT' then goto,out
	mymethod = strcompress(interstr.kind(0),/rem)
	if mymethod NE '2' then begin
	  n_interp=interstr.npts
	  int_step = $
	    (max(tmpset(0,*))-min(tmpset(0,*)))/float(n_interp-1)
	  newset = fltarr(2,n_interp)
      	  newset(0,*)=findgen(n_interp)*int_step+min(tmpset(0,*))
	endif
	case mymethod of
	'0': newset(1,*) = interpol(tmpset(1,*),tmpset(0,*),newset(0,*))
	'1': newset(1,*) = spline(tmpset(0,*),tmpset(1,*),newset(0,*),$
		interstr.sigma)
	'2': begin
	  xx=0.0 & yy=0.0
	  spline_p,tmpset(0,*),tmpset(1,*),xx,yy
	  newset = fltarr(2,n_elements(xx))
	  newset(0,*) = reform(xx)
	  newset(1,*) = reform(yy)
	  end
	endcase
	wtitle='Xplot Calculations/Interpolation/Spline-'
	CASE StrCompress(interstr.kind[0],/Rem) OF
	 '0': wtitle = wtitle+'(Linear)' 
	 '1': wtitle = wtitle+'(Spline - Tension='+ $
	      StrCompress(interstr.sigma,/REM)+' )'
	 '2': wtitle = wtitle+'(Spline)
	 else:
	ENDCASE
	IF StrCompress(interstr.overplot[0],/Rem) EQ '1' THEN BEGIN
	  p=0
	  Widget_Control,/HourGlass
	  XPlot,tmpset,WTitle=wtitle,NO_BLOCK=1,Parent=p 
	  XPlot_SaveCurrent,p
	  XPlot_LoadFile,p,newset
	  XPlot_Controls_action,p,PSYMSIGN=1,PSymbol=1 
	  XPlot_Exebuffer,p,SetBuffer=$
	    "legend,['original','Interpolation/Spline'],linestyle=[0,0]"+$
	    ",PSym=[0,1]"
	ENDIF ELSE xplot,newset,wtitle=wtitle,NO_BLOCK=state.no_block 
	end
 'Convolution & Correlation...':		begin
	Conv_data = state.calcstr.xpconvolution 
	;Conv_flag = ['1','w(0) EQ 1','1','1','w(3) EQ 0']
	Conv_flag = ['1','1','w(1) EQ 2','1',$
		'w(1) EQ 1','w(1) EQ 1','1','1','w(7) EQ 1','1']
	Conv_titles = $
	    ['Method','Type','External file:','Normalization:',$
	  'Voigt ratio (1=Lorentzian, 0=gaussian)',$
	  'Voigt Full Width at Half Maximum',$
	  'Overplot original data',$
	  'Write output to file:', 'Output file name:',$
	  'Calculation by:']
	XscrMenu,conv_data,GROUP=event.top,TITLES=conv_titles, $
	  FLAGS=conv_flag,ACTION=action,/INTER,/NOTYPE, $
	  WTITLE='Convolution data',HELP=xplot_text('CONVOLUTION'),$
	  ncol=2,Dialog_Parent=event.top
	IF (action EQ 'DONT') THEN goto,out
	IF (STRCOMPRESS(Conv_data.Type(0),/REM) EQ '2') THEN  BEGIN
	  tmp = rascii(Conv_data.Filename)
	  if not(keyword_set(tmp)) then begin
	    itmp = widget_message(DIALOG_PARENT=event.top,$
	   'Error opening file'+$
	    Conv_data.Filename,/ERROR)
            goto,OUT
          endif
	  tmpset2 = tmp(0:1,*)
	ENDIF ELSE tmpset2 = tmpset
	if strcompress(Conv_data.type(0),/rem) EQ 1 then begin
	  FWHM=[Conv_data.voigt_fwhm,Conv_data.voigt_ratio]
	endif else FWHM=0
	;conv_res = CONV2(tmpset2,tmpset, FWHM=fwhm,$
	widget_control,/HOURGLASS
	conv_res = CONV2(tmpset,tmpset2, FWHM=fwhm,$
	  NORM=fix(Conv_data.Normalization(0)),$
	  CORRELATION=fix(Conv_data.method(0)),$
	  METHOD=fix(Conv_data.calculation(0)) )
	itmp = 0L

	IF (STRCOMPRESS(Conv_data.over(0),/REM) EQ '1')  THEN BEGIN
	  xplot,tmpset,wtitle= 'Xplot Calculations/Convolution&Correlation',$
	  NO_BLOCK=state.no_block,PARENT=itmp,$
	  xrange=[max(tmpset(0,*)),min(tmpset(0,*))]
	  state.calcstr.xpconvolution = Conv_data
	  xplot_savecurrent,itmp
	  xplot_loadfile,itmp,conv_res
	ENDIF ELSE BEGIN
	  xplot,conv_res,wtitle= 'Xplot Calculations/Convolution&Correlation',$
	  NO_BLOCK=state.no_block,PARENT=itmp
	  state.calcstr.xpconvolution = Conv_data
	ENDELSE

	IF (STRCOMPRESS(Conv_data.Write(0),/REM) EQ '1')  THEN BEGIN
	  tmp = widget_message(DIALOG_PARENT=itmp,$
	  ['File '+Conv_data.Fileout+' already exists.', $
	  'Overwrite it?'],/CANCEL,/DEFAULT_CANCEL)
	  if tmp EQ 'Cancel' then goto,out
	  OPENW,Unit,Conv_data.Fileout,/GET_LUN
	  PRINTF,Unit,conv_res
	  FREE_LUN,Unit
	  ;tmp = widget_message(DIALOG_PARENT=itmp,$
	  ;'File '+Conv_data.Fileout+' written to disk.',/INFO)
	  widget_control,state.wids.message,set_value=$
		'File '+Conv_data.Fileout+' written to disk.'
	ENDIF
	end
 'Find peaks...':		begin
	;
	; calls PeakFinder
	;
	IF N_Elements(peakfinderstr) EQ 0 THEN $
	  peakfinderstr = {cutoff:['0','Default','User defined'], $
	    plot_cutoff:['0','No','Yes'], $
	    display:['0','Good peaks','All peaks'] }
	flags = Replicate('1',3)
	titles=['Cutoff value','Plot number of peaks vs cutoff',$
	  'Overplot ']
	XscrMenu,peakfinderstr,/NoType,/Interp,Flags=flags,Titles=titles, $
	  Action=action,WTitle='Find peaks',Group=group,$
	  Help=xplot_text('PEAKFINDER'),Dialog_Parent=event.top
	IF action EQ 'DONT' THEN GoTO,out

	pcutoff=0 & CLimits=0 & npeaks=0
	a=PeakFinder(tmpset[1,*],tmpset[0,*],PCutoff=pcutoff,CLimits=CLimits,$
	  NPeaks=npeaks,/Sort,/Opt)

	; 
	; 
	; plot #Peaks vs Cutoff
	;
	IF StrCompress(peakfinderstr.plot_cutoff[0],/Rem) EQ '1' THEN BEGIN
	  p=0
	  XPlot,pcutoff,XTitle='cutoff value',YTitle='Number of peaks',$
	    WTitle='PeakFinder results',No_Block=state.no_block,parent=p
	  XPlot_Controls_Action,p,PSymSign='1',PSymbol=10
	ENDIF
	;
	; plot results
	;
	xplot_plot,state
	p_old=!p
	x_old=!x
	y_old=!y
	!p=state.psysvar.p
	!x=state.psysvar.x
	!y=state.psysvar.y
	nPeaks2=nPeaks
	IF StrCompress(peakfinderstr.cutoff[0],/Rem) EQ '1' THEN BEGIN
	  pp=cLimits[0]
	  XEdit,pp,InfoText=['Found '+StrCompress(nPeaks)+$
		' peaks with stable cutoff values in the ',$
		'interval '+Vect2String(cLimits[2:3]) ], $
		Text= 'New cutoff value: ',Title='Setting PeakFinder cutoff',$
		Dialog_Parent=event.top

	  nPeaks2 = N_Elements(Where(a[4,*] GE pp)) > 0
	  text = ['','','Results of peak search: ','', $	
	    'Peaks found (total): '+String(N_Elements(a[0,*])), $
	    'Cutoff value (default): '+String(climits[0]), $
	    'Good Peaks for default cutoff: '+String(nPeaks), $
	    'Cutoff value (new): '+String(pp), $
	    'Good Peaks for new cutoff: '+String(nPeaks2), $
	    'Confidence value in cutoff: '+String(climits[1]) ]
	ENDIF ELSE BEGIN
	  text = ['','','Results of peak search: ','', $	
	    'Peaks found (total): '+String(N_Elements(a[0,*])), $
	    'Cutoff value: '+String(climits[0]), $
	    'Good Peaks: '+String(nPeaks), $
	    'Confidence value in cutoff: '+String(climits[1]) ]
	ENDELSE

	OPlot,a[1,0:nPeaks2-1],a[2,0:nPeaks2-1],PSym=6
	IF StrCompress(peakfinderstr.display[0],/Rem) EQ '1' THEN $
	  OPlot,a[1,*],a[2,*],PSym=1
	!p=p_old
	!x=x_old
	!y=y_old

	;
	; display results
	;
	tmp = StrArr(N_Elements(a[0,*]))
	FOR i=0L,N_Elements(a[0,*])-1 DO BEGIN
	   tmp[i]=String(a[*,i],Format='(I10,5G10.5)')
	ENDFOR
	
	text = [text,' ','All peaks',$
	'============================================================',$
	'     index   x value   y value    weight    norm-w     width',$
	'============================================================',$
	tmp]

	a1 = a[*,0:nPeaks-1]
	tmp = StrArr(N_Elements(a1[0,*]))
	FOR i=0L,N_Elements(a1[0,*])-1 DO BEGIN
	   tmp[i]=String(a1[*,i],Format='(I10,5G10.5)')
	ENDFOR

	text = [text,' ','Good peaks for default cutoff ',$
	'============================================================',$
	'     index   x value   y value    weight    norm-w     width',$
	'============================================================',$
	tmp]


	IF StrCompress(peakfinderstr.cutoff[0],/Rem) EQ '1' THEN BEGIN

	  a1 = a[*,0:nPeaks2-1]
	  tmp = StrArr(N_Elements(a1[0,*]))
	  FOR i=0L,N_Elements(a1[0,*])-1 DO BEGIN
	     tmp[i]=String(a1[*,i],Format='(I10,5G10.5)')
	  ENDFOR

	text = [text,' ','Good peaks for new cutoff ',$
	'============================================================',$
	'     index   x value   y value    weight    norm-w     width',$
	'============================================================',$
	tmp]
	ENDIF

	XDisplayFile1, Group=event.top, Text=text ,$
	  Title='Xplot PeakFinder results'
	end

 'Histogram...': BEGIN
        IF n_elements(histogramStr) EQ 0 then begin
	  histogramstr={col:['0','X','Y'], $
		type:['0','Number of bins','bin size'],nBins:25L,binsize:1.0}
	ENDIF
	titles=['Column to histogram','Input type','Number of bins',$
		'Bin size']
	flags = ['1','1','w(1) EQ 0','w(1) EQ 1']
	action=''
	XscrMenu,histogramstr,/NOTYPE,GROUP=event.top, $
	  WTITLE='Histogram',ACTION=action, Dialog_Parent=event.top , $
	  Titles=titles,/Interp,Flags=flags
	IF action EQ 'DONT' then GoTo,out
	CASE strCompress(histogramstr.col[0],/Rem) OF
	  '0': x = Reform(tmpset[0,*])
	  '1': x = Reform(tmpset[1,*])
	  else:
	ENDCASE
	hx = 0
	CASE strCompress(histogramstr.type[0],/Rem) OF
	  '0': hy = whistogram(x,xVal=hx,NBins=histogramstr.nBins)
	  '1': hy = whistogram(x,xVal=hx,BinSize=histogramstr.binsize)
	  else:
	ENDCASE
	; display it
	xplot,no_Block=state.no_block,Parent=p
	xplot_loadfile,p,Make_Set(hx,hy),/NoRefresh,Wtitle=$
		'Result of Calculations/Histogram'
	xplot_controls_action,p,PSymbol=4,/NoRefresh
	xplot_controls_action,p,PsymSign=1,/NoRefresh
	xplot_controls_action,p,PSymbol=10
	
	END




 'Fit - Polynomial&Gaussian...':		begin
	outtext = ''
	oplotflag = n_elements(tmpset(0,*)) 
	if oplotflag LT 100 then oplotflag = 100 else $
		oplotflag=1
	yfit = xplot_fitpolgauss(tmpset,OPLOT=oplotflag,$
		OUTTEXT=outtext,EXTRAP=tmpset1,GROUP=event.top,$
		psysvar=state.psysvar)
	if not(keyword_set(yfit)) then goto,out
	XDisplayFile1, GROUP=event.top, TEXT=outtext ,$
	  TITLE='Xplot Polynomial&Gaussian Fit Results'
	;oplot,tmpset(0,*),yfit,psym=-4
	widget_control,state.wids.message,set_value = $
	  'Diamond points are the result'+$
	  ' of the fit.'
	tmp = widget_message( DIALOG_PARENT=event.top,$
	  /QUESTION,TITLE='Temporary Fit results',[ ttext,' ',$
	  'The overplotted diamond points are the result',$
	  'of the fitting routine. This data has not been',$
	  'saved yet, thus will disappear when updating', $
	  '(ReDraw) the Xplot window...', $
	  '  ',$
	  'You can append the result of the fit to ',$
	  'the current set in a new additional column.',$
	  '   ','Do you want to do that?'])
	if tmp EQ 'No' then goto,out
	; add result as new column
	; place the "state" because it will be needed by xplot_addcol
	widget_control, event.top, SET_UVALUE = state, /NO_COPY
	xplot_addcol,event.top,yfit
	return
	end
 'Fit - Multiple peaks...':begin
	maxPeaks=5
	action=''
	XEdit,maxPeaks,Title='XPeakFit dimensions', $
	  Text='Maximum number of peaks:', Action=action, $
		Dialog_Parent=event.top
	IF action EQ 'CANCEL' THEN GoTo,out
	nPeaks = 3 < maxPeaks
	XPeakFit,tmpset, group=event.top, MaxPeaks=maxPeaks, NPeaks=nPeaks
	end
 'Fit - Non linear...':begin
        if n_elements(nlfitstr) EQ 0 then begin
	  p = [max(tmpset(1,*)),(max(tmpset(0,*))+min(tmpset(0,*)))/2.0,$
	      getfwhm(tmpset)/2.35]
          nlfitstr = ["equation='p[0]*exp(-((x-p[1])/p[2])^2/2)'",$
		'p='+vect2string(p)]
        endif

	xnlfit,tmpset, /NoDraw ,GROUP=event.top , $
	  xplotmain = state.wids.main, NO_BLOCK=state.no_block, $
	  init_pars=nlfitstr,psysvar=state.psysvar, ExtendedSet=tmpset1
	end
 'XAFS extraction (xaid_mu2chi)': $
	xaid_mu2chi,tmpset, GROUP=event.top, NO_BLOCK=state.no_block
 'XAFS Fourier Filter (xaid_ff)': $
	xaid_ff,tmpset, GROUP=event.top , NO_BLOCK=state.no_block
 'XAFS main window (XAID)': $
	xaid,tmpset, GROUP=event.top , NO_BLOCK=state.no_block
 else: print,'CALCULATIONS case not found: ',eventval
 ENDCASE
OUT:
widget_control, event.top, SET_UVALUE = state, /NO_COPY

end ;xplot_calculations
;
;=========================================================================
;
PRO xplot_controls_map,initval
on_error,2
;
; MAP the connected/Disconnected when Symbol=line (always connected)
;
if initval.Symbol EQ 0 then begin ; line selected
	widget_control,initval.wids.Connect,MAP=0 ; disappear
	widget_control,initval.wids.linestyle,MAP=1 ; appear
endif else begin
	widget_control,initval.wids.Connect,MAP=1 ; appear
	if initval.Connect EQ 0 then $ ; connected set
	  widget_control,initval.wids.linestyle,MAP=1 else $; appear
	  widget_control,initval.wids.linestyle,MAP=0 ; disappear
endelse
end ; xplot_controls_map
;
;=========================================================================
;
PRO xplot_controls_update,initval
;
; updates the control pannel 
;
if type(initval) NE 8 then begin
  message,/info,'Usage: xplot_controls_update,str_controls'
  return
endif
;values
if widget_info(initval.wids.main,/VALID_ID) then begin
  widget_control,initval.wids.Connect,set_droplist_select=initval.Connect
  widget_control,initval.wids.Symbol,set_droplist_select=initval.Symbol
  widget_control,initval.wids.Linestyle,set_droplist_select=initval.Linestyle
  widget_control,initval.wids.Axis,set_droplist_select=initval.Axis
  widget_control,initval.wids.Ticks,set_droplist_select=initval.Ticks
  widget_control,initval.wids.Font,set_droplist_select=initval.Font
  widget_control,initval.wids.Thickness,set_value=initval.Thickness
  widget_control,initval.wids.CharSize,set_value=initval.CharSize
  if widget_info(initval.wids.BackGround,/REALIZED) then $
    widget_control,initval.wids.BackGround,set_value=initval.BackGround
  if widget_info(initval.wids.Axiscolor,/REALIZED) then $
    widget_control,initval.wids.Axiscolor,set_value=initval.Axiscolor
  if widget_info(initval.wids.Color,/REALIZED) then $
    widget_control,initval.wids.Color,set_value=initval.Color
  ;
  ; MAP the connected/Disconnected when Symbol=line (always connected)
  ;
  xplot_controls_map,initval
endif
end ;xplot_controls_update
;
;=========================================================================
;
pro xplot_controls_event,event
;
; reduced event handler for Controls window (xplot_controls)
;
widget_control,event.top,get_uval=uval
        ooo = event
        ooo.top = uval
        ;if xregistered('xplot',/noshow) GT 0 then xplot_event,ooo
	xplot_controls_action,ooo
end ;xplot_controls_event
;
;=========================================================================
;
function xplot_controls,top,group=group , initval=initval, no_block=no_block
;
;
; creates and updates the controls panel.
;
;
on_error,2
if not(keyword_set(group)) then group=0
if not(keyword_set(top)) then top=0L

id = widget_base(/COLUMN,TITLE='Xplot controls')
;
; CONTROLS
;

wBquit = widget_button(id, value=  "Hide Controls",$
  UVALUE='CONTROLS')
wBdrop = widget_base(id,/COLUMN,/FRAME)

wBsymbol = widget_droplist(wBdrop,UVALUE='PSYMBOL', $
	VALUE = ['0: Line','1: +','2: *','3: Period',$
        '4: Diamond','5: Triangle','6: Square','7: X',$
	'8: User-defined     ',$
	'9: Undefined', '10:Histogram'], TITLE='   Symbol:')

; I had to add this tmp base to MAP correctly. I do not know why.  97/10/23
tmp = widget_base(wBdrop)
wBconnect = widget_droplist(tmp,UVALUE='PSYMSIGN', $
	VALUE = ['Connected','Disconnected        '], TITLE=' Sym Link:')

; I had to add this tmp base to MAP correctly. I do not know why.  97/10/23
tmp = widget_base(wBdrop)
wBlinestyle = widget_droplist(tmp, $
	VALUE=['0: Solid','1: Dotted','2: Dashed','3: Dash-Dot',$
	'4: -...','5: Long Dash        '],$
	TITLE='Linestyle:',UVALUE='LINESTYLE')

wBkind = widget_droplist(wBdrop, $
	VALUE=["Lin/Lin","Lin/Log","Log/Lin","Log/Log             "], $
	TITLE='     Axis:',UVALUE='KIND')

wBticks = widget_droplist(wBdrop, $
	VALUE=["In","Out","Grid                "], $
	TITLE='    Ticks:',UVALUE='TICKLEN',XSIZE=20)

ttFonts =  0
TT_Font_Manager,Get_List=ttFonts
ttFonts = 'TT: '+ttfonts
wBfonts = widget_droplist(wBdrop, VALUE= $
               ['3:  Simplex Roman', $    ; index-0
                '4:  Simplex Greek', $
                '5:  Duplex Roman', $
                '6:  Complex Roman', $
                '7:  Complex Greek', $
                '8:  Complex Italic', $
                '9:  Math and Special', $
                '10: Special', $
                '11: Gothic English', $
                '12: Simplex Script', $
                '13: Complex Script', $
                '14: Gothic Italian', $
                '15: Gothic German', $
                '16: Cyrillic', $
                '17: Triplex Roman', $
                '18: Triplex Italic', $
		'Hardware Font',$       ; index=16
		 ttfonts ],TITLE='     Font:',UVALUE='FONT')


wBsli = widget_base(id,/COLUMN,/FRAME)
wBthick = cw_fslider(wBsli, TITLE='Line/Symbol thickness',/FRAME, $
 	UVALUE='THICK',MIN=0.,MAX=10.0, XSIZE=205, /EDIT)

wBchar = cw_fslider(wBsli, TITLE='Character Size',/FRAME, $
 	UVALUE='CHARSIZE',MIN=0.1,MAX=3.0, XSIZE=210, /EDIT)

wBc = widget_base(id,/COLUMN,/FRAME)
wBclr_reset = widget_button(wBc, value="      Reset initial color    ",$
  UVALUE='CLR_RESET')
wBxloadct = widget_button(wBc, value=  "      Change color table     ",$
  UVALUE='XLOADCT')
wBxloadct = widget_button(wBc, value=  "      Default color table     ",$
  UVALUE='CLR_DEFAULT')
wBbackg= cw_clr_index1(wBc, LABEL='Bkgd: ', $
 	UVALUE='CLR_BACK', XSIZE=90, YSIZE=25, /UPDOWN)
wBaxiscolor= cw_clr_index1(wBc, LABEL='Axis: ', $
 	UVALUE='CLR_AXIS', XSIZE=90, YSIZE=25, /UPDOWN)
wBcolor= cw_clr_index1(wBc, LABEL='Lines:', $
 	UVALUE='CLR_LINES', XSIZE=90, YSIZE=25, /UPDOWN)

;
; set initial values
;
if keyword_set(initval) then begin
  ;wids
  initval.wids.Main = id
  initval.wids.Connect = wBconnect
  initval.wids.Symbol = wBsymbol
  initval.wids.Linestyle = wBlinestyle
  initval.wids.Axis = wBkind
  initval.wids.Ticks = wBticks
  initval.wids.Font = wBfonts
  initval.wids.Thickness = wBthick
  initval.wids.CharSize = wBchar
  initval.wids.BackGround = wBbackg
  initval.wids.Axiscolor = wBaxiscolor
  initval.wids.Color = wBcolor
  ;values
  ;tmp = initval
  xplot_controls_update,initval
  ;widget_control,wBconnect,set_droplist_select=initval.Connect
  ;widget_control,wBsymbol,set_droplist_select=initval.Symbol
  ;widget_control,wBlinestyle,set_droplist_select=initval.Linestyle
  ;widget_control,wBkind,set_droplist_select=initval.Axis
  ;widget_control,wBticks,set_droplist_select=initval.Ticks
  ;widget_control,wBfonts,set_droplist_select=initval.Font
  ;widget_control,wBthick,set_value=initval.Thickness
  ;widget_control,wBchar,set_value=initval.CharSize
endif

widget_control,id,set_uval=top
widget_control,id,/realize,group=group,  $
  event_pro='xplot_controls_event'

if keyword_set(initval) then begin
  widget_control,wBbackg,set_value=initval.BackGround
  widget_control,wBaxiscolor,set_value=initval.Axiscolor
  widget_control,wBcolor,set_value=initval.Color
endif
if keyword_set(no_block) then $
   xmanager, 'xplot_controls', id, GROUP_LEADER=group,/no_block else $
   xmanager, 'xplot_controls', id, GROUP_LEADER=group
return,[id,wBbackg,wBaxiscolor,wBcolor]
end
;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_CONTROLS_ACTION
;	PURPOSE: to change plot attributes in XPlot (i.e. Log plots,
;		symbols, thicknesses, colors, etc). In other words, 
;		permits the non-interactive manipulations of all the 
;		controls under the "Show Controls" button.
;		If the "Show Controls" window is opened, it also updates it.
;
;	CALLING SEQUENCE: XPLOT_CONTROLS_ACTION,parent [, keywords]
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;
;	KEYWORD PARAMETERS [USE AT LEAST ONE]:
;		Note that some keyword values are assigned to zero. For 
;		setting them use KEYWORD='0', because KEYWORD=0 means not 
;		setting the keyword. For non-zero values, both KEYWORD='n'
;		and KEYWORD=n are, in general, accepted.
;
;		PSYMSIGN set plot to concected (PSYMSIGN='0') or disconnected
;			(PSYMSIGN='1') line.  This keyword does not make any 
;			effect if PSYMBOL='0'
;		PSYMBOL set symbol:
;			'0' line
;			1 +
;			2 *
;			3 Period
;			4 Diamond
;			5 Triangle
;			6 Square
;			7: X
;			8: User-defined
;			9: undefined
;			10: histogram
;		LINESTYLE define the line style:
;			'0' Solid
;			1 Dotted
;			2 Dashed
;			3 Dash-Dot
;			4 -...-...
;			5 Long dash
;		KIND log or lin plots in X/Y:
;			'0' lin/lin
;			1   lin/log
;			2   log/lin
;			3   log/log
;		TICKLEN internal/external ticks or grid
;			'0' internal
;			1   external
;			2   grid
;		THICK <float> the line thickness
;		CHARSIZE <float> the character size
;		CLR_BACK <integer> the background color
;		CLR_AXIS <integer> the axis color
;		CLR_LINES <integer> the lines color
;		CLR_RESET set this keyword to reset the original background,
;			axis and lines colors.
;		CLR_DEFAULT set this keyword to set the color table to the
;			default one (B&W + Tek_Color)
;		FONT selects a font 
;                       Index 0 -> 3:  Simplex Roman
;                       Index 1 -> 4:  Simplex Greek
;                       Index 2 -> 5:  Duplex Roman
;                       Index 3 -> 6:  Complex Roman
;                       Index 4 -> 7:  Complex Greek
;                       Index 5 -> 8:  Complex Italic
;                       Index 6 -> 9:  Math and Special
;                       Index 7 -> 10: Special
;                       Index 8 -> 11: Gothic English
;                       Index 9 -> 12: Simplex Script
;                       Index 10 -> 13: Complex Script
;                       Index 11 -> 14: Gothic Italian
;                       Index 12 -> 15: Gothic German
;                       Index 13 -> 16: Cyrillic
;                       Index 14 -> 17: Triplex Roman
;                       Index 16 -> 18: Triplex Italic
;                       Index 16 -> hardware font
;                       Index 17,18,... -> TT Fonts (as returned by 
;				TT_Font_manager)
;		NOREFRESH set this keyword to avoid refreshing the graph
;			after the loading process. Useful when loading
;			many curves.
;
;	MODIFICATION HISTORY:
;		97-09-11 srio@esrf.fr initial version from xplot code
;		99-03-12 srio@esrf.fr adds Hardware and TT fonts
;
;-
PRO xplot_controls_action,input,PSYMSIGN=PSYMSIGN,PSYMBOL=PSYMBOL,$
  LINESTYLE=LINESTYLE,$
  KIND=KIND, TICKLEN=TICKLEN, THICK=THICK, CHARSIZE=CHARSIZE, $
  CLR_BACK=CLR_BACK, CLR_AXIS=CLR_AXIS, CLR_LINES=CLR_LINES, $
  CLR_RESET=CLR_RESET,CLR_Default=clr_default, $
  FONT=FONT,NOREFRESH=NOREFRESH, $
  CONTROLS=CONTROLS ; this one for use with xplot.pto, not for interactive use
  
 
catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XPLOT_CONTROLS_ACTION: error caught: '+!error_state.msg)
  catch, /cancel
  goto,out
endif

eventUVal=0
pannelupdate=0
case type(input) of
8: begin
   event = input
   widget_control, event.top , GET_UVALUE = state, /NO_COPY
   Widget_Control, event.id, get_UValue=eventUVal
   wid_top=event.top
   if keyword_set(CONTROLS) then eventUVal='CONTROLS'
   end
3: begin
   widget_control, input, GET_UVALUE = state, /NO_COPY
   wid_top=input
   eventuval = ''
   if keyword_set(PSYMSIGN) then eventUVal=[eventUVal,'PSYMSIGN']
   if keyword_set(PSYMBOL) then eventUVal=[eventUVal,'PSYMBOL']
   if keyword_set(LINESTYLE) then eventUVal=[eventUVal,'LINESTYLE']
   if keyword_set(KIND) then eventUVal=[eventUVal,'KIND']
   if keyword_set(TICKLEN) then eventUVal=[eventUVal,'TICKLEN']
   if keyword_set(THICK) then eventUVal=[eventUVal,'THICK']
   if keyword_set(CHARSIZE) then eventUVal=[eventUVal,'CHARSIZE']
   if keyword_set(CLR_BACK) then eventUVal=[eventUVal,'CLR_BACK']
   if keyword_set(CLR_AXIS) then eventUVal=[eventUVal,'CLR_AXIS']
   if keyword_set(CLR_LINES) then eventUVal=[eventUVal,'CLR_LINES']
   if keyword_set(CLR_RESET) then eventUVal=[eventUVal,'CLR_RESET']
   if keyword_set(CLR_DEFAULT) then eventUVal=[eventUVal,'CLR_DEFAULT']
   if keyword_set(FONT) then eventUVal=[eventUVal,'FONT']
   my_value = 0
   if n_elements(eventuval) GE 2 then $
     eventuval=eventuval(1:n_elements(eventuval)-1)
   ;if widget_info(state.strcontrols.wids.main,/VALID_ID) then $
   ;	pannelupdate = 1 
   end
else: begin
   message,/info,'Usage: xplot_controls_action, event (or parent_id)'
   return
   end
endcase

; srio 98/08/25
if widget_info(state.strcontrols.wids.main,/VALID_ID) then pannelupdate = 1 

for i=0,n_elements(eventUVal)-1 do begin

  event_tmp = eventUVal(i)
  if type(input) EQ 3 then begin
    command = 'my_value = '+event_tmp
     itmp = execute(command)
  endif

  CASE event_tmp of
	'PSYMSIGN':	BEGIN
		if not(keyword_set(PSYMSIGN)) then my_value=event.index
		state.xpdt.connect = my_value
		state.strcontrols.connect = my_value
		if pannelupdate then xplot_controls_map,state.strcontrols
		;xplot_plot,state
		END
	'PSYMBOL':	BEGIN
		if not(keyword_set(PSYMBOL)) then my_value=event.index
		state.xpdt.psym = my_value
		state.strcontrols.symbol = my_value
		if pannelupdate then xplot_controls_map,state.strcontrols
		;xplot_plot,state
		END
	'LINESTYLE':	BEGIN
		if not(keyword_set(LINESTYLE)) then my_value=event.index
                state.xpdt.linestyle = my_value
		state.strcontrols.LineStyle = my_value
		;xplot_plot,state
		END
	'KIND':	BEGIN
		if not(keyword_set(KIND)) then my_value=event.index
		temp = my_value
		state.strcontrols.Axis = my_value
                if temp eq 0 then state.xpdt.kind = 1 
                if temp eq 1 then state.xpdt.kind = 3 
                if temp eq 2 then state.xpdt.kind = 4 
                if temp eq 3 then state.xpdt.kind = 5 
                ;if temp eq 4 then state.xpdt.kind = 2 
		;xplot_plot,state
		END
	'TICKLEN':	BEGIN
		if not(keyword_set(TICKLEN)) then my_value=event.index
		state.strcontrols.Ticks = my_value
		temp = my_value
                if temp eq 0 then state.xpdt.sysvar(1) = '!p.ticklen = 0.02 '
                if temp eq 1 then state.xpdt.sysvar(1) = '!p.ticklen = -0.02 '
                if temp eq 2 then state.xpdt.sysvar(1) = '!p.ticklen = 0.5'
		;xplot_plot,state
		END
	'THICK':	BEGIN
		if not(keyword_set(THICK)) then $
			widget_control,event.id,GET_VALUE=my_value
		temp = my_value
		state.strcontrols.Thickness = temp
		state.xpdt.Thick = temp
		;xplot_plot,state
		END
	'CHARSIZE':	BEGIN
		if not(keyword_set(CHARSIZE)) then $
		  widget_control,event.id,GET_VALUE=my_value
		temp = my_value
		state.strcontrols.CharSize = temp
		state.xpdt.sysvar(0) = '!p.charsize = '+ $
			strcompress(temp,/rem)
		;xplot_plot,state
		END
	'CLR_BACK':	BEGIN
                if not(keyword_set(CLR_BACK)) then $
		  widget_control,event.id,GET_VALUE=my_value
		temp = my_value
		state.strcontrols.BackGround = temp
		state.xpdt.sysvar(6) = '!p.background = '+ $
                        strcompress(temp,/rem)
                ;xplot_plot,state
		END
	'CLR_AXIS':	BEGIN
		if not(keyword_set(CLR_AXIS)) then $
		  widget_control,event.id,GET_VALUE=my_value
		temp = my_value
		state.strcontrols.Axiscolor = temp
		;state.xpdt.sysvar(6) = '!p.background = '+ $
                ;        strcompress(temp,/rem)
                ;xplot_plot,state
		END
	'CLR_LINES':	BEGIN
		if not(keyword_set(CLR_LINES)) then $
		  widget_control,event.id,GET_VALUE=my_value
		temp = my_value
		state.strcontrols.Color = temp
                state.xpdt.sysvar(7) = '!p.color = '+ $
                        strcompress(temp,/rem)
		;xplot_plot,state
		END
	'CLR_RESET':	BEGIN
;		state.xpdt.sysvar(6) = state.xpdt.sysvar_old(6)
;		state.xpdt.sysvar(7) = state.xpdt.sysvar_old(7)
;		tmp = execute(state.xpdt.sysvar(6))
;		tmp = execute(state.xpdt.sysvar(7))
		if not(keyword_set(CLR_RESET)) then begin
		  widget_control,state.wids.controls(1),Set_val=!p.background
		  widget_control,state.wids.controls(2),Set_val=!p.color
		  widget_control,state.wids.controls(3),Set_val=!p.color
		endif
		state.strcontrols.BackGround = !p.background
		state.strcontrols.Axiscolor = !p.color
		state.strcontrols.Color = !p.color
		IF !d.n_colors GT 255 THEN BEGIN
		  itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
		     [ 'Color table changed.',$
		     'As you are using a 24-bit display, you must',$
		     'close and reopen the Xplot controls window to',$
		     'get there the colors updated.'])
		ENDIF
		;xplot_plot,state
		END
	'CLR_DEFAULT':	BEGIN
		LoadCT,0
		Tek_Color
		IF !d.n_colors GT 255 THEN BEGIN
		  itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
		     [ 'Color table changed.',$
		     'As you are using a 24-bit display, you must',$
		     'close and reopen the Xplot controls window to',$
		     'get there the colors updated.'])
		ENDIF
		END
	'FONT':	BEGIN
		if not(keyword_set(FONT)) then my_value=event.index
                state.strcontrols.Font = my_value
		IF my_value LE 15 THEN BEGIN
		  !p.font=-1
		  state.xpdt.sysvar[8] = '!p.font=-1'
		  xyouts,"'!"+strcompress(3+ $
			fix(state.strcontrols.Font),/REM)+"'"
		ENDIF ELSE BEGIN
		  IF my_value EQ 16 THEN BEGIN
		    Message,/Info,'Setting hardware font'
		    state.xpdt.sysvar[8] = '!p.font=0'
		  ENDIF ELSE BEGIN
		    !p.font=1
		    state.xpdt.sysvar[8] = '!p.font=1'
		    Message,/Info,'Setting TT font'+StrCompress(my_value-17)
		    TT_Font_Manager,Set_Index=my_value-17
		  ENDELSE
		ENDELSE
		;xplot_plot,state
		END
	; the next two entries are only for interactive use

	'XLOADCT': BEGIN
		Xloadct,GROUP=event.top,/Modal
		IF !d.n_colors GT 255 THEN BEGIN
		  itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
		     [ 'Color table changed.',$
		     'As you are using a 24-bit display, you must',$
		     'close and reopen the Xplot controls window to',$
		     'get there the colors updated.'])
		ENDIF
		xplot_plot,state
		END
	'CONTROLS':	BEGIN
                widget_control,event.id,get_value=tmp
                case tmp of
                  'Show Controls': begin
                     widget_control,event.id,set_value='Hide Controls'
		     tmpstr = state.strcontrols
		     state.wids.controls=xplot_controls(event.top, $
			group=event.top,initval=tmpstr, $
			no_block=state.no_block)
		     state.strcontrols = tmpstr
                     end
                  'Hide Controls': begin
                     widget_control,state.wids.showcontrols, $
			set_value='Show Controls'
                     if widget_info(state.wids.controls(0),/valid_id) then $
                       widget_control,state.wids.controls(0),/DESTROY
                     end
                  else:
                endcase
		goto,out
		END
	else:
  ENDCASE
endfor
if pannelupdate then xplot_controls_update,state.strcontrols
if not(keyword_set(norefresh)) then xplot_plot,state

out:
widget_control, wid_top , SET_UVALUE = state, /NO_COPY
end ; xplot_controls_action

;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_RELOADFILE 
;	PURPOSE: re-load a file or variable in xplot
;	CALLING SEQUENCE: XPLOT_RELOADFILE,parent 
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;
;	KEYWORD PARAMETERS:
;
;	MODIFICATION HISTORY:
;		2002-03-06 srio@esrf.fr initial version from xplot code
;
;-
PRO XPLOT_RELOADFILE,id

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, Dialog_Parent=id,$
    'XPLOT_RELOADFILE: error caught: '+!error_state.msg)
  catch, /cancel
  IF N_Elements(state) NE 0 THEN widget_control, state.wids.main , SET_UVALUE = state
  RETURN
endif
;
widget_control, id , GET_UVALUE = state, /NO_COPY


if state.fileinf.name EQ '<none>' or state.fileinf.name EQ '' then begin
  itmp = widget_message(/Error,dialog_parent=state.wids.main,$
    'Xplot data came from variable, not from file.')
  widget_control, state.wids.main , SET_UVALUE = state
  RETURN
endif

widget_control,state.wids.message,set_value='Reloading '+state.fileinf.name(0)+'...'
specon = state.spec.on
filename = state.fileinf.name
widget_control,state.wids.main,set_uvalue=state,/no_copy,/hourglass
if specon then begin
  xplot_loadfile,id,SPEC=filename
endif else begin
  xplot_loadfile,id,filename
endelse

end ;xplot_reloadfile



;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_AUTORELOADFILE 
;	PURPOSE: re-load a file or variable in xplot
;	CALLING SEQUENCE: XPLOT_RELOADFILE,parent
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named variable)
;
;	KEYWORD PARAMETERS:
;		Action = One of the following strigns (case sensitive):
;			'Preferences...' Pops-up the preferences window
;			'Start' Start timer (autoreload) mode
;			'Stop' Stop autoreload mode
;		SET_CHECK: Flag for controlling the updating (autoreload):
;			0: ALways reload file
;			1: Reload file if it has been modified 
;		SET_TIMER: Time [sec] waited between realods 
;
;	EXAMPLE:
;		
;		xplot,Parent=p,/No_Block
;		xplot_Loadfile,p,'xplot.dat'
;		xplot_AutoReLoadFile,p,Set_timer=1.0
;		xplot_AutoReLoadFile,p,Action='Start'
;		xplot_AutoReLoadFile,p,Action='Stop'
;
;	MODIFICATION HISTORY:
;		2002-03-08 srio@esrf.fr initial version 
;
;-
PRO XPLOT_AUTORELOADFILE,id,Action=action, $
   SET_CHECK=set_Check, Set_Timer=set_Timer
catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, Dialog_Parent=id,$
    'XPLOT_AUTORELOADFILE: error caught: '+!error_state.msg)
  catch, /cancel
  IF N_Elements(state) NE 0 THEN widget_control, state.wids.main , SET_UVALUE = state
  RETURN
endif
;
widget_control, id , GET_UVALUE = state, /NO_COPY

id1 = state.wids.autoReLoadFile

IF N_Elements(set_Check) NE 0 THEN state.autoreloadfile.set.check = StrCompress(set_Check,/Rem)
IF N_Elements(set_Timer) NE 0 THEN state.autoreloadfile.set.timer = set_Timer
IF N_Elements(action) EQ 0 THEN action = ''

		CASE action of
		'Preferences...':BEGIN
			IF state.autoReLoadFile.reLoadFlag EQ 1 THEN BEGIN
			  Widget_Control,id,TLB_Set_Title='Xplot '+Xplot_Version()
			  state.autoReLoadFile.reLoadFlag = 0
			ENDIF
			tmp = state.autoReLoadFile.set
			XscrMenu,tmp,/NOTYPE,/INTERP, $
			TITLES=state.autoReLoadFile.titles, $
 			ACTION=action,WTITLE='AutoReLoad option [timer]',Dialog_Parent=id ; ,$
			; HELP=xplot_text('OPERWITHSETS')
			IF action EQ 'DONT' THEN GoTo,out
			state.autoReLoadFile.set = tmp
			IF StrCompress(state.autoReLoadFile.set.start[0],/Rem) EQ '1' THEN BEGIN
			  state.autoReLoadFile.reLoadFlag = 1
			  id1 = state.wids.autoReLoadFile
			  timer = state.autoReLoadFile.set.timer
			  Widget_Control, id, SET_UVALUE = state, /NO_COPY
			  Xplot_ReLoadFile,id
			  Widget_Control,id,TLB_Set_Title= $
				'XPLOT in AutoReLoadFile mode ['+StrCompress(Fix(timer),/Rem)+' sec]'
			  ; Widget_Control,id, timer=timer
			  Widget_Control, id1 , timer=timer
			  RETURN
			ENDIF
			;state.autoReLoadFile.set.start[0]='0'
			END
		'Start': BEGIN
			state.autoReLoadFile.reLoadFlag = 1
			id1 = state.wids.autoReLoadFile
			timer = state.autoReLoadFile.set.timer
			Widget_Control, id, SET_UVALUE = state, /NO_COPY
			Xplot_ReLoadFile,id
			Widget_Control,id,TLB_Set_Title= $
				'XPLOT in AutoReLoadFile mode ['+StrCompress(Fix(timer),/Rem)+' sec]'
			;Widget_Control,id,TLB_Set_Title='XPLOT in AutoReLoadFile mode'
			;Widget_Control,id, timer=timer
			Widget_Control, id1 , timer=timer
			RETURN
			END
		'Stop': BEGIN
			state.autoReLoadFile.reLoadFlag = 0
			Widget_Control, id, SET_UVALUE = state, /NO_COPY
			Widget_Control,id,TLB_Set_Title='Xplot '+Xplot_Version()
			;Xplot_ReLoadFile,id
			RETURN
			
			END
		else: 
		ENDCASE

out:
widget_control, id , SET_UVALUE = state, /NO_COPY
END ; Xplot_AutoReLoadFile

;
;=========================================================================

;+
;
;	This routine is part of the xplot package. It is used internally
;	by xplot, but it also permits to manipulate non-interactively
;	the xplot window. Only the last feature is documented here.
;
;	NAME: XPLOT_GETCURRENTDATA 
;	PURPOSE: stores in a variable the current data in Xplot.
;	CALLING SEQUENCE: XPLOT_GETCURRENTDATA,parent,data
;	INPUTS:
;		PARENT the widget id of the main Xplot window to 
;		It has to be retrieved when the main window is created:
;		PARENT = 0 ; initializes the variable 
;		Xplot,...,PARENT=parent ;stores widget id in parent (named 
;		variable)
;
;	OUTPUTS:
;		DATA  an array with the data.
;
;	KEYWORD PARAMETERS:
;		ALLCOLUMNS: When set, returns all the columns in the 
;			current set. Otherwise (default) returns only
;			the two columns in the current graph.
;
;	MODIFICATION HISTORY:
;		2003-11-17 srio@esrf.fr initial version 
;
;-
PRO xplot_getcurrentdata,id,data,allcolumns=allcolumns

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'XPLOT_GETCURRENTDATA: error caught: '+!error_state.msg)
  catch, /cancel
  if n_elements(state) NE 0 then widget_control, wid, SET_UVALUE = state, $
    /NO_COPY
  return
endif


data=0
if not(keyword_set(group)) then group=id
;
widget_control, id , GET_UVALUE = state, /NO_COPY
plotfile,state.dthcurrent,data,/NODATA
IF Keyword_Set(allcolumns) THEN GoTo,out

data=data[[state.xpdt.Xcol-1,state.xpdt.Ycol-1],*]

out:
widget_control, id , SET_UVALUE = state, /NO_COPY
end ; xplot_GetCurrentData

;
;=========================================================================
;
PRO xplot_evaluate,set,eval_str=eval_str,group=group,no_block=no_block
;
; drives the Calculations/Operations with sets... option
;
on_error,2
;
; to evaluate an expression [Calculations Menu]
;
if not(keyword_set(eval_str)) then eval_str = $
          xplot_defaults('XPevaluate')
;
flags=['1','1','1','1','w(2) EQ 1','1','w(5) EQ 1','1']
titles=['Expression for x Cx=f(Ax): ', 'Expression for y Cy=f(Ay,By): ', $
  'Number of data sets:', $
  'A data set:', 'B data file:', $
  'Write output to file: ', 'Output file name:', $
  'Overplot original data:' ]

eval_str_old = eval_str
XscrMenu,eval_str,/NOTYPE,/INTERP,FLAGS=flags,TITLES=titles, $
  ACTION=action,WTITLE='Operations with sets',GROUP=group,$
  HELP=xplot_text('OPERWITHSETS')

if action eq 'DONT' then begin
  eval_str = eval_str_old
  return
endif
;
; start operations
;
; load data 
;
IF KEYWORD_SET(SET) THEN BEGIN
if eval_str.set1 ne 'current set' then begin
  tmp = rascii(eval_str.set1)
  if not(keyword_set(tmp)) then begin
    tmp = widget_message(DIALOG_PARENT=group, $
    ['Error opening file','Returning'],/ERROR)
    eval_str = eval_str_old
    return
  endif
  set1 = tmp
endif else  begin
  if not(keyword_set(set)) then begin
    tmp = widget_message(DIALOG_PARENT=group, $
    ['Empty data set','Returning'],/ERROR)
    eval_str = eval_str_old
    return
  endif
  set1 = set
endelse
Ax = reform(set1(0,*))
Ay = reform(set1(1,*))
;
; when using two curves 
;
if strcompress(eval_str.nset(0),/REM)  eq '1' then begin
  ; load file
  tmp = rascii(strcompress(eval_str.set2,/REM))
  if not(keyword_set(tmp)) then begin
    ;xtext,TEXT=['Error opening file','Returning'],GROUP=group
    tmp = widget_message(DIALOG_PARENT=group, $
    ['Error opening file','Returning'],/ERROR)
    eval_str = eval_str_old
    return
  endif
  set2 = tmp
  Bx = reform(tmp(0,*))
  By = reform(tmp(1,*))
endif
ENDIF
;
; evaluate the Cx expression
;
Cx = 0
cxx = Ax[where(Ax LE 4000)]
tmp = execute(eval_str.expression_x)
if tmp ne 1 then begin
  ;xtext,TEXT='Expression for x not understood',GROUP=group $
  tmp = widget_message(DIALOG_PARENT=group, $
  'Expression for x not understood',/ERROR)
  eval_str = eval_str_old
  return
endif    ; else set1(0,*) = Cx
;
;
; interpolate set1 to new abscissas (if needed) 
;
ineed_int = 0
IF N_Elements(Ax) NE N_Elements(Cx) THEN BEGIN
  ineed_int = 1
ENDIF ELSE BEGIN
  IF ABS(Total(Ax-Cx)) GT 1e-3 THEN ineed_int = 1 
ENDELSE
IF ineed_int THEN BEGIN
  Ayold = Ay
  Ay = interpol(Ay,Ax,Cx)
  Ax=Cx
ENDIF

;
;
; interpolate set2 to new abscissas (if needed) 
;
if strcompress(eval_str.nset(0),/REM)  eq '1' then begin
  ineed_int = 0
  IF N_Elements(Bx) NE N_Elements(Cx) THEN BEGIN
    ineed_int = 1
  ENDIF ELSE BEGIN
    IF ABS(Total(Bx-Cx)) GT 1e-3 THEN ineed_int = 1 
  ENDELSE
  IF ineed_int THEN BEGIN
    Byold = By
    By = interpol(By,Bx,Cx)
    Bx=Cx
  ENDIF
endif
;
print,eval_str.expression_y
Cy = 0
tmp = execute(eval_str.expression_y)
if tmp ne 1 then begin
  tmp = widget_message(DIALOG_PARENT=group, $
  'Expression for y not understood',/ERROR)
  eval_str = eval_str_old
  return
endif
;
;setnew = fltarr(2,n_elements(Cx)) ;set1
;setnew(0,*) = Cx
;setnew(1,*) = Cy
setnew = Make_Set(Cx,Cy)
;
; show and exit
;
choice = strcompress(eval_str.save_as(0),/REM)
;
; overplot old sets of data
;
if ( strcompress(eval_str.over(0),/REM) eq '1') then begin
  if strcompress(eval_str.nset(0),/REM)  eq '1' then begin
    xmin = min( [Reform(setnew(0,*)),Reform(set1(0,*)),Reform(set2(0,*))],$
	max = xmax)
    ymin = min( [Reform(setnew(1,*)),Reform(set1(1,*)),Reform(set2(1,*))],$
	max = ymax)
    Widget_Control,/HourGlass
    XPlot, NO_BLOCK=1,Parent=p, XRange=[xmin,xmax], $
       WTitle='Xplot '+xplot_version()+' Results of Operations with sets...'
    XPlot_LoadFile,p,set1,/NoRefresh
    XPlot_Controls_action,p,LineStyle=1,/NoRefresh
    XPlot_SaveCurrent,p
    XPlot_LoadFile,p,set2,/NoRefresh
    XPlot_Controls_action,p,LineStyle=2,/NoRefresh
    XPlot_SaveCurrent,p
    XPlot_LoadFile,p,setnew,/NoRefresh
    XPlot_Controls_action,p,LineStyle='0',/NoRefresh
    XPlot_Exebuffer,p,SetBuffer=$
    "legend,['Set A','Set B','Set C'],linestyle=[1,2,0]"
  endif else begin
    xmin = min( [Reform(setnew(0,*)),Reform(set1(0,*))],max = xmax)
    ymin = min( [Reform(setnew(1,*)),Reform(set1(1,*))],max = ymax)
    Widget_Control,/HourGlass
    XPlot, NO_BLOCK=1,Parent=p, XRange=[xmin,xmax], $
       WTitle='Xplot '+xplot_version()+' Results of Operations with sets...'
    XPlot_LoadFile,p,set1,/NoRefresh
    XPlot_Controls_action,p,LineStyle=1,/NoRefresh
    XPlot_SaveCurrent,p
    XPlot_LoadFile,p,setnew,/NoRefresh
    XPlot_Controls_action,p,LineStyle='0',/NoRefresh
    XPlot_Exebuffer,p,SetBuffer=$
    "legend,['Set A','Set C'],linestyle=[1,0]"
  endelse
endif else begin
  if keyword_set(no_block) then $
  xplot,setnew,/no_block,wtitle=$
	'Xplot '+xplot_version()+' Results of Operations with sets...' else $
  xplot,setnew,wtitle=$
	'Xplot '+xplot_version()+' Results of Operations with sets...'
endelse


;
if choice eq '1' then begin
  if checkfile(eval_str.filename) EQ 1 then begin
     tmp = widget_message(DIALOG_PARENT=group, $
        ['File '+eval_str.filename+' already exists',$
	'Overwrite it?'],/CANCEL,/DEFAULT_CANCEL)
     if tmp EQ 'Cancel' then return 
  endif
  openw,Unit,strcompress(eval_str.filename,/REM),/GET_LUN
  printf,Unit,setnew
  free_lun,Unit
  tmp = widget_message(DIALOG_PARENT=group, $
	'File '+eval_str.filename+' written to disk', /INFORMATION)
endif
;
end ;xplot_evaluate
;
;=========================================================================
;
FUNCTION xplot_fitgauss,set,OUTTEXT=outtext
;
; drives the Calculations/Fitting/Gaussian option
;
on_error,2
text = ['The gaussian fit option of XPlot permits to fit', $
	'the current data y=f(x) to one of the following functions: ', $
	'  ',$
	'  1) F(x) = A0*EXP(-z^2/2) + A3', $ 
	'     with         z=(x-A1)/A2', $
	'     where:', $
	'     A0 = height of exp, A1 = center of exp, A2 = sigma (the width)',$
	'     A3 = constant term', $
	'     (it uses the function GAUSS_FIT)', $
	'  ',$
	'  2) F(x) = A0*EXP(-z^2/2) + A3 + A4*x + A5*x^2', $
	'     with         z=(x-A1)/A2', $
	'     where:', $
	'     A0 = height of exp, A1 = center of exp, A2 = sigma (the width)',$
	'     A3 = constant term, A4 = linear term, A5 = quadratic term.', $
	'     (it uses the function GAUSSFIT)', $
	'  ',$
	'Please read the documentation in the Help/Routines for',$
	'detailed references of the method used for each fit.']

tmp = widget_message(text,/INFO,/CANCEL)
if tmp EQ 'Cancel' then return,0

widget_control,sensitive=0
itmp = wmenu2([' F(x) = A0*EXP(-z^2/2) + A3', $
	'  F(x) = A0*EXP(-z^2/2) + A3 + A4*x + A5*x^2'])
widget_control,sensitive=1


setx = reform(set(0,*))
sety = reform(set(1,*))

case  itmp  of
  0:BEGIN
	yfit = Gauss_Fit(setx,sety,Result)
	outtext = [ $
	'   Gaussina Fit (GAUSS_FIT) results ', $
	'   Fit result :  F(x) = A0*EXP(-z^2/2) + A3, z=(x-A1)/A2', $
	'            A0: '+strcompress(Result(0),/rem), $
	'            A1: '+strcompress(Result(1),/rem), $
	'            A2: '+strcompress(Result(2),/rem), $
	'            A3: '+strcompress(Result(3),/rem)]

	for i=0,n_elements(outtext)-1 do print,outtext(i)
    END
  1:BEGIN
	yfit = GaussFit(setx,sety,Result)
	outtext = [ $
	'   Gaussina Fit (GAUSS_FIT) results ', $
	'   Fit result :  F(x) = A0*EXP(-z^2/2) + A3 + A4*x + A5*x^2', $
	'                 z=(x-A1)/A2', $
	'            A0: '+strcompress(Result(0),/rem), $
	'            A1: '+strcompress(Result(1),/rem), $
	'            A2: '+strcompress(Result(2),/rem), $
	'            A3: '+strcompress(Result(3),/rem), $
	'            A4: '+strcompress(Result(4),/rem), $
	'            A5: '+strcompress(Result(5),/rem)]

	for i=0,n_elements(outtext)-1 do print,outtext(i)
    END
  else:
endcase

outtext = [outtext, ' ', $
  'Minimum y value for the fit: '+strcompress(min(yfit,max=yfitmax),/rem), $
  'Maximum y value for the fit: '+strcompress(yfitmax,/rem)]
return,yfit
end

;
;=========================================================================
;
function xplot_restorestate,OLDSTATE=oldstate,FILENAME=filename,IERR=ierr,$
  WSIZE=wsize,Group=group
;
; drives the FIle/Restore xplot backup file
;
on_error,2
catch,ierr
 
if ierr NE 0 then begin
  txt = 'XPLOT_RESTORESTATE: Error restoring Xplot backup.'
  message,/info,txt
  itmp=Dialog_Message(Dialog_Parent=group, /Error, txt)
  return,0
endif

if not(keyword_set(filename)) then filename = 'xplot.bck'

;
; initialize variables
;
version = 0 & state = 0 & dtcurrent = 0 & wsize=[0,0]
red = 0  &  green = 0  &  blue = 0
for i=0,10 do tmp =  execute('dtbuffer'+strcompress(i,/rem)+' = 0')

;
; restore
;
restore,filename,/VERBOSE
print,'XPLOT_RESTORESTATE: Restored Xplot backup '+version
;
; re-arrange handlers
;
if keyword_set(dtcurrent) then begin
  h1 = handle_create(value=dtcurrent)
  ;handle_value, h1,dtcurrent,/set
  state.dthcurrent=h1
endif else state.dthcurrent = 0L

h1 = 0L
print,'XPLOT_RESTORESTATE: old dthbuffer: ',state.dthbuffer
if keyword_set(state.buffer) then begin
  for i=0L,n_elements(state.buffer)-1 do begin
    h1 = handle_create()
    command = 'handle_value, h1,dtbuffer'+strcompress(i,/rem)+',/set'
    tmp = execute(command)
    state.dthbuffer(i)=h1
    ; substitute the new handler name in the buffer text
     line = state.buffer(i)
     s1= strpos(line,'L')
     newline = 'plotfile, '+strcompress(h1,/rem)+strmid(line,s1,strlen(line)-s1)
     state.buffer(i) = newline
  endfor
endif
print,'XPLOT_RESTORESTATE: new dthbuffer: ',state.dthbuffer
;
; color tables
;
tvlct,red,green,blue
;
; use correct widget identifiers
;
if keyword_set(oldstate) then state.wids = oldstate.wids

return,state
end

;
;=========================================================================
;
pro xplot_savestate,state,FILENAME=filename
;
; drives the File/Save as Xplot backup ... option
;
on_error,2
version = xplot_version() ;'Xplot version 2.0, (c) MSR-ESRF'
tvlct,red,green,blue,/get
wsize = [!d.x_size,!d.y_size]
if not(keyword_set(filename)) then filemane = 'xplot.bck'

if keyword_set(state.dthcurrent) then $
	plotfile,state.dthcurrent,dtcurrent,/nodata else $
	dtcurrent = 0

strdtbuffer = ''
if keyword_set(state.buffer) then begin
  for i=0L,n_elements(state.buffer)-1 do begin
    command = 'plotfile,state.dthbuffer('+strcompress(i,/rem)+$
		'),/nodata,dtbuffer'+strcompress(i,/rem)
    print,'XPLOT_SAVESTATE: executing: '+command
    tmp = execute(command)
    strdtbuffer = strdtbuffer+'dtbuffer'+strcompress(i,/rem)+','
  endfor
endif

command = 'save,version,state,dtcurrent,'+$
    'red,green,blue,wsize,'+strdtbuffer+$
   'FILENAME='+"'"+strcompress(filename,/rem)+"'"+',/VERBOSE'
print,'XPLOT_SAVESTATE: executing: '+command
tmp = execute(command)
end

;
;=========================================================================
;
PRO xpstr_update,state, $
   BUFFER=buffer, NOBUFFER = nobuffer, $
   DTHCURRENT=dthcurrent, NODTHCURRENT=nodthcurrent, $
   DTHBUFFER=dthbuffer, NODTHBUFFER=nodthbuffer, $
   EXEBUFFER=exebuffer, NOEXEBUFFER=noexebuffer, $
   NOCOLTITLES=nocoltitles, COLTITLES=coltitles, $
   NOPLOTNAME_BUFFER=noplotname_buffer, PLOTNAME_BUFFER=plotname_buffer, $
   AUTORELOADFILE=autoReLoadFile

;
; This procedure updates the 'date' or 'buffer' tags in the structure 'state'
;  for the XPlot program
;
;  Why a procedure to update a tag, and I do not use the simple
;  expression state.data = newdata or state.buffer = newbuffer ?
;
; Because when I create the state structure, the dimensions of
; the tags are those of the initialization values. Now, I need that
; data very dimensions depending on the data to be display.
; 
; This procedures copies and re-defines the structure
; The structure MUST be anonymous
;
; MSR 94-02-18
; MSR 95-02-18
;
on_error,2
;
state_old = state

if keyword_set(buffer) then newbuffer = buffer else newbuffer = state_old.buffer
if keyword_set(nobuffer) then newbuffer = 0

if keyword_set(dthbuffer) then newdthbuffer = dthbuffer else $
		newdthbuffer = state_old.dthbuffer
if keyword_set(nodthbuffer) then newdthbuffer = 0L

if keyword_set(dthcurrent) then newdthcurrent = dthcurrent else $
		newdthcurrent = state_old.dthcurrent
if keyword_set(nodthcurrent) then newdthcurrent = 0L
;
if keyword_set(exebuffer) then newexebuffer = exebuffer else $
		newexebuffer = state_old.exebuffer
if keyword_set(noexebuffer) then newexebuffer = 0
;
if keyword_set(coltitles) then newcoltitles = coltitles else $
		newcoltitles = state_old.coltitles
if keyword_set(nocoltitles) then newcoltitles = 0
;
if keyword_set(plotname_buffer) then newplotname_buffer = plotname_buffer $
	else newplotname_buffer = state_old.plotname_buffer
if keyword_set(noplotname_buffer) then newplotname_buffer = 0
;
if keyword_set(autoReLoadFile) then autoReLoadFile = autoReLoadFile $
	else autoReLoadFile = state_old.autoReLoadFile
;

state = { wids:state_old.wids, $
	  calcstr:state_old.calcstr, $
	  xpdt:state_old.xpdt, $
	  xpdt_limits:state_old.xpdt_limits, $
	  xpdt_titles:state_old.xpdt_titles, $
	  buffer:newbuffer, $
	  autoReLoadFile:autoReLoadFile, $
	  dthcurrent:newdthcurrent, $
	  dthbuffer:newdthbuffer, $
	  strcontrols:state_old.strcontrols, $
	  spec:state_old.spec, $
	  exebuffer:newexebuffer, $
	  version:state_old.version, $
	  cloned:state_old.cloned, $
	  coltitles:newcoltitles,$
	  fileinf:state_old.fileinf,$
	  psysvar:state_old.psysvar, $
	  no_block:state_old.no_block, $
	  plotname_current:state_old.plotname_current, $
	  plotname_buffer:newplotname_buffer }
;
end
;
;=========================================================================
;
; Makes the plot
;
PRO XPLOT_PLOT,state,SAVE = save, NOCOMMAND = nocommand, $
	PRINTBUFFER=printbuffer,Group=group
;
catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error,Dialog_Parent=group,$
    'XPLOT_PLOT: error caught: '+!error_state.msg)
  catch, /cancel
  goto,out
endif
;
; xplot 2.0:
;	/save: to update the buffer with a saved set
;	printbuffer: is the printbuffer keyword is initialized when 
;		calling xplot_plot, then the printer buffer is 
;		stored in such variable. Useful for printing.
;	/nocommand: obsolete
;
widget_control,state.wids.draw,get_value=windownum
wset,windownum
;added 98/08/27
p_old=!p
x_old=!x
y_old=!y
!p=state.psysvar.p
!x=state.psysvar.x
!y=state.psysvar.y
;
for i=0,n_elements(state.xpdt.sysvar)-1 do begin
  ;print,'<><><> ',state.xpdt.sysvar(i)
  tmp = execute(state.xpdt.sysvar(i))
endfor
;
; in the case that the image is zoomed, overwrite some variables
; already defined in state.xpdt.sysvar
if state.xpdt.zoom_on EQ 1 then begin
  !x.range = state.xpdt.zoomsel(0:1)
  !y.range = state.xpdt.zoomsel(2:3)
  !x.style=1
  !y.style=1
endif


if keyword_set(state.dthbuffer) then $
    ndatasets = n_elements(state.dthbuffer)  else $
    ndatasets = 1
    

case state.xpdt.kind of
   3: logstyle=',/ylog'
   4: logstyle=',/xlog'
   5: logstyle=',/xlog,/ylog'
   else: logstyle=''
endcase

;Note: if command1 is changed, remember to change also 
;	xplot_event/SETCURRENT
if state.xpdt.connect EQ 0 then psymsign='-' else psymsign=''
command1 = 'plotfile,'+ strcompress(state.dthcurrent) +'L'+ $
	',xcol='+strcompress(state.xpdt.Xcol,/REM)+ $
	',ycol='+strcompress(state.xpdt.Ycol,/REM)+ $
	',psym='+psymsign+strcompress(state.xpdt.psym,/REM) + $
	',linestyle='+strcompress(state.xpdt.Linestyle,/REM)+ $
	',thick='+string(state.xpdt.thick,format='(G3.2)')+ $
	',color='+strcompress(state.strcontrols.color,/rem) ;+ $
	;',shiftval='+vect2string(state.xpdt.Shift)

; changed to avoid very long line
if state.xpdt.Shift[0] EQ 0 and state.xpdt.Shift[1] EQ 0 then $
  command1 = command1+',shiftval=[0.,0.]' else $
  command1 = command1+',shiftval='+vect2string(state.xpdt.Shift)

if strcompress(state.xpdt.mesh.flag(0),/rem) EQ '1' then begin
  command1 = command1+$
    ',meshkind='+strcompress(state.xpdt.mesh.kind(0),/rem)
  if state.xpdt.mesh.keywords NE '' then $
	command1 = command1+','+state.xpdt.mesh.keywords
  case strcompress(state.xpdt.mesh.split(0),/rem) of
    '0': command1 = command1+$
	',meshpoints='+strcompress(state.xpdt.mesh.points,/rem)
    '1': command1 = command1+$
	',meshcol='+strcompress(state.xpdt.mesh.col,/rem)
    else: message,/info,'Error: MESH case not found'
  endcase
endif


if state.xpdt.axis2(0) NE state.xpdt.axis2(1) then $
	command1 = command1 + ',x2axis='+vect2string([state.xpdt.axis2(0),$
	 state.xpdt.axis2(1)])
if state.xpdt.axis2(2) NE state.xpdt.axis2(3) then $
	command1 = command1 + ',y2axis='+vect2string([state.xpdt.axis2(2),$
	 state.xpdt.axis2(3)])
if state.xpdt.ErrPlot(0) NE state.xpdt.ErrPlot(1) then $
	command1 = command1 + ',errplot='+vect2string([state.xpdt.errplot(0),$
	 state.xpdt.errplot(1)])

xtitletext = state.xpdt_titles.xtitle
if xtitletext EQ '-1' then begin
  if state.spec.on EQ 1 then begin
    if state.spec.mca EQ 1 then xtitletext='' else begin
      ;if strcompress(state.xpdt.mesh.flag[0],/rem) EQ '1' then begin
      ;  xtitletext = ' '
      ;endif else begin
      labels = state.spec.labels 
      tmp = strparse(labels,'!',labelslist)
      if tmp EQ -1 then $
        tmp = widget_message(dialog_parent=state.wids.main,$
          'Error (column separator xtitle)', /ERROR) else $
          xtitletext = ' '+labelslist(state.xpdt.Xcol-1)
     endelse
  endif else begin
    IF N_Elements(state.coltitles) GE state.xpdt.Xcol>2 THEN $
    xtitletext = state.coltitles(state.xpdt.Xcol-1) 
  endelse
endif

ytitletext = state.xpdt_titles.ytitle
if ytitletext EQ '-1' then begin
  if state.spec.on EQ 1 then begin
    if state.spec.mca EQ 1 then ytitletext='' else begin
      if strcompress(state.xpdt.mesh.flag[0],/rem) EQ '1' then begin
        ytitletext=''
      endif else begin
        labels = state.spec.labels
        tmp = strparse(labels,'!',labelslist)
        if tmp EQ -1 then $
          tmp = widget_message(dialog_parent=state.wids.main,$
	    'Error (column separator xtitle)', /ERROR) else $
  	    ytitletext = ' '+labelslist(state.xpdt.Ycol-1)
      endelse
    endelse
  endif else begin
    IF N_Elements(state.coltitles) GE state.xpdt.Ycol>2 THEN $
    ytitletext = state.coltitles(state.xpdt.Ycol-1) 
  endelse
endif
;iok
titletext = state.xpdt_titles.title
if strcompress(titletext,/rem) NE '' AND state.spec.on EQ 1 then  begin
  handle_value, state.spec.handle, hspec
  titletext=xplot_spectitle(titletext,hspec)
endif

command = command1 + $
	;',title='+"'"+state.xpdt_titles.title+"'" + $
	',title='+'"'+titletext+'"' + $
	;',xtitle='+"'"+state.xpdt_titles.xtitle+"'" + $
	',xtitle='+'"'+xtitletext+'"' + $
	;',ytitle='+"'"+state.xpdt_titles.ytitle+"'" + $
	',ytitle='+'"'+ytitletext+'"' + $
	',axiscolor='+strcompress(state.strcontrols.Axiscolor,/rem) + $
	logstyle

;

if keyword_set(save) then begin
  if state.spec.on  then begin
     handle_value, state.spec.handle, hspec
     currentscan=' [#S '+$
	strcompress(spec_scan(hspec,'CURRENT'),/rem)+'] ' 
  endif else currentscan=''
  ;copy handle and places the copy as current
  plotfile,state.dthcurrent,/nodata,tmp
  hh = handle_create(value=tmp)
  ;handle_value,hh,tmp,/set
  if not(keyword_set(state.buffer)) then begin
        ;<>print,'XPLOT: ....... writting buffer .............'
	xpstr_update,state,buffer = command1+',ki=2' , dthbuffer= $
	    [state.dthcurrent],plotname_buffer=state.plotname_current+$
	    currentscan
  endif else begin
        ;<>print,'XPLOT: ....... adding to  buffer .............'
	xpstr_update,state,buffer = [state.buffer,command1+',ki=2'], $
	    dthbuffer= [state.dthbuffer,state.dthcurrent]
  	xpstr_update,state,plotname_buffer=[state.plotname_buffer,$
	   state.plotname_current+currentscan]
  endelse
  state.dthcurrent = hh
  goto,OUT
endif
;
if keyword_set(printbuffer) then begin
  tmp0 = ['c0=!p','c1=!x','c2=!y']
  tmp1 = ['!p=c0','!x=c1','!y=c2']
  ; sysvar from 0 to 5 to avoid screwing the colors! 98/09/24
  printbuffer = [tmp0,state.xpdt.sysvar[0:5]]
  ;
  ; in the case that the image is zoomed, overwrite some variables
  ; already defined in state.xpdt.sysvar
  if state.xpdt.zoom_on EQ 1 then begin
     printbuffer = [printbuffer, $
    '!x.range = '+vect2string(state.xpdt.zoomsel(0:1)), $
    '!y.range = '+vect2string(state.xpdt.zoomsel(2:3)), $
    '!x.style=1',$
    '!y.style=1']
  endif
  printbuffer = [printbuffer,command]
  if keyword_set(state.buffer) then printbuffer = [printbuffer,state.buffer]
  if keyword_set(state.exebuffer) then $
		printbuffer = [printbuffer,state.exebuffer]
  printbuffer = [printbuffer,tmp1]
  ;print,'XPLOT: ############ printbuffer: #############'
  ;for i=0L,n_elements(printbuffer)-1 do print,printbuffer[i]
  ;print,'XPLOT: ############ end printbuffer: #############'
  ;goto,OUT
  return
endif
;
; executing
;
if not(keyword_set(nocommand)) then begin
  ;print,'XPLOT: ...... executing command ...........'
  erase
  ;print,command
  a=execute(command)
endif

; if BUFFER buffer exist, execute it
if keyword_set(state.buffer) then begin
  ;<>print,'XPLOT: ...... executing buffer ...........'
  ;<>print,state.buffer
    for i=0,n_elements(state.buffer)-1 do a=execute(state.buffer(i))
endif

; if EXEBUFFER buffer exist, execute it
if keyword_set(state.exebuffer) then begin
  ;<>print,'XPLOT: ...... executing exebuffer ...........'
  ;<>print,state.exebuffer
    for i=0,n_elements(state.exebuffer)-1 do a=execute(state.exebuffer(i))
endif

OUT:
state.psysvar.p=!p
state.psysvar.x=!x
state.psysvar.y=!y
;added 98/08/27
!p=p_old
!x=x_old
!y=y_old
end
;
;=========================================================================
;
PRO XPLOT_EVENT,event

FORWARD_FUNCTION LookupManagedWidget

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error,Dialog_Parent=event.top,$
    'XPLOT_EVENT: error caught: '+!error_state.msg)
  catch, /cancel
  GoTo,out
endif

;
; the following lines put artificially eventUVal='RESIZE' when the
; user resizes the window (event created by the top base widget (tbw) ).
; This is not elegant: one should store the "state" in the first 
; child of tbl, but in xplot is done directly in tbl. 
; One day that I will have the time I'll fix it! 
; MSR 97/10/16
;  
if tag_names(event,/structure_name) EQ 'WIDGET_BASE' then begin
  eventUVal = 'RESIZE'
endif else begin
  Widget_Control, event.id, get_UValue=eventUVal
endelse
;
; First the actions that do not require the "state" explicitely
; =============================================================
;
CASE eventuval OF
  'LOADFILE':     BEGIN
    xplot_loadfile,event.top
    return
    END
  'CALCULATIONS':	BEGIN
    xplot_calculations,event
    return
    END
  'SAVE':	BEGIN
    xplot_savecurrent,event.top
    return
    END
  'CONTROLS':	begin
    xplot_controls_action,event,/CONTROLS
    return
    end
  'XCOL':	begin
    xplot_changecol,event
    return
    end
  'YCOL':	begin
    xplot_changecol,event
    return
    end
  'ERRPLOT':	begin
    xplot_errorbars,event
    return
    end
  'EXEBUFFERCLEAN': begin
    xplot_exebuffer,event
    return
    end
  'EXEBUFFEREDIT': begin
    xplot_exebuffer,event
    return
    end
  'SETLABELS': begin
    xplot_exebuffer,event
    return
    end
  'SETLEGEND': begin
    xplot_exebuffer,event
    return
    end
  'EXIT':	begin
    xplot_quit,event.top
    return
    end
  'PLOTMGR': begin
    xplot_plotmgr,event.top
    return
    end
  'CLEARDATA': begin
    xplot_cleardata,event.top,/confirm
    return
    end
  'SETLIMITS': begin
    xplot_setlimits,event.top
    return
    end
  'SAVEASCII': begin
    xplot_saveascii,event.top
    return
    end
  'SHIFT': begin
    xplot_shift,event.top,/openwindow
    return
    end
  'MESH': begin
    xplot_mesh,event
    return
    end
  'RELOADFILE': begin
    xplot_reloadfile,event.top
    return
    end
  'ID': begin
    itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
           'Xplot ID number is: '+StrCompress(event.top))
    return
    end
  else:
ENDCASE


;
; Second the actions that do require the "state" 
; ==============================================
;
widget_control, event.top, GET_UVALUE = state, /NO_COPY

; this section is now in xplot_plot and in 'CURSOR'
;widget_control,state.wids.draw,get_value=windownum 
;wset,windownum
;!p=state.psysvar.p
;!x=state.psysvar.x
;!y=state.psysvar.y

CASE eventuval OF
        'ATTRIBUTES': BEGIN
		tmp = ''
		Widget_Control,event.id,Get_Value=tmp
		CASE tmp OF
		  'Write...': BEGIN
			Xplot_WriteAttributes,state,Group=event.top
			END
		  'Load...': BEGIN
			iflag=0
		  	file = Dialog_Pickfile( Dialog_Parent=event.top,$
			  Filter='*.att',Title=$
			  'Select a file for loading Xplot attributes')
	                if (file eq '') then goto,out
			txt = read_textfile(file)
			itmp = 0
			FOR i=0,N_Elements(txt)-1 DO BEGIN
			  IF StrMid(txt[i],0,1) NE ';' THEN BEGIN
			    itmp = Execute('state.'+txt[i])
			    IF itmp NE 1 THEN itmp = Dialog_Message(/Error,$
			      Dialog_Parent=event.top, $
			      'XPLOT_EVENT: Error executing: '+'state.'+txt[i])
			  ENDIF
			ENDFOR
			;
			; this is an artificial way to update the 
			; widgets containg xcol and ycol
			;
			itmp1 = FilePosition(file,'xpdt.xcol')
			itmp2 = FilePosition(file,'xpdt.ycol')
			xCol=0 & yCol=0
			IF itmp1[0] NE -1 AND N_Elements(itmp1) EQ 3 THEN BEGIN
			  tmp = txt[itmp1[2]-1]
			  xCol=Fix( StrMid(tmp,StrPos(tmp,'=')+1,StrLen(tmp)) )
			ENDIF
			IF itmp2[0] NE -1 AND N_Elements(itmp2) EQ 3 THEN BEGIN
			  tmp = txt[itmp2[2]-1]
			  yCol=Fix( StrMid(tmp,StrPos(tmp,'=')+1,StrLen(tmp)) )
			ENDIF
			IF xCol NE 0 OR yCol NE 0 THEN BEGIN
			  Widget_Control,event.top,Set_UValue=state,/No_Copy
			  Xplot_ChangeCol,event.top,XCol=xCol,YCol=ycol
			  Widget_Control,event.top,Get_UValue=state,/No_Copy
			  ;RETURN
			ENDIF

			itmp = FilePosition(file,'strcontrols.font')
			IF itmp [0] NE -1 THEN BEGIN
			  tmp = txt[itmp[0]-1]
			  myFont=(StrMid(tmp,StrPos(tmp,'=')+1,StrLen(tmp)) )
			  itmp = execute('myfont='+myfont)
			  IF myfont[0] GT 16 THEN $
			    TT_Font_manager,Set_Index=myfont[0]-17
			ENDIF
			xplot_plot,state
			END
		  'Clean Session Default': BEGIN
			Xop_SetEnv,'XPLOT_DEFAULTS='
			END
		ENDCASE
	END
	;
	; resize
	;
        'RESIZE': BEGIN
		if sdep() EQ 'WINDOWS' then widget_control,state.wids.draw,$
		  scr_xsize=event.x-5,scr_ysize=event.y-110 else $
		  widget_control,state.wids.draw,$
		  scr_xsize=event.x-5,scr_ysize=event.y-165 
		xplot_plot,state
		goto,out
	END
	; 
	; these are in the menu  bar
	;
        'SSCAN': BEGIN
		; added 98/08/20 to clean the FWHM etc when changing scans
		xpstr_update,state, /noexebuffer
		;
                widget_control,state.spec.scanbase,/show, bad_id=bad
                event_name = tag_names(event, /structure_name)
                if event_name eq 'NEW_SCAN_EVENT' then begin
                   widget_control,/HOURGLASS
                   tmp=xplot_readspec(state, event.newscan, event.mca, /oldcol)
		   IF N_Elements(tmp) EQ 1 AND tmp[0] EQ 0 THEN BEGIN
		     itmp = Dialog_Message(/Error,Dialog_Parent=event.top,$
			'Scan contains no data')
		     GoTo,out
		   ENDIF
                   state.dthcurrent = handle_create(value=tmp)
                   xplot_mappanels, state.wids, state.spec
                   xplot_plot,state
                endif else if bad eq state.spec.scanbase then begin
                   xspecscan, state.spec.handle, caller=event.id,             $
                              tlb_id=tmp, mca=state.spec.mca, /not_all,       $
                              group_leader=event.top, title='Scan Selection', $
			      no_block=state.no_block,xplotmessageid=$
			      state.wids.message
                   state.spec.scanbase = tmp
                endif
                END
	'RESTORE':	BEGIN
		if sdep() eq 'UNIX' then filter='*.bck' else filter=0
		file = Dialog_pickfile(FILTER=filter,GROUP=event.top,/NOCONF,$
		title='Select a file with an Xplot backup')
		if file eq '' then GOTO,out
		newstate = xplot_restorestate(OLDSTATE=state, $
			IERR=IERR,FILENAME=file,WSIZE=wsize,$
			Group=event.top)
		if ierr NE 0 then begin
		  tmp = widget_message(dialog_parent=event.top,$
			'XPLOT_EVENT: Error restoring Xplot backup file',/ERROR)
		  goto, out
		endif
                ; close control panel if open
                if widget_info(state.wids.controls(0),/valid_id) then begin
		  widget_control,state.wids.showcontrols, $
			set_value='Show Controls'
		  widget_control,state.wids.controls(0),/DESTROY
                endif

		; substitute state by the new one
		state = newstate
                ;copy_structure,newstate,state

		state.spec.on = 0
		; resizes window if necessary
		if ((wsize(0) NE !d.x_size) OR (wsize(1) NE !d.y_size)) $
			then begin
		   widget_control,state.wids.draw,xsize=wsize(0), ysize=wsize(1)
		endif
		xplot_plot,state
		; update info widgets
		widget_control,state.wids.message,set_value = $
		 'Xplot restored backup file: '+file

		if keyword_set(state.buffer) then $
			nsaved = n_elements(state.buffer) else $
			nsaved = 0
		widget_control,state.wids.saved,set_value = $
		     strcompress(nsaved,/rem)
		if  nsaved GE 1 then widget_control,state.wids.plotmgr, $
		     /sensitive
		plotfile,state.dthcurrent,tmp,/nodata
		ncol = n_elements(tmp(*,0))
		widget_control,state.wids.ncol,set_value = $
                     strcompress(ncol,/rem)
		if widget_info(state.wids.xcol,/NAME) EQ 'DROPLIST' then begin
		  if not(keyword_set(state.coltitles)) then begin
		    coltitles = 'Col: '+$
			strcompress((sindgen(Ncol+1))(1:Ncol),/rem)
		    xpstr_update,state,coltitles=coltitles
		  endif
		  if n_elements(state.coltitles) NE Ncol then begin
		    coltitles = (sindgen(Ncol+1))(1:Ncol)
		    xpstr_update,state,coltitles=coltitles
		  endif
		  widget_control,state.wids.xcol,set_value=state.coltitles, $
			set_droplist_sel=state.xpdt.xcol-1
		  widget_control,state.wids.ycol,set_value=state.coltitles, $
			set_droplist_sel=state.xpdt.ycol-1
		endif else begin
		  widget_control,state.wids.xcol,set_value = $
		     strcompress(state.xpdt.xcol,/rem)
		  widget_control,state.wids.ycol,set_value = $
		     strcompress(state.xpdt.ycol,/rem)
		endelse

		END
	'AUTORELOADFILE': 	BEGIN
		mainId = state.wids.main

		IF Tag_Names(event,/Structure_Name) EQ 'WIDGET_TIMER' THEN BEGIN
		  IF state.autoReLoadFile.reLoadFlag EQ 1 THEN BEGIN
		    timer = state.autoReLoadFile.set.timer
		    do_realod_flag = 1
		    IF (Fix(state.autoReLoadFile.set.check[0]) EQ 1) $
			AND (Float(!version.release) GE 5.5) THEN BEGIN
			IF (File_Info(state.fileinf.name)).mtime LE state.fileinf.mtime THEN $
				do_realod_flag = 0
		    ENDIF
		    
		    Widget_Control, event.top, SET_UVALUE = state, /NO_COPY
		    IF do_realod_flag EQ 1 THEN BEGIN
		      Xplot_ReLoadFile,event.top
		      Widget_Control,mainId,TLB_Set_Title= $
				'XPLOT in AutoReLoadFile mode ['+StrCompress(Fix(timer),/Rem)+' sec]'
		      Message,/Info,'realoading...'
		    ENDIF ELSE BEGIN
		      Message,/Info,'Checked. Not realoaded.'
		    ENDELSE
		    Widget_Control,event.id, timer=timer
		  ENDIF ELSE BEGIN
		    Widget_Control, event.top, SET_UVALUE = state, /NO_COPY
		  ENDELSE
		  RETURN
		ENDIF

		Widget_Control, event.top, SET_UVALUE = state, /NO_COPY
		Widget_Control,event.id,Get_Value=value
		Xplot_AutoReLoadFile, event.top , Action=value

		END
	'PRINT': 	BEGIN
		; initialize printing buffer
		printbuffer = '  '
		; load printing buffer
		xplot_plot,state,printbuffer=printbuffer
		; print it
		xprint,BUFFER=printbuffer,GROUP=event.top,/COLOR
		END
	'EDITDATA':	BEGIN
                if not(keyword_set(state.dthcurrent)) then begin
                  tmp = widget_message(dialog_parent=event.top,$
                        'No data in current set',/ERROR)
                  goto,out
                endif
                plotfile,state.dthcurrent,tmpset,/nodata
		ycol = state.xpdt.ycol-1
		Csel = [ycol,ycol]
		if (state.xpdt.zoomsel(0) ne state.xpdt.zoomsel(1)) and $
		   (state.xpdt.zoomsel(2) ne state.xpdt.zoomsel(3)) then begin
		 xcol = state.xpdt.xcol-1
		 xmin = min([state.xpdt.zoomsel(0),state.xpdt.zoomsel(1)])
		 xmax = max([state.xpdt.zoomsel(0),state.xpdt.zoomsel(1)])
		 xi0 = where(tmpset(xcol,*) LT xmin)  & xi0 = n_elements(xi0)
		 xie = where(tmpset(xcol,*) LT xmax)  & xie = n_elements(xie)
		 Rsel=[xi0,xie-1]
		endif else Rsel=0
		action=''
		xtable,tmpset,group=event.top,Wtitle='Editing Xplot data',$
		  Rsel=Rsel,Csel=Csel,Action=action
		if action EQ 'Cancel' then goto,out
		widget_control, event.top, SET_UVALUE = state, /NO_COPY
		xplot_loadfile,event.top,tmpset
		return
		END

	'EDITCOLUMNS':	BEGIN
                if not(keyword_set(state.dthcurrent)) then begin
                  tmp = widget_message(dialog_parent=event.top,$
                        'No data in current set',/ERROR)
                  goto,out
                endif
		; if spec/mca is set, error...
		if state.spec.on AND state.spec.mca then begin
		  itmp = Dialog_Message(/Error, $
		  'Operation not available eith SPEC/MCA data',$
		  Dialog_Parent=group)
		  GoTo,out
		endif

                plotfile,state.dthcurrent,tmpset,/nodata
		ncol = n_elements(tmpset[*,0]) 
		txtcol=Vect2String(IndGen(ncol)+1)
		str={txtcol:txtcol}
		action=''
                XscrMenu,str,/NOTYPE,GROUP=event.top, $
                  WTITLE='Columns to keep',ACTION=action,FIELDLEN=80,$
		  Dialog_Parent=event.top , $
		  Titles=['Column numbers for columns to be kept']
                  ; HELP=xplot_text('XX')
		if action EQ 'DONT' then goto,out
		tmpset2=tmpset
		indices = 0
		command = 'indices = '+str.txtcol+'-1'
		itmp = Execute(command)
		IF itmp NE 1 THEN BEGIN
		  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		    'Error executing: '+command)
		  GoTo,out
		ENDIF
		tmpset2 = tmpset[indices,*]

		ncol = n_elements(tmpset2[*,0])
		xcol = state.xpdt.xcol
		ycol = state.xpdt.ycol
		xcol = xcol < ncol
		ycol = ycol < ncol
		state.xpdt.xcol = xcol
		state.xpdt.ycol = ycol

		if widget_info(state.wids.xcol,/NAME) EQ 'TEXT' then begin
		   Widget_Control,state.wids.xcol,Set_Value= $
			StrCompress(xcol,/Rem)
		   Widget_Control,state.wids.ycol,Set_Value= $
			StrCompress(ycol,/Rem)
		   Widget_Control,state.wids.ncol,Set_Value= $
			StrCompress(ncol,/Rem)
		endif else begin
		   tmp  = state.coltitles[indices]
		   xpstr_update,state,coltitles=tmp
		   widget_control,state.wids.xcol,set_value=$
			Xplot_ColTitles_Short(tmp)
		   widget_control,state.wids.ycol,set_value=$
			Xplot_ColTitles_Short(tmp)
		   widget_control,state.wids.xcol,set_droplist_select=$
			state.xpdt.xcol-1
		   widget_control,state.wids.ycol,set_droplist_select=$
			state.xpdt.ycol-1
		endelse

		; spec setup
		if state.spec.on EQ 1 then begin
		  labels = state.spec.labels ;+'!Column Oper'
		  ;state.spec.labels = labels
		  tmp = strparse(labels,'!',labelslist)
		  if tmp EQ -1 then begin
		    tmp = widget_message(dialog_parent=event.top,$
			'Error (column separator)', /ERROR)
		    goto,out
		  endif
		  labelslist = labelslist[indices]
		  labels = labelslist(0)
		  for i=1,n_elements(labelslist)-1 do $
			labels = labels + '!' + labelslist(i)
		  state.spec.labels = labels
                  labelslist=strcompress((sindgen(n_elements( $
                    labelslist)+1))(1:n_elements(labelslist)) ) + $
                    ': '+labelslist
      		  widget_control,state.wids.xcolSPEC,SET_VALUE=$
			labelslist, $
			SET_DROPLIST_SELECT =  state.xpdt.xcol-1
      		  widget_control,state.wids.ycolSPEC,SET_VALUE=$
			labelslist, $
			SET_DROPLIST_SELECT =  state.xpdt.ycol-1
		endif

		handle_free,state.dthcurrent
		h1 = handle_create(value=tmpset2)
		state.dthcurrent = h1
		xplot_plot,state
		widget_control,state.wids.message,set_value = $
		  'Current set has been modified '+$
		  "with the option 'Edit|Columns...'"

		widget_control, event.top, SET_UVALUE = state, /NO_COPY
		return
		END
	'EDITCUT':	BEGIN
                if not(keyword_set(state.dthcurrent)) then begin
                  tmp = widget_message(dialog_parent=event.top,$
                        'No data in current set',/ERROR)
                  goto,out
                endif
                plotfile,state.dthcurrent,tmpset,/nodata
		xcol = state.xpdt.xcol-1
		xx = Reform(tmpset[xcol,*])
		xmin = min(xx) & xmax=max(xx)

		; get extrema
		if (state.xpdt.zoomsel(0) ne state.xpdt.zoomsel(1)) and $
		   (state.xpdt.zoomsel(2) ne state.xpdt.zoomsel(3)) then begin
		 xmin = min([state.xpdt.zoomsel(0),state.xpdt.zoomsel(1)])
		 xmax = max([state.xpdt.zoomsel(0),state.xpdt.zoomsel(1)])
		 xi0 = where(tmpset(xcol,*) LT xmin)  & xi0 = n_elements(xi0)
		 xie = where(tmpset(xcol,*) LT xmax)  & xie = n_elements(xie)
		 Rsel=[xi0,xie-1]
		endif else Rsel=[xmin,xmax]

		Rsel=Float(Rsel)
		str={min:Rsel[0],max:Rsel[1], $
                  select:['0','Keep selected data only', $
			'Exclude selected data'], $
                  output:['1','Rewrite current set','New xplot window']}
		action=''
                XscrMenu,str,/NOTYPE,GROUP=event.top, $
                  WTITLE='Cut data',ACTION=action, $
		  Dialog_Parent=event.top , $
		  Titles=['Minimum X value for selected data',$
		          'Maximum X value for selected good data',$
                          'Data to write: ',$
			  'Output to: '], /Interp
                  ; HELP=xplot_text('XX')
		if action EQ 'DONT' then goto,out
		tmpset2=tmpset

		CASE StrCompress(str.select[0],/Remove_All) OF
                '0': igood = Where(xx GE str.min AND xx LE str.max)
                '1': igood = Where(xx LT str.min OR xx GT str.max)
		ENDCASE
		IF igood[0] EQ -1 THEN BEGIN
		  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		    'Error in selecting interval: no good points ')
		  GoTo,out
		ENDIF
		tmpset2=tmpset[*,igood]
		npoints1 = N_Elements(tmpset[0,*])
		npoints2 = N_Elements(tmpset2[0,*])
		CASE StrCompress(str.output[0],/Remove_All) OF
                '0': BEGIN
		     itmp = Dialog_Message(/Question,Dialog_Parent=event.top, $
		       ['Number of abscissa points reduced from '+$
		       StrCompress(npoints1,/Rem)+' to '+ $
			StrCompress(npoints2,/Rem),$
		       ' ','Accept the operation?'])
		     IF itmp EQ 'No' THEN GoTo,out
		     widget_control, event.top, SET_UVALUE = state, /NO_COPY
		     xplot_loadfile,event.top,tmpset2
		     return
		     END
                '1': BEGIN
		     Xplot,tmpset2,WTitle='Cutted data',no_block=state.no_block
		     widget_control, event.top, SET_UVALUE = state, /NO_COPY
		     RETURN
		     END
                 ENDCASE
		END
	'SAVEIDL': 	BEGIN
		iflag=1
		WHILE iflag EQ 1 DO BEGIN
		  iflag = 0
		  gfile = Dialog_Pickfile(FILE='xplot.bck',Group=event.top,$
		    title='Select a file for writing Xplot backup')
                  if (gfile eq '') then goto,out
		  IF CheckFile(gfile) EQ 1 THEN BEGIN
		    itmp = Dialog_Message(/Question,Dialog_Parent=event.top,$
			['File '+gfile+' already exists.',' ','Overwrite it?'])
		    IF itmp EQ 'No' THEN iflag=1
		  ENDIF
		ENDWHILE
		if keyword_set(state.buffer) then $
		  if (n_elements(state.buffer) GT 10) then $
		    tmp = widget_message(dialog_parent=event.top,$
			['Xplot MAY fail in restoring a',$
			' Xplot backup file when the number of saved sets', $
			' is greater than 10.'])
                xplot_savestate,state,FILENAME=gfile
                widget_control,state.wids.message,set_value = $
                  'Backup file '+gfile+' written to disk.'
		;tmp = widget_message(dialog_parent=event.top,/INFO, $
		; 'File '+gfile+' written to disk.')
		END
        'SETLIMITSDEF':    BEGIN
		tmp = XRegistered('Xplot_SetLimits_Interactive')
		IF tmp GE 1 THEN BEGIN
		  FOR jj=0,tmp-1 DO BEGIN
		  id = LookupManagedWidget('Xplot_SetLimits_Interactive')
		  Widget_Control,id,/Destroy
		  ENDFOR
		ENDIF
                state.xpdt_limits.Xmin = 0.0
                state.xpdt_limits.Xmax = 0.0
                state.xpdt_limits.Ymin = 0.0
                state.xpdt_limits.Ymax = 0.0

                state.xpdt.sysvar(2) = '!x.range = [0.0,0.0]'
                state.xpdt.sysvar(3) = '!y.range = [0.0,0.0]'
		state.xpdt.zoom_on=0
		Xplot_plot,state
		END
        'SETLIMITSZOOM':    BEGIN
		IF XRegistered('Xplot_SetLimits_Interactive') THEN BEGIN
		  id = LookupManagedWidget('Xplot_SetLimits_Interactive')
		  Widget_Control,id,/Destroy
		ENDIF
                state.xpdt_limits.Xmin = state.xpdt.zoomsel(0)
                state.xpdt_limits.Xmax = state.xpdt.zoomsel(1)
                state.xpdt_limits.Ymin = state.xpdt.zoomsel(2)
                state.xpdt_limits.Ymax = state.xpdt.zoomsel(3)
                  state.xpdt.sysvar(2) = '!x.range = [' + $
                        strcompress(state.xpdt.zoomsel(0),/rem)+','+ $
                        strcompress(state.xpdt.zoomsel(1),/rem)+']'
                  state.xpdt.sysvar(3) = '!y.range = [' + $
                        strcompress(state.xpdt.zoomsel(2),/rem)+','+ $
                        strcompress(state.xpdt.zoomsel(3),/rem)+']'
		state.xpdt.zoom_on=0
		Xplot_plot,state
		END
	'SETTITLES':	BEGIN
		tmp = state.xpdt_titles
		XscrMenu,tmp,/NOTYPE,GROUP=event.top, $
		  WTITLE='Xplot titles',ACTION=action,FIELDLEN=80,$
		  HELP=xplot_text('TITLES'),/APPLY_BUTTON
		if action EQ 'DONT' then goto,out
		state.xpdt_titles = tmp
		Xplot_plot,state
		if action eq 'DO_APPLY' then begin
		  widget_control, event.top, SET_UVALUE = state, /NO_COPY
		  xplot_event,event
		  return
		endif
		END
	'PLTNAME_CUR':	BEGIN
		tmp = state.plotname_current
		tmpstr = {tmp:tmp}
		XscrMenu,tmpstr,/NOTYPE,GROUP=event.top, FIELDLEN=84, $
		  WTITLE='Edit name of the current set',ACTION=action,$
		  TITLES='Please edit the names for the current sets'
		if action EQ 'DONT' then goto,out
		state.plotname_current = tmpstr.tmp
		END
	'PLTNAME_SAV':	BEGIN
		tmp = state.plotname_buffer
		if not(keyword_set(tmp)) then begin
		  itmp = widget_message(/ERROR,dialog_parent=event.top,$
		    'No saved plots to edit.')
		  goto,out
		endif
		tmpstr = {tmp:tmp}
		XscrMenu,tmpstr,/NOTYPE,GROUP=event.top, FIELDLEN=84, $
		  WTITLE='Edit names of the saved set',ACTION=action,$
		  TITLES='Please edit the names for the saved sets'
		if action EQ 'DONT' then goto,out
		state.plotname_buffer = tmpstr.tmp
		END
	'SETCOLTITLES':	BEGIN
		if state.spec.on EQ 1 then begin 
                 labels = state.spec.labels 
                  junk = strparse(labels,'!',tmp)
                  if junk EQ -1 then begin
                    junk = widget_message(dialog_parent=event.top,$
			'Error (column separator)',/ERROR)
                    goto,out
                  endif
		endif else begin
		  if not(keyword_set(state.coltitles)) then begin
		    junk = widget_message(dialog_parent=event.top,$
			['Option only available',$
			'when starting xplot ',	"with the 'COLTILES'", $
			'keyword, or when ','using SPEC files'])
		    goto,OUT
		  endif
		  ; MSR debugging 23 Apr 96:
		  tmp = state.coltitles
		endelse
		; 
		; create a structure holding the information and use
		; XscrMenu to edit it
		;
		tmp_n=n_elements(tmp)
		tmp_s=''
		for i=0,tmp_n-1 do tmp_s=tmp_s+',tmp['+StrCompress(i,/Rem)+']'
		tmp_strtitles='Col'+StrCompress(IndGen(tmp_n)+1,/Rem)
		tmp_titles='Column '+StrCompress(IndGen(tmp_n)+1,/Rem)
		tmpstr=0
		command = 'tmpstr=create_struct(tmp_strtitles'+tmp_s+')'
		itmp = execute(command)
		XscrMenu,tmpstr,/NOTYPE,GROUP=event.top, FIELDLEN=84, $
		  WTITLE='Xplot column titles',ACTION=action,/Scroll,$
		  help=xplot_text('COLTITLES'),Titles=tmp_titles,/Interp
		if action EQ 'DONT' then goto,OUT
		FOR i=0,tmp_n-1 do tmp[i]=tmpstr.(i)
		tmpnew=tmp
		cmax = Xplot_ColTitles_Short(/Get_Maximum)
		IF Max(StrLen(tmp)) GT cmax THEN BEGIN
		  XEdit,cmax,InfoText=['The selected titles contains more',$
			'than 23 characters. For better displaying they ',$
			'should be reduced to '],Text= '# characters: ',$
			Title='Setting column titles'
		  tmpnew = Xplot_ColTitles_Short(tmp,Max=cmax)
		ENDIF
		if (state.spec.on) EQ 1 then begin
		  labels = tmp(0)
                  for i=1,n_elements(tmp)-1 do $
                        labels = labels + '!' + tmp(i)
                  state.spec.labels = labels
                  tmp=strcompress((sindgen(n_elements( $
                    tmp)+1))(1:n_elements(tmp)) ) + ': '+tmp
                  widx = state.wids.xcolSPEC
                  widy = state.wids.ycolSPEC
		endif else begin
		  state.coltitles = tmp
		  widx = state.wids.xcol
		  widy = state.wids.ycol
		endelse
		widget_control,widx,SET_VALUE=tmpnew, $
		  set_droplist_select=state.xpdt.xcol-1
		widget_control,widy,SET_VALUE=tmpnew, $
		  set_droplist_select=state.xpdt.ycol-1
		Xplot_plot,state
		END
        'ANNOTATE':    BEGIN
		text = xplot_text('ANNOTATE')
		tmp = widget_message(dialog_parent=event.top,$
		  text,/CANCEL,/DEFAULT_CANCEL)
		if tmp EQ 'Cancel' then goto,out
		annotate
                END

	'SHOWDATAXY':	BEGIN
                plotfile,state.dthcurrent,/nodata,tmpset_pre
                tmpset = tmpset_pre([state.xpdt.xcol-1,state.xpdt.ycol-1],*)
		Widget_Control,/HourGlass
		Xdisplayfile1,text=tmpset,GROUP=event.top,title='Xplot XY data.'
		END
	'SHOWDATASET':	BEGIN
                plotfile,state.dthcurrent,/nodata,tmpset
		Widget_Control,/HourGlass
		Xdisplayfile1,text=tmpset,GROUP=event.top, $
		  title='Xplot set data.'
		END
	'DRAWDIM':	BEGIN
		tmp = { Horizontal:!d.x_size, $
			Vertical:!d.y_size }
		XscrMenu,tmp,GROUP=event.top,/NOTYPE,ACTION=action, $
		  WTITLE='Xplot Drawing size (pixels)',$
		  HELP=xplot_text('DRAWDIM')
		if action eq 'DONT' then goto,out
		widget_control,state.wids.draw,xsize=tmp.Horizontal, $
			ysize=tmp.Vertical
		xplot_plot,state
		END
	'COLPANEL': BEGIN
		if state.spec.on then begin
		  tmp = widget_message(dialog_parent=event.top,/ERROR, $
			'This option is not allowed with SPEC data')
		  goto,out
		endif
		wkind = widget_info(state.wids.xcol,/NAME)
		plotfile,state.dthcurrent,tmp,/NODATA
		ncol = n_elements(tmp(*,0))
		if wkind EQ 'TEXT' then begin
		  if not(keyword_set(state.coltitles)) then begin
		    coltitles = (sindgen(Ncol+1))(1:Ncol)
		    xpstr_update,state,coltitles= coltitles
		  endif
		  if ncol NE n_elements(state.coltitles) then begin
			tmp = widget_message(dialog_parent=event.top,/ERROR, $
			 'Number of columns not equal to stored labels')
			goto,out
		  endif
		  widget_control,state.wids.xcol,/DESTROY
		  widget_control,state.wids.ycol,/DESTROY
		  new1 = widget_droplist(state.wids.xcolBase, $
			value=state.coltitles,UVALUE='XCOL')
		  new2 = widget_droplist(state.wids.ycolBase, $
			value=state.coltitles,UVALUE='YCOL')
		  widget_control,new1,set_droplist_sel=state.xpdt.xcol-1
		  widget_control,new2,set_droplist_sel=state.xpdt.ycol-1
		  state.wids.xcol = new1
		  state.wids.ycol = new2
		endif else begin
		  widget_control,state.wids.xcol,/DESTROY
		  widget_control,state.wids.ycol,/DESTROY
		  new1 = widget_text(state.wids.xcolBase, $
			value=strcompress(state.xpdt.xcol,/Rem), $
			XSIZE=4,YSIZE=1,/EDIT,/FRAME,UVALUE='XCOL')
		  new2 = widget_text(state.wids.ycolBase, $
			value=strcompress(state.xpdt.ycol,/Rem), $
			XSIZE=4,YSIZE=1,/EDIT,/FRAME,UVALUE='YCOL')
		  state.wids.xcol = new1
		  state.wids.ycol = new2
		endelse
		END
	'SECONDAXIS':	BEGIN
		tmp = state.xpdt.axis2
		tmpstr = { Xmin:tmp(0), Xmax:tmp(1), Ymin:tmp(2), Ymax:tmp(3)}
		XscrMenu,tmpstr,GROUP=event.top,/NOTYPE,ACTION=action, $
		  WTITLE='Xplot 2nd axis limits',HELP=xplot_text('SECONDAXIS')
		if action ne 'DO' then goto,out
		state.xpdt.axis2 = $
			[tmpstr.Xmin,tmpstr.Xmax,tmpstr.Ymin,tmpstr.Ymax]
		xplot_plot,state
		END
	'MESH':	BEGIN
		widget_control,state.wids.saved,get_value=tmp
		if strcompress(tmp,/rem) NE '0' then begin
		  itmp = widget_message(dialog_parent=event.top, $
			/error,'MESH option not available'+$
		      ' with Saved sets.')
		  goto,out
		endif
		tmp = state.xpdt.mesh
		XscrMenu,tmp,GROUP=event.top,/NOTYPE,ACTION=action, $
		  WTITLE='Mesh values',FIELDLEN=40,/INTERP,/APPLY_BUTTON,$
		  TITLES=$
		  ['Mesh plot:','Kind','Split plots','N for splitting',$
		   'Column with mesh label values',$
		   'Additional keywords'],HELP=xplot_text('MESH'), $
		  FLAGS=['1','w(0) EQ 1','w(0) EQ 1','w(0) EQ 1 and w(2) EQ 0',$
			'w(0) EQ 1 and w(2) EQ 1','w(0) EQ 1']
		if action eq 'DONT' then goto,out
		state.xpdt.mesh = tmp
		xplot_plot,state
		; autocall xplot_event when Apply button is pressed
		if action eq 'DO_APPLY' then begin
		  widget_control, event.top, SET_UVALUE = state, /NO_COPY
		  xplot_event,event
		  return
		endif
		END
	'SYSVAR':	BEGIN
		widget_control, event.id, GET_VALUE=eventval
		TEXT=['This is an advanced option which use ', $
		'requires a deep knowdledge of IDL.', $
		'Its use is not recommended for the general user.', $
		'   ',$
		'Some of the modifications done here will remain ', $
		'after quiting xplot']
		tmp = widget_message(text,/CANCEL,/DEFAULT_CANCEL,$
			DIALOG_PARENT=event.top)
		if tmp EQ 'Cancel' then goto,out
		case eventval of
                   '!x': BEGIN
			tmp = !x
			XscrMenu,tmp,GROUP=event.top,NCOL=4,ACTION=action
			if action eq 'DONT' then goto,out
			!x = tmp
			xplot_plot,state
			END
                   '!y': BEGIN
			tmp = !y
			XscrMenu,tmp,GROUP=event.top,NCOL=4,ACTION=action
			if action eq 'DONT' then goto,out
			!y = tmp
			xplot_plot,state
			END
                  '!p': BEGIN
			tmp = !p
			XscrMenu,tmp,GROUP=event.top,NCOL=4, $
		  	  ACTION=action, /NOTYPE
			if action eq 'DONT' then goto,out
			!p = tmp
			xplot_plot,state
			END
		endcase
		END
	'COLUMNOPER':	BEGIN
		if not(keyword_set(state.dthcurrent)) then begin
		  tmp = widget_message(dialog_parent=event.top,$
			'No data in current set',/ERROR)
		  goto,out
		endif
		plotfile,state.dthcurrent,tmpset,/nodata
		tmpcol = 0
		ncol_old = n_elements(tmpset(*,0))
		action=0
		xplot_columnoper,tmpset,res=tmpcol,group=event.top,$
		  no_block=state.no_block, action=action
		if not(keyword_set(tmpcol)) then goto,out
		if tmpcol EQ -1 then goto,out
		ncol = n_elements(tmpset(*,0))

		; if spec/mca is set, inset column instead of appending it.
		ncoladd=''
		if state.spec.on AND state.spec.mca then begin
		  ncol = ncol-2
		  tmpcol = tmpcol-2
		  ncoladd='+2' 
		  ; commented to avoid changing column 98/08/20
		  ;widget_control,state.wids.ycolMCA,set_value = $
		  ;  strcompress(tmpcol,/rem)
		endif

		widget_control,state.wids.ncol,set_value = $
			strcompress(ncol,/rem)+ncoladd

		if widget_info(state.wids.xcol,/NAME) EQ 'TEXT' then begin
		   ; comments are to avoid changing column 98/08/20
		   ; widget_control,state.wids.ycol,set_value = $
		   ; strcompress(tmpcol,/rem) 
		endif else begin
		   tmptxt = ' '+$
		     strcompress(n_elements(state.coltitles)+1,/rem)+': '
		   tmp  = [state.coltitles,tmptxt+'Oper with columns']
		   if ncol_old NE ncol then begin
		     xpstr_update,state,coltitles=tmp
		     widget_control,state.wids.xcol,set_value=$
			Xplot_ColTitles_Short(tmp)
		     widget_control,state.wids.ycol,set_value=$
			Xplot_ColTitles_Short(tmp)
		     widget_control,state.wids.xcol,set_droplist_select=$
			state.xpdt.xcol-1
		     widget_control,state.wids.ycol,set_droplist_select=$
			state.xpdt.ycol-1
		   endif
		endelse

		; spec setup
		if state.spec.on EQ 1 then begin
		  labels = state.spec.labels ;+'!Column Oper'
		  ;state.spec.labels = labels
		  tmp = strparse(labels,'!',labelslist)
		  if tmp EQ -1 then begin
		    tmp = widget_message(dialog_parent=event.top,$
			'Error (column separator)', /ERROR)
		    goto,out
		  endif
		  if (ncol EQ ncol_old) then begin
		    labelslist(tmpcol-1) = 'Column Oper'
		  endif else begin
		    labelslist = [labelslist,'Column Oper']
		  endelse
		  labels = labelslist(0)
		  for i=1,n_elements(labelslist)-1 do $
			labels = labels + '!' + labelslist(i)
		  state.spec.labels = labels
                  labelslist=strcompress((sindgen(n_elements( $
                    labelslist)+1))(1:n_elements(labelslist)) ) + $
                    ': '+labelslist
      		  widget_control,state.wids.xcolSPEC,SET_VALUE=$
			labelslist, $
			SET_DROPLIST_SELECT =  state.xpdt.xcol-1
      		  widget_control,state.wids.ycolSPEC,SET_VALUE=$
			labelslist, $
			SET_DROPLIST_SELECT =  state.xpdt.ycol-1
		  ; commented to avoid changing column 98/08/20
      		  ;widget_control,state.wids.ycolSPEC,SET_VALUE=$
		  ;  labelslist, SET_DROPLIST_SELECT = tmpcol-1
		endif

		handle_free,state.dthcurrent
		h1 = handle_create(value=tmpset)
		state.dthcurrent = h1
		xplot_plot,state
		widget_control,state.wids.message,set_value = $
		  'Current set has been modified '+$
		  "with the option 'Calculations/Operation with columns'"

		if action eq 'DO_APPLY' then begin
		  widget_control, event.top, SET_UVALUE = state, /NO_COPY
		  xplot_event,event
		  return
		endif
		END
	'XPLOT':	XPlot,WTITLE= $
		'XPLOT '+xplot_version()+' -child of xplot-',$
			no_block=state.no_block
	'XPLOTCLONE':	BEGIN	
		state.cloned=1
		tmp = state
		XPlot,WTITLE='XPLOT '+xplot_version()+' -clone of xplot-',$
			state=tmp,WINDOW_SIZE=[!d.x_size,!d.y_size],$
			no_block=state.no_block
		END
	'IDLCOMM': BEGIN
		itmp = Dialog_Message(/Question,Dialog_Parent=event.top,$
		 'Do you want to load the current xplot graph in the '+$
		 'XOP MACRO?')
		case itmp of
		'Yes': begin
                	; initialize printing buffer
                	printbuffer = '  '
                	; load printing buffer
                	xplot_plot,state,printbuffer=printbuffer
                	; print it
                	Xop_Macro,printbuffer ; ,GROUP=event.top
                	end
		'No': Xop_Macro,group=event.top
		endcase

		END
	'XLOADCT': BEGIN
		Xloadct,GROUP=event.top,/Modal
		IF !d.n_colors GT 255 THEN BEGIN
		  itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
		     [ 'Color table changed.',$
		     'As you are using a 24-bit display, you must',$
		     'close and reopen the Xplot controls window to',$
		     'get there the colors updated.'])
		ENDIF
		xplot_plot,state
		END
	'FONT':	BEGIN
                  state.strcontrols.Font = event.index
;		  xyouts,"'!"+strcompress(3+ $
;			fix(state.strcontrols.Font),/REM)+"'"
		IF my_value LE 15 THEN BEGIN
		  state.xpdt.sysval[8]='!p.font=-1'
		  xyouts,"'!"+strcompress(3+ $
			fix(state.strcontrols.Font),/REM)+"'"
		ENDIF ELSE BEGIN
		  IF my_value EQ 16 THEN BEGIN
		    Message,/Info,'Setting hardware font'
		    state.xpdt.sysval[8]='!p.font=0'
		  ENDIF ELSE BEGIN
		    state.xpdt.sysval[8]='!p.font=1'
		    Message,/Info,'Setting TT font'+StrCompress(my_value-17)
		    TT_Font_Manager,Set_Index=my_value-17
		  ENDELSE
		ENDELSE
		  xplot_plot,state
		END
	'WCALC':	scicalc,GROUP=event.top
	'HELPABOUT':		BEGIN
		tmp = widget_message(dialog_parent=event.top,$
		 xplot_text('HELPABOUT'),/info,title='About Xplot')
		END
	'HELPWWW':		BEGIN
		;command = Xop_GetEnv('XOP_WEB_BROWSER')
		;web_address = $
		;  'http://www.esrf.fr/computing/scientific/xop/xplot'
		;IF command EQ '' THEN BEGIN
		;  itmp = Dialog_Message(/Info,Dialog_Parent=event.top,$
		;    ['Point your browser to the XPLOT Web page at:','',$
		;     web_address,' ',' ',$
		;     '(for launching automatically the web browser, define it',$
		;     'in the XOP_WEB_BROWSER environment variable.)'])
		;ENDIF ELSE BEGIN
		;  action=0
		;  command = command + ' ' + web_address
		;  IF SDep() EQ 'UNIX' THEN command=command+' &'
		;  xedit,command,infotext='Command to open WWW browser: ',$
		;    action=action, text = ' ',Title='WWW conection'
		;    if action EQ 'CANCEL' then GoTo,out
		;    message,/Info,'Executing: '+command
		;  widget_control,/hourglass
		;  message,/info,'Executing: '+command
		;  output=''
  		;  spawn,command,output
		;  IF output[0] NE '' THEN $
		;   itmp = Dialog_Message(/Info,Dialog_Parent=event.top,$
		;	Title='Spawn output',output)
		;ENDELSE
		Widget_Control,/HourGlass
		ds = SDep(/ds)
                IF sdep() EQ 'WINDOWS' THEN BEGIN
                  spawn,'start '+xop_getenv('XOP_HOME')+ds+'XOP-DOC'+ds+'xplot.pdf'
                ENDIF ELSE BEGIN
                  spawn,'acroread '+xop_getenv('XOP_HOME')+ds+'XOP-DOC'+ds+'xplot.pdf &'
                ENDELSE

		END
	'HELP':		BEGIN
			widget_control, event.id, get_value=eventval
			xhelp,strlowcase(eventval),group=event.top
		END

;
; these are in the top button menu
;
	'REDRAW':	BEGIN
		Widget_control,state.wids.Message,set_value= ''
		state.xpdt.zoom_on=0
		xplot_plot,state
		END
	'ZOOM':	BEGIN
		state.xpdt.zoom_on = 1
		;xrangeold = state.xpdt.sysvar(2)
		;yrangeold = state.xpdt.sysvar(3)
		;state.xpdt.sysvar(2) = '!x.range = '+ $
		;  vect2string(state.xpdt.zoomsel(0:1))
		;state.xpdt.sysvar(3) = '!y.range = '+ $
		;  vect2string(state.xpdt.zoomsel(2:3))
		xplot_plot,state
		;state.xpdt.sysvar(2) = xrangeold
		;state.xpdt.sysvar(3) = yrangeold
		;itmp = execute(xrangeold)
		;itmp = execute(yrangeold)
                Widget_control,state.wids.Message,set_value= $
                '[x1,x2,y1,y2]='+ $
                vect2string(state.xpdt.zoomsel)+ $
		' Use ReDraw button to zoom back to the top level'
		END
	'CURSOR':	BEGIN
		widget_control,state.wids.draw,get_value=windownum
		wset,windownum
		;added 98/08/27
		p_old = !p
		x_old = !x
		y_old = !y
		!p=state.psysvar.p
		!x=state.psysvar.x
		!y=state.psysvar.y
		if state.no_block then  begin
                  itmp = execute(state.xpdt.sysvar(2))
                  itmp = execute(state.xpdt.sysvar(3))
		endif

		coords_d = [event.x,event.y]
		if (coords_d(0) GT !d.x_size or coords_d(0) LT 0 or $
		  coords_d(1) GT !d.y_size or coords_d(1) LT 0) then goto,out
		if keyword_set(state.dthcurrent) then begin
		  coords = convert_coord(event.x,event.y,/device,/to_data)
		endif else coords = coords_d
		widget_control,state.wids.cursor,SET_VALUE='X:'+$
			strcompress(coords(0),/REM)+' Y:'+$
			strcompress(coords(1),/REM)

                ;
                ; add point to "calculations->list of points"
                ;
		IF event.press EQ 4  THEN BEGIN 
		  IF state.calcstr.getListOfPoints EQ 1 THEN $
		    XDisplayFile1_Append,string(coords[0])+string(coords[1])
		  GOTO,out
                ENDIF
		IF event.press EQ 2  THEN  BEGIN; close list
		  IF state.calcstr.getListOfPoints EQ 1 THEN BEGIN
		    XDisplayFile1_Append,string(coords[0])+string(coords[1])
		    state.calcstr.getListOfPoints=0
		    GOTO,out
                  ENDIF
                ENDIF
                ;
                ; zoom
                ;
                if event.type EQ 0 then begin ;initialize zoom origin
		  xplot_plot,state
                  state.xpdt.zoomsel=[-1000,-1000,-1000,-1000]
                  state.xpdt.zoomsel(0)=coords_d(0)
                  state.xpdt.zoomsel(2)=coords_d(1)
		  ; this is needed for creating a pixmap an storing it
		  ; which contains the graph with no rubberband. This 
		  ; pixmap will be restored while dragging the cursor in
		  ; the zoom selection
		  if state.xpdt.zoompixmap GE 0 THEN $
			wdelete,state.xpdt.zoompixmap
		  window,/free,xsize=!d.x_size,ysize=!d.y_size,/pixmap
		  device,copy=[0,0,!d.x_size,!d.y_size,0,0,windownum]
		  state.xpdt.zoompixmap = !d.window
		  goto,out
                endif
                if (state.xpdt.zoomsel(1) eq state.xpdt.zoomsel(3)) and $
                  (state.xpdt.zoomsel(1) EQ -1000)  then zoom_sel_flag = 1 $
                  else zoom_sel_flag = 0
                if zoom_sel_flag EQ 1 then begin  ; zoom rectangle
                  distx=abs(coords_d(0)-state.xpdt.zoomsel(0))
                  disty= abs(coords_d(1)-state.xpdt.zoomsel(2))
                  if distx NE 0.0 and disty NE 0.0 then begin
		    ; old zoom part
                    ;tvbox,min([coords_d(0),state.xpdt.zoomsel(0)]), $
                    ;min([coords_d(1),state.xpdt.zoomsel(2)]), distx,disty,-2
		    ; new zoom part
		    xd=coords_d(0)
		    yd=coords_d(1)
		    xs=state.xpdt.zoomsel(0)
		    ys=state.xpdt.zoomsel(2)
		    ;print,'start: ',xs,ys,' end: ',xd,yd
		    device,copy=$	
			[0,0,!d.x_size,!d.y_size,0,0,state.xpdt.zoompixmap]
		    plots,[xs,xs,xd,xd,xs],[ys,yd,yd,ys,ys],/device,linestyle=2
                  endif
                endif
                if ((event.type EQ 1) AND (zoom_sel_flag EQ 1)) then begin 
		;end zoom region
		  if keyword_set(state.dthcurrent) then begin
                    coords0 = convert_coord(state.xpdt.zoomsel(0),$
                      state.xpdt.zoomsel(2),/device,/to_data)
		  endif else coords0 = [state.xpdt.zoomsel(0),$
			state.xpdt.zoomsel(2)]
                  x0 = min([coords0(0),coords(0)])
                  y0 = min([coords0(1),coords(1)])
                  distx=abs(coords0(0)-coords(0))
                  disty= abs(coords0(1)-coords(1))
                  xrange = [x0,x0+distx] 
                  yrange = [y0,y0+disty]
                  state.xpdt.zoomsel=[x0,x0+distx,y0,y0+disty]
                  Widget_control,state.wids.Message,set_value= $
                  'Selection Limits: [x1,x2,y1,y2]='+ $
                  vect2string(state.xpdt.zoomsel)+' ; in columns '+$
		  Vect2String([state.xpdt.xcol,state.xpdt.ycol])
                endif
		END
ENDCASE

OUT:
if n_elements(p_old) NE 0 then !p=p_old
if n_elements(x_old) NE 0 then !x=x_old
if n_elements(y_old) NE 0 then !y=y_old
IF N_Elements(state) NE 0 THEN $
  IF Widget_Info(event.top,/Valid_Id) THEN $
    Widget_Control, event.top, SET_UVALUE = state, /NO_COPY

end ;xplot_event
;
;=======================================================================
;
PRO XPLOT, indata, indataopt, GROUP=group, WINDOW_SIZE=w_size, $
	Xcol = xcol, YCOL = ycol, QUIT_BUTTON=tmp, $
	WTITLE = wtitle, TITLE = title, XTITLE = xtitle, YTITLE = ytitle, $
	LEGEND = legend, STATE=pSTATE, $
	XRANGE=xrange, YRANGE=yrange, PARENT = wXplotWindow, $
	COLTITLES=coltitles, XLOG=xlog, YLOG=ylog, SPEC=spec, $
	NO_BLOCK=no_block,FONT=font
;
;
forward_function spec_access, spec_scan ,spec_labels,spec_data,$
  spec_headers,spec_columns,spec_motors,spec_name,spec_points,spec_save


Catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!error_state.msg
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'XPLOT: error caught: '+!error_state.msg)
   catch, /cancel
   on_error,2
endif

if keyword_set(group) then widget_control,/hourglass

;
; check colors
;

Device,Get_Decomposed=usingDecomposed
IF usingDecomposed EQ 1 THEN BEGIN
  itmp = Dialog_Message(/Question, $
     ['XPLOT does not work properly with decomposed colors',$
     'Turn decomposed colors off?'])
  IF itmp EQ 'Yes' THEN Device,Decomposed=0  
ENDIF
;
; create data structures
;

;
; the data structure xpdt
;

;
; store the system variables that may be affected by xplot and reset when exit
;
xpdt = xplot_defaults('xpdt')
xpdt_limits = xplot_defaults('xpdt_limits')
xpdt_titles = xplot_defaults('xpdt_titles')

strcontrols = xplot_defaults('strcontrols')

IF Keyword_set(font) THEN strcontrols.font=font ELSE font=strcontrols.font

IF font LE 15 THEN BEGIN
  xpdt.sysvar[8]='!p.font=-1'
ENDIF ELSE IF font EQ 16 THEN BEGIN
  xpdt.sysvar[8]='!p.font=0'
ENDIF  ELSE BEGIN
  xpdt.sysvar[8]='!p.font=1'
ENDELSE
  
data = 0
;
;insert defaults
if n_elements(no_block) EQ 0  then no_block=0
if keyword_set(Xcol) 	then xpdt.Xcol = Xcol
if keyword_set(Ycol) 	then xpdt.Ycol = Ycol
;if keyword_set(Wtitle) 	then Wtitle = Wtitle else Wtitle='Xplot '+$
;		xplot_version()
if keyword_set(Title) 	then xpdt_titles.Title = Title
if keyword_set(Xtitle) 	then begin
  xpdt_titles.Xtitle = Xtitle
  if (Xtitle EQ '-1' and not(keyword_set(coltitles))) then coltitles=1
endif
if keyword_set(Ytitle) 	then begin
  xpdt_titles.Ytitle = Ytitle
  if (Ytitle EQ '-1' and not(keyword_set(coltitles))) then coltitles=1
endif


if keyword_set(xrange) 	then begin
  xpdt_limits.Xmin = xrange(0)
  xpdt_limits.Xmax = xrange(1)
  xpdt.sysvar(2) = '!x.range = [' + $
    strcompress(xpdt_limits.Xmin,/rem)+','+ $
    strcompress(xpdt_limits.Xmax,/rem)+']'
endif

if keyword_set(yrange) 	then begin
  xpdt_limits.Ymin = yrange(0)
  xpdt_limits.Ymax = yrange(1)
  xpdt.sysvar(3) = '!y.range = [' + $
    strcompress(xpdt_limits.Ymin,/rem)+','+ $
    strcompress(xpdt_limits.Ymax,/rem)+']'
endif

fileinf={name:'<none>',dir:'', mtime:long64(0) }
plotname='<variable>'
if keyword_set(indata) 	then begin
  if type(indata) eq 7 then begin
      plotname = indata
      fileinf.name=indata
		; MSR 97/02/10 This section is commented because if spec_acess 
		; is precompiled with xop.sav, this option does not work. 
		;
      ;if sdep() EQ 'WINDOWS' then tmpf='SPEC_A~1.DAT' else $
      ;  tmpf='spec_access.pro'
      ;if filepath1(tmpf) NE '' then $
      ;	spec_access_flag = spec_access(hspec, indata) else $
      ;  spec_access_flag = -1
      IF Not(keyword_Set(wTitle)) THEN wTitle=indata+' - '+'Xplot '+$
                xplot_version()
      if not(keyword_set(spec)) then begin  ; ASCII input
	spec_access_flag = -1
	data = rascii(indata) 
                  if not(keyword_set(data)) then begin
		     tmp = widget_message(dialog_parent=group,$
			'Error opening file',/ERROR)
                     return
                  endif
	xpdt.infile=indata
      endif else begin     ; spec input
        spec_access_flag = spec_access(hspec, indata) 
        tmp = widget_message(dialog_parent=group,['SPEC input file.', $
		'Please use File/Load Data file option to store it'],/ERROR)
	indata = 0
      endelse
  endif else begin
    ; when calling xplot with two arrays (as plot) then collapse them in
    ; only one.
    if n_params() EQ 2 then begin
      print,'XPLOT: Using two input arrays.'
      data = fltarr(2,n_elements(indata))
      if  n_elements(indata) NE n_elements(indataopt) then begin
        print,'XPLOT: Error: the two arrays must be of the same dimension.'
        return
      endif
      data(0,*) = reform(indata)  &  data(1,*) = reform(indataopt)
    endif else data = indata
  endelse
  xpdt.Nfiles = 1
endif
if keyword_set(Wtitle) 	then Wtitle = Wtitle else Wtitle='Xplot '+$
		xplot_version()
;
;window
if n_elements(w_size) lt 2 then begin     ; default window size
  device, GET_SCREEN = w_size
  w_size = (w_size(0) * [5,4]) /9
endif

;
; widget definition
;
wXplotWindow = $
  Widget_base(title=Wtitle,MBAR=wMenuBar, $
	/TLB_SIZE_EVENTS)	; base window
wXplotBase = Widget_base(wXplotWindow,/Column)	; base widget

wXplotButtons   = Widget_base(wXplotBase,/Row)	; buttons
wXplotDraw = Widget_draw(wXplotBase, XSIZE=w_size(0), YSIZE=w_size(1), $
	/MOTION_EVENTS,UVALUE='CURSOR',RETAIN=2, $ ; /NO_COPY, $
	BUTTON_EVENTS=1)
;
; BUG IN IDL: If we use /frame here (as done up to xplot 2.5) the
; xplot window changes size when doing some actions, like pressing the
; REDRAW button. To avoid that, Do Not Use /frame
;
;wXplotInfo   = Widget_base(wXplotBase, ROW=1, /FRAME)	; buttons
wXplotInfo   = Widget_base(wXplotBase, ROW=1)	; buttons

wXplotMessage   = Widget_label(wXplotBase,Value='Xplot Message: ', $
	/ALIGN_LEFT,/DYNAMIC_RESIZE)

;
; build the menu bar
;

wFileMenu =  WIDGET_BUTTON(wMenuBar, VALUE='File', /MENU)
  wLoadFile=WIDGET_BUTTON(wFileMenu, VALUE='Load Data File...', $
	UVALUE='LOADFILE')
  wReLoadFile=WIDGET_BUTTON(wFileMenu, VALUE='Re-Load Data File', $
	UVALUE='RELOADFILE')
  wAutoReLoadFileMain=WIDGET_BUTTON(wFileMenu, VALUE='AutoRe-Load Data File',/Menu)
     wAutoReLoadFile = $
	WIDGET_BUTTON(wAutoReLoadFileMain, VALUE='Preferences...',UVALUE='AUTORELOADFILE')
     wtmp = WIDGET_BUTTON(wAutoReLoadFileMain, VALUE='Start',UVALUE='AUTORELOADFILE')
     wtmp = WIDGET_BUTTON(wAutoReLoadFileMain, VALUE='Stop',UVALUE='AUTORELOADFILE')
  wSaveDataAs=WIDGET_BUTTON(wFileMenu, /Separator, VALUE= $
	'Write ASCII file with current set...', UVALUE='SAVEASCII')
  wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Xplot Backup File', /MENU)
    wSavePlotAs=WIDGET_BUTTON(wtmp, VALUE='Write...',  UValue='SAVEIDL')
    wRestoreFile=WIDGET_BUTTON(wtmp, VALUE='Load...', UValue='RESTORE')
  wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Xplot Attributes File', /MENU)
    wtmp1=WIDGET_BUTTON(wtmp, VALUE='Write...', UValue='ATTRIBUTES')
    wtmp1=WIDGET_BUTTON(wtmp, VALUE='Load...', UValue='ATTRIBUTES')
    wtmp1=WIDGET_BUTTON(wtmp, VALUE='Clean Session Default',UValue='ATTRIBUTES')
  wPrintPlot=WIDGET_BUTTON(wFileMenu, VALUE='Print Plot...', UVALUE='PRINT',$
	/SEPA)
  wExit=WIDGET_BUTTON(wFileMenu, VALUE='Quit', UVALUE='EXIT',/SEPA)


wEditMenu =  WIDGET_BUTTON(wMenuBar, VALUE='Edit', /MENU)
  wClearAll=WIDGET_BUTTON(wEditMenu, VALUE='Clear All', UVALUE='CLEARDATA')
  wLimits=WIDGET_BUTTON(wEditMenu, VALUE='Limits', /MENU)
  wData=WIDGET_BUTTON(wEditMenu, VALUE='Data...', UVALUE='EDITDATA')
  wtmp=WIDGET_BUTTON(wEditMenu, VALUE='Columns...', UVALUE='EDITCOLUMNS')
  wtmp=WIDGET_BUTTON(wEditMenu, VALUE='Cut...', UVALUE='EDITCUT')
  wLimitsSet=WIDGET_BUTTON(wLimits, VALUE='Set Limits and Styles...', $
	UVALUE='SETLIMITS')
  wLimitsSel=WIDGET_BUTTON(wLimits, VALUE='Set Defaults', $
	UVALUE='SETLIMITSDEF')
  wLimitsSel=WIDGET_BUTTON(wLimits, VALUE='Set Limits as Selection Area', $
	UVALUE='SETLIMITSZOOM')

  wTitles=WIDGET_BUTTON(wEditMenu, VALUE='Titles...',UVALUE='SETTITLES')
  wColTitles=WIDGET_BUTTON(wEditMenu, VALUE='Column Titles...',$
	UVALUE='SETCOLTITLES')
  wLegend=WIDGET_BUTTON(wEditMenu, VALUE='Legend...',UVALUE='SETLEGEND')
  wLabels=WIDGET_BUTTON(wEditMenu, VALUE='Labels...',UVALUE='SETLABELS')

  wPlotNames=WIDGET_BUTTON(wEditMenu, VALUE='Plot Names',/MENU)
    wtmp=WIDGET_BUTTON(wPlotNames, VALUE='Current Set...',UVALUE='PLTNAME_CUR')
    wtmp=WIDGET_BUTTON(wPlotNames, VALUE='Saved Set...',UVALUE='PLTNAME_SAV')

  wShift=WIDGET_BUTTON(wEditMenu, VALUE='Shift data...', UVALUE='SHIFT')
  wErrPlot=WIDGET_BUTTON(wEditMenu, VALUE='Error bars...', UVALUE='ERRPLOT')
  wExeBuffer=WIDGET_BUTTON(wEditMenu, VALUE='Legend/Label buffer',/MENU)
  wExeBE=WIDGET_BUTTON(wExeBuffer, VALUE='Clean', UVALUE='EXEBUFFERCLEAN')
  wExeBC=WIDGET_BUTTON(wExeBuffer, VALUE='Edit...', UVALUE='EXEBUFFEREDIT')
  wSysVar=WIDGET_BUTTON(wEditMenu, VALUE='System Variables', /MENU)
    wSysVarx=WIDGET_BUTTON(wSysVar, VALUE='!x', UVALUE='SYSVAR')
    wSysVary=WIDGET_BUTTON(wSysVar, VALUE='!y', UVALUE='SYSVAR')
    wSysVarp=WIDGET_BUTTON(wSysVar, VALUE='!p', UVALUE='SYSVAR')

wViewMenu =  WIDGET_BUTTON(wMenuBar, VALUE='View', /MENU)
  wShowData=WIDGET_BUTTON(wViewMenu, VALUE='Show Data', /MENU)
    wShowDataXY=WIDGET_BUTTON(wShowData, VALUE= $
	'Plotted X Y columns (current set)', UVALUE='SHOWDATAXY')
    wShowDataSet=WIDGET_BUTTON(wShowData, VALUE= $
	'All columns of the current set', UVALUE='SHOWDATASET')
    wDrawDim=WIDGET_BUTTON(wViewMenu, VALUE='Draw window size...', $
	UVALUE='DRAWDIM')
    wColPanel=WIDGET_BUTTON(wViewMenu, VALUE='Change Column Panel', $
	UVALUE='COLPANEL')
    w2ndAxis=WIDGET_BUTTON(wViewMenu, VALUE='Second X/Y axis...', $
	UVALUE='SECONDAXIS')
    wMesh=WIDGET_BUTTON(wViewMenu, VALUE='Mesh plot...', $
	UVALUE='MESH')
    wTmp=Widget_Button(wViewMenu,VALUE='Xplot ID number',UVALUE='ID')

wCalc=WIDGET_BUTTON(wMenuBar, VALUE='Calculations', /MENU)
  wWidthInt=WIDGET_BUTTON(wCalc,VALUE='Width/Integral/MinMax', $
	UVALUE='CALCULATIONS')
  wWidthInt=WIDGET_BUTTON(wCalc,VALUE='Moments', UVALUE='CALCULATIONS')
  wWidthInt=WIDGET_BUTTON(wCalc,VALUE='List of points...', UVALUE='CALCULATIONS')

  wtmp=WIDGET_BUTTON(wCalc,VALUE='Create data set...', $
	UVALUE='CALCULATIONS',/Separator)
  wtmp=WIDGET_BUTTON(wCalc,VALUE='New window with current set', $
	UVALUE='CALCULATIONS')
  wColumnOper=WIDGET_BUTTON(wCalc,VALUE='Operations with columns...', $
	UVALUE='COLUMNOPER')
  wEvaluation=WIDGET_BUTTON(wCalc,VALUE='Operations with sets...',$
	UVALUE='CALCULATIONS')

  wtmp=WIDGET_BUTTON(wCalc,VALUE='Derivative',UVALUE='CALCULATIONS', $
	/Separator)
  wtmp=WIDGET_BUTTON(wCalc,VALUE='CDF...',UVALUE='CALCULATIONS')
  wtmp=WIDGET_BUTTON(wCalc,VALUE='Power Spectral Density...',UVALUE='CALCULATIONS')
  wFouFil=WIDGET_BUTTON(wCalc,VALUE='Fourier Filtering...', $
	UVALUE='CALCULATIONS')
  wConvolution=WIDGET_BUTTON(wCalc,VALUE='Convolution & Correlation...',$
	UVALUE='CALCULATIONS')

  wtmp=WIDGET_BUTTON(wCalc,VALUE='Smooth...',UVALUE='CALCULATIONS',$
	/Separator)
  wInterpolation=WIDGET_BUTTON(wCalc,VALUE='Interpolation/Spline...', $
	UVALUE='CALCULATIONS')

  wConvolution=WIDGET_BUTTON(wCalc,VALUE='Find peaks...',$
	UVALUE='CALCULATIONS', /Separator)
  wtmp=WIDGET_BUTTON(wCalc,VALUE='Histogram...', UVALUE='CALCULATIONS')

  wFitPoly=WIDGET_BUTTON(wCalc,VALUE='Fit - Polynomial&Gaussian...', $
	UVALUE='CALCULATIONS',/Separator)
  wtmp=WIDGET_BUTTON(wCalc,VALUE='Fit - Multiple peaks...', $
	UVALUE='CALCULATIONS')
  wNLFit=WIDGET_BUTTON(wCalc,VALUE='Fit - Non linear...', $
	UVALUE='CALCULATIONS')

    IF Xop_GetEnv('XOP_EXTENSION_XAID') NE '' THEN BEGIN
      wXAFS=WIDGET_BUTTON(wCalc,VALUE='XAFS', /MENU)
      wXcnpick=WIDGET_BUTTON(wXAFS,VALUE='XAFS extraction (xaid_mu2chi)', $
	UVALUE='CALCULATIONS')
      wXcnpick=WIDGET_BUTTON(wXAFS,VALUE='XAFS Fourier Filter (xaid_ff)', $
	UVALUE='CALCULATIONS')
      wtmp=WIDGET_BUTTON(wXAFS,VALUE='XAFS main window (XAID)', $
	UVALUE='CALCULATIONS')
    ENDIF
wToolsMenu =  WIDGET_BUTTON(wMenuBar, VALUE='Tools', /MENU)
  ;wXloadct=WIDGET_BUTTON(wToolsMenu, VALUE='Xloadct', UVALUE='XLOADCT')
  wAnnotate=WIDGET_BUTTON(wToolsMenu, VALUE='Annotate...',UVALUE='ANNOTATE')
  wCalculator=WIDGET_BUTTON(wToolsMenu, VALUE='Calculator', UVALUE='WCALC')
  wXplot=WIDGET_BUTTON(wToolsMenu, VALUE='Xplot (Empty)', UVALUE='XPLOT')
  ; wXplotClone=WIDGET_BUTTON(wToolsMenu, VALUE='Xplot (Clone)', $
  ;    UVALUE='XPLOTCLONE')
  wIdlCommand=WIDGET_BUTTON(wToolsMenu, VALUE='XOP macro', UVALUE='IDLCOMM')

wHelpMenu = WIDGET_BUTTON(wMenuBar, VALUE='Help', /HELP)
  wHelptmp = WIDGET_BUTTON(wHelpMenu, VALUE='About Xplot', UVALUE='HELPABOUT')
  wHelptmp = WIDGET_BUTTON(wHelpMenu, VALUE='Xplot', UVALUE='HELP')
  wHelptmp = WIDGET_BUTTON(wHelpMenu, VALUE='Xplot manual', $
	UVALUE='HELPWWW')
  ;wHelpRoutines = WIDGET_BUTTON(wHelpMenu, $
  ;	VALUE='Math Routines used by Xplot',/MENU)
  ;n=strparse((xplot_defaults('math_routines')).math_routines,':',mylist)
  ;for i=0,n do $
  ;  wHelptmp = WIDGET_BUTTON(wHelpRoutines, VALUE=mylist(i), UVALUE='HELP')
;
; build the buttons panel
;

wB0 = widget_base(wXplotButtons,/Row)

wB1 = widget_base(wB0,/ROW,/FRAME)
wBquit = widget_button(wB1, VALUE="Quit",UVALUE='EXIT')
wBredraw =  widget_button(wB1, VALUE="ReDraw",UVALUE='REDRAW')
wBsave =  widget_button(wB1, VALUE="Save Plot",UVALUE='SAVE')
wBplotmgr =  widget_button(wB1, VALUE="Plot Mgr",UVALUE='PLOTMGR')
;wBsetcurrent =  widget_button(wB1, VALUE="Set Current",UVALUE='SETCURRENT')
;wBzoomsel = cw_bgroup(wB1, "Zoom Sel",UVALUE='ZOOMSEL')
wBzoom = cw_bgroup(wB1, "Zoom",UVALUE='ZOOM')
wBcontrols = widget_button(wB1, VALUE="Show Controls",UVALUE='CONTROLS')
wBcursor = widget_text(wB0,/FRAME,XSIZE=25)


;
; Info (bottom) line
;
if keyword_set(data) then begin
  if n_elements(data) EQ 1 then plotfile,data,tmp,/nodata else tmp=data
  Ncol = n_elements(tmp(*,0)) 
  if keyword_set(Ycol) then xpdt.Ycol = Ycol
  if keyword_set(Xcol) then xpdt.Xcol = Xcol
  IF xpdt.Xcol LE 0 THEN xpdt.Xcol=1
  IF xpdt.Ycol LE 0 THEN xpdt.Ycol=NCol
  ;current_set = '1'
endif else begin
  Ncol=0
  ;current_set = '0'
endelse

if keyword_set(coltitles) then begin
  if n_elements(coltitles) EQ 1 then coltitles=['<none>']
  if keyword_set(indata) then begin
    if n_elements(coltitles) EQ 1 then begin
	coltitles= (sindgen(Ncol+1))(1:Ncol)
      ;print,'Im now here'
    endif
    if n_elements(coltitles) NE Ncol then tmp = widget_message($
	dialog_parent=group,$
	'The number of column titles is different to the column number.')
  endif
endif else coltitles=0
;====================================================================
wBsetBase = widget_base(wXplotInfo,/Row,/align_center)
 wBsetsaved = widget_label(wBsetBase,value=strcompress(0,/rem))
 wBset_tmp =  widget_label(wBsetBase,value=' saved sets. Current set has ')
 wBsetncol = widget_label(wBsetBase,value=strcompress(Ncol,/REM),/Dynamic_Resize)
 wBset_tmp = widget_label(wBsetBase,value=' columns,')
wBsetcol = widget_base(wXplotInfo)
 wBsetcolASCII = widget_base(wBsetcol,/Row, /align_center, MAP=1)
  wBset_tmp = widget_label(wBsetcolASCII,value=' X: ')
  if not(keyword_set(coltitles)) then begin
    wBsetBxcol = widget_base(wBsetcolASCII)
    wBsetxcol = widget_text(wBsetBxcol,value= $
      strtrim(string(xpdt.Xcol),2),XSIZE=4,YSIZE=1,/EDIT,/FRAME,UVALUE='XCOL')
    wBset_tmp = widget_label(wBsetcolASCII,value=' Y: ')
    wBsetBycol = widget_base(wBsetcolASCII)
    wBsetycol = widget_text(wBsetBycol,value= $
      strtrim(string(xpdt.Ycol),2),XSIZE=4,YSIZE=1,/EDIT,/FRAME,UVALUE='YCOL')
  endif else begin
    wBsetBxcol = widget_base(wBsetcolASCII)
    wBsetxcol = widget_droplist(wBsetBxcol,value= $
      Xplot_ColTitles_Short(coltitles), UVALUE='XCOL')
    wBset_tmp = widget_label(wBsetcolASCII,value=' Y: ')
    wBsetBycol = widget_base(wBsetcolASCII)
    wBsetycol = widget_droplist(wBsetBycol,value= $
      Xplot_ColTitles_Short(coltitles), UVALUE='YCOL')
  endelse

 wBsetcolSPEC = widget_base(wBsetcol,/Row,MAP=0)
  wBset_tmp0 = widget_base(wBsetcolSPEC,MAP=1)
   wBsetcolNormal = widget_base(wBset_tmp0,MAP=1, row=1)
    wBset_tmp = widget_label(wBsetcolNormal,value=' X:')
    wBsetxcolSPEC = widget_droplist(wBsetcolNormal,value='<none>', $
	UVALUE='XCOL',/DYNAMIC_RESIZE)
    wBset_tmp = widget_label(wBsetcolNormal,value=' Y:')
    wBsetycolSPEC = widget_droplist(wBsetcolNormal,value='  <none>  ', $
	UVALUE='YCOL',/DYNAMIC_RESIZE)
   wBsetcolMCA = widget_base(wBset_tmp0,MAP=0, row=1)
    wBsetxmca = cw_bgroup(wBsetcolMCA, ['Channel','Energy'], $
                      ROW=1, /EXCLUSIVE, UVALUE='XCOL', set_value=0, /frame)
    wBset_tmp = widget_label(wBsetcolMCA,value=' Y: ')
    wBsetymca = widget_text(wBsetcolMCA,value='0', $
                                   XSIZE=4,YSIZE=1,/EDIT,/FRAME,UVALUE='YCOL')
  wBset_tmp0 = widget_button(wBsetcolSPEC,value= 'Scan sel...', UVALUE='SSCAN')

;====================================================================
;
; activate widgets
;
widget_control, wXplotWindow, /NO_COPY, /REALIZE

IF font LE 15 THEN XYOuts,'!'+StrCompress(font+3,/Rem) ELSE IF $
   font GT 16 THEN TT_Font_Manager,Set_Index=font-17

widget_control, wBplotmgr, SENSITIVE=0

if keyword_set(data) then begin
  if type(data) NE 3 then begin
    h1 = handle_create(value=data)
    ;handle_value, h1, data, /set
    ;if handle_info(h1) then print,'Handle data -> handle h1=',h1
    dthcurrent=h1
  endif else dthcurrent=data
endif else dthcurrent = 0L


if keyword_set(Xlog) then begin
   xpdt.kind = 4
   strcontrols.axis=2
endif
if keyword_set(Ylog) then begin
   xpdt.kind = 3
   strcontrols.axis=1
endif
if (keyword_set(Xlog) AND keyword_set(Ylog)) then begin
   xpdt.kind = 5
   strcontrols.axis=3
endif

;
; create the state structure
;
;structure to store the widget ids to be change during the
;running time
wids = { XPWIDS, main:wXplotWindow, cursor:wBcursor, $
        asciipanel:wBsetcolASCII, specpanel:wBsetcolSPEC, $
        normalpanel:wBsetcolNormal, mcapanel:wBsetcolMCA, $
        saved:wBsetsaved, ncol:wBsetncol, $
	xcolBase:wBsetBxcol,ycolBase:wBsetBycol, $
	xcol:wBsetxcol,ycol:wBsetycol, $
        xcolspec:wBsetxcolSPEC, ycolspec:wBsetycolSPEC, $
        xcolmca:wBsetxmca, ycolmca:wBsetymca, $
        plotmgr:wBplotmgr, showcontrols:wBcontrols, $
        controls:[0L,0L,0L,0L], message:wXplotMessage, draw:wXplotDraw, $
	autoReLoadFile:wAutoReLoadFile  }

xpevaluate = xplot_defaults('XPevaluate')
;xpcolumnoper = xplot_defaults('XPcolumnoper')
xpconvolution = xplot_defaults('XPconvolution')
;xpcorrelation = xplot_defaults('XPcorrelation')
;calcstr = { xpevaluate:xpevaluate, xpcolumnoper:xpcolumnoper, $
calcstr = { xpevaluate:xpevaluate, xpconvolution:xpconvolution, getListOfPoints:0}
autoReLoadFile = xplot_defaults('AutoReLoadFile')

if keyword_set(Legend) 	then exebuffer = Legend else exebuffer = 0

spechandle = handle_create(value=0)
specstr = {on:0, handle:spechandle, mca:0, scan:0, scanbase:0L, labels:''}

; ************************* VERY IMPORTANT ***************************
; note: when updating state, update also xpstr_update!!!
; ************************* VERY IMPORTANT ***************************
;spechandle = handle_create(value=0)
version = float(xplot_version())
state = { wids:wids, calcstr:calcstr, $
  xpdt:xpdt, xpdt_limits:xpdt_limits, $
  xpdt_titles:xpdt_titles, $
  autoReLoadFile:autoReLoadFile, $ 
  buffer:0, $ 
  dthcurrent:dthcurrent, dthbuffer:0L, strcontrols:strcontrols, $
  spec:specstr, exebuffer:exebuffer, version:version, cloned:0, $
  coltitles:coltitles, fileinf:fileinf, psysvar:{p:!p,x:!x,y:!y}, $
  no_block:no_block,plotname_buffer:0,plotname_current:plotname}
; ********************************************************************

xplot_mappanels, state.wids, state.spec

if keyword_set(indata) then state1 = state
if keyword_set(pstate) then state1 = state

;if keyword_set(spec) then xplot_loadfilespec,state,spec
if keyword_set(spec) then begin
  widget_control, wXplotWindow, set_uvalue=state, /NO_COPY
  xplot_loadfile,wXplotWindow,spec=spec
  widget_control, wXplotWindow, get_uvalue=state, /NO_COPY
endif

;<><>widget_control, wXplotWindow, set_uvalue=state, /NO_COPY

if keyword_set(coltitles) then begin
  widget_control,wBsetxcol,set_droplist_select=xpdt.xcol-1
  widget_control,wBsetycol,set_droplist_select=xpdt.ycol-1
endif

if keyword_set(pstate) then BEGIN
  print,'<><> XPLOT: clonning..'
  wids = state1.wids
  widget_control,pstate.wids.saved,get_value=saved
  widget_control,pstate.wids.ncol,get_value=ncol
  widget_control,pstate.wids.xcol,get_value=xcol
  widget_control,pstate.wids.ycol,get_value=ycol
  widget_control,pstate.wids.message,get_value=message
  widget_control,pstate.wids.xcolmca,get_value=xcolmca
  widget_control,pstate.wids.ycolmca,get_value=ycolmca


  state1 = pstate
  state1.wids = wids
  widget_control,state1.wids.saved,set_value=saved
  widget_control,state1.wids.ncol,set_value=ncol
  widget_control,state1.wids.xcol,set_value=xcol
  widget_control,state1.wids.ycol,set_value=ycol
  widget_control,state1.wids.message,set_value=message
  widget_control,state1.wids.xcolmca,set_value=xcolmca
  widget_control,state1.wids.ycolmca,set_value=ycolmca

  if state1.spec.on then begin
    xplot_mappanels, state1.wids, state1.spec
    tmp = strparse(state1.spec.labels,'!',labelslist)
    widget_control,state1.wids.xcolspec,set_value=labelslist, $
	set_droplist_select=state1.xpdt.xcol-1
    widget_control,state1.wids.ycolspec,set_value=labelslist, $
	set_droplist_select=state1.xpdt.ycol-1
  endif
  
  widget_control,state1.wids.saved,get_value=saved
  if fix(saved) GE 1 then $
      widget_control,state1.wids.plotmgr,/sensitive
  xplot_plot,state1
  ;widget_control, wXplotWindow, set_uvalue=state1, /NO_COPY
ENDIF

IF keyword_set(indata) then BEGIN
  ;print,'XPLOT: call to xplot_plot from main program'
  xplot_plot,state1
  state.psysvar = state1.psysvar
ENDIF
;<><>
if keyword_set(pstate) then $
  widget_control, wXplotWindow, set_uvalue=state1, /NO_COPY else $
  widget_control, wXplotWindow, set_uvalue=state, /NO_COPY 

if no_block then $
  xmanager, 'xplot', wXplotWindow, GROUP_LEADER=group,/no_block else $
  xmanager, 'xplot', wXplotWindow, GROUP_LEADER=group

;handle_free, spechandle
end
