FUNCTION debyewaller, element, Temper=temper, Ratio=ratio, Edit=edit, $
  Verbose=verbose

; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       DEBYEWALLER
;
; PURPOSE:
;	This finction returns the Debye-Waller factor that affects the 
;	structure factor. 
;	
; CATEGORY:
;       X-Ray optics. 
;
; CALLING SEQUENCE:
;	out = debyeWaller([element])
; INPUTS:
;	element: an identifier of an element: atomic number (i.e. 14) or 
;		symbol, i.e. 'Si', or an array of identifiers
;		(default, Z=14)
;	
; KEYWORDS:
;	TEMPER = The temperature in K (default = 300 K)
;	RATIO = The ratio x=sin(theta)/lambda=1/(2 d) for the calculation
;	EDIT = Set this flag to allow interactive changing of the 
;		calculation parameters, i.e., the ratio, debye temperatures
;		and atomic mass(es). By default, the values for the debye
;		temperature and atomic mass correspond to those of the
;		selected element, available in the DABAX file 
;		AtomicConstants.dat
;	VERBOSE = Set this flag to print in the screen some intermediate
;		calculation values. 
;
; OUTPUT:
;	out: the value(s) of the debye-waller parameter. 
;
; PROCEDURE:
;	It applies the formula DW=exp[-M], with 
;	M = (Bo+BT) (ratio)^2
;	Bo=3 h^2 / 2 Kb Td A
;	   h = Plank's constant, Kb = Boltzmann's constant,
;	   Td = Debye temperatute, A = atomic mass
;	Bt = Bo Phi(y)/y
;	   y = T/Td, T = temperature, and the Phi(y) function is
;	  Phi(y) = y^-1 Integral[0,y,x dx /(exp(x)-1)]
;	  Phi is calculated by numerical integration of the 
;	  function evaluated over a 5001 points grid. 
;	
;	References, 
;	See for example A. Freund, Nucl. Instrum, and Meth. 213 (1983) 495-501
;	or International Tables for X-ray crystallography, etc. 
;
; LIMITATIONS: 
;	This treatment is only approximate. It is only valid for cubic 
;	structures.  Each atom has its own value of M, which itself may 
;	be dependent upon the orientation of the reflecting planes. 
;	For much X-ray work, however, it is sufficient
;	to assume that a mean value of M (e.g., (n1 M1+n2 M2)/(n1 + n2) 
;	is appropriate; The mean DW is 
;	DW = (DW1)^(n1/(n1+n2)) (DW2)^(n2/(n1+n2))
;
; EXAMPLES:
;	print,DebyeWaller('Si',ratio=0.6105)
;	    0.88384834
;	print,DebyeWaller(['Si','Ga'],ratio=0.6105)
;	     0.88384834      0.83062927
;
; MODIFICATION HISTORY:
;       2002-10-08 Written by M. Sanchez del Rio (srio@esrf.fr)
;		cased in the program Mos_GradX.pro (Lucia Alianelli).
;
;
;-

IF N_Elements(element) EQ 0 THEN element=14     ; Default=Si
IF N_Elements(temper) EQ 0 THEN temper=300.0	; TEMPERATURE (K)

IF N_Elements(ratio) EQ 0 THEN BEGIN
  ratio = 1D/(2D*0.819) ; corresponding, for instance, Quartz (330)
ENDIF

t=temper

a_mass = atomic_constants(element,RETURN='AtomicMass')
t_deb = atomic_constants(element,RETURN='DebyeTemperature')

; 
; Allow interactive modification of the parameters, in the case
; that the EDIT keyword is set
;
IF Keyword_Set(edit) THEN BEGIN
  tmp = {ratio:ratio, temper:t, a_mass:a_mass, t_deb:t_deb}
  titles=['sin(theta)/lambda=1/(2d)','Temperature [K]',$
	'Atomic mass(es)','Debye temperature(s)']
  XScrMenu,tmp,/NoType,Action=action ,WTitle='DebyeWaller inputs',Titles=titles
  IF action EQ 'DO' THEN BEGIN
    ratio = tmp.ratio
    t=tmp.temper
    a_mass=tmp.a_mass
    t_deb=tmp.t_deb
  ENDIF
ENDIF

;**************Formulas from (5) to (12) of Freund, NIM (1983):******** 
cte = 2873.

; cte = 3/2 * h^2/(K A) 
; in SI units [L]^2
cte = (3D/2D)*(physical_constants('h')^2)/physical_constants('kb')/ $
      physical_constants('am')
; in A^2
cte = cte * 1d20

B_zero = cte/(T_deb*A_mass) 
X = T_deb/T  

phix = DblArr(N_Elements(x))
FOR i=0L,N_Elements(x)-1 DO BEGIN
  y = DIndGen(5001)*x[i]/5000. 
  y[0] = 0.001*y[1] 
  fy = y/(exp(y)-1.) 
  IF (x[i] GE 7.0) THEN phix[i] = 1.642/x[i]  ELSE $
    phix[i] = (INT_TABULATED(y,fy,/double))/x[i]
ENDFOR


  ;B_T = 4.*B_zero*phix/x  
  B_T = (B_zero/x)*4D*phix
  M = (B_zero+B_T)*ratio^2.
  DW = exp(-M)


  FOR i=0,N_Elements(x)-1 DO BEGIN
    IF t_deb[i]-0.01 LE 1D-5 THEN BEGIN
      IF SDep(/W) EQ 1 THEN BEGIN
        itmp = Dialog_Message(/Error, $
	['DEBYEWALLER: Debye temperature not valid for element: '+ $
	 StrCompress(element[i],/Rem),'Return exp[-M]=1'] ) 
       ENDIF ELSE BEGIN
        Message,/Info,'Debye temperature not valid. Return exp[-M]=1'
      ENDELSE
      dw[i]=1.0
    ENDIF
  ENDFOR

IF Keyword_Set(verbose) THEN BEGIN
  print,'************** DebyeWaller inputs *********************'
  print,'T [K]                            ',T
  print,'ratio=1/(2d) [A^-1]:             ',ratio
  print,'A_mass [units of 12C/12]         ',a_mass 
  print,'T_deb [K]                        ',T_deb
  print,'************** DebyeWaller output *********************'
  print,'B_zero   		         ',B_zero
  print,'B_T   		                 ',B_T
  print,'M   		                 ',M
  print,'Debye Waller factor exp(-M)     ',DW 
  print,'*******************************************************'
  print,'                                                  '
ENDIF


IF Keyword_Set(edit) THEN BEGIN
  itmp = Dialog_Message(/Info,'Using DebyeWaller exp[-M]: '+Vect2String(dw))
ENDIF

RETURN,dw  
END
