Function dabax_ls,path, Group=group, NoSpawn=noSpawn, NF=nn

; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       DABAX_LS
;
; PURPOSE:
;	Returns a string array with the name of the DABAX data sets
;	accessible in the path.
;
;	IFf no files are found, it returns ''
;	
; CATEGORY:
;       DABAX (DAta Base for Atomic X-rays properties) applications.
;
; CALLING SEQUENCE:
;	List = DABAX_LS([path])
; OPTIONAL INPUTS:
;	PATH:  DABAX data path. It is a string with the directories
;		separated by ':'
;		If this input is not set, dabax_ls will read the path
;		from the DABAX_PATH environment variable.
;	
; KEYWORDS:
;	GROUP = The widget ID of the widget that calls DABAX_LS.
;
; OUTPUT KEYWORDS:
;	NF: The number of files found. It is zero if not files are found.
;
; OUTPUT:
;	A string with the file list.
;
; PROCEDURE:
;
; EXAMPLE:
;	IDL> List = dabax_ls( $
;	'/pgexp/applications1/XRayOptics/DABAX/data/')
;	IDL> print,list
;	Constant.dat CrystalCell.dat CrystalStructures.dat
;
; MODIFICATION HISTORY:
;       96-07-10 Written by M. Sanchez del Rio (srio@esrf.fr)
;       97-01-14 srio@esrf.fr ports to Windows (for these platforms,
;		it MUST exist a file calles dabax.lis with the DABAX
;		data file list. This is due to the fact that the 
;		spawn command does not return values to IDL.)
;       97-02-04 srio@esrf.fr uses file dabax.lis also in Unix when this
;		file exists.
;       98-12-14 srio@esrf.fr added Group kw. Use of Catch.
;       99-03-17 srio@esrf.fr Mostly rewritten. Now accepts DABAX_PATH
;		also under Windows. Removed Spawn call.
;       00-07-12 srio@esrf.fr fixed bug: unsort files when using dabax.lis
;       00-07-13 srio@esrf.fr reverse the sorting when dabax.lis is not used
;       02-06-06 srio@esrf.fr changes RStrPos by StrPos(/Reverse_Search)
;		
;-

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!error_state.msg
   IF SDep(/W) THEN $
     itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'DABAX_LS: error caught: '+!error_state.msg)
   Catch, /Cancel
   On_Error,2
   RETURN,''
ENDIF

if not(keyword_set(path)) then path = Xop_GetEnv('DABAX_PATH')
message,/info,'Path: '+path

IF  StrCompress(path,/Rem) EQ '' THEN BEGIN
  action=''
  IF SDep(/W) THEN $
    XPickFile,Title='Input DABAX data path: ',WTitle='Undefined DABAX Path',$
    Action=action,File=path,Group=group ELSE Message,'DABAX_PATH not set.'
  IF action NE 'DO' THEN RETURN,''
  IF path EQ '' THEN RETURN,''
  Xop_Setenv,'DABAX_PATH='+path
ENDIF


separator = sdep(/ps)
bar = sdep(/ds)
;path=strsubstitute(path,separator,' ')
;n = strparse(path,' ',list)
n = strparse(path,separator,list)

stmp=''
use_listfile = 0
FOR i=0,n DO BEGIN
  filelist = list[i]+bar+'dabax.lis'
  IF checkfile(filelist) EQ 1 THEN BEGIN
    use_listfile = 1
    openr,Unit,filelist,/get_lun
    while not eof(Unit) do begin
      readf,Unit,stmp
      IF N_Elements(a) EQ 0 THEN a=stmp ELSE a=[a,stmp]
    endwhile
  ENDIF ELSE BEGIN
    tmp = FindFile(list[i]+bar+'*.dat')
    IF N_Elements(a) EQ 0 THEN a=tmp ELSE a=[a,tmp]
  ENDELSE
ENDFOR

FOR i=0,N_Elements(a)-1 DO a[i]=StrMid(a[i],1+StrPos(a[i],bar,/Reverse_Search))
;IF N_Elements(a) GT 1 THEN a=a(Uniq(a,Sort(a)))
IF N_Elements(a) GT 1 THEN a=a(Uniq(a))
IF N_Elements(a) EQ 0 THEN a='' ELSE nn=N_Elements(a)

nn = N_Elements(a)
IF nn EQ 1 AND a[0] EQ '' THEN nn=0
;
; This is set to present as default f1f2_Windt in xop 2.0 instead 
; of f1f2_BrennanCowan wothout using dabax.lis
;
IF use_listfile EQ 0 THEN BEGIN
  idx = Sort(StrUpCase(a))
  a=a[idx]
  a = Reverse(a) 
ENDIF 

RETURN,a

END ; Dabax_Ls
