; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
;+
; NAME:
;     silo
;
; PURPOSE:
;     An object that implements a random access data storage mechanism.  
;     Any IDL datatype can be added, and types can be mixed within a 
;     single instance of the silo.
;
; TYPE:
;     OBJECT
;
; CATEGORY:
;     Data structures
;
; CALLING SEQUENCE:
;     mySilo = OBJ_NEW ('silo')
;
; INPUTS:
;     NONE
;
; KEYWORDS:
;     NONE
;
; COMMON BLOCKS:
;     NONE
;
; SIDE EFFECTS:
;     None known
;
; RESTRICTIONS:
;     None known
;
; DEPENDENCIES:
;     NONE
;
; METHODS:
;     
;     set (PROCEDURE) - Add data to the silo
;         Inputs: KEY  : STRING name denoting the data
;                 DATA : data to add to the silo (can be any IDL datatype)
;        Outputs: NONE 
;       Keywords: NONE 
;
;     get (FUNCTION) - Retrieve data from the silo
;         Inputs: KEY : STRING name denoting the data
;        Outputs: DATA corresponding to the specified KEY
;       Keywords: NUMBER : Set this keyword to return the number
;                     of data elements currently in the silo.  No input argument 
;                     is required in this case.
;                 NAMES  : Set this keyword to return a STRARR that holds the
;                     names of the  data elements currently in the silo.  No 
;                    input argument is required in this case.
;                 ERROR  : Set to a named variable to return the status of
;                     the GET.  If data could not be extracted, ERROR is set
;                     to 1, else ERROR = 0.
;
;     print (PROCEDURE) - Print all keys currently stored
;         Inputs: NONE
;        Outputs: NONE
;       Keywords: NONE 
;
;     delete (PROCEDURE) - Delete data from the silo
;         Inputs: KEY : STRING name denoting the data
;        Outputs: NONE
;       Keywords: NONE 
;
; EXAMPLE:
;
;     ; Create an instance of a data silo
;     ;
;     silo = OBJ_NEW ('silo')
;
;     ; Add some data to the silo.  The data names are not case-sensitive.
;     ;
;     silo->set, 'intData', 1
;     silo->set, 'floatData', 1.0
;     silo->set, 'structData', { DATA, x: 0, y: PTR_NEW (), z: OBJ_NEW () }
;
;     ; Print the current entries
;     ;
;     silo->print
;
;     ; Retrieve some data from the silo
;     ;
;     myInt = silo->get ('intData')
;
;     ; Retrieve some data from the silo, and check for error
;     ;
;     myStruct = silo->get ('structData', ERROR = error)
;     IF (error) THEN $
;        PRINT, 'Could not find data in silo: structData'
;
;     ; Delete an entry from the silo
;     ;
;     silo->delete, 'floatData'
;
;     ; All done
;     ;
;     OBJ_DESTROY, silo
;
; MODIFICATION HISTORY:
;
;     Modified by Roger J. Dejus (dejus@aps.anl.gov 01/28/99.
;     Added SILENT keyword to the GET method to return silently (with a NULL
;     string) when the KEY is not found. Added the "Message, /Reset" statement
;     to the LOOKUPKEY method to avoid setting the !error_state system variable.
;     See occurrences "RJD" below.
;     1.02: RSM, fixed NULL pointer references in cleanup and print methods for
;           the case where no data were in the silo.  Fixed heap memory
;           leak.  Thanks to Roger Dejus.
;           
;           Added ERROR keyword to the GET method (see EXAMPLE).
;
;     1.01: RSM, added NUMBER, NAMES keywords to get() method to return number
;           or string names of data elements currently in the silo.
;
;     1.0 : Written, 1998 July, Robert.Mallozzi@msfc.nasa.gov
;
;-
; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 


;------------------------------------------------------------------------------
; Initialize the data silo object
;------------------------------------------------------------------------------

FUNCTION silo::init

    RETURN, 1

END


;------------------------------------------------------------------------------
; Destroy the data silo object
;------------------------------------------------------------------------------

PRO silo::cleanup


    numData = (PTR_VALID (self.data) ? N_ELEMENTS (*self.data) : 0)
    FOR i = 0L, numData - 1 DO BEGIN

        PTR_FREE, (*(*self.data)[i]).data
        PTR_FREE, (*self.data)[i]

    ENDFOR

    PTR_FREE, self.data

END



;------------------------------------------------------------------------------
; Set a silo data element.  userData can be data of any IDL type.
;------------------------------------------------------------------------------

PRO silo::set, key, userData


    ; Ensure request is a string variable
    ;
    IF (STRUPCASE (SIZE (key, /TNAME)) NE 'STRING') THEN BEGIN
       MESSAGE, /CONT, 'Argument must be of type STRING: ' + STRTRIM (key, 2)
       RETURN
    ENDIF

    str = { USER_DATA, $
        key  : key, $
        data : PTR_NEW (userData) $
    }


    IF (NOT PTR_VALID (self.data)) THEN BEGIN

       self.data = PTR_NEW (PTR_NEW (str))

    ENDIF ELSE BEGIN

       index = self->lookupKey (key)

       IF (index EQ -1) THEN BEGIN
          
	  *self.data = [*self.data, PTR_NEW (str)]

       ENDIF ELSE BEGIN
       
          PTR_FREE, (*(*self.data)[index]).data
          PTR_FREE, (*self.data)[index]
	  (*self.data)[index] = PTR_NEW (str)

       ENDELSE

    ENDELSE

END



;------------------------------------------------------------------------------
; Given a key, find the index into the self.data structure.
; This could be a fairly expensive operation, since it REPEATs until
; the key is found, or until all keys are checked (a loop is
; required, since only scalar pointer dereferences are allowed).
;
; TODO: Implement a hash table?
;------------------------------------------------------------------------------

FUNCTION silo::lookupKey, key


    CATCH, keyNotFound
    IF (keyNotFound NE 0) THEN BEGIN
       CATCH, /CANCEL
    ; Modified RJD, 01/28/99
       MESSAGE, /RESET
       RETURN, -1
    ENDIF

    key = STRUPCASE (key)

    i = -1L
    REPEAT BEGIN

        i = i + 1L
        testKey = STRUPCASE ((*(*self.data)[i]).key)

    ENDREP UNTIL (testKey EQ key)

    CATCH, /CANCEL
    RETURN, i


END



;------------------------------------------------------------------------------
; Print names of all currently stored data.  If silo is empty, print nothing.
;------------------------------------------------------------------------------

PRO silo::print

    
    numData = (PTR_VALID (self.data) ? N_ELEMENTS (*self.data) : 0)

    FOR i = 0L, numData - 1 DO $
        PRINT, (*(*self.data)[i]).key
    
END



;------------------------------------------------------------------------------
; Retrieve a silo data element.  Sets ERROR to 1 if data not found in silo.
;------------------------------------------------------------------------------

; Modified RJD, 01/28/99
FUNCTION silo::get, key, NUMBER = number, NAMES = names, ERROR = error, SILENT = silent

    
    ERROR  = 0
     
    IF (KEYWORD_SET (number) AND $
        KEYWORD_SET (names)) THEN BEGIN
       MESSAGE, /CONTINUE, 'Conflicting keywords: NUMBER, NAMES'
       ERROR = 1
       RETURN, 0
    ENDIF   

    ; Return the number of data elements currently in the silo
    ;
    IF (KEYWORD_SET (number)) THEN $
       RETURN, (PTR_VALID (self.data) ? N_ELEMENTS (*self.data) : 0)

    ; Return STRARR of the names of data elements currently in the silo
    ;
    IF (KEYWORD_SET (names)) THEN BEGIN

       s = ''
       IF (NOT PTR_VALID (self.data)) THEN $
          RETURN, s
       FOR i = 0, N_ELEMENTS (*self.data) - 1 DO $
           s = [s, (*(*self.data)[i]).key]   
       s = s[1:*]
       RETURN, s

    ENDIF
    
       
    ; Ensure request is a string variable
    ;
    IF (STRUPCASE (SIZE (key, /TNAME)) NE 'STRING') THEN BEGIN
       MESSAGE, /CONT, 'Argument must be of type STRING: ' + STRTRIM (key, 2)
       ERROR = 1
       RETURN, 0
    ENDIF

    index = self->lookupKey (key)

    IF (index NE -1) THEN $
       RETURN, *(*(*self.data)[index]).data

    ; Modified RJD, 01/28/99
    ERROR = 1
    IF (KEYWORD_SET (silent)) THEN RETURN, ''
    MESSAGE, /CONTINUE, 'Unknown data: ' + STRTRIM (key, 2)
    RETURN, 0

END



;------------------------------------------------------------------------------
; Delete a silo data element.  Returns silently if data not found in silo.
;------------------------------------------------------------------------------

PRO silo::delete, key


    ; Ensure request is a string variable
    ;
    IF (STRUPCASE (SIZE (key, /TNAME)) NE 'STRING') THEN BEGIN
       MESSAGE, /CONT, 'Argument must be of type STRING: ' + STRTRIM (key, 2)
       RETURN
    ENDIF

    numData = (PTR_VALID (self.data) ? N_ELEMENTS (*self.data) : 0)

    index = self->lookupKey (key)

    IF (index NE -1) THEN BEGIN

       PTR_FREE, (*(*self.data)[index]).data
       PTR_FREE, (*self.data)[index]

       IF (numData EQ 1) THEN BEGIN

          PTR_FREE, *self.data 
          PTR_FREE, self.data 
          self.data = PTR_NEW ()

       ENDIF ELSE BEGIN
       
          lookup = INTARR (numData) + 1
          lookup[index] = 0
           
          *self.data = (*self.data)[WHERE (lookup)]

       ENDELSE      

    ENDIF

END



;------------------------------------------------------------------------------
; Object that stores arbitrary data
;------------------------------------------------------------------------------

PRO silo__define

    obj = { SILO, $

        data : PTR_NEW () $

    }


END
