;+
;	NAME:
;		XPOWELLFIT
;	PURPOSE:
;		Widget interface to powellfit procedure.
;	CALLING SEQUENCE:
;		xpowellfit,x,y  or xpowellfit,data
;	OPTIONAL INPUT PARAMETERS:
;	KEYWORD PARAMETERS:
;		GROUP = The widget ID of the widget that calls XPowellFit.
;		When this ID is specified, a death of the caller results
;		in a death of PowellFit.
;	OUTPUTS:
;		Open a widget to input data and runs PowellFit
;	COMMON BLOCKS:
;               COMMON powellfit,equation,x,y, sigma, xrange, chi2fit, fpar
;		(set PowellFit help for a description)
;	SIDE EFFECTS:
;		If not active, starts Xmanager
;	RESTRICTIONS:
;		The limitations of nr_powell.
;	PROCEDURE:
;		Use XScrMenu to get the parameters and then calls PowellFit
;	MODIFICATION HISTORY:
;       	by  Manuel Sanchez del Rio. ESRF. 95-02-13
;		95-03-16 MSR adds statistic weight, covariance, external
;			input of sigma...
;		97-01-10 MSR puts fpar in a new common block xpow
;			(it conflicted with the common b def in powellfit)
;		97-09-30 srio@esrf.fr READ_ASCII renamed to RASCII
;-
PRO xpowellfit, set, optinp, group = group
common powellfit, equation, x, y, sigma, xrange, chi2fit
common xpow, fpar

on_error,2
if n_params() EQ 0 then begin
  print,'XPOWELLFIT: Usage: xpowellfit, set, keyword_pars'
  return
endif

text = ['This option permits to fit the current set of data to ',$
   'a user-defined function, by using a non-linear fit.', $
   '   ', $
   'For the non-linear fit it uses the the NR_POWELL idl function to find',$
   'the minimum f the function ChiSquare= Sum{ ([y-f(x;a)]/sigma)^2 }.', $
   '   ', $
   'The user must define the fitting function in IDL syntax as an ',$
   'expression of the independent variable x and a parameter array a.',$
   'For example a straight line is : a(0)*x+a(1)', $
   '              gaussian is a(0)*exp( (x-a(1))^2/a(2)) etc etc.', $
   'For fitting with complicated functions, the user may write an',$
   'IDL customized function that has to be typed in the Equation',$
   'entry. For example, write a file with:', $
   '       FUNCTION MYFITTEST,x,a', $
   '       return,a(0)*exp( (x-a(1))^2/a(2))', $
   '       end', $
   " and type in the 'Equation: y=f(x;a())' entry: myfittest(x,a)",$
   '   ',$
   'The initial parameter guess must be provided in array notation:',$
   '[a(0),a(1),a(2),etc], and it also needs the x range defining the',$
   'points to be used for the fit. By setting xmin=xmax, the program ',$
   'uses all the x interval for the fit.',$
   '   ',$
   'The sigma values can be the unit for all points, or can be read',$
   'from an external file, or can be set to statistical noise: ',$
   'sigma = sqrt(yfit).', $
   '   ',$
   'In addition, the user may select the option to calculate covariance ',$
   'and error estimation on the resulting parameters of the fit.']
;
tmp = widget_message(text,/INFO,/CANCEL)
if tmp EQ 'Cancel' then return

if  n_params() EQ 2 then begin
  x = reform(set)
  y = reform(optinp)
endif else begin
  x = set(0,*)
  y = set(1,*)
endelse
if not(keyword_set(xrange)) then xrange=[0.,0.]

;
; XscrMenu part
;
if not(keyword_set(fpar)) then fpar = { xpowellfit, $
           equ:'a(0)*exp(-(x-a(1))^2/a(2))', $
           guess:'[100.,20.,1.]',            $
           xmin:xrange(0), $
           xmax:xrange(1), $
           fileflag:['0','No','Yes'],        $
           file:'xpowellfit.dat' ,	     $
           sigmaflag:['0','1','Statistic','From file'],        $
           sigmafile:'sigma.dat', $
	   covar:['0','No','Yes']  }

if keyword_set(equation) then fpar.equ=equation

flags = ['1','1','1','1','1','w(4) EQ 1','1','w(6) eq 2','1']
titles= ['Equation: y=f(x;a())','Initial parameter guess:','xmin:',$
         'xmax (=xmin for all interval)','Output to file?', $
         'Output file name:','Sigma values:','Sigma file', $
	 'Covariance calculation:' ]
Xscrmenu, fpar, Action=Action, group=group, /notype, /interp, $
    flags=flags,titles=titles, wtitle='Powell fit parms' ;,nrow=2
if ACTION NE 'DO' then return

widget_control,/hourglass
;
; prepare PowellFit call
;
error = not( execute('a='+fpar.guess) )
if error then begin
  print,'XPOWELLFIT: Error defining initial parameters.'
  ;Xtext, text=['','  Powell fit FAILED.  ', '  Returning.'], $
  ;  size=[25,4], GROUP=group
  tmp  = widget_message('  Powell fit FAILED.  ',/ERROR)
  return
endif
;define equation
equation = fpar.equ
xrange=[fpar.xmin,fpar.xmax]

if strcompress(fpar.sigmaflag(0),/rem) EQ '0' then sigma=0
if strcompress(fpar.sigmaflag(0),/rem) EQ '1' then statsigma=1 else statsigma=0

if strcompress(fpar.sigmaflag(0),/rem) EQ '2' then begin
  print,'XPOWELLFIT: Reading sigma from file: ',fpar.sigmafile
  ss = rascii(fpar.sigmafile)
  if not(keyword_set(ss)) then begin
    ;Xtext, text=['File reading error', '  Returning.'], $
    ;size=[25,4], GROUP=group
    tmp =  widget_message('File reading error',/ERROR)
    Return
  endif
  if n_elements(ss(0,*)) NE n_elements(x) then begin
    ;Xtext, text=['Sigma File dimensions',' not valid', '  Returning.'], $
    ;size=[25,4], GROUP=group
    tmp  = widget_message('Sigma File dimensions not valid',/ERROR)
    Return
  endif
  nncol = n_elements(ss(*,0))-1
  print,'XPOWELLFIT: Reading sigma from last column of the file: ',nncol+1
  sigma=ss(nncol,*)  ;last column
endif

print,'XPOWELLFIT: calling powellfit.'
powellfit,a,yfit=y2,error=error,statsigma=statsigma ;,/show
print,'XPOWELLFIT: back from powellfit.'
;
; sort the outputs
;
if error then begin
  print,'XPOWELLFIT: Error in the fitting. Returning.'
  ;Xtext, text=['','  Powell fit FAILED.  ', '  Returning.'], $
  ;  size=[25,4], GROUP=group
  tmp = widget_message(' Powell fit FAILED. ',/ERROR)
  Return
endif

print,'XPOWELLFIT: fitting parms: '+vect2string(a)
fpar.guess=vect2string(a)

if strcompress(fpar.covar(0),/rem) EQ '1' then begin
  covar_fit,x,y,a,equation,sigma=sigma,statsigma=statsigma,$
	covar=covar,outsigma=outsigma
endif

out1 = fltarr(2,n_elements(x))
out2 = fltarr(2,n_elements(x))
out1(0,*) = x
out2(0,*) = x
out1(1,*) = y
out2(1,*) = y2


; define handles to plot and overplot using 'xplot'
if keyword_set(hh) then handle_free,hh
hh = handle_create()
handle = handle_create(hh,value=out2,/FIRST_CHILD)
handle = handle_create(handle,value=out1,/SIBLING)
;plotfile,hh
print,'XPOWELLFIT: Calling xplot with handle: ',hh
xplot,hh,group=group,wtitle='Powell fit results', $
  title='PowellFit: y='+fpar.equ+'   a='+vect2string(a)+ $
  ' !4v!X!S!E2!R/dof='+strcompress(chi2fit,/rem)


if strcompress(fpar.covar(0),/rem) EQ '1' then begin
  nco = n_elements(covar(0,*))
  atext = strarr(nco)
  for i=0,nco-1 do atext(i)=vect2string(covar(*,i))
  text = [ ' Powell fit results:', $
  ' Equation: y = '+fpar.equ, $
  ' Fit parameters: a = '+vect2string(a), $
  ' xrange = '+vect2string(xrange), $
  ' ChiSquare = '+strcompress(chi2fit,/rem), $
  ' ',$
  'Sigma of the Fit parameters: '+vect2string(outsigma), $
  'covariance matrix:', atext]
  ;xdisplayfile1,'tmp.covar_fit',title='Fit covariance results',/remove
  ShowInfo,TITLE='Xplot Non linear Fit Results',GROUP=group, $
    INFOTEXT=text, WIDTH=max(strlen(text)), HEIGH=n_elements(text)
endif


if (strcompress(fpar.fileflag(0),/rem) EQ '1') then begin
  openw,Unit,fpar.file,/get_lun
  printf,Unit,'; '
  printf,Unit,'; Powell fit results:'
  printf,Unit,'; y = '+fpar.equ
  printf,Unit,'; a = '+vect2string(a)
  printf,Unit,'; xrange = '+vect2string(xrange)
  printf,Unit,'; ChiSquare = '+strcompress(chi2fit)
  printf,Unit,'; '
  printf,Unit,'; next the x, y, yfit values: '
  for i=0l,n_elements(x)-1 do begin
    printf,Unit,out1(0,i),out1(1,i),out2(1,i)
  endfor
  free_lun,Unit
  ;xtext,text=['','  File    '+fpar.file,$
  ;  ' written to current directory '],size=[25,4], GROUP=group
  tmp = widget_message('  File    '+fpar.file+' written to disk',/INFO)
endif


end
