;+
;
;                         ===============================
;                         = Functional Fitting Facility =
;                         ===============================
;
; FuFiFa 1.0Beta 1  December 2001
;
;
; Fufifa is an application to fit interactively a SEVERAL spectra to a FUNCTION.
;
; The goal is to provide with a point-and-click application for fitting many scans
; (usually from a SPEC file) with a user-defined function. The function can be built
; interactively using a set of components (straight line, polynomial, peaks
; (gaussian, lotentzian and pseudovoigt), steps (error function and arc tangent) or
; any user-define function). The application allows to fit a bunch of scans and to
; refine individually the individual fits. At the end, a plot of the fitted
; parameters and chi square versus the scan index can help to estimate the quality
; of the fits. 
;
; To operate with fufifa follow the following steps:
;
; 1) Start the application by either: 
;
; i) Start xop and from the Xop->Tools->FuFiFa, This will create the fufifa window.
; ii) Enter "xop fufifa" at the unix prompt
;
; 2) Import your data.
;   Press the load button in the data sector (top right of the main window).
;   You should select a file (SPEC type) which has been previously "cleaned".
;   "Cleaned SPEC file" means that:
;     i) you have removed the unwanted scans (only kept those used for fitting)
;     ii) Each scans has ONLY two or three columns
;         The first one is the abscissas X
;         The second one is the ordinates Y
;         The third (optional) one contains the errors of Y. The errors used for
;            the fit are defined in the Fit->Preferences menu item. 
;    You can use the EXODUS application (available in the XOP/XAID - XAFS toolbox)
;         for this purpose (i.e., cleaning your SPEC file).
;
;    Example: Using the file /scisoft/users/srio/Working/IDL/fufifa/cat3.spec I
;             extracted (using EXODUS) the scans [82,83,92,93,111,112 and 113] I used 
;             column 1 for X and column 16 for Y. The result was written to file 
;             exodus.spec. This file was imported into fififa. Then I selected all
;             scans (by pressing ctrl-click iteratively or doing multiple selection 
;             with shift-click). The result of these operations are in Fig1.
;
; 3) Define your model:
;    Using the bottom sector of your fufifa window, I defined a function consisting 
;    in a step function (errorF) and a lorentzian peak. For this I clicked "Add
;    function" twice, one for each functional component. I did not adjust the input 
;    parameters. 
;    First I select my first scan (clicking on the list at the top right sector).
;    Then I click in one of the small "G" buttons of the first line of the function. 
;    That allows me to define, by clicking three points the initial guess parameters 
;    for this function. I do the same thing for the second line, in order to define 
;    my lorentzian peak. The result of this selection is in Fig2. You see your data
;    in red, your model in green, your model components in blue and the residuals
;    in cyan. The "F" buttons is used to fix (freeze) the respective parameter.
;
; 4) First attempt to fit.
;
;    If you press "**Fit**" you fit the function parameters to data. Then you can 
;    press "propagate model" that copies the fitted values for the first scan to
;    the rest of scans. 
;
;    If you want to fit all spectra, just make a multiple selection and press the
;    fit button. Then you can select individual scans to see better the results.
;    Perhaps some scans are not well fitted. You can select by hand the guess
;    parameters and fit again. 
;    With these operations I obtained the plot in Fig3. 
;    Not very good, I admit.
;
; 5) Refinement of the fit. 
;    You can preprocess the data using the "Data" menu items to:
;      i) perform some operation (like normalization). Try it!
;      ii) Cut unwanted data and redefine your interval. Use it to remove 
;          points of the pre-edge and the incomplete last oscillation. 
;      iii) Substract a linear background (not needed in this case). 
;
;      After doing that, I added two more lorentzian peaks. I got results shown
;      in Fig4. Much better!
;
; 6) Visualizing and saving results:
;     By pressing "Pars vs Scans" you can mahe plots of the parameters and the reduced 
;     Chi square versus the scan number. This permits you to see if there are drastic 
;     changes in some parameters.  In our case I see that scan 3 is the worst fitted. 
;
;     The function used can be saved to a file using Model-Save to a file. This 
;     file can be edited and reloaded. This option saves only the definition of the
;     function and one parameter sets. It does not save the parameters fitted for all
;     scans.
;     In order to save the full information, you can use FuFiFa->Save as... It will
;     create a SPEC-type file with all scans, error used, and models. In the scan
;     header you can find the information on the fitted parameters and chi**2. The
;     last scan (numbered 1000) contains the varuiation of the parameters versus
;     the scan index. Of course, you can use Xplot to view this resulting file. 
;
;
;
;
; This is a beta version. Please send bugs and ideas to improve it to srio@esrf.fr
;
;
;
;
;
; TO DO
; =====
;
; - Allow MCI inputs
; - Load components of the function from an external (customizable) file
; - Access to boundaries of parameters for the fit?
; - Abort button (when fitting...)
; - Customize colors
; - Polinomial inputs from mouse
; - "Expert" mode (do not show information windows)
; - Integrate in xplot and in xop 
; - Interactive help
;
;
;
;    
; srio@esrf.fr 7 December 2001
; Modified:
;     2003/12/09 srio@esrf.fr Fix an incompatibility problem for IDL >=5.6
;
;-

;
; FuFIFA  M. Sanchez del Rio. 7 December 2001
;

;
;
;========================================================================
;

FUNCTION FuFiFa_version
RETURN,'1.0Beta1.0.1'
END ; FuFiFa_version

;
;========================================================================
;

;
;========================================================================
;

FUNCTION FuFiFa_GetModelParsVariation,tlb,error=error

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_GetModelParsVariation: error caught: '+!error_state.msg)
  Catch, /Cancel
  error=1
  RETURN,0
ENDIF

error=0
Widget_Control,tlb, Get_UValue=state

ndata = state.oData->info(/N_Elements)
oData = state.oData

IF ndata EQ 1 and state.oData->value(0,/IsDataDef) NE 1 THEN BEGIN
  Message,/Info,'No data'
  error=1
  RETURN,0
ENDIF

FOR i=0,nData-1 DO BEGIN
  IF state.oData->value(i,/IsAuxDef) EQ 1 THEN BEGIN
    tmp = state.oData->value(i,/Aux)
    npars = N_Elements(tmp)
    IF i EQ 0 THEN BEGIN
      set = DblArr(npars+2,nData)
    ENDIF 
    set[0,i]=i+1
    set[1:npars,i]=tmp
    FuFiFa_chisquare,state,i,chi2=chi2
    set[npars+1,i]=chi2
  ENDIF ELSE BEGIN
    error=1
    Message,'Some data sets have not assigned parameters'
    ;itmp = Dialog_message(/Error,Dialog_Parent=event.top, $
    ;['Some data sets have not assigned parameters',$
    ;'Apply fit to them'])
    RETURN,0
  ENDELSE
ENDFOR

RETURN,set
END ; FuFiFa_GetModelParsVariation
;
;========================================================================
;

FUNCTION FuFiFa_GetErrorArray, data, mpfitpars, errTxt, Group=group

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_GetErrorArray: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN,0
ENDIF
errorsDefined = 0

IF StrCompress( (mpfitpars.errorsFrom)[0], /Rem) EQ '0' THEN BEGIN
  errtxt = 'Requested using errors from third column'
  Message,/Info,errtxt
  IF (size(data))[1] EQ 3 THEN BEGIN
    errorsDefined =1 
    dataErr = data[2,*]
    errtxt = 'Using errors from third column'
    Message,/Info,errtxt
  ENDIF ELSE BEGIN
    errtxt = 'Undefined third column'
    Message,/Info,errtxt
  ENDELSE
ENDIF


IF errorsDefined EQ 0 THEN BEGIN
  CASE StrCompress(mpfitpars.errors[0],/Remove_All) OF
    '0': BEGIN
        errtxt = 'Using constant errors'
        Message,/Info,errtxt
        dataErr=Reform(data[1,*])*0+1
        END
    '1': BEGIN
        errtxt = 'Using statistical errors'
        Message,/Info,errtxt
        dataErr = Sqrt(Reform(data[1,*]))  ; 1*sigma = sqrt(y)
        END
    else:
  ENDCASE
ENDIF

RETURN,dataErr
END ; FuFiFa_GetErrorArray
;
;============================================================================
;
PRO FuFiFa_chisquare, state, indexLine, Group=group, Ask=ask, chi2=chi2


Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_chisquare: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

;
; get data
;
data = state.oData->value(indexLine)
dataX = Reform(data[0,*])
dataY = Reform(data[1,*])

;
; get model expression
;
oeq = state.oequation
model = oeq->getEquation()
IF model EQ '' THEN BEGIN
  itmp = Dialog_message(/Error,Dialog_parent=group, $
	'No model to fit.')
  RETURN
ENDIF
;  print,'>>> Model equation is: '+model

IF  state.oData->value(indexLine,/IsAuxDef) EQ 1 THEN BEGIN
  aux = state.oData->value(indexLine,/Aux)
  Widget_Control,state.wids.dataModel,Set_Value='Model: defined.'
  oeq->SetProperty,Param=aux   ; ,/UpdateWidgets
ENDIF ELSE BEGIN
  itmp = Dialog_message(/Error,Dialog_Parent=group,$
	['No parameters assigned to data set with index: '+$
	StrCompress(indexLine),'Apply fit to it.'])
  RETURN
ENDELSE



IF N_Elements(datax) EQ 0 THEN BEGIN
 itmp = Dialog_Message(/Error,'FuFiFa_plot: abscissas not defined')
 RETURN
ENDIF

;plot,datax,data[1,*],YRange=yRange,/NoData

oeq = state.oequation
nlines = oeq->GetProperty(/Nlines)

IF nlines EQ 0 THEN BEGIN
  itmp = Dialog_message(/Error,Dialog_Parent=group,$
    'No model defined')
  RETURN
ENDIF

dataFit=oeq->evaluate(datax)
dataErr = FuFiFa_GetErrorArray(data,state.MPFitPars)

deviates = (dataY-dataFit)/dataErr
nn = N_Elements(dataX)
chi2 = Total(deviates^2)/nn/(nn-1)

END ; FuFiFa_chisquare

;
;========================================================================
;

PRO fufifa_addFunction,wid, str=tmp, param=param1

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_addFunction: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF
        Widget_Control,wid, Get_UValue=state

	oeq = state.oequation

IF Not(Keyword_Set(tmp)) THEN BEGIN
	list = ['0',oeq->getProperty(/ListEqua)]
	tmp = {list:list, degree:1, center:10.0, fwhm:11.0, jump:12.0, $
		equation:'x*p[!]+p[!]',pars:'[1.0,1.0]' }
	flags=['1','w(0) EQ 1', $
		'w(0) GE 2 AND w(0) LE 6', $
		'w(0) GE 2 AND w(0) LE 6', $
		'w(0) GE 2 AND w(0) LE 6', $
		'w(0) EQ 7','w(0) EQ 7']
	titles=['Function','Degree','Center','FWHM','Height or Jump', $
		'Function f(x,p[!],...p[!])','Guess pars']

        XScrmenu,tmp,/Interp,/NoType,Action=action,Dialog_Parent=wid, $
          Flags=flags, Titles=titles, WTitle='Function definition', $
	  FieldLen=30
        IF action EQ 'DONT' THEN RETURN

	;  print,oeq->info()
ENDIF ELSE BEGIN
	;tmp = {list:list, degree:1, center:10.0, fwhm:11.0, jump:12.0, $
	;	equation:'x*p[!]+p[!]',pars:'[1.0,1.0]' }
ENDELSE

	index = Fix( (tmp.list)[0] )
	name = (tmp.list)[1+Fix((tmp.list)[0])]
print,'*****************'+name+'****************'
	CASE StrCompress(StrUpCase(name),/Rem) OF 
	  'STRAIGHTLINE': BEGIN	; StraightLine
		oeq->AddLine,/straight
		isApeak = 0
		END
	  'POLYNOMIAL': BEGIN	; Polynomial
		oeq->AddLine,Poly=tmp.degree
		isApeak = 0
		END
	  'GAUSSIAN': BEGIN	; Gaussian
		oeq->AddLine,/Gauss,Para=[tmp.jump,tmp.center,tmp.fwhm]
		isApeak = 1
		END
	  'LORENTZIAN': BEGIN	; Lorentzian
		oeq->AddLine,/Lorentzian,Para=[tmp.jump,tmp.center,tmp.fwhm]
		isApeak = 1
		END
	  'PSEUDOVOIGT': BEGIN	; PseudoVoigt
		oeq->AddLine,/PseudoVoigt, $
			Para=[tmp.jump,tmp.center,tmp.fwhm,0.5]
		isApeak = 1
		END
	  'STEPERRORF': BEGIN	; StepErrorF
		oeq->AddLine,/StepErrorF,Para=[tmp.jump,tmp.center,tmp.fwhm]
		isApeak = 0
		END
	  'STEPATAN': BEGIN	; StepAtan
		oeq->AddLine,/StepAtan,Para=[tmp.jump,tmp.center,tmp.fwhm]
		isApeak = 0
		END
	  'USERDEFINED': BEGIN	; UserDefined
		para = 0.0
		command = 'para = '+tmp.pars
		itmp = Execute(command)
		oeq->AddLine,tmp.equation,Para=para
		isApeak = 0
		END
	  else: Message,'Function not implemented: '+name
	ENDCASE

	IF Keyword_Set(param1) THEN BEGIN
	  oeq->SetPropertyLine,param=param1
	ENDIF
	;
	; create and store widget information
	;
	nlines=oeq->GetProperty(/NLines)
	line = oeq->GetPropertyLine(nlines-1,/equation)
	values = oeq->GetPropertyLine(nlines-1,/parLine)
	flags=Fix(values*0+1)
	;wtmpbase = Widget_Base(state.wids.ModelBase)
	value={coefs:values, flags:flags }
	wtmp =CW_PField(state.wids.ModelBase, $
		UValue=nlines-1, $	; line number info in UValue
		Value=value,Title=name)
	Widget_Control,wtmp,/Realize
	oeq->SetPropertyLine,wId=wtmp

	; 
	; set boundaries
	;
	IF StrUpCase(StrCompress(name,/Rem)) EQ 'PSEUDOVOIGT' THEN BEGIN
	  itmp = Dialog_message(/Question, Dialog_Parent=wid, $
		['Setting boundaries:',$
		'Do you want to limit the Pseudovoigt weigth',$
		'between 0 (pure gaussian) and 1 (pure lorentzian)? '])
	  IF itmp EQ 'Yes' THEN $
	  oeq->setPropertyItem,nlines-1,3,Limited=[1,1],Limits=[0.0,1.0]
	ENDIF

	IF isApeak EQ 1 THEN BEGIN
	  itmp = Dialog_message(/Question, Dialog_Parent=wid, $
		['Setting boundaries:',$
		'Do you want to limit the peak amplitude',$
		'to positive values?'])
	  IF itmp EQ 'Yes' THEN $
	  oeq->setPropertyItem,nlines-1,0,Limited=[1,0],Limits=[0.0,0.0]
	ENDIF



	;
	; update data auxiliar
	;
	sel = Widget_Info(state.wids.dataList,/List_Select)
	IF sel[0] NE -1 THEN BEGIN
	nn =  state.oData->info(/N_Elements)
	IF nn EQ 0 THEN Message,'No data input.'
	; clear all stored data
	FOR i=0L,nn-1 DO BEGIN
	  state.oData->Set,i,/ClAux
	ENDFOR
	goodIndices = IndGen(nn)
	goodIndices = goodIndices(sel)
	all_pars=oeq->GetProperty()
	; set new auxiliary data
	FOR i=0L,N_Elements(goodIndices)-1 DO BEGIN
	  state.oData->Set,goodIndices[i],Aux=all_pars
	ENDFOR
	ENDIF
	; 
	; Plot
	;
	FuFiFa_Plot,wid

	END
;
;========================================================================
;
PRO FuFiFa_quit,event


Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_quit: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

Widget_Control,event.top, Get_UValue=state

Obj_Destroy,state.oData
Obj_Destroy,state.oEquation

Widget_Control,event.top,/Destroy
END ; FuFiFa_quit

;
;========================================================================
;
PRO FuFiFa_plot,tlb,index,Group=group,Print=print


Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_plot: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

Widget_Control,tlb,Get_UValue=state

IF N_Elements(index) EQ 0 THEN BEGIN
index=Widget_Info(state.wids.dataList,/List_Select)
  IF index[0] EQ -1 THEN BEGIN
;    itmp = Dialog_message(/Error,Dialog_parent=group, $
;	'No data to plot. Please Load/Select data')
    RETURN
  ENDIF
ENDIF

IF NOT(Keyword_Set(print)) THEN BEGIN
  ;Widget_Control,event.top, Get_UValue=state
  widget_control,state.wids.draw, get_val = wnum
  wset,wnum
ENDIF

flag_which=state.plotDropList
oeq = state.oEquation

;
; get data
;

FOR i=0L,N_Elements(index)-1 DO BEGIN
  data = state.oData->value(index[i])
  datax = Reform(data[0,*])

  IF  state.oData->value(index[i],/IsAuxDef) EQ 1 THEN BEGIN
    aux = state.oData->value(index[i],/Aux)
    IF NOT(Keyword_Set(print)) THEN $
      Widget_Control,state.wids.dataModel,Set_Value='Model: defined.'

    oeq->SetProperty,Param=aux,/UpdateWidgets
    ; ?????   state.oData->set,i,Aux=aux
  ENDIF ELSE BEGIN
    IF NOT(Keyword_Set(print)) THEN $
      Widget_Control,state.wids.dataModel,Set_Value=' Model: undefined. '
  ENDELSE

  IF N_Elements(datax) EQ 0 THEN BEGIN
   itmp = Dialog_Message(/Error,'FuFiFa_plot: abscissas not defined')
   RETURN
  ENDIF

  ymin = min(data[1,*],max=ymax)
  yint = Abs(ymax-ymin)
  yRange=[ymin-0.2*yint,ymax+0.1*yint]
  IF i EQ 0 THEN plot,datax,data[1,*],YRange=yRange,/NoData

  ;
  ; show data
  ;
  IF flag_which[0] EQ 1 THEN BEGIN
     oplot,datax,data[1,*] ,Color=2
  ENDIF

  ;
  ; show errors
  ;
  IF flag_which[3] EQ 1 THEN BEGIN
     dataErr = fufifa_GetErrorArray(data,state.mpfitpars,Group=group)
     plotErr,datax,data[1,*]-dataErr,data[1,*]+dataErr
  ENDIF

  oeq = state.oequation
  nlines = oeq->GetProperty(/Nlines)

  IF nlines NE 0 THEN BEGIN
    ; 
    ; show individual models
    ; 
    IF flag_which[2] EQ 1 THEN BEGIN
      FOR j=0L,nlines-1 DO BEGIN
        oplot,datax,oeq->evaluate(datax,Line=j),Color=4
      ENDFOR
    ENDIF
    ; 
    ; show model
    ; 
    IF flag_which[1] EQ 1 THEN BEGIN
      oplot,datax,oeq->evaluate(datax),Color=3
    ENDIF
    ; 
    ; show residuals
    ; 
    IF flag_which[4] EQ 1 THEN BEGIN
      oplot,datax,data[1,*]-oeq->evaluate(datax),Color=5
    ENDIF
  ENDIF
ENDFOR


END ; FuFiFa_plot


;
;============================================================================
;
FUNCTION fufifa_plotDropList,values


Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_plotDropList: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN,0
ENDIF

  tmp = ['Show/Hide:','Data ','Model ','Components ','Error bars ','Residuals ']

FOR i=0,N_Elements(values) -1 DO BEGIN
  IF values[i] EQ 1 THEN tmp[i+1] = tmp[i+1]+'(shown)' ELSE $
	tmp[i+1] = tmp[i+1]+'(hidden)' 
ENDFOR

RETURN,tmp
END




;
;============================================================================
;

PRO FuFiFa_MPFit_Display, fcn, x, iter, FunctArgs=fcnargs, FMT=fmt, $
         Quiet=quiet, _Extra=iterargs

Forward_Function mpfit_enorm
; this routine is called by MPFit

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_MPFit_Display: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

IF Keyword_Set(quiet) THEN RETURN
fvec = Call_Function(fcn, x, _Extra=fcnargs)
fnorm = mpfit_enorm(fvec)

XDisplayfile1_Append, [String(iter, fnorm^2, $
  Format='("Iter ",I6,"   CHI-SQUARE = ",G20.8)'), $
  '           Timing [sec]:  '+$
	String(SysTime(1)-iterargs.t0,Format='(G6.2)')]
IF N_Elements(fmt) GT 0 THEN BEGIN
    XDisplayfile1_Append, String(p, format=fmt)
ENDIF ELSE BEGIN
    p = '  P('+StrTrim(LIndGen(N_Elements(x)),2)+') = ' $
      + StrTrim(String(x,Format='(G20.6)'),2) + '  '
    XDisplayfile1_Append, String('         '+p, Format='(A)')
ENDELSE

IF (SysTime(1)-iterargs.t0) GE iterargs.tmax THEN BEGIN
;       itmp = Widget_Message(Dialog_Parent=iterargs.parent, $
;	['Maximum time ('+StrCompress(Long(iterargs.tmax))+' sec) reached.',$
;	 'Fitting process aborted.'])
  !err=-1
ENDIF

RETURN
END ; FuFiFa_MPFit_Display



;
;============================================================================
;
PRO FuFiFa_fit, state, indexLine, Group=group, Ask=ask


Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_fit: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF
;
; get data
;
data = state.oData->value(indexLine)
dataX = Reform(data[0,*])
dataY = Reform(data[1,*])

;
; get model expression
;
oeq = state.oequation
model = oeq->getEquation()
IF model EQ '' THEN BEGIN
  itmp = Dialog_message(/Error,Dialog_parent=group, $
	'No model to fit.')
  RETURN
ENDIF
;  print,'>>> Model equation is: '+model

;
; get parameters structure
;
guess = oeq->getProperty(/str)
print,'guess: ',guess

;
; Overwrite initial parameteres
;


IF  state.oData->value(indexLine,/IsAuxDef) EQ 1 THEN BEGIN
  aux = state.oData->value(indexLine,/Aux)
  Widget_Control,state.wids.dataModel,Set_Value='Model: defined.'
  oeq->SetProperty,Param=aux,/UpdateWidgets
  ; ?????   state.oData->set,i,Aux=aux
ENDIF ELSE BEGIN
  Widget_Control,state.wids.dataModel,Set_Value=' Model: undefined. '
ENDELSE


;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

;
;  MPFit parameters
;
tol = state.MPFitPars.tol
tMax=state.MPFitPars.tMax
covar = 0 
status = 0

dataErr = FuFiFa_GetErrorArray(data,state.MPFitPars,errTxt,Group=group) 

nPoints = N_Elements(dataX)
dof = nPoints-N_Elements(guess.value)+Long(Total(guess.fixed))
;
; append info to log window
;
Widget_Control,/HourGlass
XDisplayfile1_Append,['',$
  '*****************************'+$
  'Fitting data set # '+StrCompress(indexLine+1)+' '+$
  '*****************************',$
  'Function Expression: '+model, errtxt, $
  'Number of data points: '+StrCompress(nPoints), $
  'Initial parameters: '+Vect2String(guess.value),$
  'Fixed params (0=free,1=fix):'+Vect2String(guess.fixed),$
  'Bottom limits (0=No,1=Yes):'+Vect2String( (guess.limited[0]) ),$
  'Top limits (0=No,1=Yes):'+Vect2String( (guess.limited[1]) ),$
  'Bottom limit:'+Vect2String( (guess.limits[0]) ),$
  'Top limit:'+Vect2String( (guess.limits[1]) ),$
  'Degrees of Freedom:'+ StrCompress(dof), $
  'Tolerance value: '+StrCompress(tol), $
  'Maximum fitting time [sec]: '+StrCompress(tmax), $
  'Maximum iterations: '+StrCompress(state.MPFitPars.maxIter), $
  '','']

;
; Call MPFit
;
print,guess
r = mpfitexpr(model, dataX, dataY, dataErr, ParInfo=guess, $
  FTol=tol, XTol=tol, GTol=tol, $
  NPrint=state.MPFitPars.nprint, MaxIter=state.MPFitPars.maxIter, $
  IterProc='fufifa_mpfit_display', $
  IterArgs={t0:systime(1), tmax:tmax, parent:group}, $
  Status=status, $	; outputs
  PError=pError, $	; output
  nIter=nIter, $	; output
  yFit=yFit $	; output
   )

CASE status OF
  -1:  text='Error/Aborted'
  0:  text='improper input parameters.'
  1:  text=['both actual and predicted relative reductions',$
        'in the sum of squares are at most FTOL(=TOL).']
  2:  text=['relative error between two consecutive iterates',$
        'is at most XTOL(=TOL)']
  3:  text=['both actual and predicted relative reductions',$
        'in the sum of squares are at most FTOL(=TOL). And, ',$
        'relative error between two consecutive iterates',$
        'is at most XTOL(=TOL)']
  4:  text=['the cosine of the angle between fvec and any column',$
        'of the jacobian is at most GTOL(=TOL) in absolute value.']
  5:  text='the maximum number of iterations has been reached'
  6:  text=['FTOL(=TOL) is too small. no further reduction in the ',$
        'sum of squares is possible.']
  7:  text=['XTOL(=TOL) is too small. no further improvement in the ',$
        'approximate solution x is possible.']
  8:  text=['GTOL(=TOL) is too small. fvec is orthogonal to the ',$
        'columns of the jacobian to machine precision.']
  else: text='Unknown.'
ENDCASE


dataFit=0
p=r
x=dataX
cmd = 'dataFit = '+model
itmp = Execute(cmd)
IF itmp NE 1 THEN BEGIN
  tmp = Dialog_Message(/Error,Dialog_Parent=group, $
		'Error executing : '+cmd+'. Aborted.')
  RETURN
ENDIF

widget_control,state.wids.draw, get_val = wnum
wset,wnum
OPlot,dataX,DataFit,Color=6

; pause
deviates = (dataY-yFit)/dataErr
chi2 = Total(deviates*deviates)
	

XDisplayFile1_Append,['','MPFit termination status: ',text,$
	  'Number of iterations: '+StrCompress(nIter),$
	  'Chi-square: '+StrCompress(chi2),$
	  'Chi-square/N/(N-1): '+StrCompress(chi2/npoints/(npoints-1)),$
	  'Goodness of fit (=igamma(0.5*dof,chi2) ): '+ $
		StrCompress(IGamma(0.5*dof,0.5*chi2)),$
	  '','Resulting params: ',Vect2String(r),$
	  '','1-sigma errors in params: ',Vect2String(pError),$
  '***************************'+$
  'End fitting data set # '+StrCompress(indexLine+1)+' '+$
  '***************************', ' ',$
	  '']


;
; update results
;
IF N_Elements(ask) EQ 0 THEN ask=0
IF Keyword_Set(ask) THEN BEGIN
  itmp = Dialog_Message(Dialog_Parent=group, /Question,$
	'Update parameters with fit results?' )
  IF itmp EQ 'No' THEN RETURN
ENDIF

oeq->SetProperty,Param=r,/UpdateWidgets
state.oData->set,indexLine,Aux=r

END ; FuFiFa_fit


;
;============================================================================
;
PRO FuFiFa_WriteSpecFile, tlb, Group=group, File=file

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_WriteSpecFile: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

Widget_Control,tlb,Get_UValue=state

oData = state.oData
oeq = state.oEquation

nn =  oData->info(/N_Elements)
IF nn EQ 1 and oData->value(0,/IsDataDef) NE 1 THEN BEGIN
  Message,'No data'
  RETURN
ENDIF 


;
; get data
;

FOR i=0L,nn-1 DO BEGIN
  data = oData->value(i)
  dataX = Reform(data[0,*])
  dataY = Reform(data[1,*])

  IF  state.oData->value(i,/IsAuxDef) EQ 1 THEN BEGIN
    aux = state.oData->value(i,/Aux)
  ENDIF ELSE BEGIN
    IF NOT(Keyword_Set(print)) THEN $
    Message,'No model defined for data index: '+StrCompress(i)
    RETURN
  ENDELSE

  IF N_Elements(datax) EQ 0 THEN BEGIN
   Message,'Abscissas not defined'
   RETURN
  ENDIF

  dataErr = fufifa_GetErrorArray(data,state.mpfitpars,Group=group)

  oeq = state.oequation
  nlines = oeq->GetProperty(/Nlines)

  IF nlines NE 0 THEN BEGIN
    ; 
    ; model
    ; 
    oeq->SetProperty,Param=aux ; ,/UpdateWidgets
    dataModel= oeq->evaluate(datax)
    ; 
    ; residuals
    ; 
    dataResiduals = dataY-oeq->evaluate(datax)
  ENDIF ELSE BEGIN
    Message,'No model available.'
    RETURN
  ENDELSE

  ; 
  ; write file
  ; 
  IF i EQ 0 THEN BEGIN ; main header
     OpenW,unit,file,/Get_Lun
     PrintF,unit,'#F '+file
     PrintF,unit,'#C File created by XOP/fufifa (functional fitting facility)'
     PrintF,unit,'#D '+SysTime()
     PrintF,unit,'#UFUFIFA_VERSION '+FuFiFa_Version()
     ; 
     ; Model information (as in file Model->Save to file)
     ; 
     PrintF,unit,'#UFUFIFA_MODEL ; '
     PrintF,unit,'#UFUFIFA_MODEL ; fufifa-model data'
     PrintF,unit,'#UFUFIFA_MODEL ; Comments start with ";"'
     PrintF,unit,'#UFUFIFA_MODEL ; fufifa version: '+fufifa_version()
     PrintF,unit,'#UFUFIFA_MODEL ; Created on: '+SysTime()
     cd,Current=pwd
     PrintF,unit,'#UFUFIFA_MODEL ; Working on: '+pwd
     PrintF,unit, $
      '#UFUFIFA_MODEL ;'+$
      ' The lines below define the model functions and their fit/guess '
     PrintF,unit, $
      '#UFUFIFA_MODEL ;'+$
      '    parameters. They can be edited keeping the format unchanged'
     PrintF,unit,'#UFUFIFA_MODEL ; '

     FOR ii=0L,nlines-1 DO BEGIN
       name = oeq->GetPropertyLine(ii,/Names)
              tmp = name+' | '+oeq->GetPropertyLine(ii,/Template)+' | '+ $
                vect2string(oeq->GetPropertyLine(ii,/parLine))
              printf,unit,'#UFUFIFA_MODEL '+tmp
       ; print,tmp
     ENDFOR
  ENDIF ; main header

  nPoints = N_Elements(dataX)
  chi2 = Total( (dataResiduals/dataErr)^2 )
  chi2 = chi2/nPoints/(nPoints-1)
  
  PrintF,unit,''
  PrintF,unit,'#S '+oData->value(i,/Title)
  PrintF,unit,'#UFUFIFA_PAR '+Vect2String(aux)
  PrintF,unit,'#UFUFIFA_CHI2 '+StrCompress(chi2)
  PrintF,unit,'#N 5'
  PrintF,unit,'#L X  Y  ErrorY  Residuals  Model'
  FOR ii=0L,nPoints-1 DO BEGIN
    PrintF,unit, $
	dataX[ii],dataY[ii],dataErr[ii],dataResiduals[ii],dataModel[ii],$
	Format='(5(G0.6," "))'
  ENDFOR

ENDFOR


pars = FuFiFa_GetModelParsVariation(tlb,error=error)
IF error NE 1 THEN BEGIN
  npars = N_Elements(pars[*,0])
  colTitles = '  Par'+StrCompress(SIndGen(npars-2),/Rem)
  colTitles=['ScanNumber',colTitles,'  R_Chi**2']
  colTitles2 = ''
  FOR ii=0L,N_Elements(colTitles)-1 DO BEGIN
    colTitles2 = colTitles2+colTitles[ii]
  ENDFOR
	  
  PrintF,unit,''
  PrintF,unit,'#S 1000 Parameter variation'
  PrintF,unit,'#N '+StrCompress(npars)
  PrintF,unit,'#L '+colTitles2
  FOR ii=0L,nn-1 DO BEGIN
    PrintF,unit, pars[*,ii], Format='('+StrCompress(npars,/rem)+'(G0.6," "))'
  ENDFOR
ENDIF


Free_Lun,unit


END ; FuFiFa_WriteSpecFile

;
;============================================================================
;
PRO FuFiFa_fit_event, event

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_cwfit_event: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

Widget_Control,event.top,Get_UValue=state
Widget_Control,event.id,Get_UValue=uValue


indexLine=Widget_Info(state.wids.dataList,/List_Select)
IF indexLine[0] EQ -1 THEN BEGIN
  itmp = Dialog_message(/Error,Dialog_parent=event.top, $
	'No data to fit. Please Load/Select data')
  RETURN
ENDIF


;
; Start log window
;
Widget_Control,/HourGlass
XDisplayfile1,Text=['','Running MPFit on '+SysTime(),''], $
  Group=event.top, Title='Fitting results'

;
; Performs fitting 
;

FOR i=0L,N_Elements(indexLine)-1 DO BEGIN
  CASE uValue OF
    '**Fit**': FuFiFa_Fit,state,indexLine[i], Group=event.top, Ask=0
    else:
  ENDCASE
ENDFOR


END ; FuFiFa_fit_event

;
;============================================================================
;
PRO FuFiFa_parField_event, event

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_parField_event: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

Widget_Control,event.top,Get_UValue=state
Widget_Control,event.id,Get_Value=value
Widget_Control,event.id,Get_UValue=linenumber

oeq = state.oequation
oData = state.oData
;  print,oeq->info()

;
; set new numerial parameters
;
IF event.type EQ '' THEN BEGIN
  ;
  ; update equation object
  ;
  oeq->setPropertyLine,lineNumber,param=value.coefs

  ;
  ; update data auxiliar
  ;
  sel = Widget_Info(state.wids.dataList,/List_Select)
  IF sel[0] NE -1 THEN BEGIN
    nn =  oData->info(/N_Elements)
    IF nn EQ 1 and oData->value(0,/IsDataDef) NE 1 THEN BEGIN
      Message,/Info,'No data'
      RETURN
    ENDIF 
    goodIndices = IndGen(nn)
    goodIndices = goodIndices(sel)
    all_pars=oeq->GetProperty()
    FOR i=0L,N_Elements(goodIndices)-1 DO BEGIN
      oData->Set,goodIndices[i],Aux=all_pars
    ENDFOR
  ENDIF
  ;
  ; Plot
  ;
  FuFiFa_Plot,event.top
ENDIF


;
; set new fix/free flag
;
IF event.type EQ 'F' THEN BEGIN
  ; event.index EQ 0 means unsensitive
  ; fixed_flag EQ 0 means "free"
  fixed_flag = abs(value.flags[event.index]-1)
  oeq->setPropertyItem,lineNumber,event.index,Fixed=fixed_flag
ENDIF
  
;
; get "guess" values
;
IF event.type EQ 'G' THEN BEGIN
  ;oeq->setPropertyItem,lineNumber,event.index,Fixed=fixed_flag
  name=oeq->getPropertyLine(lineNumber,/Names)

  CASE StrUpCase(name) OF
    'STRAIGHTLINE': mytype = 'TWOPOINTS'
    'GAUSSIAN': mytype='PEAK'
    'LORENTZIAN': mytype='PEAK'
    'PSEUDOVOIGT': mytype='PEAK'
    'STEPATAN': mytype='STEP'
    'STEPERRORF': mytype='STEP'
    else: mytype='NONE'
  ENDCASE
  
  CASE mytype OF
    'NONE': BEGIN
	itmp = Dialog_message(/Info,Dialog_Parent=event.top, $
		['No mouse selection allowed.', $
		 'Please type values and <enter>'])
	END
    'TWOPOINTS': BEGIN
	itmp = Dialog_message(/Info,Dialog_Parent=event.top, $
		'Enter two points with mouse')
	Cursor,x1,y1,Wait=3,/Data
	OPLot,[x1,x1],[y1,y1],Color=1,Psym=2
	Wait,0.2
	Cursor,x2,y2,Wait=3,/Data
	OPLot,[x2,x2],[y2,y2],Color=1,Psym=2
	p = value.coefs
	p[1]= (y2-y1)/(x2-x1)
	p[0]= y1 - p[1]*x1
	value.coefs = p
	Widget_Control,event.id,Set_Value=value
	oeq->setPropertyLine,lineNumber,param=p
	END
    'PEAK': BEGIN
	itmp = Dialog_message(/Info,Dialog_Parent=event.top, $
		['Enter three points with mouse: ', $
		 ' 1st at the left side of the peak, at half width',$
		 ' 2nd at the top of the peak',$
		 ' 3rd at the right side of the peak, at half width'])
	Cursor,xL,yL,Wait=3,/Data
	OPLot,[xL,xL],[yL,yL],Color=1,Psym=2
	Wait,0.2
	Cursor,xC,yC,Wait=3,/Data
	OPLot,[xC,xC],[yC,yC],Color=1,Psym=2
	Wait,0.2
	Cursor,xR,yR,Wait=3,/Data
	OPLot,[xR,xR],[yR,yR],Color=1,Psym=2
	Wait,0.2
	p = value.coefs
	p[0] = 2.0*yC-(yL+yR)
	p[1] = xC
	p[2] = abs(xR-xL)
	value.coefs = p
	Widget_Control,event.id,Set_Value=value
	oeq->setPropertyLine,lineNumber,param=p
	END
    'STEP': BEGIN
	itmp = Dialog_message(/Info,Dialog_Parent=event.top, $
		['Enter three points with mouse: ', $
		 ' 1st at the bottom-left side of the step',$
		 ' 2nd at the center (half-height) of the step',$
		 ' 3rd at the right side of the step, at top '])
	Cursor,xL,yL,Wait=3,/Data
	OPLot,[xL,xL],[yL,yL],Color=1,Psym=2
	Wait,0.2
	Cursor,xC,yC,Wait=3,/Data
	OPLot,[xC,xC],[yC,yC],Color=1,Psym=2
	Wait,0.2
	Cursor,xR,yR,Wait=3,/Data
	OPLot,[xR,xR],[yR,yR],Color=1,Psym=2
	Wait,0.2
	p = value.coefs
	p[0] = yR
	p[1] = xC
	p[2] = abs(xR-xL)
	value.coefs = p
	Widget_Control,event.id,Set_Value=value
	oeq->setPropertyLine,lineNumber,param=p
	END
  ENDCASE

  ;
  ; update data auxiliar
  ;
  sel = Widget_Info(state.wids.dataList,/List_Select)
  IF sel[0] NE -1 THEN BEGIN
    nn =  oData->info(/N_Elements)
    IF nn EQ 1 and oData->value(0,/IsDataDef) NE 1 THEN BEGIN
      Message,/Info,'No data'
      RETURN
    ENDIF 
    goodIndices = IndGen(nn)
    goodIndices = goodIndices(sel)
    all_pars=oeq->GetProperty()
    FOR i=0L,N_Elements(goodIndices)-1 DO BEGIN
      oData->Set,goodIndices[i],Aux=all_pars
    ENDFOR
  ENDIF
  ; 
  ; Plot
  ;
  FuFiFa_Plot,event.top
ENDIF
  
END



;
;============================================================================
;
PRO FuFiFa_event, event

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_event: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

IF Tag_Names(event,/Structure_Name) EQ 'WIDGET_KILL_REQUEST' THEN BEGIN
  FuFiFa_Quit,event
  RETURN
ENDIF

Widget_Control,event.id,Get_UValue=uvalue

CASE uValue OF
  'SaveFile': BEGIN
	itmp = Dialog_Message(/Info,/Cancel,Dialog_Parent=event.top,$
	  ['This option creates a file (SPEC format) containing all data ',$
	   'plus model information.'])
	IF itmp EQ 'Cancel' THEN RETURN

	Widget_Control,event.top,Get_UValue=state
        dir = state.importDir
	file = Dialog_PickFile(/Write, file='fufifa.spec',$
            Dialog_Parent=event.top,PATH=dir,GET_PATH=dir)
        IF file EQ '' THEN RETURN
        Widget_Control,/Hourglass
	fufifa_WriteSpecFile,event.top,File=file,Group=event.top
        ;state.specFile=file
        state.importDir=dir
	Widget_Control,event.top, Set_UValue=state
	END

  'Quit': BEGIN
	FuFiFa_Quit,event
	END

  'Refresh': BEGIN
	Widget_Control,event.top,Get_UValue=state
	fufifa_plot,event.top,Group=event.top
	END

  'PlotPrint': BEGIN
	printBuffer = 'fufifa_plot,'+StrCompress(event.top,/Rem)+'L,/Print'
	xprint,BUFFER=printbuffer,GROUP=event.top,/COLOR
	END

  'DataLoad': BEGIN
	Widget_Control,event.top,Get_UValue=state
        dir = state.importDir
	file = Dialog_PickFile(Title='Select a SPEC file',$
            Dialog_Parent=event.top,PATH=dir,GET_PATH=dir)
        IF file EQ '' THEN RETURN
        Widget_Control,/Hourglass
        Widget_Control,event.top,TLB_Set_Title= $
            'FuFiFa '+FuFiFa_Version()+' - Imported: '+file
        n = Spec_Access(hSpec,file)
        IF n EQ 0 THEN BEGIN
          itmp = Dialog_Message(/Error, DIALOG_PARENT=event.top, $
            ['FUFIFA_EVENT: Error reading SPEC data file: ',file])
          RETURN
        ENDIF
        list0 = StrCompress(hSpec.scan.scan_n)+' - '
        list1 = hSpec.scan.name
        list01= list0+list1

	nData = N_Elements(list01)


	oData = state.oData

	nn =  oData->info(/N_Elements)
	IF nn EQ 1 and oData->value(0,/IsDataDef) NE 1 THEN BEGIN
	  append = 0
	ENDIF ELSE BEGIN
          itmp = Dialog_Message(/Question, DIALOG_PARENT=event.top, $
            ['Append new data to existing one ?',$
	     '( if not, existing data will be removed )'])
	  IF itmp EQ 'Yes' THEN append=1 else append=0
	ENDELSE

	IF append EQ 0 THEN BEGIN
	  Obj_Destroy,oData
	  oData = Obj_New('ompp',nTimes=nData)
	  FOR i=0L,nData-1 DO BEGIN
             data = Spec_Data(hSpec,i+1,/Index)
	     oData->set,i,Title=list01[i],Value=data
	  ENDFOR
          ; Widget_Control,state.wids.dataList,Set_Value=list01
	ENDIF ELSE BEGIN
	  FOR i=0L,nData-1 DO BEGIN
             data = Spec_Data(hSpec,i+1,/Index)
	     oData->set,/Add,Title=list01[i],Value=data
	  ENDFOR
	ENDELSE

	; update list widget
	nData = oData->info(/N_Elements)
	list01 = StrArr(nData)
	FOR i=0L,nData-1 DO BEGIN
	  list01[i] = oData->value(i,/Title)
	ENDFOR
        Widget_Control,state.wids.dataList,Set_Value=list01


        state.specFile=file
        state.importDir=dir
	state.oData=oData
	Widget_Control,event.top, Set_UValue=state

	widget_control,state.wids.draw, get_val = wnum
	wset,wnum
	erase
	
	END

  'DataLoadASCII': BEGIN
	Widget_Control,event.top,Get_UValue=state
        dir = state.importDir
	file = Dialog_PickFile(Title='Select an ASCII file',$
            Dialog_Parent=event.top,PATH=dir,GET_PATH=dir)
        IF file EQ '' THEN RETURN
        Widget_Control,/Hourglass
        Widget_Control,event.top,TLB_Set_Title= $
            'FuFiFa '+FuFiFa_Version()+' - Imported: '+file

	data = RAscii(file)


	oData = state.oData

	nn =  oData->info(/N_Elements)
	IF nn EQ 1 and oData->value(0,/IsDataDef) NE 1 THEN BEGIN
	  append = 0
	ENDIF ELSE BEGIN
          itmp = Dialog_Message(/Question, DIALOG_PARENT=event.top, $
            ['Append new data to existing one ?',$
	     '( if not, existing data will be removed )'])
	  IF itmp EQ 'Yes' THEN append=1 else append=0
	ENDELSE

	IF append EQ 0 THEN BEGIN
	  Obj_Destroy,oData
	  oData = Obj_New('ompp',nTimes=1)
	  oData->set,0,Title=' ASCII file: '+file,Value=data
	ENDIF ELSE BEGIN
	   oData->set,/Add,Title=' ASCII file: '+file,Value=data
	ENDELSE

	; update list widget
	nData = oData->info(/N_Elements)
	list01 = StrArr(nData)
	FOR i=0L,nData-1 DO BEGIN
	  list01[i] = oData->value(i,/Title)
	ENDFOR
        Widget_Control,state.wids.dataList,Set_Value=list01


        state.specFile=file
        state.importDir=dir
	state.oData=oData
	Widget_Control,event.top, Set_UValue=state

	widget_control,state.wids.draw, get_val = wnum
	wset,wnum
	erase
	
	END

  'DataOperations': BEGIN
	Widget_Control,event.top, Get_UValue=state
	str = {apply:['0','On all scans','On selected scans only'],$
		expression:'y/max(y)'}
        flags=Replicate('1',N_tags(str))
        titles=['Apply operation','Expression']
        XScrmenu,str,/Interp,/NoType,Action=action,group=event.top, $
          Flags=flags, Titles=titles, WTitle='Data operations',$
	  Dialog_Parent=event.top
        IF action EQ 'DONT' THEN RETURN

	nn =  state.oData->info(/N_Elements)
	oData = state.oData
	IF nn EQ 1 and oData->value(0,/IsDataDef) NE 1 THEN BEGIN
	  Message,/Info,'No data'
	  RETURN
	ENDIF
	goodIndices = IndGen(nn)
	CASE StrCompress(str.apply[0],/Rem) OF
	  '0':BEGIN 	; on all scans
		END
	  '1':BEGIN 	; on selected scans
		sel = Widget_Info(state.wids.dataList,/List_Select)
		IF sel[0] EQ -1 THEN BEGIN
		  itmp = Dialog_message(/Error, Dialog_Parent=event.top, $
			'Please select a data set.')
		  RETURN
		ENDIF
		goodIndices = goodIndices(sel)
		END
	ENDCASE

	FOR i=0L,N_Elements(goodIndices)-1 DO BEGIN
	   oData = state.oData
	   data = oData->value(goodIndices[i])
	   x = Reform(data[0,*])
	   y = Reform(data[1,*])
	   itmp = Execute('y = '+str.expression)
	   IF itmp NE 1 THEN BEGIN
	     tmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		'Error executing: '+str.expression)
	     RETURN
	   ENDIF
	   oData->Set,goodIndices[i],Value=Make_Set(x,y)
	ENDFOR

	FuFiFa_Plot,event.top
	
	END

  'DataCut': BEGIN
	itmp = Dialog_Message(/Info,/Cancel,Dialog_Parent=event.top, $
	  ['This options permits to select an abscissas interval.',$
          'Data points external to this interval will be eliminated.',$
          'This option could be used to eliminate unwanted points before fitting.'])
	IF itmp EQ 'Cancel' THEN RETURN
	

	Widget_Control,event.top, Get_UValue=state

	str = {apply:['0','On all scans','On selected scans only'],$
		selection:['0','with mouse','with keyboard'],$
		min:0.0, max:0.0 }
        flags=['1','1','w(1) EQ 1','w(1) EQ 1']
        titles=['Apply operation','Point selection', $
	   'Minumum value:','Maximum value']
        XScrmenu,str,/Interp,/NoType,Action=action,group=event.top, $
          Flags=flags, Titles=titles, WTitle='Data cut',$
	  Dialog_Parent=event.top
        IF action EQ 'DONT' THEN RETURN

	nn =  state.oData->info(/N_Elements)
	IF nn EQ 1 and state.oData->value(0,/IsDataDef) NE 1 THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
          'No data found.')
	  RETURN
	ENDIF

	goodIndices = IndGen(nn)
	CASE StrCompress(str.apply[0],/Rem) OF
	  '0':BEGIN 	; on all scans
		END
	  '1':BEGIN 	; on selected scans
		sel = Widget_Info(state.wids.dataList,/List_Select)
		IF sel[0] EQ -1 THEN BEGIN
		  itmp = Dialog_message(/Error, Dialog_Parent=event.top, $
			'Please select a data set.')
		  RETURN
		ENDIF
		goodIndices = goodIndices(sel)
		END
	ENDCASE

	IF StrCompress(str.selection[0],/Rem) EQ '0' THEN BEGIN ; mouse
	  itmp = Dialog_message(/Info,/Cancel,Dialog_Parent=event.top, $
		'Enter two points with mouse')
	  IF itmp EQ 'Cancel' THEN RETURN
	  Cursor,x1,y1,Wait=3,/Data
	  OPLot,[x1,x1],[y1,y1],Color=1,Psym=2
	  Wait,0.2
	  Cursor,x2,y2,Wait=3,/Data
	  OPLot,[x2,x2],[y2,y2],Color=1,Psym=2
	  Wait,0.2
	  str.min=x1
	  str.max=x2
	ENDIF

	IF goodIndices[0] EQ -1 THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
          'No data selected.')
          RETURN
	ENDIF

	FOR i=0L,N_Elements(goodIndices)-1 DO BEGIN
	  oData = state.oData
	  data = oData->value(goodIndices[i])
	  xx = Reform(data[0,*])
	  igood = Where(xx GE str.min AND xx LE str.max)
	  IF igood[0] EQ -1 THEN BEGIN
	    itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	      'Error in selecting interval: no good points ')
	    RETURN
	  ENDIF
	  tmpset=data[*,igood]
	  npoints1 = N_Elements(data[0,*])
	  npoints2 = N_Elements(tmpset[0,*])
	  IF i EQ 0 THEN BEGIN
	    IF N_Elements(goodIndices) GT 1 THEN $
		add='(for the first selected data set)' ELSE add=''
	    itmp = Dialog_Message(/Question,Dialog_Parent=event.top, $
	      ['Number of abscissa points reduced from '+$
	      StrCompress(npoints1,/Rem)+' to '+ $
	      StrCompress(npoints2,/Rem),add,$
	      ' ','Accept the operation (No Undo)?'])
	    IF itmp EQ 'No' THEN RETURN
	  ENDIF
	  oData->Set,goodIndices[i],Value=tmpset
	ENDFOR
	FuFiFa_Plot,event.top
	END


  'DataBackground': BEGIN
	itmp = Dialog_Message(/Info,/Cancel,Dialog_Parent=event.top, $
	  ['This options permits to substract a (fitted) straigt line. '])
	IF itmp EQ 'Cancel' THEN RETURN

	Widget_Control,event.top, Get_UValue=state

	str = {apply:['0','On all scans','On selected scans only'],$
		selection:['0','with mouse','with keyboard'],$
		min:0.0, max:0.0 }
        flags=['1','1','w(1) EQ 1','w(1) EQ 1']
        titles=['Apply operation','Point selection', $
	   'Min value for fitting interval:',$
	   'Max value for fitting interval:']
        XScrmenu,str,/Interp,/NoType,Action=action,group=event.top, $
          Flags=flags, Titles=titles, WTitle='Substract background',$
	  Dialog_Parent=event.top
        IF action EQ 'DONT' THEN RETURN

	nn =  state.oData->info(/N_Elements)
	IF nn EQ 1 and state.oData->value(0,/IsDataDef) NE 1 THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
          'No data found.')
	  RETURN
	ENDIF

	goodIndices = IndGen(nn)
	CASE StrCompress(str.apply[0],/Rem) OF
	  '0':BEGIN 	; on all scans
		END
	  '1':BEGIN 	; on selected scans
		sel = Widget_Info(state.wids.dataList,/List_Select)
		IF sel[0] EQ -1 THEN BEGIN
		  itmp = Dialog_message(/Error, Dialog_Parent=event.top, $
			'Please select a data set.')
		  RETURN
		ENDIF
		goodIndices = goodIndices(sel)
		END
	ENDCASE

	IF StrCompress(str.selection[0],/Rem) EQ '0' THEN BEGIN ; mouse
	  itmp = Dialog_message(/Info,/Cancel,Dialog_Parent=event.top, $
		'Enter two points with mouse')
	  IF itmp EQ 'Cancel' THEN BEGIN
	    FuFiFa_Plot,event.top
	    RETURN
	  ENDIF
	  Cursor,x1,y1,Wait=3,/Data
	  OPLot,[x1,x1],[y1,y1],Color=1,Psym=2
	  Wait,0.2
	  Cursor,x2,y2,Wait=3,/Data
	  OPLot,[x2,x2],[y2,y2],Color=1,Psym=2
	  Wait,0.2
          str.min = x1
          str.max = x2
	ENDIF

	IF goodIndices[0] EQ -1 THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
          'No data selected.')
          RETURN
	ENDIF

	FOR i=0L,N_Elements(goodIndices)-1 DO BEGIN
	  oData = state.oData
	  data = oData->value(goodIndices[i])
	  xx = Reform(data[0,*])
	  igood = Where(xx GE str.min AND xx LE str.max)
	  IF igood[0] EQ -1 THEN BEGIN
	    itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	      'Error in selecting interval: no good points ')
	    RETURN
	  ENDIF
	  tmpset=data[*,igood]
	  lincoeffs = LinFit(tmpset[0,*],tmpset[1,*],/Double)
	  yFit = lincoeffs[1]*xx+lincoeffs[0]
	  plot,xx,data[1,*],/NoData
	  oplot,xx,data[1,*],Color=2
	  oplot,xx,yFit,Color=10
	  itmp = Dialog_Message(/Question,/Cancel,Dialog_Parent=event.top, $
	    ['Using data set index: '+StrCompress(goodIndices[i]), $
	    'Substract the fit from data?'])
	  IF itmp EQ 'Cancel' THEN BEGIN
	    FuFiFa_Plot,event.top
	    RETURN
	  ENDIF
	  IF itmp EQ 'Yes' THEN BEGIN
	    data[1,*]=data[1,*]-yFit
	    oData->Set,goodIndices[i],Value=data
	  ENDIF
	ENDFOR
	FuFiFa_Plot,event.top
	END

  'DataDeleteScan': BEGIN

	itmp = Dialog_message(/Info,/Cancel,Dialog_parent=event.top, $
	  ['This option removes permanently the selected scans',$
	  'Continue?'])
	IF itmp EQ 'Cancel' THEN RETURN
	Widget_Control,event.top,Get_UValue=state
	indices=Widget_Info(state.wids.dataList,/List_Select)
	IF indices[0] EQ -1 THEN BEGIN
	    itmp = Dialog_message(/Error,Dialog_parent=event.top, $
		'No data to plot. Please Load/Select data')
	    RETURN
	ENDIF
	;
	; remove data
	;
	oData = state.oData
        nData = oData->info(/N_Elements)
	IF nData EQ N_Elements(indices) THEN BEGIN
	  itmp = Dialog_message(/Error,Dialog_parent=event.top, $
          'I cannot remove all scans.')
           RETURN
	ENDIF
	oData->remove,indices
	;
	; update widgets
	;
	titles = oData->value(/title)
        Widget_Control,state.wids.dataList,Set_Value=titles

	widget_control,state.wids.draw, get_val = wnum
	wset,wnum
	erase

	END


  'ModelLoad': BEGIN
	Widget_Control,event.top, Get_UValue=state
	oeq = state.oequation

	;
	; clear equtions
	;
	nlines=oeq->GetProperty(/NLines)
	FOR i=0L,nlines-1 DO oeq->deleteLine

	; clear all stored data
	nn =  state.oData->info(/N_Elements)
	IF nn EQ 1 and state.oData->value(0,/IsDataDef) NE 1 THEN BEGIN
	  Message,/Info,'No data'
	  RETURN
	ENDIF
	FOR i=0L,nn-1 DO state.oData->Set,i,/ClAux

	;
	; access to file
	;

	dir = state.importDir
	file = Dialog_PickFile(Title='Select a fififa-model data file',$
            Dialog_Parent=event.top,PATH=dir,GET_PATH=dir, $
	    FILTER='*.dat')
        IF file EQ '' THEN RETURN
        Widget_Control,/Hourglass
	state.importDir = dir
	stmp=''
	OpenR,unit,file,/Get_Lun
	readF,unit,stmp
	WHILE NOT(EOF(unit)) DO BEGIN
	  ReadF,unit,stmp
	  IF StrMid(StrCompress(stmp,/Rem),0,1) NE ';' THEN BEGIN
	    n = StrParse(stmp,'|',list)
	    IF n NE 2 THEN Message,'File command not understood: '+tmp
	    name = list[0]
            equation = list[1] 
            pars = list[2]
	    pars2 = 0
	    itmp = Execute('pars2='+list[2])
	    degree=N_Elements(pars2)-1
	    IF itmp NE 1 THEN Message,'Error executing '+'pars2='+list[2]
  
	    tmp = {list:['0',name], degree:degree, center:pars2[1], $
		  fwhm:pars2[2], jump:pars2[0], equation:equation,pars:pars }
	    tmp = {list:['0',name], degree:degree, center:0.0, $
		  fwhm:0.0, jump:0.0, equation:equation,pars:pars }
	
	    fufifa_addfunction,event.top,str=tmp,param=pars2
	  ENDIF
	ENDWHILE

	Free_Lun,unit
	END

  'ModelSave': BEGIN
	Widget_Control,event.top,Get_UValue=state

	dir = state.importDir
	file = Dialog_PickFile(Title='Write model to a fififa-model data file',$
            Dialog_Parent=event.top,PATH=dir,GET_PATH=dir, $
	    File='fufifa_model.dat')
        IF file EQ '' THEN RETURN
        Widget_Control,/Hourglass
	IF CheckFile(file) EQ 1 THEN BEGIN
	  itmp = Dialog_Message(/Cancel,/Info,Dialog_Parent=event.top, $
		['File exists: '+file,'Overwrite it?'])
	  IF itmp EQ 'Cancel' THEN RETURN
	ENDIF
	state.importDir = dir

	oeq = state.oEquation
        nlines=oeq->GetProperty(/NLines)
	IF nlines EQ 0 THEN RETURN

	OpenW,unit,file,/Get_Lun
	PrintF,unit,'; '
	PrintF,unit,'; fufifa-model data'
	PrintF,unit,'; Comments start with ";"'
	PrintF,unit,'; fufifa version: '+fufifa_version()
	PrintF,unit,'; Created on: '+SysTime()
	cd,Current=pwd
	PrintF,unit,'; Working on: '+pwd
	PrintF,unit, $
	 '; The lines below define the model functions and their fit/guess '
	PrintF,unit, $
	 ';    parameters. They can be edited keeping the format unchanged'
	PrintF,unit,'; '

	FOR i=0L,nlines-1 DO BEGIN
	  name = oeq->GetPropertyLine(i,/Names)
          tmp = name+' | '+oeq->GetPropertyLine(i,/Template)+' | '+ $
            vect2string(oeq->GetPropertyLine(i,/parLine))
          printf,unit,tmp
	  print,tmp
	ENDFOR

	Free_Lun,unit
	Print,'File fufifa_model.dat written to disk.'

	END

  'ModelDeleteAllFunctions': BEGIN
	Widget_Control,event.top, Get_UValue=state
	oeq = state.oequation

	; clear equtions
	nlines=oeq->GetProperty(/NLines)
	FOR i=0L,nlines-1 DO oeq->deleteLine

	; clear all stored data
	nn =  state.oData->info(/N_Elements)
	IF nn EQ 1 and state.oData->value(0,/IsDataDef) NE 1 THEN BEGIN
	  Message,/Info,'No data'
	  RETURN
	ENDIF 
	FOR i=0L,nn-1 DO state.oData->Set,i,/ClAux
	END

  'ModelDeleteFunction': BEGIN
	Widget_Control,event.top, Get_UValue=state
	oeq = state.oequation
	index=0
	xEdit,index,Text='Index: ',$
		InfoText='Index of the line to delete (starting from zero)',$
		Dialog_Parent=event.top, Action=action,Title='Delete model line'
	IF action EQ 'CANCEL' THEN RETURN
	oeq->deleteLine,index


	;
	; update data auxiliar
	;
	nn =  state.oData->info(/N_Elements)
	IF nn EQ 1 and state.oData->value(0,/IsDataDef) NE 1 THEN BEGIN
	  Message,/Info,'No data'
	  RETURN
	ENDIF
	; clear all stored data
	FOR i=0L,nn-1 DO BEGIN
	  state.oData->Set,i,/ClAux
	ENDFOR
	END

  'ModelAddFunction': BEGIN
	fufifa_addFunction,event.top
	END

  'ModelPropagate': BEGIN
	Widget_Control,event.top, Get_UValue=state
	nn=state.oEquation->GetProperty(/NPars)
	IF nn EQ 0 THEN Message,'No equation input.'
	mm = state.oData->Info(/N_Elem)
        IF mm EQ 1 THEN BEGIN
	  IF  state.oData->value(0,/IsDataDef) NE 1 THEN $
		Message,'No data input.'
        ENDIF
	all_pars=state.oEquation->GetProperty()
	;Widget_Control,state.wids.dataList,Get_Value=value
	;nn = Widget_Info(state.wids.dataList,/List_Number)
	;IF nn EQ 0 THEN Message,'No data input.'
	FOR i=0L,mm-1 DO state.oData->Set,i,Aux=all_pars
	END

  'ModelFix': BEGIN
	Widget_Control,event.top, Get_UValue=state
	; value.flags in cw_pfield EQ 0 means unsensitive (i.e. fixed)
  	; fixed=0 in oEquation means "free"
	nlines = state.oEquation->getProperty(/NLines)
	FOR i=0L,nLines-1 DO BEGIN
	  nIndices = state.oEquation->getPropertyLine(i,/NIndices)
	  wId = state.oEquation->getPropertyLine(i,/wId)
	  Widget_Control,wId,Get_Value=value
	  value.flags = 0
	  Widget_Control,wId,Set_Value=value
	  FOR j=0L,nIndices-1 DO BEGIN
	    state.oequation->setPropertyItem,i,j,Fixed=1
	  ENDFOR
	ENDFOR
	END

  'ModelFree': BEGIN
	Widget_Control,event.top, Get_UValue=state
	; value.flags in cw_pfield EQ 0 means unsensitive (i.e. fixed)
  	; fixed=0 in oEquation means "free"
	nlines = state.oEquation->getProperty(/NLines)
	FOR i=0L,nLines-1 DO BEGIN
	  nIndices = state.oEquation->getPropertyLine(i,/NIndices)
	  wId = state.oEquation->getPropertyLine(i,/wId)
	  Widget_Control,wId,Get_Value=value
	  value.flags = 1
	  Widget_Control,wId,Set_Value=value
	  FOR j=0L,nIndices-1 DO BEGIN
	    state.oequation->setPropertyItem,i,j,Fixed=0
	  ENDFOR
	ENDFOR
	END

  'ModelParsVariation': BEGIN
	pars = FuFiFa_GetModelParsVariation(event.top,error=error)
        npars = N_Elements(pars[*,0])
	colTitles = 'Par'+StrCompress(SIndGen(npars-2),/Rem)
	colTitles=['ScanNumber',colTitles,'Chi**2']
	Xplot,pars,/No_Block,ColTitles=colTitles,Wtitle='Parameter plot',$
	  Xtitle='-1',Ytitle='-1'
	  
	Widget_Control,event.top, Get_UValue=state
	END



  'Preferences': BEGIN
	Widget_Control,event.top,Get_UValue=state
	str0=state.MPFitPars
        flags=Replicate('1',N_tags(str0))
        titles=['Fitting tolerance','Max of fitting iterations',$
          'Use errors from third column (if exists)',$
          'Data errors', 'Report printing frequency', $
	  'Fit time limit (per scan) [sec]']
        XScrmenu,str0,/Interp,/NoType,Action=action,group=event.top, $
          Flags=flags, Titles=titles, WTitle='Fitting preferences',$
	  Dialog_Parent=event.top
        IF action EQ 'DONT' THEN RETURN
        ;Copy_Structure,str,str0
        state.MPFitPars = str0
	Widget_Control,event.top,Set_UValue=state
	END

  'PlotDropList': BEGIN
	Widget_Control,event.top, Get_UValue=state
	dl = state.plotDropList
	index = event.index
	IF index EQ 0 THEN RETURN
	;flip index
	dl[index-1]=abs(dl[index-1]-1)
	tmp = fufifa_PlotDropList(dl)
	state.plotDropList = dl
	Widget_Control,state.wids.plotDropList,Set_Value=tmp
	Widget_Control,state.wids.plotDropList,Set_DropList_Select=0
	Widget_Control,event.top, Set_UValue=state

	widget_control,state.wids.draw, get_val = wnum
	wset,wnum
	erase
	fufifa_plot,event.top,Group=event.top
	
	END

  'APPLYVALUE': BEGIN
	Widget_Control,event.id,Get_Value=value
	CASE StrUpCase(value) OF
          'XOP': xop
	  'XPLOT': xplot,/no_block	
	  'XAID': xaid,/no_block	
	  'EXODUS': exodus,/no_block	
	  else: itmp = Execute(uvalue)
	ENDCASE
	END
  'Help': BEGIN
	tmp=''
        Widget_Control,event.id,Get_Value=tmp
        Xhelp,StrLowCase(tmp[0]),group=event.top
	END

  else:
ENDCASE

END  ; FuFiFa_event

;
;============================================================================
;
PRO FuFiFa, group = group, no_block=no_block

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

Xop_Default_CT,Group=group

IF N_Elements(no_block) EQ 0 THEN no_block=1

; 
; Widget definition
;
tlb = Widget_base(TITLE='FuFiFa (Functional Fitting Facility) '+$
	FuFiFa_version(),MBAR=wMenu, /TLB_KILL_REQUEST_EVENTS, $
	Event_Pro='FuFiFa_Quit')

;
; define menu bar
;
wtmp0 = WIDGET_BUTTON(wMenu, VALUE='FuFiFa', /MENU)
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Save as...', UVALUE='SaveFile')
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Quit', UVALUE='Quit',/SEPARATOR)

wtmp0 = WIDGET_BUTTON(wMenu, VALUE='Plot', /MENU)
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Refresh',UValue='Refresh')
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Print', UVALUE='PlotPrint')

wtmp0 = WIDGET_BUTTON(wMenu, VALUE='Data', /MENU)
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Load data (Spec cleaned file/exodus)', $
	UVALUE='DataLoad')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Load data (ASCII)', $
	UVALUE='DataLoadASCII')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Operations...', /Separator, $
	UVALUE='DataOperations')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Cut data...', UVALUE='DataCut')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Substract background...', $
		UVALUE='DataBackground')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Delete selected scans', /Separator,$
	UVALUE='DataDeleteScan')

wtmp0 = WIDGET_BUTTON(wMenu, VALUE='Model', /MENU)
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Load from file...', UVALUE='ModelLoad')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Save to file...', UVALUE='ModelSave')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Add Function...', $
		UVALUE='ModelAddFunction',/Separator)
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Delete All Functions', $
		UVALUE='ModelDeleteAllFunctions')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Delete Function...', $
		UVALUE='ModelDeleteFunction')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Propagate parameters', $
		UVALUE='ModelPropagate')

wtmp0 = WIDGET_BUTTON(wMenu, VALUE='Fit', /MENU)
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Preferences...',UValue='Preferences')

wtmp0 = WIDGET_BUTTON(wMenu, VALUE='Tools', /MENU)
 wtmpb = Widget_Button(wtmp0, VALUE='Xop', UValue='APPLYVALUE')
 wtmpb = Widget_Button(wtmp0, VALUE='Xplot', UValue='APPLYVALUE')
 wtmpb = Widget_Button(wtmp0, VALUE='XAID', UValue='APPLYVALUE')
 wtmpb = Widget_Button(wtmp0, VALUE='EXODUS', UValue='APPLYVALUE')
 wtmpb = Widget_Button(wtmp0, VALUE='Color table',/Separator, /Menu)
   wtmp1 = WIDGET_BUTTON(wtmpb, VALUE='Set Default', UVALUE='ColorTable')
   wtmp1 = WIDGET_BUTTON(wtmpb, VALUE='Change table...', UVALUE='ColorTable')
wtmp0 = WIDGET_BUTTON(wMenu, VALUE='Help', /HELP)
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='FuFiFa', UVALUE='Help')



wBase = Widget_Base(tlb,/Column)

  ; draw + list
  wBase1 = Widget_Base(wBase,/Row)

    wBase11 = widget_base(wBase1,/Column,/Frame)
      wDraw = widget_draw(wBase11, XSIZE=400, YSIZE=350, RETAIN=2)
      wBase111 = widget_base(wBase11,/Row)
        wtmp = Widget_Button(wBase111,Value='Refresh',UValue='Refresh')
	plotDropListFlags = [1,1,1,0,1]
        tmp = FuFiFa_plotDropList(plotDropListFlags)
        wPlotDropList = Widget_DropList(wBase111,Value=tmp, $
		UValue='PlotDropList')

    wBase12 = widget_base(wBase1,/Column,/Frame)
      wBase121 = widget_base(wBase12,/Row)
        wtmp = Widget_Button(wBase121,Value='Load...',UValue='DataLoad')
        wtmp = Widget_Label(wBase121,Value=' Data sets  ')
      wDataModel = Widget_Label(wBase12,Value=' Model: undefined. ')
      IF float(!version.release) LE 5.5 THEN $
        wDataList = Widget_List(wBase12, Value=Replicate('',25), $
	  ySize=115,xSize=20,/Multiple,UValue='Refresh') ELSE $
        wDataList = Widget_List(wBase12, Value=Replicate('',25), $
	  ySize=22,xSize=20,/Multiple,UValue='Refresh')

  ; Controls (buttons)
  wBase2 = Widget_Base(wBase,/Row,/Frame)

    wtmp = Widget_Button(wBase2,Value='Quit',UValue='Quit')
    wtmp = Widget_Button(wBase2,Value='Add function',UValue='ModelAddFunction')
    wtmp = Widget_Button(wBase2,Value='Propagate model', $
	UValue='ModelPropagate')
    wtmp = Widget_Button(wBase2,Value='**Fit**',UValue='**Fit**',$
	Event_Pro='fufifa_fit_event')

  ; Models
  wBase3 = Widget_Base(wBase,/Column,/Frame)
    wBase31 = Widget_Base(wBase3,/Row)
      wtmp = Widget_Label(wBase31,Value='Models ')
      wtmp = Widget_Button(wBase31,Value='Free all pars',UValue='ModelFree')
      wtmp = Widget_Button(wBase31,Value='Fix all pars',UValue='ModelFix')
      wtmp = Widget_Button(wBase31,Value='Pars vs Scans', $
		UValue='ModelParsVariation')
    wModelBase = Widget_Base(wBase3,/Column, $
	Scr_XSize=600,Scr_YSize=300,$
      	XSize    =600,YSize=    300, $
	/Scroll,Event_Pro='fufifa_parfield_event')

wids={tlb:tlb, draw:wDraw, plotDropList:wPlotDropList, dataList:wDataList, $
	modelBase:wModelBase, dataModel:wDataModel}

odata = Obj_New('ompp')
oequation = Obj_New('f3equation')

MPFitPars = {tol:1D-10, maxIter:200, $
  errorsFrom:['1','Yes','No'], $
  errors:['0','Constant','Statistical'],$ 
  nPrint:10, tMax:30.0  }

state = {wids:wids, specFile:'', importDir:'', $
   odata:odata, oequation:oequation,  $
   plotDropList:plotDropListFlags, $	; flags show/hide for 
					;data-model-components-error
   mpFitPars: mpFitPars $
   } 

Widget_Control,tlb,Set_UValue=state

Widget_Control,tlb,/Realize

xmanager, 'fufifa', tlb, GROUP_LEADER=group,No_Block=no_block
END
