FUNCTION xop_macro_compact, inCode, Verbose=iVerbose, Execute=iExecute, $
  File=file, Group=group, HighlightId=highLightId


;+
; NAME:
;	Xop_Macro_Compact
;
; PURPOSE:
;	This function compacts and optionally executes an "XOP macro", 
;	An XOP macro is like an IDL macro (i.e., a bunch of command to
;	be exacuted sequentiually). IDL macros, on the contrary to IDL 
;	programs, do not accept FOR or IF blocks. However, an XOP macro 
;	does accepts blocks, because in the "compacting" action carried 
;	out by Xop_Macro_Compact reduces BEGIN...END blocks to a single 
;	line code. 
;
; CATEGORY:
;	Utilities. Widgets.
;
; CALLING SEQUENCE:
;	outCode = Xop_Macro_Compact(inCode) 
;
;
; OPTIONAL INPUTS:
;	inCode:	An array of strings with the code to be processed. 
;	
; KEYWORD PARAMETERS:
;	VERBOSE = set this keyword to print the processed code.
;	EXECUTE = set this keyword to execute the processed code.
;	FILE = set this keyword to a name of a file containing the 
;		code (in this case inCode is ignored).
;       GROUP = The widget ID of the widget that calls MyIdlMacro.
;       	When this ID is specified, a death of the caller results
;       	in a death of widgets created by MyIdlMacro.
;
; OUTPUTS:
;	the processed code
;
;
; RESTRICTIONS:
;       It is possible to use variables in the buffer, but this variables
;       must have a name included in the list: a0,...a9,b0,...b9,
;       c0,...,c9,d0,...,d9 (this is due to the fact that EXECUTE cannot 
;	create variables on the fly. The variable must be pre-initializad.
;	The comands are executed using "EXECUTE" commands, thus apply all the
;	limitations of this function (i.e. no command splitting in two
;	lines, no programming blocks, etc.)
;
; PROCEDURE:
;	Straightforward
;
; EXAMPLE:
;
;	inCode = ["; this is a comment", $
;		"tmp=''", $
;		"Read,'Execute loop? [Y/N]?',tmp", $
;		"IF tmp EQ 'Y' OR tmp EQ 'y' THEN BEGIN", $
;		"  FOR i=0,10 DO BEGIN ; start loop", $
;		"    print,'i=',i", $
;		"    print,'i*5=',i*5", $
;		"  ENDFOR", $
;		"ENDIF"]
;
;	outCode = Xop_Macro_Process(inCode)
;
; MODIFICATION HISTORY:
; 	Written by:	Manuel Sanchez del Rio (srio@esrf.fr) 00/05/22
;
;-

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'XOP_MACRO_COMPACT: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   RETURN,inCode
endif

IF N_Elements(iVerbose) EQ 0 THEN iVerbose=0
IF N_Elements(iExecute) EQ 0 THEN iExecute=0

IF Keyword_Set(file) THEN BEGIN
  prog1 = read_textfile(file)
ENDIF ELSE prog1=inCode

nProg1 = N_Elements(prog1)
iPos=-1
prog2=''
lab_begin=LonArr(nProg1)
lab_end=LonArr(nProg1)
j=-1

;
; remove comments
;
FOR i=0L,nProg1-1 DO BEGIN
  line = prog1[i]  
  iPos = StrPos(line,';')
  IF iPos GE 0 THEN line=StrMid(line,0,iPos)
  ;print,'**'+line+'***'
  IF StrLen(line) GT 0 THEN BEGIN
    j=j+1
    prog2=[prog2,line]
  ENDIF
ENDFOR

prog2 = prog2(1:N_Elements(prog2)-1)

print,prog2

nbegin = StrPos(StrLowCase(prog2),' begin')
nend = StrPos(StrLowCase(prog2),'end')
icopy = 0
j=-1
prog3=''

;
; main process loops
;

line=''
FOR i=0,N_Elements(prog2)-1 DO BEGIN
  line_i = prog2[i]

  IF nbegin[i] GE 0 THEN icopy = icopy + 1

  IF icopy GT 0 THEN BEGIN
    ; add new line to the line buffer
    ; (pay attention not to insert the & symbol in the case that
    ; the command finishes by BEGIN)
    list=''
    n = StrParse(StrCompress(StrLowCase(line)),' ',list)
    IF line EQ '' THEN add_and=0 ELSE add_and = 1
    IF n GE 0 THEN IF StrCompress(list[n],/Rem) EQ 'begin' THEN add_and = 0
    IF add_and EQ 1 THEN line = line + ' & ' + line_i  ELSE $
                         line = line + ' ' + line_i 
  ENDIF ELSE BEGIN
    prog3 = [prog3,line_i]
    line = ''
  ENDELSE 

  IF icopy EQ 1 THEN BEGIN
    ; if section is closed, add buffer line to the new program 
    ; otherwise, add the current line
    ;prog3 = [prog3,line]
    ;line = line_i
    IF nend[i] GE 0 THEN prog3 = [prog3,line] 
  ENDIF

  IF nend[i] GE 0 THEN icopy = icopy - 1
ENDFOR

prog3 = prog3(1:N_Elements(prog3)-1)
iverbose=1

IF iverbose EQ 1 THEN BEGIN
  print,'=============================='
  FOR i=0,N_Elements(prog3)-1 DO BEGIN
    print,prog3[i]
  ENDFOR
  print,'=============================='
ENDIF

outCode=prog3
IF iExecute EQ 1 THEN BEGIN
  ;
  ;  define allowed variables
  ;
  a0 = 0 & a1 = 0 & a2 = 0 & a3 = 0 & a4 = 0 & a5 = 0 
  a6 = 0 & a7 = 0 & a8 = 0 & a9 = 0 
  b0 = 0 & b1 = 0 & b2 = 0 & b3 = 0 & b4 = 0 & b5 = 0 
  b6 = 0 & b7 = 0 & b8 = 0 & b9 = 0 
  c0 = 0 & c1 = 0 & c2 = 0 & c3 = 0 & c4 = 0 & c5 = 0 
  c6 = 0 & c7 = 0 & c8 = 0 & c9 = 0 
  d0 = 0 & d1 = 0 & d2 = 0 & d3 = 0 & d4 = 0 & d5 = 0 
  d6 = 0 & d7 = 0 & d8 = 0 & d9 = 0 

  FOR iiii=0,N_Elements(prog3)-1 DO BEGIN
    IF Keyword_Set(highLightId) THEN  BEGIN
      ixy = Widget_Info(highLightId,Text_XY_To_OffSet=[0,iiii])
      Widget_Control,highLightId,Set_Text_Select=[ixy,0]
      Widget_Control,highLightId,Set_Text_Select=[ixy,StrLen(prog3[iiii])]
    ENDIF
    itmp = Execute(prog3[iiii])
    IF itmp EQ 0 THEN BEGIN
      jtmp = Dialog_Message(/Question,Dialog_Parent=group, $
	['','Error executing command line: ',prog3[iiii],$
	 'Continue?'])
      IF jtmp EQ 'No' THEN RETURN,outCode
    ENDIF
    ;Wait,0.1
  ENDFOR
  IF Keyword_Set(highLightId) THEN Widget_Control,highLightId,Set_Text_Select=[0,0]
ENDIF

RETURN,outCode
END

