FUNCTION RandomN1, ixx, d1,d2
;+
; NAME:
;	RANDOMN1
;
; PURPOSE:
;       This function returns one or more normal-distributed,
;	floating-point, pseudo-random numbers with variance one.
;	(Note: this routine has similar and more limited functionality
;	than the IDL's RANDOMN routine. However, it allows one to fix
;	the seed (i.e., correctly initialize it). )
;
; CATEGORY:
;	Mathematics
;
; CALLING SEQUENCE:
;       Result = RandomN1( [seed, d1, d2] )
;
; OPTIONAL INPUTS:
;       
;	Seed: A long integer used to initialize the random number generator. 
;		You can use Seed to start the sequence. If Seed is a named 
;		variable, RANDOMN1 update it to the next long integer in 
;		the sequence. To generate a random seed from the system time, 
;		set Seed equal to either zero or to an undefined named variable.
;
;	d1,d2: The dimensions of the result. The dimension parameters can 
;		be any scalar expression. Up to TWO dimensions can be specified.
;		If no dimensions are specified, it sets d1=d2=1 (default)
;
; OUTPUTS:
;       This function return the pseudo-random normal-distributed number.
;
; PROCEDURE:
;       
;       Uses the Mox-Muller method, adapted from Numerical Recipes
;	(Press, Flannery, Teukilsky & Vetterling, Cambridge Univ. Press,
;	1989 (1st edition)).
;	
;	Used RandomU1 as random number generator
;
;	It is obvious that this procedure is much slower than the 
;	"compiled-in" RandomN.
;
; EXAMPLE:
;       print,variance(RandomN1(55541L,10000))
;	1.01375
;
; MODIFICATION HISTORY:
; 	Written by:	Manuel Sanchez del Rio (srio@esrf.fr), 98-12-21
;
;-

On_Error,2

IF Keyword_Set(ixx) EQ 0 THEN ixx=Long(SysTime(1))
IF N_Elements(d1) EQ 0 THEN d1=1L
IF N_Elements(d2) EQ 0 THEN d2=1L

iset=0
IF iset EQ 0 THEN BEGIN
one:
  v1 = 2.0*RandomU1(ixx,d1*d2)-1.0
  v2 = 2.0*RandomU1(ixx,d1*d2)-1.0
  r = v1^2 + v2^2
two:
  bad = Where(r GE 1 OR r EQ 0)
  nbad = N_Elements(bad)
  IF nbad GE 1 AND bad[0] NE -1 THEN BEGIN
    vv1 = 2.0*RandomU1(ixx,nbad)-1.0
    vv2 = 2.0*RandomU1(ixx,nbad)-1.0
    rr = vv1^2 + vv2^2
    v1[bad]=vv1
    v2[bad]=vv2
    r[bad]=rr
    GoTo,two
  ENDIF
  fac = Sqrt(-2.0*alog(r)/r)
  gset = v1*fac
  gasdev = v2*fac
  iset=0
ENDIF ELSE BEGIN
  gasdev=gset
  iset=0
ENDELSE
IF d2 NE 1 THEN gasdev = Reform(gasdev,d1,d2)
RETURN,gasdev

END


