;+
;
;	NAME:
;		POWELLFIT
;	PURPOSE:
;		to make a fit of a data set (x,y) with an specified function
;		y=f(x;p) where a are the free parameters.
;	CALLING SEQUENCE:
;		POWELLFIT,a,keyword_parameters
;	INPUT PARAMETERS:
;		P = initial guess of the parameters
;	OPTIONAL INPUT PARAMETERS:
;	KEYWORD PARAMETERS:
;
;		SHOW = when set to 1, plot the data points and overplots
;		the initial guess.
;
;		YFIT = yfit returns the array with the best fit
;		evaluation points.
;
;		ERROR = logical with the error status
;
;		STATSIGMA = when set to 1, then takes 
;		sigma = sqrt(yfit) (statistical error)
;
;		FTOLERANCE = the tolerance value for minimization (def 1e-3)
;
;		ITER = output (number of iterations performed)
;
;		ITMAX = number of maximum iterations allowed (def=200,
;			5000 if AMOEBA keywors is set)
;	
;		AMOEBA = When this keyword is set the minimization
;			method is the SIMPLEX method (instead of powell), and 
;			it used the AMOEBA IDL routine. This keyword must be 
;			set to a ndim array (ndim is the dimesion of P) 
;			containing the problem's characteristic length scale 
;			for each dimension. This array is used with P to build 
;			an initial (ndim+1) point simplex. 
;			If all dimensions have the same scale, set AMOEBA 
;			equal to a scalar.
;
;	OUTPUTS:
;		Overwrite the fitting values on the initial estimation a, 
;		and (optionally) puts the evaluated function in the yfit 
;		keyword.
;	COMMON BLOCKS:
;		COMMON powellfit,equation,x,y, sigma, xrange, chi2fit, fpar
;		where:
;		equation: a string text with the f(x;a) equation
;		x: the abscissas array of the data points
;		y: the ordinates array of the data points
;		sigma: an array with the experimental error for y points
;			(the program takes sigma=x*0.+1 per default)
;		xrange: the x range where to evaluate the ChiSquare
;			function to be minimized
;		chi2fit: (output) value of the ChiSquate for the
;			parameter values obtained in the fitting process.
;		
;		
;	SIDE EFFECTS:
;		Uses internally the powellfit6 function.
;
;	RESTRICTIONS:
;		Unknown.
;	PROCEDURE:
;		Uses the the powell (or amoeba) idl function to find the 
;		minimum of the function 
;		ChiSquare= Sum{ ( (y-f(x;p))/sigma )^2 }. 
;		The data are passed trough common blocks because the 
;		"user defined function" which calculates ChiSquare must 
;		know them. It uses the function powellfit6 internally 
;		to evaluate ChiSquare.
;
;		See Numerical Recipes in C: The Art of Scientific Computing 
;		(Second Edition), published by Cambridge University Press
;		for description of POWELL and AMOEBA routines. 
;
;	EXAMPLE: 
;		; define common block
;		COMMON powellfit,equation,x,y, sigma, xrange, chi2fit
;		;create data set (gaussian + noise)
;		x=findgen(100) & y= 10.*exp( -(x-55.)^2 / 50.)
;		y = y+2.*randomu(seed,100)
;		;define the equation to be fitted:
;		equation='p[0]*exp(-(x-p[1])^2/p[2])'
;		p=[6,40,20]  ; the initial guess
;		powellfit,p,yfit=yfit
;		; for simplex minimization use:
;		; powellfit,p,yfit=yfit,amoeba=[100.,50,100]
;		; plot the result:
;		plot,x,y & oplot,x,yfit
;               print,p ;print the fitted values=[10.6253,55.0800,68.7902]
;
;	MODIFICATION HISTORY:
;       by  Manuel Sanchez del Rio. ESRF. 95-02-13
;	95-03-15 MSR Adds statsigma keyword. Changes chi2fit->chi2fit/dof
;		Other small changes.
;	95-11-20 MSR adds ftolerance,iter and itmax  keyword and change 
;		default ftolerance 1e-4 -> 1e-3. Removes fpar in common block.
;	99-03-05 srio@esrf.fr adds catch. Now the parameter is P instead
;		of A. The new array subscripts [] is used instead ().
;	99-05-25 srio@esrf.fr adds the AMOEBA keyword (simplex method).
;
;-
FUNCTION powellfit6,p,error=error,statsigma=statsigma
common powellfit,equation,x,y, sigma, xrange, chi2fit



catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!error_state.msg
   catch, /cancel
   on_error,2
   RETURN,0
endif


error = not(execute('y1 = '+equation) )
if error then begin
  print,'POWELLFIT: Error interpreting equation'
  return,0
endif else begin
  if keyword_set(statsigma) then begin
    sigma = sqrt(y1)
    print,'POWELLFIT6: considering statistical error.'
  endif
  data = [1,0]#x
  data(1,*) = ( (y-y1)/sigma )^2
  if xrange(0) NE xrange(1) then  cutset,data,data2,xrange=xrange $
	else data2=data
  nterms = n_elements(p) 			; npoints
  nfree = n_elements(data(1,*)) - nterms ; degrees of freedom
  if nfree le 0 then print,'POWELLFIT: Not enough data points.'
  return,total(data2(1,*)) / nfree
endelse
end

;
;
; 

PRO powellfit,p,show=show,yfit=yfit,error=error,statsigma=statsigma, $
ftolerance = ftolerance, iter=iter, itmax=itmax,Amoeba=iamoeba

common powellfit,equation,x,y, sigma, xrange, chi2fit



catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!error_state.msg
   catch, /cancel
   on_error,2
   error=1
   RETURN
endif

if not(keyword_set(sigma)) then sigma = x*0.+1.

if not(keyword_set(xrange)) then xrange=[0.,0]
if not(keyword_set(itmax)) then begin
  if Keyword_Set(iamoeba) then itmax=5000 ELSE itmax=200
endif
if xrange(0) EQ xrange(1) then xrange=[min(x),max(x)]

if keyword_set(show) then begin
  plot,x,y
  error = not( execute ('y1 = '+equation) )
  if error then begin
    print,'POWELLFIT: Error evaluating equation'
    print,'POWELLFIT: Returning'
    return
  endif 
  oplot,x,y1
endif

npar = n_elements(p)
xi = fltarr(npar,npar)
for i=0,npar-1 do xi(i,i) = 1.0

IF NOT(keyword_set(ftolerance)) THEN ftolerance = 1e-3
iter=1

IF Keyword_Set(iamoeba) THEN BEGIN
  Message,/info,'Using Simplex method (AMOEBA)'
  pp = Amoeba(ftolerance, P0=p, SCALE=iamoeba,  FUNCTION_NAME='powellfit6',$
  NCalls=iter, NMax=itmax )
  IF N_Elements(pp) GT 1 AND pp[0] NE -1 THEN p=pp ELSE BEGIN
    Message,/Info,'AMOEBA failed to find minimum.'
    error=1
  ENDELSE
ENDIF ELSE BEGIN
 powell, p, xi ,ftolerance, fmin, 'powellfit6', iter=iter, $
  itmax=itmax
ENDELSE



chi2fit = powellfit6(p,error=error,statsigma=statsigma)

error = not( execute ('yfit = '+equation) )
if error then begin
  print,'POWELLFIT: Error evaluating equation'
  print,'POWELLFIT: Returning'
  return
endif 
if keyword_set(show) then begin
  oplot,x,yfit
endif

end
