FUNCTION infl,f1,f2,method=method
;+
; NAME:
;	INFL
;
; PURPOSE:
;	This function performs the mathematical correlation of two 
;	input functions f1 and f2, each one defined as fltarr(2,npts).
;	Both functions must have the same abscissas axes and the same
;	number of points. (Use INFL2 if the abscissas are not the same)
;
; CATEGORY:
;	Mathematics
;
; CALLING SEQUENCE:
;
;	result = INFL(f1,f2)
;
; INPUTS:
;	f1 and f2: the input functions
;
; OPTIONAL INPUTS:
;	
; KEYWORD PARAMETERS:
;       METHOD: 0 (default) calculates convolution using the FFT
;               1 calculates convolution performing the integrals
;
; OUTPUTS:
;	the correlation function defined as 
;	h(t) = int[ f2(y) f1(y-t) dy ]
;	Be careful with the order, f2 is "fixed" and f1 is "moving"!
;
; SIDE EFFECTS:
;	the array of the correlation function is fltarr(2,2*npts), being
;	npts the number of points for the input functions
;
; PROCEDURE:
;	Straightforward.
;
; KNOWN BUGS:
;	The result of infl(f1,f2) with f1 symmetrical must be the 
;	same as the result of conv(f1f2). However a small shift is
;	appreciated (MSR 97/10/20)
;
; EXAMPLE:
;
;		a = infl(f1,f2)
;
; MODIFICATION HISTORY:
; 	Written by:	M. Sanchez del Rio
;	June 27, 1994
;	96/01/05 PF+MSR correct abscissas shift and add fft option.
;	97/10/20 MSR makes integration result agree with FTT one.
;
;-
;
;
; makes the correlation function of two sets of data (each set is a 
; fltarr(2,npts) )
;
;
on_error,2
;
; test correctness of array dimension and abscissas arrays
;
if (n_elements(f1(0,*)) ne n_elements(f2(0,*))) then begin
  print,' Error from CONV: mismatch of array dimensions. Abort'
  return,0
endif
diff = abs(f1(0,1) - f2(0,1))
if (diff gt 1e-4) then begin
  print,' Error from CONV: mismatch of abscissas arrays. Abort'
  return,0
endif
;
; define abscissas arrays
;
npts = n_elements(f1(0,*))
npts2 = 2*npts
step = f1(0,1) - f1(0,0)
xx=fltarr(npts2)
xx(0:npts-1)=f1(0,*)
for i=0,npts-1 do xx(npts+i) = xx(npts-1)+step*(i+1)
;;;xx = xx  - 0.5*( f1(0,npts-1) - f1(0,0) )
xx = xx  -  f1(0,npts-1) 
; calculates convolution of z2 and z3 functions
;
z2=fltarr(npts2)
z3=fltarr(npts2)

z2(0:npts-1)=f1(1,*)
;z2(npts:npts2-1)=f1(1,*)
;z3(0:npts-1)=f2(1,*)
z3(npts:npts*2-1)=f2(1,*)
h = fltarr(npts2)
;

if not(keyword_set(method)) then begin
  ;
  ; makes the convolution using the convolution theorem
  ;
  ff1 = fft(z2,-1)
  ff2 = fft(z3,-1)
  ff3 = conj(ff1)*ff2
  h = fft(ff3,+1)
  h = float(h) *npts2
endif else begin
;
; makes the integral
;
;help,z2
;help,z3
  for j=0,npts2-1 do begin
    h(j) = 0
    for i=j,npts2-1 do begin
      ;;;;;;h(j) = h(j) + z2(i)*z3(i-j)
      ;print,i-j
      h(j) = h(j) + z2(i-j)*z3(i)
    endfor
  endfor
endelse
;
; normalization factor = 1/sqrt( integral(f1)*integral(f2) )
;
int1 = total(f1(1,*))  ;*step 
int2 = total(f2(1,*))  ;*step
aa= total(h)*step/(int1*int2)
h = h*aa
;
; prepare the output
;
res = fltarr(2,npts2)
res(0,*) = xx
res(1,*) = h
;
return,res
;
end
