
; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       XSPECVIEW
;
; PURPOSE:
;       This procedure provides a graphical user interface to explore the
;       contents of a SPEC data file. It allows to visualize headers and
;	data blocks.
;
; CATEGORY:
;       Widgets.
;
; CALLING SEQUENCE:
;       XSPECVIEW , SpecHandle
;
; INPUTS:
;       SpecHandle - Named variable that contains a handle to a SPEC data
;             file as generated by SPEC_ACCESS(). 
;
; KEYWORDS:
;       GROUP_LEADER: Set this keyword to the widget ID of an existing widget
;               that serves as "group leader" for XSPECVIEW. When the group
;               leader is killed XSPECVIEW is also destroyed.
;
;       TITLE:  String that will appear as window title of XSPECVIEW.
;       NOSEL:  When this keyword is set, the 'Scan sel...' button does
;		not appear in the main window.
;       NO_BLOCK:  If set the NO_BLOCK keyword is passed to Xmanager.
;
; OUTPUT:
;
; PROCEDURE:
;	Uses xspecscan.
;
; EXAMPLE:
;
; MODIFICATION HISTORY:
;       96-07-10  Written by M. Sanchez del Rio (srio@esrf.fr)
;       96-09-18  MSR allows window resizing, refresh, adds TITLE kw,etc.
;       97-01-13  MSR ports to Windows (change widget sizes).
;       97-09-09  MSR adds no_block keyword.
;       98-09-23 MSR (srio@esrf.fr) some fixes for mca data. Use catch.
;	03-01-29 srio@esrf.fr fixes a bug when using multiple windows. 
;-
;
;=====================================================================
;
PRO xspecview_display,state,scanindex=scanindex, group=group, mcdata=mcdata
;

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught '+!err_string
  itmp=Dialog_Message(/Error, DIALOG_PARENT=group, $
    'XSPECVIEW_DISPLAY: error caught '+!err_string)
  catch, /cancel
  return
endif

widget_control,/HOURGLASS
handle_value,state.spechandle,h
if not(keyword_set(mcdata)) then mcdata=0
if not(keyword_set(scanindex)) then scanindex = $
   spec_scan(h,'current',/return_index)
case strcompress(state.sel,/rem) of
   '0': begin   ; all
   	text = spec_headers(h,scanindex,/index)
	tmp = spec_data(h,scanindex,/index,/string,mcdata=mcdata)
	n = n_elements(tmp(0,*))
	tmp1 = strarr(n)
	for i=0,n-1  do tmp1(i) = straccumulate(tmp(*,i),' ')
	text = [text,tmp1]
	if spec_points(h,scanindex,/index) EQ 1 then begin
	  ncol = spec_columns(h,scanindex,/index)
	  labels = spec_labels(h,scanindex,/index)
	  tmp1 = ['','','','*************** arranged data **************']
	  for i=0,ncol-1 do tmp1=[tmp1,labels(i)+': '+strcompress(tmp(i),/rem)]
	  text = [text,tmp1]
	endif
	end
   '1': begin   ; data
	tmp = spec_data(h,scanindex,/index,/string,mcdata=mcdata)
	n = n_elements(tmp(0,*))
	text = strarr(n)
	for i=0,n-1  do text(i) = straccumulate(tmp(*,i),' ')
	end
   '2':text = spec_headers(h,scanindex,/index)   ; header
   else: 
endcase
widget_control,state.wids.text,set_value=text
end
;
;=====================================================================
;
PRO xspecview_event,event

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught '+!err_string
  itmp=Dialog_Message(/Error, DIALOG_PARENT=event.top, $
    'XSPECVIEW_EVENT: error caught '+!err_string)
  catch, /cancel
  if n_elements(state) NE 0 then goto,out else return
endif

Widget_Control, event.id, get_UValue=eventUValue

if n_elements(eventuvalue) EQ 0 then eventuvalue = ''
if not(keyword_set(eventuvalue)) then eventuvalue = ''

stateid = Widget_Info(event.handler,/Child)
Widget_Control, stateid, get_UValue=state  ; , /No_Copy

if event.top eq event.id then begin ; resize drawing area
if sdep() eq 'WINDOWS' then  $
  widget_control,state.wids.text,scr_xsize=event.x-25,scr_ysize=event.y-65 $
  else $
  widget_control,state.wids.text,scr_xsize=event.x-15,scr_ysize=event.y-50
  goto,out
endif

case eventuvalue of
  'QUIT':begin	
	handle_free,state.spechandle
	widget_control,/destroy,event.top
	return
	end
  'PRINT':begin	
	widget_control,state.wids.text,get_value=text
	print_ascii,text,group=event.top
	end
  'SEL': begin
	if event.select NE 1 then goto,out   ; *for Windows*
	state.sel=event.value
	xspecview_display,state, group=event.top, mcdata=state.mcdata
	end
  'SSCAN': begin
                widget_control,state.scanbase,/show, bad_id=bad
                event_name = tag_names(event, /structure_name)
                if event_name eq 'NEW_SCAN_EVENT' then begin
		   xspecview_display,state,scanindex=event.newscan, $
		   group=event.top, mcdata=state.mcdata
                endif else if bad eq state.scanbase then begin
                   xspecscan, state.spechandle, caller=event.id,             $
                              tlb_id=tmp, mca=state.specmca, /not_all,       $
                              group_leader=event.top, title='Scan Selection'
                   state.scanbase = tmp
                endif
	end
  else:
endcase

out:
IF Widget_Info(stateid,/Valid_Id) THEN $
  Widget_Control, stateid, set_UValue=state ; , /No_Copy
end
;
;=======================================================================
;
PRO xspecview,hspec,GROUP=group,title=title,nosel=nosel,no_block=no_block, $
  mcdata=mcdata
;


catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught '+!err_string
  itmp=Dialog_Message(/Error, DIALOG_PARENT=group, $
    'XSPECVIEW: error caught '+!err_string)
  catch, /cancel
  return
endif

if n_elements(no_block) EQ 0 then no_block=0
if n_params() EQ 0 then message,'Usage: xspecview,hspec'
if not(keyword_set(title)) then title='XSpecView'
if not(keyword_set(mcdata)) then mcdata=0
;
base = widget_base(/COLUMN,TITLE=title,/TLB_SIZE_EVENTS)
;
;
; the blocks box
;

Bbox = widget_base(base,/COLUMN,/FRAME)
 
tmpbase = widget_base(Bbox,/Row)

tmp = widget_button(tmpbase,VALUE='Quit',UVALUE='QUIT')
tmp = widget_button(tmpbase,VALUE='Print/SaveAs...',UVALUE='PRINT')
if not(keyword_set(nosel)) then $
tmp = widget_button(tmpbase,VALUE='Scan sel...',UVALUE='SSCAN')
if mcdata then sel=2 else sel = 0
tmp = cw_bgroup(tmpbase,['All','Data','Header'],UVALUE='SEL', $
	/EXCLUSIVE,/FRAME,/RETURN_INDEX,ROW=1,SET_VALUE=sel)

Btext = widget_text(Bbox,VALUE=replicate('    ',30),XSIZE=80, $
; changed for Windows 	YSIZE=40,/SCROLL,/EDIT)
	YSIZE=20,/SCROLL,/EDIT)

wids = {text:Btext}
currentscan = spec_scan(hspec,'current',/return_index)
spechandle= handle_create(value=hspec)

state = { wids:wids, scanbase:0L, spechandle:spechandle, specmca:0, $
	sel:sel, mcdata:mcdata}
widget_control,Widget_Info(base,/Child),set_uvalue=state


widget_control,base,/REALIZE
xspecview_display,state,scanindex=currentscan, group=group,mcdata=mcdata

if no_block then $
  xmanager,'XSPECVIEW',base,GROUP=group, /no_block else $
  xmanager,'XSPECVIEW',base,GROUP=group
;
end
