; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       XSPECSCAN
;
; PURPOSE:
;       This procedure provides a graphical user interface to explore the
;       contents of a SPEC data file and select scans. Depending on the way
;       it is invoked, it runs in one of the following three different modes:
;
;       STANDALONE - It runs as an independent widget, no output, no
;               communication with other procedures.
;
;       MODAL - It runs as a modal widget, all other widgets are desensitized.
;               When a scan is selected, XSPECSCAN dies and returns the index
;               of the scan in the SPEC data file.
;
;       SLAVE - It runs as a slave of another widget procedure or function
;               that acts as master. Whenever a scan is selected in XSPECSCAN,
;               it sends and event to the master widget. The event structure
;               sent is defined by the following statement:
;                 {NEW_SCAN_EVENT, ID:Id, TOP:0L, HANDLER:0L,                $
;                          FHANDLE:IdlHandle, NEWSCAN:ScanIndex, MCA:McaFlag }
;               where IdlHandle is the IDL handle that points to the SPEC file
;               handle (a data structure that holds an index of the SPEC data
;               file, see SPEC_ACCESS), ScanIndex is the position of the scan
;               in the data file and McaFlag is set if the MCA button was
;               activated in XSPECSCAN.
;               If XSPECSCAN receives any event from the master widget (see
;               TLB_ID keyword below), it reinitializes itself. In this way
;               the master procedure can inform XSPECSCAN that something
;               changed (new data file, new scans, etc..).
;
; CATEGORY:
;       Widgets.
;
; CALLING SEQUENCE:
;       XSPECSCAN [, SpecHandle | IdlHandle]                  (STANDALONE)
;       XSPECSCAN, [SpecHandle | IdlHandle], SpecScan         (MODAL)
;       XSPECSCAN, IdlHandle, /IDL_HANDLE, CALLER=Caller_Id   (SLAVE)
;
; INPUTS:
;       SpecHandle - Named variable that contains a handle to a SPEC data
;             file as generated by SPEC_ACCESS(). If SpecHandle is not a
;             valid SPEC file handle, XSPECSCAN initializes it internally.
;             When SpecHandle is not initialized it must be of any data type
;             except LONG in order to avoid ambiguities with IdlHandle.
;             In MODAL mode SpecHandle is also used as output parameter,
;             returning the updated handle.
;
;       IdlHandle - IDL handle that points to a SPEC file handle. In SLAVE
;             mode the SPEC file handle must be previously initialized by the
;             master procedure. In MODAL mode IdlHandle can be used as output
;             parameter to obtained the updated SPEC file handle.
;
;       Caller_Id - ID of the master widget when XSPECSCAN runs in SLAVE mode.
;
; KEYWORDS:
;       GROUP_LEADER: Set this keyword to the widget ID of an existing widget
;               that serves as "group leader" for XSPECSCAN. When the group
;               leader is killed XSPECSCAN is also destroyed.
;
;       NOT_ALL: Set this keyword to not display the "All" button in the 
;               selection scan panel.
;
;       TITLE:  String that will appear as window title of XSPECSCAN.
;
;       MCA:    Set this keyword to activate the MCA button in the selection
;               scan panel. In MODAL mode if this keyword is set to a named
;               variable, it also returns the final state of the MCA button.
;
;       CALLER: (SLAVE mode) ID of the master widget as explained above. 
;
;       TLB_ID: (SLAVE mode) Set this keyword to a named variable that will
;               receive the ID of the Top Level Base of XSPECSCAN.
;               The master procedure can use this value to send events to
;               XSPECSCAN and force it to reinitialize. No information is
;               extracted from the event structure, any event is accepted
;               therefore.
;       NO_BLOCK: When set, the NO_BLOCK keyword is passed to Xmanager.
;
; OUTPUT:
;       SpecScan - (only MODAL mode) Returns the index of the scan in the
;             SPEC file. The index is an integer number that indicates the
;             relative position of the scan in the file starting from 1.
;             If the selection is cancelled, SpecScan is set to zero.
;
; EXAMPLE:
;
; MODIFICATION HISTORY:
;       1.00  Oct./95,  Written by P. Fajardo
;	96-09-20 MSR (srio@esrf.fr) adds the 'Options/View scan header/data'
;	and shorts the scan list label text when it is too long.
;	97-09-09 MSR (srio@esrf.fr) adds NO_BLOCK keyword
;	98-06-18 MSR (srio@esrf.fr) renames "Load" button to "ReLoad"
;	  when CALLER kwyword is set. Show names by default. Added common
;	  block xspecscan_shownames to store a value indicating that the
;	  names are shown (1) of not shown (1) for successive calls.
;	  New buttons (first, last, netx scans). Automatic selection in
;	  the list. Other cosmetics.
;	98-09-23 MSR (srio@esrf.fr) some fixes for mca data
;	98-11-20 MSR (srio@esrf.fr) adds the "add to file" functions.
;	02-03-13 MSR (srio@esrf.fr) changes RStrPos by StrPos(/Reverse_Search)
;-

;
;=========================================================================
;
pro xspecscan_cleanup, id
   common xspecscan, scan_index, mca_data

   widget_control, id, get_uvalue=state
   if state.cancel then begin
      scan_index = 0
      mca_data = 0
   endif else begin
      scan_index = state.scanidx
      mca_data = state.datatype eq 1
   endelse
   if state.destroy_handle then handle_free, state.handle
   handle_free, state.listhandle
end

;
;=========================================================================
;
pro xspecscan_createlist, state, spechandle, update=update
   names = widget_info(state.w.names, /child)
   if keyword_set(update) and names eq 0 then return

   current = spechandle.currscan + 1
   indx_list = spec_scan(spechandle, '-', /return_index, $
                 mcdata=state.datatype eq 1, no_empty=state.datatype eq 0)
   tmp = spec_scan(spechandle, current, /index)
   handle_value, state.listhandle, old_list
   if n_elements(old_list) eq n_elements(indx_list) and   $
      n_elements(old_list) gt 0 then                      $
      if where(old_list ne indx_list) eq -1 then          $
         return

   handle_value, state.listhandle, indx_list, /set
   if indx_list(0) lt 0 then begin
      name_list = '<none>'
   endif else begin
      name_list = string(spechandle.scan(indx_list-1).scan_n)+ '.   ' +  $
                  spechandle.scan(indx_list-1).name
   endelse
   if names ne 0 then widget_control, names, /destroy
   tmp = widget_list(state.w.names, value=name_list, uvalue='ACCEPT',  $
                     ysize=min([9, n_elements(name_list)-1]))
   state.w.select_list = tmp
   aux = (where(indx_list eq current))(0)
   if aux ge 0 then widget_control, tmp, set_list_select=aux  
end

;
;=========================================================================
;
pro xspecscan_update, state, error_status, reload=reload, group=group
;error_status = 0


   catch, error_status
   if error_status ne 0 then begin
     message,/info,'error caught '+!err_string
     itmp=Dialog_Message(/Error, DIALOG_PARENT=group, $
       'XSPECSCAN_UPDATE: error caught '+!err_string)
     catch, /cancel
     if n_elements(spechandle) NE 0 then goto,out else return
   endif

   handle_value, state.handle, spechandle, /no_copy

   valid = 0
   if (size(spechandle))(2) eq 8 then begin
      if (tag_names(spechandle))(0) eq 'SPECFILE' then begin
         valid = 1
      endif
   endif

   if valid then begin
      file = spechandle.specfile
      state.scanidx = spechandle.currscan + 1
      scan = strtrim(spec_scan(spechandle, state.scanidx, /index), 2)
      widget_control, /hourglass
      list = spec_access(spechandle,/list,/compress,/str,readalways=reload, $
                   mcdata=state.datatype eq 1, no_empty=state.datatype eq 0)
      xspecscan_createlist, state, spechandle, /update
      widget_control, state.w.input, sensitive=1
   endif else begin
      file = '<no file>'
      state.scanidx = 0
      scan='<none>'
      list = ''
      widget_control, state.w.input, sensitive=0
   endelse
   if list eq '' then list = '<none>'
   nlist = strlen(list)
   if nlist GT 160 then begin
     list = strmid(list,0,40)+'...(skipped)...'+strmid(list,nlist-40,40)
   endif
   
   widget_control, state.w.field, set_value=scan
   widget_control, state.w.current, set_value='Current scan :  '+scan

   widget_control, state.w.file, set_value=file
   widget_control, state.w.list, set_value=list

Out:
   handle_value, state.handle, spechandle, /set, /no_copy
end
;
;=========================================================================
;
pro xspecscan_event, event
common xspecscan_shownames,shownames_value


catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught '+!err_string
  itmp=Dialog_Message(/Error, DIALOG_PARENT=event.top, $
    'XSPECSCAN_EVENT: error caught '+!err_string)
  if n_elements(spechandle) NE 0 then $
    handle_value, state.handle, spechandle, /set
  catch, /cancel
  return
endif

   widget_control, event.top, get_uvalue=state
   if event.id eq event.top then begin
      xspecscan_update, state, error, group=event.top
      widget_control, event.top, set_uvalue=state
      if error then widget_control, event.top, /destroy
      return
   endif

   handle_value, state.handle, spechandle, /no_copy
   reload=0
   widget_control, event.id, get_uvalue=evuval

   case evuval of
      'OPEN': begin
            file = Dialog_Pickfile(title='Select a spec data file', /must_exist)
            if file ne '' then begin
               widget_control, /hourglass
               if spec_access(newhandle, file) gt 0 then begin
                  aux = spec_scan(newhandle, 1, /index)
                  spechandle = newhandle
               endif else begin
                  tmp = widget_message('File is not a SPEC data file', /error)
               endelse
            end
         end

      'RELOAD': begin
            reload = 1
         end

      'QUIT': begin  ; Do nothing here and wait till the end of the procedure
            state.cancel = 1
            evuval = 'EXIT'
         end

      'SHOW': begin
            if state.scanidx gt 0 then begin
               xspecscan_createlist, state, spechandle
               widget_control, event.id,set_value='Hide Names',set_uvalue='HIDE'
	       shownames_value = 1
            endif else begin
               tmp = widget_message('No data file.')
            endelse
         end
      'VIEW': begin
            title = spechandle.specfile
            if strpos(title,'/') ne -1 then begin
              n = strparse(title,'/',flist)
              title = flist(n)
            endif
	    if state.datatype EQ 1 then mcdata=1 else mcdata=0
            title = title+'(scan:'+$
              strcompress(spec_scan(spechandle,'current'),/rem)+')'
            xspecview,spechandle,group=event.top,/nosel,title=title,$
		no_block=state.no_block, mcdata=mcdata
         end
      'HIDE': begin
            widget_control, widget_info(state.w.names, /child), /destroy
            handle_value, state.listhandle, -1, /set
            widget_control, event.id, set_value='Show Names', set_uvalue='SHOW'
	    shownames_value = 0
	    state.w.select_list = 0L
         end

      'ACCEPT': begin
         index = -1
         event_name = tag_names(event, /structure_name)
         handle_value, state.listhandle, listhandle
         if event_name ne 'WIDGET_LIST' then begin
            widget_control, state.w.field, get_value=scan
            scan=strtrim(scan(0),2)
            if strpos(scan, '-') ge 0 or strpos(scan, ',') ge 0 then begin
               tmp = widget_message('Scan ranges are not supported', /error, $
                                                      dialog_parent=event.top)
            endif else begin
               index = (spec_scan(spechandle, scan, /return_index, /no_range,  $
                   mcdata=state.datatype eq 1, no_empty=state.datatype eq 0))(0)
            endelse
            if event_name eq 'WIDGET_BUTTON' then begin
              widget_control,event.id,get_value=tmp
	      ; be sure that "listhandle" contains the right list
	      if n_elements(listhandle) EQ 0 then listhandle=-1
	      if listhandle[0] eq -1 then listhandle = spec_scan(spechandle, $
		'-', /return_index, mcdata=state.datatype eq 1, $
		no_empty=state.datatype eq 0)
	      ;
              case tmp[0] of 
                'Next Scan': begin
			tmpindex = where(listhandle EQ scan) 
			if tmpindex[0] EQ n_elements(listhandle)-1 then begin
			  itmp = dialog_message(/info,dialog_parent=event.top,$
			    'Botton of the list reached')
			  handle_value, state.handle, spechandle, /set
			  return
			endif
			scan = listhandle[tmpindex+1]
			end
                'Previous Scan': begin
			tmpindex = where(listhandle EQ scan) 
			if tmpindex[0] EQ 0 then begin
			  itmp = dialog_message(/info,dialog_parent=event.top,$
			    'Top of the list reached')
			  handle_value, state.handle, spechandle, /set
			  return
			endif
			scan = listhandle[tmpindex-1]
			end
                'First Scan': scan = listhandle[0]
                'Last Scan': scan = listhandle[n_elements(listhandle)-1]
                else: 
              endcase
	      index = (spec_scan(spechandle, scan , /return_index, $
		  /no_range,  mcdata=state.datatype eq 1, $
		  no_empty=state.datatype eq 0))(0)
            endif
	    ; update list widget
	    if widget_info(state.w.select_list,/Valid_id) then begin
	      tmpindex = where(listhandle EQ index)
	      if tmpindex[0] NE -1 then $
                widget_control,state.w.select_list,set_list_select=tmpindex[0]
	    endif
         endif else begin
            ;handle_value, state.listhandle, tmp
            ;tmp = tmp(event.index)
            tmp = listhandle(event.index)
            if tmp ge 0 then begin
               index = spec_scan(spechandle, tmp, /index, /return_index)
            endif else begin
               scan='<none>'
            endelse
         endelse
         if index gt 0 then begin
            state.scanidx=index
            case state.mode of
               0:
               1: evuval='EXIT'
               2: begin
                     dummyevent={NEW_SCAN_EVENT, ID:0L, TOP:0L,     $
                                 HANDLER:0L, FHANDLE:state.handle,  $
                                 NEWSCAN:state.scanidx,             $
                                 MCA:(state.datatype eq 1)}
                     widget_control, state.w.caller, send_event=dummyevent
                  end

            endcase
         endif else begin
            ;msg='Scan "'+scan+'" is not valid.'
            msg='Selected scan not valid or contains no data.'
            tmp = widget_message(msg, /error, dialog_parent=event.top)
            handle_value, state.handle, spechandle, /set
            return
         endelse
      end

      'DATATYPE': begin
         widget_control, event.id, get_value=aux
         state.datatype = aux
      end
      'CLEAN_FILE': BEGIN
	 IF StrCompress(state.cleanfile,/Remove_All) NE '' THEN BEGIN
	   filein = state.cleanfile
	 ENDIF ELSE BEGIN
	   file = spechandle.specfile
	   itmp = StrPos(file,sdep(/DS),/Reverse_Search)
	   if itmp NE -1 then file = StrMid(file,itmp+1,StrLen(file)-itmp)
	   filein=file+'.cleaned'
	 ENDELSE
back:
	 file = Dialog_Pickfile(group=event.top, File=filein, Title=$
	  'Select a file to receive cleaned scans')
	 IF file EQ '' THEN GoTo,out
	 IF CheckFile(file) EQ 1 THEN BEGIN
	   itmp = Dialog_Message(Dialog_Parent=event.top,/Question, $
		['File '+file+' already exists.','Append scans to it?'])
	   IF itmp EQ 'No' THEN GoTo,back
	 ENDIF ELSE BEGIN
	   header1=spec_headers(spechandle,'CURRENT',/All)
	   header2=spec_headers(spechandle,'CURRENT')
	   goodlines = n_elements(header1)-n_elements(header2)
	   Openw,unit,file,/Get_Lun
	   FOR i=0L,goodlines-1 DO PrintF,unit,header1[i]
	     PrintF,unit,'#C**************************************'+$
		'****************'
	   FOR i=0L,4 DO PrintF,unit,'#C'
	   PrintF,unit,'#C File cleaned by XOP/Xplot on '+SysTime()
	   FOR i=0L,4 DO PrintF,unit,'#C'
	   PrintF,unit,'#C**************************************'+$
		'****************'
	   PrintF,unit,''
	   Free_Lun,unit
	ENDELSE
	state.cleanfile=file
	; update screen
        ;dummyEvent={NEW_SCAN_EVENT, ID:0L, TOP:0L,     $
        ;                         HANDLER:0L, FHANDLE:state.handle,  $
        ;                         NEWSCAN:state.scanidx,             $
        ;                         MCA:(state.datatype eq 1)}
	;Widget_Control, state.w.caller, Send_Event=dummyevent
      END
      'CLEAN_ADD': BEGIN
	IF state.cleanfile EQ '' THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top,$
	    ['File to receive scans not defined.',$
	     'Define it using menu "File/Define file for storing cleaned scans..."'])
	  GoTo,out
	ENDIF
	IF CheckFile(state.cleanfile) NE 1 THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top,$
	    ['File to receive scans ('+state.cleanfile+') does not exist.',$
	     'Define it using menu "File/Define file for storing cleaned scans..."'])
	  GoTo,out
	ENDIF
	tmp=''
	Widget_Control,state.xplotMessageId,Get_Value=tmp
	list=''
	n = StrParse(tmp[0],'[]',list)
	cleanFileShort = state.cleanFile
	itmp = StrPos(cleanFileShort,sdep(/DS),/Reverse_Search)
	if itmp NE -1 then cleanFileShort = $
	  StrMid(cleanFileShort,itmp+1,StrLen(cleanFileShort)-itmp)
	IF n NE 5 THEN BEGIN
	  itmp = Dialog_Message(Dialog_Parent=event.top, /Question,$
	    ['Scan limits selection not found.',$
	     '(Select with the mouse the scan limits.)',$
	     ' ','Do you want to add the whole Scan to the file?'] )
	  IF itmp EQ 'No' THEN GoTo,out
	  ; Get Spec header and data
	  header=Spec_Headers(specHandle,'CURRENT')
	  dataf=Spec_Data(spechandle,'CURRENT')
	  msg = 'Whole current scan added to file: '+cleanFileShort
	ENDIF ELSE BEGIN
	  cmd1='['+list[3]+']'
	  cmd2='['+list[5]+']'
	  a0=0 & a1=0
	  ierror = Execute('a0='+cmd1)
	  IF ierror EQ 1 THEN ierror = Execute('a1='+cmd2)
	  IF ierror EQ 0 THEN BEGIN
	    itmp = Dialog_Message(Dialog_Parent=event.top, /Error,$
	     ['Error executing: ',cmd1,cmd2])
	    GoTo,out
	  ENDIF
	  xmin = a0[0] & xmax=a0[1]
	  xcol = a1[0]
	  ; Get Spec header and data
	  xColData=Spec_Data(specHandle,'CURRENT',xcol,/Float)
	  dataf=Spec_Data(spechandle,'CURRENT')
	  good = Where(xColData GE xmin AND xColData LE xmax)
	  IF good[0] EQ -1 THEN BEGIN
	    itmp = Dialog_Message(Dialog_Parent=event.top, /Error,$
	     'Empty selection. Nothing added to file.')
	    GoTo,out
	  ENDIF
	  header=Spec_Headers(specHandle,'CURRENT')
	  dataf = dataf[*,good]
	  msg = 'Current scan in the interval '+vect2String([xmin,xmax])+$
	   ' added to file: '+cleanFileShort
	ENDELSE
	OpenW,unit,state.cleanFile,/Get_Lun,/Append
	FOR i=0L,N_Elements(header)-1 DO PrintF,unit,header[i]
	format='('+string(spec_columns(specHandle,'CURRENT'))+'(G0.6," "))'
	FOR i=0L,N_Elements(dataf[0,*])-1 DO PrintF,unit,dataf[*,i],$
		Format=format
	PrintF,unit,''
	Free_Lun,unit
	Widget_Control,state.xplotMessageId,Set_Value=msg
	; update screen
        dummyEvent={NEW_SCAN_EVENT, ID:0L, TOP:0L,     $
                                 HANDLER:0L, FHANDLE:state.handle,  $
                                 NEWSCAN:state.scanidx,             $
                                 MCA:(state.datatype eq 1)}
	Widget_Control, state.w.caller, Send_Event=dummyevent
      END
   endcase
out:
   handle_value, state.handle, spechandle, /set
   xspecscan_update, state, error, reload=reload, group=event.top
   widget_control, event.top, set_uvalue=state
   if evuval eq 'EXIT' or (state.mode eq 0 and error ne 0) then begin
      widget_control, event.top, /destroy
   endif
end
;
;=========================================================================
;
pro xspecscan, handle, scan_idx, CALLER=caller, GROUP_LEADER=group, $
TLB_ID=MainBase, MCA=mca, NOT_ALL=not_all, TITLE=title, no_block=no_block,$
XplotMessageId=xplotmessageid
;, IDL_HANDLE=idl_h
   common xspecscan
   common xspecscan_shownames,shownames_value

   if n_elements(shownames_value) EQ 0 then shownames_value=1
   if n_elements(no_block) EQ 0 then no_block = 0
   if n_elements(xplotmessageid) EQ 0 then xplotmessageid = 0L
   if keyword_set(caller) and  n_params() eq 1 then begin
      mode = 2
   endif else if n_params() le 1 then begin
      mode = 0
      caller = 0
   endif else if n_params() eq 2 then begin
      mode = 1
      caller = 0
   endif else begin
      message, 'Wrong parameters.'
   endelse

   if n_elements(handle) eq 0 then handle=0 else handle=handle(0)
   notall= keyword_set(not_all)
   mca= keyword_set(mca)
   if mca then datatype=1 else if not notall then datatype=2 else datatype=0

   idl_h = ((size(handle))(1) eq 3)   ; set idl_h if handle is of LONG type
   destroy = 0
   if idl_h then begin
      if not handle_info(handle, /valid) then begin
         if mode eq 2 then message, 'Handle must be initialized in SLAVE mode'
         handle = handle_create(value=0)
         if mode eq 0 then destroy = 1
      end
      idl_handle = handle 
      handle_value, idl_handle, spec_handle
   endif else begin
      if mode eq 2 then message, 'An IDL handle must be used in SLAVE mode'
      spec_handle = handle
      idl_handle = handle_create(value=spec_handle)
      if mode eq 0 then destroy = 1
   endelse

   if not keyword_set(title) then title='Xspecscan'
   MainBase = widget_base(title=title, mbar=tmp0, $
                   column=1,uvalue=0)
      ; menubar
      tmp1 = widget_button(tmp0, value='File')
         if mode ne 2 then $
            tmp2 = widget_button(tmp1, value='Open ...', uvalue='OPEN')
         tmp2 = widget_button(tmp1, value='Reload', uvalue='RELOAD')
         tmp2 = widget_button(tmp1, value='Define file for storing cleaned scans...',$
		 uvalue='CLEAN_FILE')
         if mode ne 2 then $
            tmp2 = widget_button(tmp1, value='Exit', uvalue='QUIT',/separator) 
	 ; added by srio
         if mode eq 2 then $
            tmp2 = widget_button(tmp1, value='Close', uvalue='QUIT',/separator)
      tmp1 = widget_button(tmp0, value='Options')
         wShowHide = widget_button(tmp1, value='Show names', uvalue='SHOW')
         tmp3 = widget_button(tmp1, value='View scan header/data', $
		uvalue='VIEW')
      ; buttons
      tmp1 = widget_base(MainBase, row=1)
         if mode eq 1 then tmp = 'Cancel' else tmp = 'Close'
         tmp2 = widget_button(tmp1, value=tmp, uvalue='QUIT')
	 if keyword_set(caller) then tmp = 'ReLoad' else tmp = 'Load'
         tmp2 = widget_button(tmp1, value=tmp, uvalue='ACCEPT') 
         tmp2 = widget_button(tmp1, value='Add to file', uvalue='CLEAN_ADD') 
      tmp1 = widget_base(MainBase, row=1)
         tmp2 = widget_button(tmp1,value='Next Scan', uvalue='ACCEPT')
         tmp2 = widget_button(tmp1,value='Previous Scan', uvalue='ACCEPT')
         tmp2 = widget_button(tmp1,value='First Scan',uvalue='ACCEPT')
         tmp2 = widget_button(tmp1,value='Last Scan', uvalue='ACCEPT')

      ;infos
      FileName = widget_label(MainBase,/dynamic_resize)
      ScanCurrent = widget_label(MainBase,/dynamic_resize)
      InputBase = widget_base(MainBase, column=1, /frame)
         ScanList = widget_label(InputBase,/dynamic_resize)
         tmp1 = widget_base(InputBase, row=1)
            ScanField = cw_field(tmp1, title='Scan id:', uvalue='ACCEPT', $
                                           xsize=7, /return_events)
            if notall then              $
               tmp=['Counters','MCA']   $
            else                        $
               tmp=['Counters','MCA','All']
            tmp2 = cw_bgroup(tmp1, tmp, /exclusive, $
                           /row, set_value=datatype, uvalue='DATATYPE', /frame)

         NamePanel = widget_base(InputBase, /column, /align_center)

   widgets = {caller:caller,       $
              input:InputBase,     $
              field:ScanField,     $
              file:FileName,       $
              current:ScanCurrent, $
              names:NamePanel,     $
              list:ScanList,       $
	      select_list:0L }
   state = {mode:mode,                  $
            handle:idl_handle,          $
            destroy_handle:destroy,     $
            scanidx:0,                  $
            listhandle:handle_create(), $
            datatype:datatype,          $
            cancel:0,                   $
            w:widgets,			$
	    no_block:no_block,		$
	    cleanfile:'',                $
	    xplotMessageId:xplotMessageId }

   xspecscan_update, state, error
   if error ne 0 and mode eq 2 then begin
      ; do something smart
      widget_control, MainBase, /destroy
      return
   end
   ;======================
   if shownames_value EQ 1 then begin
     if state.scanidx gt 0 then begin
       handle_value, state.handle, spechandle
       xspecscan_createlist, state, spechandle
       widget_control, wShowHide,set_value='Hide Names',set_uvalue='HIDE'
     endif else begin
       tmp = widget_message(/Error,Dialog_parent=group,'No data file.')
     endelse
   endif
   ;======================

   widget_control, MainBase, /realize, set_uvalue=state
   if mode eq 1 then begin
      if keyword_set(no_block) then $
      xmanager, 'xspecscan', MainBase, group_leader=group,  $
                cleanup='xspecscan_cleanup', /modal ,/no_block else $
      xmanager, 'xspecscan', MainBase, group_leader=group,  $
                cleanup='xspecscan_cleanup', /modal 
      scan_idx = scan_index   
      mca = mca_data
      if not idl_h then begin
         handle_value, idl_handle, handle
         handle_free, idl_handle
      endif
   endif else begin
      if keyword_set(no_block) then $
      xmanager, 'xspecscan', MainBase, group_leader=group,  $
                cleanup='xspecscan_cleanup', /no_block else $
      xmanager, 'xspecscan', MainBase, group_leader=group,  $
                cleanup='xspecscan_cleanup'
   endelse

   return
end
