;______________________________________________________________________________
;+
; NAME:
;       SPEC_SAVE()
;
; PURPOSE:
;       This function saves the data contained in a 2D numeric array into a file
;       following the SPEC data file format. If the file does not exist it is created.
;       I the file exits, it must be a valid SPEC data file and the data is appended
;       as a new scan. 
;
; CATEGORY:
;       Input/Output.
;
; CALLING SEQUENCE:
;       Result = SPEC_SAVE(Handle, Data [, File])
;
; INPUTS:
;       Handle - Handle to the SPEC data file. It is initialized, if necessary, and
;                updated after the data is saved.
;
;       Data   - Numeric array containing the data to be saved. The first dimension
;                correspond to the data columns in the SPEC file.
;
; OPTIONAL INPUT/OUTPUT
;       File   - If Handle is not a valid SPEC file handle, this is parameter is the
;                name of the file where the data will be saved. Otherwise File
;                specifies a named variable where the actual name of the file will be
;                returned.
;
; KEYWORDS:
;       COLUMNS: Vector that specifies the columns in the array Data that will be
;               saved. If this keyword is not set all the columns are saved.
;
;       LABELS: String vector that contains the labels for the data columns.
;
;       COMMENTS: String vector that contains comment lines that will be included
;               in the header section of the scan.
;
;       NAME:   String that will appear as scan name in the `#S' line.
;
;       OVERWRITE: Set this keyword to overwrite the file if it exists. By default
;               data is appended to existing files.
;
;       NO_CONFIRM: Set this keyword to not ask for confirmation when writting data
;               on existing files.
;
;       NO_ERROR: Set this keyword to not produce an IDL error if an error is found
;               when writting the data. 
;
;	DIALOG_PARENT: Set this keyword equal to a string that specifies the name 
;		of the parent widget (to be passed to Dialog_Message)
;
; OUTPUT: 
;       This function returns the scan number under which the data is saved in the
;       file. If the operation is cancelled, the function returns zero. If an error
;       happens and NO_ERROR is set the function returns -1 and the error message
;       is stored in the system variable !ERR_STRING.
;
; EXAMPLE:
;       To save the data in the array NewData in the file 'Newfile', enter:
;
;       PRINT, SPEC_SAVE(DummyHandle, NewData, 'Newfile')
;______________________________________________________________________________
;-
function spec_save, handle, data, file, COLUMNS=columns, LABELS=labels,      $
                     COMMENTS=comm, NAME=name, NO_ERROR=noerror,             $
                     OVERWRITE=overwrite, NO_CONFIRM=no_conf,                $
		     DIALOG_PARENT=dialog_parent
   catch, error_status
   if error_status ne 0 then begin
      catch, /cancel
      on_error, 2
      if keyword_set(noerror) then return, -1
      message, __cleanmsg(!err_string);, /traceback
   endif

   aux = size(data)
   if aux(0) ne 2 then begin
      message, 'Data is not a 2D array.'
   end
   totcol = aux(1)

   if n_elements(columns) eq 0 then columns = indgen(totcol)
   ncol = n_elements(columns)
   if min(columns) lt 0 or max(columns) ge totcol then begin
      message, 'Column values are out of range.'
   endif

   nlab = n_elements(labels)
   if nlab lt ncol then begin
      extralbl= ('Column'+strtrim(indgen(ncol),2))(nlab:ncol-1)
      if nlab eq 0 then begin
         labels = extralbl
      endif else begin
         labels = [labels, extralbl]
      endelse
   endif
   labels = labels(0:ncol-1)

   if not keyword_set(name) then name = ''

   if not keyword_set(file) then begin
      if __isaspechandle(handle) then begin
         file = handle.specfile
      endif else begin
         ;Select a filename
         file = Dialog_Pickfile(/write)
         if file eq '' then begin
            return, 0
         endif
      endelse
   endif

   ; Check if writable and if new or SPEC file 

   openw, Unit, file, /get_lun, /append, error=err
   if err then begin
      message, 'Cannot open "'+file+'" for writting.'
   endif
   status = fstat(Unit)
   if status.size gt 0 then begin
      if spec_access(handle, file) eq 0 then begin
         close, Unit
         message, 'Exists and is not a SPEC file'
      endif else begin
         scan = spec_scan(handle, 'last') + 1
      endelse
   endif else begin
      scan = 1
   endelse

   if keyword_set(overwrite) then begin
      if not keyword_set(no_conf) and scan gt 1 then begin
         tmp = Dialog_Message('SPEC file exists, overwrite data ?', /default_no, $
                              title='Writting data', /question, $
			      DIALOG_PARENT=dialog_parent)
         if tmp eq 'No' then begin
            close, Unit
            return, 0
         endif
      endif
      close, Unit
      openw, Unit, file, /get_lun
      scan = 1
   endif else begin
      if not keyword_set(no_conf) and scan gt 1 then begin
         tmp = Dialog_Message('SPEC file exists, append data ?', /question, $
                               title='Writting data',		       $
			       DIALOG_PARENT=dialog_parent)
         if tmp eq 'No' then begin
            close, Unit
            return, 0
         endif
      endif
   endelse

   ; Write the data

   if scan eq 1 then begin
      printf, Unit, '#F ',file
      printf, Unit, '#E',strcompress(long(systime(1)))
      printf, Unit, '#D ',systime(0)
   endif 
   printf, Unit, ''
   printf, Unit, '#S',strcompress(scan),' ',name
   printf, Unit, '#D ',systime(0)
   if keyword_set(comm) then printf, Unit, transpose('#C ' + comm)
   printf, Unit, format='("#N ", I2)' ,ncol
   printf, Unit, format= '("#L",'+string(ncol)+'("  ",A))', labels
   printf, Unit, format='('+string(ncol)+'(G0.6," "))', data(columns, *)
   free_lun, Unit

   return, scan
end
