;______________________________________________________________________________
;+
; NAME:
;       SPEC_POINTS()
;
; PURPOSE:
;       This function returns the number of data points of a particular scan
;       in a SPEC file.
;
; CATEGORY:
;       Input/Output.
;
; CALLING SEQUENCE:
;       Result = SPEC_POINTS(Handle, Scan_Id)
;
; INPUTS:
;       Handle   - Handle to the SPEC data file initialised by a previous call
;                  to SPEC_ACCESS().
;
;       Scan_Id  - Scan identifier. This parameter can be a numeric or string
;                  scalar value that accepts various syntax and keywords as
;                  it is explained in the `SCAN IDENTIFIERS' section at the
;                  beginning of this file). The simplest case is to use
;                  the SPEC scan number as Scan_Id.
;
; KEYWORDS:
;       INDEX:  Set this keyword to interpret a numeric value in Scan_Id
;               as the index of the scan instead of the scan number.
;
;       MCDATA: Set this keyword to make the function return the number of
;               points with multichannel data (MCA, ...) instead of the 
;               conventional data. 
;
; OUTPUT: 
;       The number of data points in the scan specified by Scan_Id.
;
; RESTRICTIONS:
;       This function requires a valid handle to a SPEC data file obtained by
;       a previous call to SPEC_ACCESS().
;       If Scan_Id does not represent a valid scan the function produces an
;       error.
;
; EXAMPLE:
;       To print the number of data points in the scan 999 from a file called
;       'sosada', enter:
;
;       Dummy = SPEC_ACCESS(File1_handle, 'sosada')
;       PRINT, SPEC_POINTS(File1_handle, 999)
;______________________________________________________________________________
;-

function spec_points, handle, scan_id, INDEX=idx, MCDATA=mcdata
   catch, error_status
   if error_status ne 0 then begin
      catch, /cancel
      on_error, 2
      message, __cleanmsg(!err_string);, /traceback
   endif

   ; Check arguments
   if N_PARAMS() ne 2 then message, 'Incorrect number of arguments.'

   __speccheck, handle
   __specload, handle, scan_id, errmsg, INDEX=idx
   if !ERR then message, errmsg

   if keyword_set(mcdata) then begin
      npoints = handle.scan(handle.currscan).a_datapoints
   endif else begin
      npoints = handle.scan(handle.currscan).n_datapoints
   endelse
   return, npoints 
end
