;______________________________________________________________________________
;+
; NAME:
;       SPEC_COLUMNS()
;
; PURPOSE:
;       This function returns the number of data columns of a particular scan
;       in a SPEC file.
;
; CATEGORY:
;       Input/Output.
;
; CALLING SEQUENCE:
;       Result = SPEC_COLUMNS(Handle, Scan_Id)
;
; INPUTS:
;       Handle   - Handle to the SPEC data file initialised by a previous call
;                  to SPEC_ACCESS().
;
;       Scan_Id  - Scan identifier. This parameter can be a numeric or string
;                  scalar value that accepts several syntax and keywords as
;                  it is explained in the `SCAN IDENTIFIERS' section at the
;                  beginning of this file). The simplest case is to use
;                  the SPEC scan number as Scan_Id.
;
; KEYWORDS:
;       INDEX:  Set this keyword to interpret a numeric value in Scan_Id
;               as the index of the scan instead of the scan number.
;
;       MCDATA: Set this keyword to return the number of channels in the
;               multichannel data lines instead of the number of standard
;               data columns.
;
; OUTPUT: 
;       The number of columns in the selected scan. If the MCDATA keyword is
;       not set the information is obtained from the `#N' line in the SPEC
;       data file and corresponds to the number of standard data columns.
;       When MCDATA is set, the returned value is the number of channels in
;       the multichannel data lines obtained from the `#@CHANN' line.
;
; RESTRICTIONS:
;       This function requires a valid handle to a SPEC data file obtained by
;       a previous call to SPEC_ACCESS().
;       If Scan_Id does not represent a valid scan the function produces an
;       error.
;
; EXAMPLE:
;       To print the number of data columns in the second occurrence of the
;       scan 150 in the file 'baddata', enter:
;
;       Dummy = SPEC_ACCESS(File1_handle, 'baddata')
;       PRINT, SPEC_COLUMNS(File1_handle, 150.02)
;______________________________________________________________________________
;-

function spec_columns, handle, scan_id, INDEX=idx, MCDATA=mcdata
   catch, error_status
   if error_status ne 0 then begin
      catch, /cancel
      on_error, 2
      message, __cleanmsg(!err_string) ;, /traceback
   endif

   ; Check arguments
   if N_PARAMS() ne 2 then message, 'Incorrect number of arguments.'
   __speccheck, handle
   __specload, handle, scan_id, errmsg, INDEX=idx
   if !ERR then message, errmsg

   if keyword_set(mcdata) then begin
      return, handle.scan(handle.currscan).n_chan
   endif else begin
      return, handle.scan(handle.currscan).n_col 
   endelse
end
