;+
; NAME:
;       FINTK53
;
; PURPOSE:
;       Calculates the function consisting of the integral, from x to infinity, 
;	of the Bessel function K5/3(x). 
;	G_ONE=fintk53*x is the universal curve from which the energy spectrum 
;	of the synchrotron bending magnet is calculated.
;
; CATEGORY:
;       Mathematics.
;
; CALLING SEQUENCE:
;       Result = fintk53(x)
;
; INPUTS:
;       x:      the argument of the function. All calculations are done in doble
;               precision.
;
; KEYWORD PARAMETERS:
;
; OUTPUTS:
;       returns the value  of the fintk53 function
;
; PROCEDURE:
;       Translated from a Fortran program, original from Umstatter. 
; 	C
; 	C Routines taken from 
; 	C
; 	C http://www.slac.stanford.edu/grp/arb/tn/arbvol2/ARDB162.pdf
; 	C The reference  1981 CERN/PS/SM/81-13  is cited in "Synchrotron Radiation Spectra" by G.Brown and W.
; 	C Lavender pp.37-61 Handbook on Synchrotron Radiation, vol. 3 edited by G.S. Brown and D.E. Moncton (Elsevier
; 	C Science Publishers B.V. 1991 North-Holland, Amsterdam) 
; 	C 
;
;       I have performed a comparison of the result with Mathematica
;       with very good agreement (note that Mathematica values diverge
;       for x> 20. I do not know why):
;       (Mathematica evaluation N[g_one[0.001],20])
;       (x = 620d0 & print,g_one(x),x*fintk53(x),format='(2G32.18)')
;
;     x      mathematica              idl (g_one)          idl (x*fintk53(x))
;   0.001 0.2131390650914501     0.213139063960598951 0.213139096577768417
;   0.01  0.4449725041142102     0.444972501836887357 0.444972550630643671
;   0.1   0.818185534872854      0.818185545849020146 0.818185588215680770
;   1.0   0.6514228153553639697  0.651422768295592536 0.651422821506926542
;   5.0   0.021248129774982      0.0212480218360103570 0.0212481300729910755
;   10.0  0.00019223826428       0.000192110311048920184 0.000192238266987909711
;   20.0  5.960464477539063E-7   1.22537071794711683E-08 1.19686346217633044E-08
;   50.0  6.881280000000002E7    2.42083894036140624E-21 1.73478522828932108E-21
;   100.0 4.642275147320176E29   9.30025460217969216E-43 4.69759373162073832E-43
;   1000.0 -1.7E424              Floating underflow (<620 OK)
;
;   Timing:
;   x=makearray1(100,0.001,100)
;   for i=1,10000 do a=g_one(x)  (23.99 sec)
;   for i=1,10000 do a=x*fintk53(x) (5.84 sec)
;
;
; MODIFICATION HISTORY:
;       Written by:     M. Sanchez del Rio, srio@esrf.fr, 2002-04-22
;
;-

;
;C
;C Computes Integral (from x to infinity) {K5/3(y) dy}
;C
function fintk53,xd
	a=1.0D0 
	b=1.0D0 
	p=1.0D0 
	q=1.0D0 
	x=1.0D0 
	y=1.0D0 
	z=1.0D0 

	xi = Where(xd GE 5.0D0, count1)
	fintk53=xd*0.0D0

	IF (count1 GT 0) THEN BEGIN
          x = Double(xd[xi])
	  z=20./x-2.
	  a= 0.0000000001d0
	  b=z*a - 0.0000000004d0
	  a=z*b-a + 0.0000000020d0
	  b=z*a-b - 0.0000000110d0
	  a=z*b-a + 0.0000000642d0
	  b=z*a-b - 0.0000004076d0
	  a=z*b-a + 0.0000028754d0
	  b=z*a-b - 0.0000232125d0
	  a=z*b-a + 0.0002250532d0
	  b=z*a-b - 0.0028763680d0
	  a=z*b-a + 0.0623959136d0
	  p=0.5*z*a-b + 1.0655239080d0
	  p=p*sqrt(1.5707963268/x)/exp(x)
	  fintk53[xi]=p
	ENDIF

	xi = Where(xd LT 5.0D0, count2)
	IF (count1+count2) NE N_Elements(xd) THEN Message,'Error: (count1+count2) NE N_Elements(xd)'
	IF (count2 GT 0) THEN BEGIN
          x = Double(xd[xi])
	  z=x^2/16.-2.
	  a= 0.0000000001d0
	  b=z*a + 0.0000000023d0
	  a=z*b-a + 0.0000000813d0
	  b=z*a-b + 0.0000024575d0
	  a=z*b-a + 0.0000618126d0
	  b=z*a-b + 0.0012706638d0
	  a=z*b-a + 0.0209121680d0
	  b=z*a-b + 0.2688034606d0
	  a=z*b-a + 2.6190218379d0
	  b=z*a-b + 18.6525089687d0
	  a=z*b-a + 92.9523266592d0
	  b=z*a-b + 308.1591941313d0
	  a=z*b-a + 644.8697965824d0
	  p=0.5*z*a-b + 414.5654364883d0
	  a= 0.0000000012d0
	  b=z*a + 0.0000000391d0
	  a=z*b-a + 0.0000011060d0
	  b=z*a-b + 0.0000258145d0
	  a=z*b-a + 0.0004876869d0
	  b=z*a-b + 0.0072845620d0
	  a=z*b-a + 0.0835793546d0
	  b=z*a-b + 0.7103136120d0
	  a=z*b-a + 4.2678026127d0
	  b=z*a-b + 17.0554078580d0
	  a=z*b-a + 41.8390348678d0
	  q=0.5d0*z*a-b + 28.4178737436d0
	  y=x^0.666666667
	  p=(p/y-q*y-1.)*1.8137993642
	  fintk53[xi]=p
	ENDIF
	RETURN,fintk53
END

; ;C
; ;C Computes x^(2/3)*fintk53(x)
; ;C
; ;
; function fintk531,zd
; a=1.0D0 
; b=1.0D0 
; p=1.0D0 
; q=1.0D0 
; x=1.0D0 
; y=1.0D0 
; z=1.0D0 

; 	x=double(zd)^3
; 	IF (x GE 5.d0) THEN BEGIN
; 	  z=20./x-2.
; 	  a= 0.0000000001d0
; 	  b=z*a - 0.0000000004d0
; 	  a=z*b-a + 0.0000000020d0
; 	  b=z*a-b - 0.0000000110d0
; 	  a=z*b-a + 0.0000000642d0
; 	  b=z*a-b - 0.0000004076d0
; 	  a=z*b-a + 0.0000028754d0
; 	  b=z*a-b - 0.0000232125d0
; 	  a=z*b-a + 0.0002250532d0
; 	  b=z*a-b - 0.0028763680d0
; 	  a=z*b-a + 0.0623959136d0
; 	  p=0.5*z*a-b + 1.0655239080d0
; 	  p=p*x^0.166666666*1.25331413731550/exp(x)
; 	  fintk531=p
; 	ENDIF ELSE BEGIN
; 	  z=x^2/16.-2.
; 	  a= 0.0000000001d0
; 	  b=z*a + 0.0000000023d0
; 	  a=z*b-a + 0.0000000813d0
; 	  b=z*a-b + 0.0000024575d0
; 	  a=z*b-a + 0.0000618126d0
; 	  b=z*a-b + 0.0012706638d0
; 	  a=z*b-a + 0.0209121680d0
; 	  b=z*a-b + 0.2688034606d0
; 	  a=z*b-a + 2.6190218379d0
; 	  b=z*a-b + 18.6525089687d0
; 	  a=z*b-a + 92.9523266592d0
; 	  b=z*a-b + 308.1591941313d0
; 	  a=z*b-a + 644.8697965824d0
; 	  p=0.5*z*a-b + 414.5654364883d0
; 	  a= 0.0000000012d0
; 	  b=z*a + 0.0000000391d0
; 	  a=z*b-a + 0.0000011060d0
; 	  b=z*a-b + 0.0000258145d0
; 	  a=z*b-a + 0.0004876869d0
; 	  b=z*a-b + 0.0072845620d0
; 	  a=z*b-a + 0.0835793546d0
; 	  b=z*a-b + 0.7103136120d0
; 	  a=z*b-a + 4.2678026127d0
; 	  b=z*a-b + 17.0554078580d0
; 	  a=z*b-a + 41.8390348678d0
; 	  q=0.5d0*z*a-b + 28.4178737436d0
; 	  y=x^0.666666667
; 	  p=(p-q*y^2-y)*1.8137993642
; 	  fintk531=p
; 	ENDELSE
; 	RETURN,fintk531
; end
;
; ;C The third routine ransf is a function of argument in the range [0,1], and 
; ;C returns, a value for x E c =g w /h, distributed according to the single-photon 
; ;C synchrotron distribution. It was obtained as a polynomial fit to the
; ;C integral provided by fintk531. 
; ;C
; function ransf,r
; p1=1.23158896184547d0
; p2=0.239365368240860d0
; a0=-8.7639738592d-4
; a1= 6.4290365191d-1
; a2=-2.282345395d0
; a3= 2.6466388166d1
; a4=-1.5645360744d2
; a5= 5.4167195057d2
; a6=-1.1173871087d3
; a7= 1.3591634604d3
; a8=-8.9941859853d2
; a9= 2.507411971d2
; b0= 7.3244951459d1
; b1=-8.4319130076d1
; b2=-8.7355315547d2
; b3= 2.7986791494d3
; b4=-3.4037250632d3
; b5= 1.8951442528d3
; b6=-4.0444403512d2
;
;
; 	r = double(r)
; 	IF (r le 0.805375529273446) THEN BEGIN
; 	  poly=a0+r*(a1+r*(a2+r*(a3+r*(a4+r*(a5+r*(a6+r*(a7+r*(a8+r*a9))))))))
; 	  ransf=(r/p1)^3*(1.+r*poly)
; 	ENDIF ELSE BEGIN
; 	  poly=b0+r*(b1+r*(b2+r*(b3+r*(b4+r*(b5+r*b6)))))
; 	  x=-log((1.-r)/p2)
; 	  x=x-0.5*log(x)
; 	  ransf=x*poly
; 	ENDELSE
; RETURN,ransf
; end
;
;
